/*
        dates.cpp -- A date object.

        The original code for this object is from the June 1990 issue of
        The C Users Journal and was written by David Clark.

        Several improvements and modifications were added by Bill
        Pemberton.  The current code is written for Turbo C++.
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <dos.h>
#include <string.h>
#include <iostream.h>
#include "dates.h"

static char ShortMonths[MAXMONTH][4] =
{
"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};

static char LongMonths[MAXMONTH][10] =
{
    "January", "February", "March", "April", "May", "June",
    "July", "August", "September", "October", "November", "December"
};

static unsigned MonthDays[MAXMONTH] =
{
    31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31
};

static char ShortWeekDays[MAXWEEKDAY+1][4] =
{
    "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"
};

static char LongWeekDays[MAXWEEKDAY+1][10] =
{
    "Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday"
};

static const char *CurrentDateFormat = "m-dd-yyy";
static int OnHeap = 0;



/*
    ChangeDefaultDateFormat -- Change the format string used to initialize
    the DateFormatPtr
*/

int ChangeDefaultDateFormat(const char *s)
{
    char *t;

    if (( t = strdup(s)) == NULL)
        return(0);
    if (OnHeap)
        free(CurrentDateFormat);
    else
        OnHeap = 1;
    CurrentDateFormat = t;
    return(1);
}

static int IsLeap(int Year)
{
    return ( (Year % 4 == 0) && (Year % 4000 != 0) &&
             ((Year % 100 != 0) || (Year % 400 == 0)) );
}

static unsigned char DaysInMonth(unsigned char Month, int Year)
{
    if ( (Month == 2) && IsLeap(Year))
        return(29);
    else
        return(MonthDays[Month-1]);
}

static int CheckForValidDate(unsigned char Day, unsigned char Month, int Year)
{
    if ((Year < MINYEAR) || (Year > MAXYEAR) || (Year == 0) || (Month < MINMONTH) ||
        (Month > MAXMONTH))
        return(0);
    return(Day <= DaysInMonth(Month,Year));
}

static unsigned char JulianToDayOfWeek(long Jul)
{
    return((Jul + 1) % 7);
}

/*
    DMYtoJulian ---- Validate the date and then convert it to Julian
*/

#define IGREG (15+31L*(10+12L*1582))

static long DMYtoJulian(unsigned char Day, unsigned char Month, int Year)
{
	long Ja, Jm, Jy, Jul;

    if (!CheckForValidDate(Day, Month, Year))
        return(BADDATE);

	if (Year < 0)
		Year++;

    if (Month > 2)
    {
		Jy = (long) Year;
        Jm = Month + 1;
    }
    else
    {
		Jy = (long) Year - 1;
        Jm = Month + 13;
    }

    Jul = (long) (floor(365.25*Jy) + floor(30.6001*Jm) + Day + 1720995);

    if (Day + 31L*(Month + 12L*Year) >= IGREG)
    {
        Ja = 0.01 * Jy;
        Jul += 2 - Ja + (int)(0.25*Ja);
    }
    return(Jul);
}

/*
    JulianToDMY  -- Converts a Julian to Date Month Year
*/

#define GREGOR 2299161

static void JulianToDMY(long Jul, unsigned char &Day, unsigned char &Month, int &Year)
{
    long Ja, JAlpha, Jb, Jc, Jd, Je;

    if ((Jul != BADDATE) && (Jul >= MINDATE) && (Jul <= MAXDATE))
    {
        if (Jul >= GREGOR)
        {
            JAlpha = ((double) (Jul - 1867216) - 0.25)/36524.25;
            Ja = Jul + 1 + JAlpha - (long) (0.25 * JAlpha);
        }
        else
            Ja = Jul;

        Jb = Ja + 1524;
        Jc = 6680.0 + ((double) (Jb - 2439870) - 122.1)/365.25;
        Jd = 365*Jc + (0.25*Jc);
        Je = (Jb - Jd)/30.6001;
        Day = Jb - Jd - (int)(30.6001*Je);
        Month = Je - 1;

        if (Month > 12)
            Month -= 12;

        Year = Jc - 4715;

        if (Month > 2)
            --Year;

        if (Year <= 0)
			--Year;
    }
}

/*
    ChangeDate -- Change a date to reflect the new date passed in the args.
    If the date is legal, return 1, else return 0.
*/

int DateObject::ChangeDate(unsigned char NMonth, unsigned char NDay, int NYear)
{
    long t;

    t = DMYtoJulian(NDay, NMonth, NYear);
	if ( t != BADDATE)
    {
        Julian = t;
        Day = NDay;
        Month = NMonth;
        Year = NYear;
        DayOfWeek = JulianToDayOfWeek(Julian);
        return(1);
    }
    return(0);
}

/*
    DateObject -- Constructor!  If no args, init to today's date.
*/

DateObject::DateObject(void)
{
    struct date d;

    getdate(&d);
    Day = d.da_day;
    Month = d.da_mon;
    Year = d.da_year;
    Julian = DMYtoJulian(Day, Month, Year);
    DayOfWeek = JulianToDayOfWeek(Julian);
    DateFormatPtr = strdup(CurrentDateFormat);
    if (DateFormatPtr == NULL)
        Julian = BADDATE;
}

DateObject::DateObject(DateObject &OtherDate)
{
    Day = OtherDate.Day;
    Month = OtherDate.Month;
    Year = OtherDate.Year;
    DayOfWeek = OtherDate.DayOfWeek;
    Julian = OtherDate.Julian;
    DateFormatPtr = strdup(OtherDate.DateFormatPtr);
    if (DateFormatPtr == NULL)
        Julian = BADDATE;
}

DateObject::DateObject(unsigned char InitMonth, unsigned char InitDay, int InitYear)
{
    ChangeDate(InitMonth, InitDay, InitYear);
    if (Julian != BADDATE)
    {
        DateFormatPtr = strdup(CurrentDateFormat);
        if (DateFormatPtr == NULL)
            Julian = BADDATE;
    }
}

DateObject::DateObject(unsigned char InitMonth, unsigned char InitDay,
     int InitYear, const char *FormatStr)
{
    ChangeDate(InitMonth, InitDay, InitYear);
    if (Julian != BADDATE)
    {
        DateFormatPtr = strdup(FormatStr);
        if (DateFormatPtr == NULL)
            Julian = BADDATE;
    }
}

/*
    operator =
*/

DateObject DateObject::operator = (DateObject &d)
{
	Day = d.Day;
    Month = d.Month;
    Year = d.Year;
    Julian = d.Julian;
    DayOfWeek = d.DayOfWeek;
    DateFormatPtr = strdup(d.DateFormatPtr);
    if (DateFormatPtr == NULL)
        Julian = BADDATE;
    return (*this);
}

/*
    operator +  This is the fundamental operator. All other arithmetic operators
    call this function.
*/

DateObject operator + (DateObject &d, long x)
{
    DateObject sum;

    sum.Julian = d.Julian + x;
    if ( (sum.Julian < MINDATE) || (sum.Julian > MAXDATE) )
        sum.Julian = BADDATE;
    else
    {
        JulianToDMY(sum.Julian, sum.Day, sum.Month, sum.Year);
        sum.DayOfWeek = JulianToDayOfWeek(sum.Julian);
        sum.DateFormatPtr = strdup(d.DateFormatPtr);
        if (sum.DateFormatPtr == NULL)
            sum.Julian = BADDATE;
    }
    return(sum);
}

DateObject operator + (long x, DateObject &d)
{
    return(d + x);
}

DateObject DateObject::operator - (long x)
{
    return(*this + (-x));
}

DateObject DateObject::operator ++ (void)
{
    return((*this) = (*this) + 1L);
}

DateObject DateObject::operator -- (void)
{
    return((*this) = (*this) + (-1L));
}

DateObject DateObject::operator += (long x)
{
    return((*this) = (*this) + x);
}

DateObject DateObject::operator -= (long x)
{
    return((*this) = (*this) + (-x));
}

/*
    GetFormat -- return a dynamically allocated copy of the format string
*/

const char * DateObject::GetFormat()
{
    return(strdup(DateFormatPtr));
}

void DateObject::ChangeFormat(const char *s)
{
    char *t;

    if ( (t=strdup(s)) != NULL)
    {
        free(DateFormatPtr);
        DateFormatPtr = t;
    }
}

/*
    Destructor -- releases the space held be *DateFormatPtr
*/

DateObject::~DateObject(void)
{
    free(DateFormatPtr);
}

char *DateObject::GetShortDayOfWeek(void)
{
	return(strdup(ShortWeekDays[DayOfWeek]));
}

char *DateObject::GetLongDayOfWeek(void)
{
	return(strdup(LongWeekDays[DayOfWeek]));
}



char *DateObject::GetShortMonth(void)
{
	return(strdup(ShortMonths[Month-1]));
}

char *DateObject::GetLongMonth(void)
{
	return(strdup(LongMonths[Month-1]));
}



/*
    CountLetters -- count the run of letters in s matching the first letter
                    in s.  This is a helper for DateToString.
*/

static int CountLetters(char **s)
{
	int n,c;

    for (n=0, c= **s; **s == c; n++,(*s)++);

    return(n);
}

/*
    DateToString -- convert Self to a printable string based on the contents of
        DateFormatPtr.

        The format string works by replacement:

        d       Replaced by a one or two digit day
        dd      Replaced by a two digit day (with leading 0)
        ddd     Replaced by a three char name for the day ie "Wed"
        dddd    Replaced by the complete word of the day
        m       One or two digit month
        mm      Two digit month (with leading 0)
        mmm     Three char name of month
        mmmm    Full name of month
        yy      Two digit year
        yyy     Four digit year, with - used for BC
        yyyy    Full year with either AD or BC appended
        \       Places next char in the string
*/

char *DateObject::DateToString(void)
{
	int Count;
	char    *str,   // A work string
			*sptr,  // A moving pointer
			*ret,   // pointer to the value to be returned
			*fptr;  // pointer into format string

    if (Julian == BADDATE)
        return(strdup("Bad Date"));

	sptr = str = (char *) calloc((MAXDATESTRLEN+1), sizeof(char));
    fptr = DateFormatPtr;
    while (*fptr)
    {
        switch (*fptr)
        {
            case 'd':
            case 'D':
                Count = CountLetters(&fptr);
                if (Count >= 4)
                {
					strcat(sptr, GetLongDayOfWeek());
					while (*sptr)
						++sptr;
				}
				else if (Count == 3)
				{
					strcat(sptr, GetShortDayOfWeek());
					while (*sptr)
						++sptr;
				}
				else
				{
					if ( (Count == 2) && (Day < 10) )
						strcat(sptr++, "0");
					itoa(Day, sptr, 10);
					while (*sptr)
						++sptr;
				}
				break;
			case 'm':
			case 'M':
				Count = CountLetters(&fptr);
				if (Count >= 4)
				{
					strcat(sptr, GetLongMonth());
					while (*sptr)
						++sptr;
				}
				else if (Count == 3)
				{
					strcat(sptr, GetShortMonth());
                    while (*sptr)
                        ++sptr;
                }
                else
                {
                    if ( (Count == 2) && (Month < 10) )
                        strcat(sptr++, "0");
                    itoa(Month, sptr, 10);
                    while (*sptr)
                        ++sptr;
                }
                break;
            case 'y':
            case 'Y':
                Count = CountLetters(&fptr);
                if (Count >= 4)
                {
                    itoa(Year, sptr, 10);
                    if (Year < 0)
                        strcpy(sptr, sptr + 1);
                    while (*sptr)
                        ++sptr;
                    if (Year > 0 )
                        strcat(sptr, " AD");
                    else
                        strcat(sptr, " BC");
                    while (*sptr)
                        ++sptr;
                }
                else
                {
                    if (Count == 2)
                    {
                        if ( (Year % 100) < 10)
                            strcat(sptr++,"0");
                        itoa((Year % 100), sptr, 10);
                        while (*sptr)
                        ++sptr;
                    }
                    else
                    {
                        itoa(Year, sptr, 10);
                        while (*sptr)
                        ++sptr;
                    }
                }
                break;
            case '\\': fptr++;
            default:
                *sptr = *fptr;
                ++sptr;
                ++fptr;
				break;
		}
    }
    ret = strdup(str);
    free(str);
    return(ret);
}

ostream& operator << (ostream& s, DateObject &d)
{
	s << d.DateToString();
	return s;
}





