        Name CRCUtils
        Title CRC Utilities
        Page 65,132
        DOSSEG
        .MODEL  SMALL
        .CODE
;************************************************************************
;
; TurboC callable CRC utilities for general 16 and 32 bit polynomials
;
; 1993 Gary Mussar
; This code is released to the public domain.
;
; CRC calculations are performed one byte at a time using a table lookup
; mechanism.  Two routines are provided: one to initialize the CRC table;
; and one to perform the CRC calculation over an array of bytes.
;
; A CRC is the remainder produced by dividing a generator polynomial into
; a data polynomial using binary arthimetic (XORs). The data polynomial
; is formed by using each bit of the data as a coefficient of a term in
; the polynomial. These utilities assume the data communications ordering
; of bits for the data polynomial, ie. the LSB of the first byte of data
; is the coefficient of the highest term of the polynomial, etc..
;
Page
;
; TurboC-callable small-model procedure that initializes the table to be
; used in the following CRC calculations.
;
; Prototype:
;   extern void I_CRC16(CRC16 Table[256],
;                       CRC16 *GenPolynomial);
;
; Input:
;   CRC16 Table[256]    :  pointer to CRC table (256x16 bits)
;   CRC16 *GenPolynomial:  pointer to generator polynomial (16 bits)
        PUBLIC  _I_CRC16
_I_CRC16     PROC
        push    bp                 ;save previous stack frame ptr
        mov     bp,sp              ;make a new stack frame
        push    si                 ;save possible register vars
        push    di
;
        mov     si,[bp+4]          ;pointer to CRC table
        mov     bx,[bp+6]          ;pointer generator polynomial
        mov     bx,[bx]            ;generator polynomial
;
; Loop through all chars from 00-FF
;
        xor     cx,cx
ICRC16_Loop:
        mov     ax,cx              ;get value of next char
;
        shr     ax,1               ;check if MSB of data poly set
        jnc     ICRC16_End0        ;If not, check next bit
        xor     ax,bx              ;else subtract gen polynomial
ICRC16_End0:
        shr     ax,1               ;check if MSB of data poly set
        jnc     ICRC16_End1        ;If not, check next bit
        xor     ax,bx              ;else subtract gen polynomial
ICRC16_End1:
        shr     ax,1               ;check if MSB of data poly set
        jnc     ICRC16_End2        ;If not, check next bit
        xor     ax,bx              ;else subtract gen polynomial
ICRC16_End2:
        shr     ax,1               ;check if MSB of data poly set
        jnc     ICRC16_End3        ;If not, check next bit
        xor     ax,bx              ;else subtract gen polynomial
ICRC16_End3:
        shr     ax,1               ;check if MSB of data poly set
        jnc     ICRC16_End4        ;If not, check next bit
        xor     ax,bx              ;else subtract gen polynomial
ICRC16_End4:
        shr     ax,1               ;check if MSB of data poly set
        jnc     ICRC16_End5        ;If not, check next bit
        xor     ax,bx              ;else subtract gen polynomial
ICRC16_End5:
        shr     ax,1               ;check if MSB of data poly set
        jnc     ICRC16_End6        ;If not, check next bit
        xor     ax,bx              ;else subtract gen polynomial
ICRC16_End6:
        shr     ax,1               ;check if MSB of data poly set
        jnc     ICRC16_End7        ;If not, check next bit
        xor     ax,bx              ;else subtract gen polynomial
ICRC16_End7:
;
        mov     word ptr [si],ax   ;save this result in table
        add     si,2               ;point to next table entry
        inc     cl                 ;move on to next char
        jnz     ICRC16_Loop        ;Loop if not back to 00
;
        pop     di                 ;restore possible register vars
        pop     si
        mov     sp,bp              ;discard locals
        pop     bp                 ;restore previous stack frame
        ret
_I_CRC16     ENDP
Page
;
; TurboC-callable small-model procedure to perform a CRC calculation over
; an array of bytes using a pre-generated lookup table. The number of bytes
; in the data array is limited to 64K (size of count), however, this value
; may actually be smaller since the entire data array must be kept in
; one segment (small model, nothing fancy). Large blocks of data can be
; checked by breaking the data into smaller pieces and calling this
; routine multiple times. The result of the calculation on previous
; pieces is passed into this routine as the running CRC value. It is
; standard practice to initialize the running CRC to FFFF on the first
; call to this routine (this is done to catch certain classes of errors).
;
; Prototype:
;   extern void F_CRC16(CRC16 Table[256],
;                       CRC16 *CRC,
;                       unsigned char *dataptr,
;                       unsigned int count);
;
; Input:
;   CRC16 Table[256]            :  pointer to CRC table (256x16 bits)
;   CRC16 *CRC                  :  pointer to running CRC result (16 bits)
;   unsigned char *dataptr      :  pointer to array of byte to check
;   unsigned int count          :  number of bytes in array
;
; Note:
;   Algorithm is most efficient if the CRC table and the data array are
;   on word boundaries. In addition,
;
        PUBLIC  _F_CRC16
_F_CRC16     PROC
        push    bp                 ;save previous stack frame ptr
        mov     bp,sp              ;make a new stack frame
        push    si                 ;save possible register vars
        push    di
;
        mov     si,[bp+4]          ;pointer to CRC table
        mov     di,[bp+8]          ;pointer to array of data bytes
        mov     bx,[bp+6]          ;get result of previous CRC calculation
        mov     ax,word ptr [bx]
        mov     cx,word ptr [bp+10];calculate number of full words to do
        shr     cx,1
        jcxz    CRC16_notfull
;
; Loop through all full words
;
CRC16_FullLoop:
        xor     ax,word ptr [di]   ;modify with data bytes
        add     di,2               ;adjust data pointer to next bytes
;
; We want bx to become an index into the CRC table and we need to do a
; SHR CRC,8 as well.
;
        xor     bx,bx              ;zero all of bx
        xchg    bl,al              ;bx contains low byte of CRC
        xchg    ah,al              ;and finish rest of shr CRC,8
        shl     bx,1               ;CRC table is made of words
        xor     ax,word ptr [si][bx] ;modify by table value
;
; Now we do it all again.
;
        xor     bx,bx              ;zero all of bx
        xchg    bl,al              ;bx contains low byte of CRC
        xchg    ah,al              ;and finish rest of shr CRC,8
        shl     bx,1               ;CRC table is made of words
        xor     ax,word ptr [si][bx] ;modify by table value
;
        loop    CRC16_FullLoop     ;handle next full word
;
; Done all full words of data. Now handle any remaining data
CRC16_notfull:
        mov     cx,word ptr [bp+10];total number of bytes to do
        and     cx,1               ;number of odd bytes to do
        jcxz    CRC16_Done
;
        xor     al,byte ptr [di]   ;modify with data byte
        xor     bx,bx              ;zero all of bx
        xchg    bl,al              ;bx contains low byte of CRC
        xchg    ah,al              ;and finish rest of shr CRC,8
        shl     bx,1               ;CRC table is made of words
        xor     ax,word ptr [si][bx] ;modify by table value
;
CRC16_Done:
        mov     bx,[bp+6]          ;save result from this calculation
        mov     word ptr [bx],ax
;
        pop     di                 ;restore possible register vars
        pop     si
        mov     sp,bp              ;discard locals
        pop     bp                 ;restore previous stack frame
        ret
_F_CRC16     ENDP
Page
;
; TurboC-callable small-model procedure to perform a CRC calculation over
; an array of bytes without a lookup table (slower). The number of bytes
; in the data array is limited to 64K (size of count), however, this value
; may actually be smaller since the entire data array must be kept in
; one segment (small model, nothing fancy). Large blocks of data can be
; checked by breaking the data into smaller pieces and calling this
; routine multiple times. The result of the calculation on previous
; pieces is passed into this routine as the running CRC value. It is
; standard practice to initialize the running CRC to FFFF on the first
; call to this routine (this is done to catch certain classes of errors).
;
; Prototype:
;   extern void S_CRC16(CRC16 *GenPolynomial,
;                       CRC16 *CRC,
;                       unsigned char *dataptr,
;                       unsigned int count);
;
; Input:
;   CRC16 *GenPolynomial        :  pointer to generator Polynomial (16 bits)
;   CRC16 *CRC                  :  pointer to running CRC result (16 bits)
;   unsigned char *dataptr      :  pointer to array of byte to check
;   unsigned int count          :  number of bytes in array
;
;
        PUBLIC  _S_CRC16
_S_CRC16     PROC
        push    bp                 ;save previous stack frame ptr
        mov     bp,sp              ;make a new stack frame
        push    si                 ;save possible register vars
;
        mov     si,[bp+8]          ;pointer to array of data bytes
        mov     bx,[bp+6]          ;get result of previous CRC calculation
        mov     ax,word ptr [bx]
        mov     bx,[bp+4]          ;pointer to generator polynomial
        mov     bx,[bx]            ;generator polynomial
        test    word ptr [bp+10],0ffffh ;any bytes to do?
        jz      SCRC16_Done
;
; Loop through all bytes
;
        xor     cx,cx
SCRC16_Loop:
        xor     al,byte ptr [si]   ;modify with data bytes
        inc     si                 ;adjust data pointer to next bytes
        mov     cl,8               ;run 8 bits through CRC
;
SCRC16_Bits:
        shr     ax,1               ;check if MSB set
        jnc     SCRC16_EndBit      ;If not, check next bit
        xor     ax,bx              ;else subtract gen polynomial
SCRC16_EndBit:
        loop    SCRC16_Bits
;
        dec     word ptr [bp+10]   ;finished a byte
        jnz     SCRC16_Loop        ;if more, then handle them
;
SCRC16_Done:
        mov     bx,[bp+6]          ;save result from this calculation
        mov     word ptr [bx],ax
;
        pop     si                 ;restore possible register vars
        mov     sp,bp              ;discard locals
        pop     bp                 ;restore previous stack frame
        ret
_S_CRC16     ENDP
Page
;
; TurboC-callable small-model procedure that initializes the table to be
; used in the following CRC calculations.
;
; Prototype:
;   extern void I_CRC32(CRC32 Table[256],
;                       CRC32 *GenPolynomial);
;
; Input:
;   CRC32 Table[256]    :  pointer to CRC table (256x32 bits)
;   CRC32 *GenPolynomial:  pointer to generator polynomial (32 bits)
        PUBLIC  _I_CRC32
_I_CRC32     PROC
        push    bp                 ;save previous stack frame ptr
        mov     bp,sp              ;make a new stack frame
        push    si                 ;save possible register vars
        push    di
;
        mov     si,[bp+4]          ;pointer to CRC table
        mov     bx,[bp+6]          ;pointer to generator polynomial
        mov     di,[bx+2]          ;generator polynomial high 16 bits
        mov     bx,[bx]            ;generator polynomial low 16 bits
;
; Loop through all chars from 00-FF
;
        xor     cx,cx
ICRC32_Loop:
        mov     ax,cx              ;get value of next char
        xor     dx,dx
;
        shr     dx,1               ;check of MSB of data poly set
        rcr     ax,1
        jnc     ICRC32_End0        ;If not, check next bit
        xor     ax,bx              ;else subtract gen polynomial
        xor     dx,di
ICRC32_End0:
        shr     dx,1               ;check of MSB of data poly set
        rcr     ax,1
        jnc     ICRC32_End1        ;If not, check next bit
        xor     ax,bx              ;else subtract gen polynomial
        xor     dx,di
ICRC32_End1:
        shr     dx,1               ;check of MSB of data poly set
        rcr     ax,1
        jnc     ICRC32_End2        ;If not, check next bit
        xor     ax,bx              ;else subtract gen polynomial
        xor     dx,di
ICRC32_End2:
        shr     dx,1               ;check of MSB of data poly set
        rcr     ax,1
        jnc     ICRC32_End3        ;If not, check next bit
        xor     ax,bx              ;else subtract gen polynomial
        xor     dx,di
ICRC32_End3:
        shr     dx,1               ;check of MSB of data poly set
        rcr     ax,1
        jnc     ICRC32_End4        ;If not, check next bit
        xor     ax,bx              ;else subtract gen polynomial
        xor     dx,di
ICRC32_End4:
        shr     dx,1               ;check of MSB of data poly set
        rcr     ax,1
        jnc     ICRC32_End5        ;If not, check next bit
        xor     ax,bx              ;else subtract gen polynomial
        xor     dx,di
ICRC32_End5:
        shr     dx,1               ;check of MSB of data poly set
        rcr     ax,1
        jnc     ICRC32_End6        ;If not, check next bit
        xor     ax,bx              ;else subtract gen polynomial
        xor     dx,di
ICRC32_End6:
        shr     dx,1               ;check of MSB of data poly set
        rcr     ax,1
        jnc     ICRC32_End7        ;If not, check next bit
        xor     ax,bx              ;else subtract gen polynomial
        xor     dx,di
ICRC32_End7:
;
        mov     word ptr [si],ax   ;save this result in table
        mov     word ptr [si+2],dx
        add     si,4               ;point to next table entry
        inc     cl                 ;move on to next char
        jnz     ICRC32_Loop        ;Loop if not back to 00
;
        pop     di                 ;restore possible register vars
        pop     si
        mov     sp,bp              ;discard locals
        pop     bp                 ;restore previous stack frame
        ret
_I_CRC32     ENDP
Page
;
; TurboC-callable small-model procedure to perform a CRC calculation over
; an array of bytes using a pre-generated lookup table. The number of bytes
; in the data array is limited to 64K (size of count), however, this value
; may actually be smaller since the entire data array must be kept in
; one segment (small model, nothing fancy). Large blocks of data can be
; checked by breaking the data into smaller pieces and calling this
; routine multiple times. The result of the calculation on previous
; pieces is passed into this routine as the running CRC value. It is
; standard practice to initialize the running CRC to FFFFFFFF on the first
; call to this routine (this is done to catch certain classes of errors).
;
; Prototype:
;   extern void F_CRC32(CRC32 Table[256],
;                       CRC32 *CRC,
;                       unsigned char *dataptr,
;                       unsigned int count);
;
; Input:
;   CRC32 Table[256]            :  pointer to CRC table (256x32 bits)
;   CRC32 *CRC                  :  pointer to running CRC result (32 bits)
;   unsigned char *dataptr      :  pointer to array of byte to check
;   unsigned int  count         :  number of bytes in array
;
; Note:
;   Algorithm is most efficient if the CRC table and the data array are
;   on word boundaries. In addition,
;
        PUBLIC  _F_CRC32
_F_CRC32     PROC
        push    bp                 ;save previous stack frame ptr
        mov     bp,sp              ;make a new stack frame
        push    si                 ;save possible register vars
        push    di
;
        mov     si,[bp+4]          ;pointer to CRC table
        mov     di,[bp+8]          ;pointer to array of data bytes
        mov     bx,[bp+6]          ;get result of previous CRC calculation
        mov     ax,word ptr [bx]
        mov     dx,word ptr [bx+2]
        mov     cx,word ptr [bp+10];calculate number of full words to do
        shr     cx,1
        shr     cx,1
        jcxz    CRC32_notfull
;
; Loop through all full words
;
CRC32_FullLoop:
        xor     ax,word ptr [di]   ;modify with data bytes
        xor     dx,word ptr [di+2]
        add     di,4               ;adjust data pointer to next bytes
;
; We want bx to become an index into the CRC table and we need to do a
; SHR CRC,8 as well.
;
        xor     bx,bx              ;zero all of bx
        xchg    bl,al              ;bx contains low byte of CRC
        xchg    ah,al              ;and finish rest of shr CRC,8
        xchg    dl,ah
        xchg    dh,dl
        shl     bx,1               ;CRC table is made of long words
        shl     bx,1
        xor     ax,word ptr [si][bx] ;modify by table value
        xor     dx,word ptr [si][bx+2]
;
; Now we do it all again.
;
        xor     bx,bx              ;zero all of bx
        xchg    bl,al              ;bx contains low byte of CRC
        xchg    ah,al              ;and finish rest of shr CRC,8
        xchg    dl,ah
        xchg    dh,dl
        shl     bx,1               ;CRC table is made of long words
        shl     bx,1
        xor     ax,word ptr [si][bx] ;modify by table value
        xor     dx,word ptr [si][bx+2]
;
; Now we do it all again.
;
        xor     bx,bx              ;zero all of bx
        xchg    bl,al              ;bx contains low byte of CRC
        xchg    ah,al              ;and finish rest of shr CRC,8
        xchg    dl,ah
        xchg    dh,dl
        shl     bx,1               ;CRC table is made of long words
        shl     bx,1
        xor     ax,word ptr [si][bx] ;modify by table value
        xor     dx,word ptr [si][bx+2]
;
; Now we do it all again.
;
        xor     bx,bx              ;zero all of bx
        xchg    bl,al              ;bx contains low byte of CRC
        xchg    ah,al              ;and finish rest of shr CRC,8
        xchg    dl,ah
        xchg    dh,dl
        shl     bx,1               ;CRC table is made of long words
        shl     bx,1
        xor     ax,word ptr [si][bx] ;modify by table value
        xor     dx,word ptr [si][bx+2]
;
        loop    CRC32_FullLoop     ;handle next full word
;
; Done all full words of data. Now handle any remaining data
CRC32_notfull:
        mov     cx,word ptr [bp+10];total number of bytes to do
        and     cx,3               ;number of odd bytes to do
        jcxz    CRC32_Done
;
CRC32_OddLoop:
        xor     al,byte ptr [di]   ;modify with data byte
        inc     di
        xor     bx,bx              ;zero all of bx
        xchg    bl,al              ;bx contains low byte of CRC
        xchg    ah,al              ;and finish rest of shr CRC,8
        xchg    dl,ah
        xchg    dh,dl
        shl     bx,1               ;CRC table is made of long words
        shl     bx,1
        xor     ax,word ptr [si][bx] ;modify by table value
        xor     dx,word ptr [si][bx+2]
        loop    CRC32_OddLoop
;
CRC32_Done:
        mov     bx,[bp+6]          ;save result from this calculation
        mov     word ptr [bx],ax
        mov     word ptr [bx+2],dx
;
        pop     di                 ;restore possible register vars
        pop     si
        mov     sp,bp              ;discard locals
        pop     bp                 ;restore previous stack frame
        ret
_F_CRC32     ENDP
Page
;
; TurboC-callable small-model procedure to perform a CRC calculation over
; an array of bytes without a lookup table (slower). The number of bytes
; in the data array is limited to 64K (size of count), however, this value
; may actually be smaller since the entire data array must be kept in
; one segment (small model, nothing fancy). Large blocks of data can be
; checked by breaking the data into smaller pieces and calling this
; routine multiple times. The result of the calculation on previous
; pieces is passed into this routine as the running CRC value. It is
; standard practice to initialize the running CRC to FFFF on the first
; call to this routine (this is done to catch certain classes of errors).
;
; Prototype:
;   extern void S_CRC32(CRC32 *GenPolynomial,
;                       CRC32 *CRC,
;                       unsigned char *dataptr,
;                       unsigned int count);
;
; Input:
;   CRC32 *GenPolynomial        :  pointer to generator Polynomial (32 bits)
;   CRC32 *CRC                  :  pointer to running CRC result (32 bits)
;   unsigned char *dataptr      :  pointer to array of byte to check
;   unsigned int count          :  number of bytes in array
;
;
        PUBLIC  _S_CRC32
_S_CRC32     PROC
        push    bp                 ;save previous stack frame ptr
        mov     bp,sp              ;make a new stack frame
        push    si                 ;save possible register vars
        push    di
;
        mov     si,[bp+8]          ;pointer to array of data bytes
        mov     bx,[bp+6]          ;get result of previous CRC calculation
        mov     ax,word ptr [bx]   ;low 16 bits
        mov     dx,word ptr [bx+2] ;high 16 bits
        mov     bx,[bp+4]          ;pointer to generator polynomial
        mov     di,[bx+2]          ;generator polynomial high 16 bits
        mov     bx,[bx]            ;generator polynomial low 16 bits
        test    word ptr [bp+10],0ffffh ;any bytes to do?
        jz      SCRC32_Done
;
; Loop through all bytes
;
        xor     cx,cx
SCRC32_Loop:
        xor     al,byte ptr [si]   ;modify with data bytes
        inc     si                 ;adjust data pointer to next bytes
        mov     cl,8               ;run 8 bits through CRC
;
SCRC32_Bits:
        shr     dx,1               ;check if MSB set
        rcr     ax,1
        jnc     SCRC32_EndBit      ;If not, check next bit
        xor     ax,bx              ;else subtract gen polynomial
        xor     dx,di
SCRC32_EndBit:
        loop    SCRC32_Bits
;
        dec     word ptr [bp+10]   ;finished a byte
        jnz     SCRC32_Loop        ;if more, then handle them
;
SCRC32_Done:
        mov     bx,[bp+6]          ;save result from this calculation
        mov     word ptr [bx],ax
        mov     word ptr [bx+2],dx
;
        pop     di                 ;restore possible register vars
        pop     si
        mov     sp,bp              ;discard locals
        pop     bp                 ;restore previous stack frame
        ret
_S_CRC32     ENDP
        END
