/***********************************************************************
*
*  CRCCHK - A simple program to check the correct functioning of
*           the CRC utility routines. This program calculates the CRC
*           of the specified file. CRCs are calculated least significant
*           bit first using both the slow, memory efficient algorithm
*           and the faster table lookup algorithm.
*
*  Arguments: File Name   - The name of the file to be checked.
*
***********************************************************************/
#include <stdio.h>
#include "crcutil.h"

/* The coefficient of each term of the generator polynomial is represented
   by a bit in the polynomial array (except for the highest order term
   which is always assumed to be 1). The coefficient for the second
   highest term in the LSB of the first byte of the polynomial array. */

/***********************************************************************
*  CCITT polynomial (Sec 2.2.7 Rec. X.25 1984):
*    Polynomial   x^16 + x^12 + x^5 + 1
*    Generator is pre-initialized to 0xffff
*    CRC is inverted before transmission on link.
*  CCITT examples from Appendix 1, Rec. X.25:
*    Note: bytes are MSB to LSB and are transmitted LSB
*    first (FCSs do not include 0 bits inserted for
*    transparency).
*    eg. byte 03 would be transmitted as 11000000
*                                        Time ->
*
*    Data    FCS
*
*    03 3f   5b ec
*    01 73   83 57
*    01 3f   eb df
*    03 73   33 64
***********************************************************************/
CRC16 ccitt_tbl[256];
CRC16 poly_ccitt = {0x8408};
CRC16 crc_ccitt = {0xffff};
CRC16 Scrc_ccitt = {0xffff};

                                              
/***********************************************************************
*  CRC16 (Bisync)
*    Polynomial   x^16 + x^15 + x^2 + 1
*    Generator is pre-initialized to 0x0000
*    CRC is transmitted as is on link.
***********************************************************************/
CRC16 crcbsc_tbl[256];
CRC16 poly_crcbsc = {0xa001};
CRC16 crc_crcbsc = {0x0000};
CRC16 Scrc_crcbsc = {0x0000};


/***********************************************************************
*  Ethernet CRC
*    Polynomial   x^32 + x^26 + x^23 + x^22 + x^16 + x^12 + x^11 + x^10 +
*                       x^8 + x^7 + x^5 + x^4 + x^2 + x^1 + 1
*    Generator is pre-initialized to 0xFFFFFFFF
*    CRC is inverted before transmission on link. ???
*
*  I have no data to verify that the generation of the Ethernet CRC is
*  correct (yet). Some day I will get a line trace with real data to
*  verify this program.
***********************************************************************/
CRC32 ether32_tbl[256];
CRC32 poly_ether32 = {0xedb88320};
CRC32 crc_ether32 = {0xffffffff};
CRC32 Scrc_ether32 = {0xffffffff};

FILE *fp;
char buff[8192];

main(argc, argv)
   int argc;
   char *argv[];
{
   unsigned len,i,c;
   unsigned hi_byte_cnt,lo_byte_cnt;

   /* We expect 1 argument */

   if (argc < 2)
   {
      printf("Usage: %s file\n", argv[0]);
      exit(1);
   }
   fp=fopen(argv[1],"rb");

   /* Set up  CRC tables for lookup method */
   I_CRC16(ccitt_tbl, &poly_ccitt);
   I_CRC16(crcbsc_tbl, &poly_crcbsc);
   I_CRC32(ether32_tbl, &poly_ether32);

   lo_byte_cnt = 0;
   hi_byte_cnt = 0;

   /* Read input file in blocks over which the CRC is calculated since
	  this is more efficient. */
   while((len = fread(buff, 1, sizeof(buff), fp)) != 0)
   {
	  F_CRC16(ccitt_tbl, &crc_ccitt, buff, len);
	  F_CRC16(crcbsc_tbl, &crc_crcbsc, buff, len);
	  F_CRC32(ether32_tbl, &crc_ether32, buff, len);
      S_CRC16(&poly_ccitt, &Scrc_ccitt, buff, len);
      S_CRC16(&poly_crcbsc, &Scrc_crcbsc, buff, len);
      S_CRC32(&poly_ether32, &Scrc_ether32, buff, len);

      /* Byte count of file (up to 65M) */

      if((lo_byte_cnt += len) >= 10000)
      {
         ++hi_byte_cnt;
         lo_byte_cnt -= 10000;
      }
   }
   fclose(fp);

   printf("\n%s completed:\n", argv[0]);
   printf(" bytes       = %d%04d\n",hi_byte_cnt,lo_byte_cnt);
   printf(" CRC16 CRC   Fast = 0x%04x  Slow = 0x%04x\n", crc_crcbsc, Scrc_crcbsc);
   printf(" CCITT CRC   Fast = 0x%04x  Slow = 0x%04x\n", crc_ccitt, Scrc_ccitt);
   printf(" Ether32 CRC   Fast = 0x%08lx  Slow = 0x%08lx\n", crc_ether32,  Scrc_ether32);

   printf("\nFor data comm transmission:\n");
   printf(" CRC16 CRC  byte(1) 0x%02x, byte(2) 0x%02x\n", 
          crc_crcbsc & 0xff, 
          crc_crcbsc>>8);
   /* These CRCs are inverted before transmision in data comm */
   crc_ccitt ^= 0xffff;
   crc_ether32 ^= 0xffffffff;
   printf(" CCITT CRC  byte(1) 0x%02x, byte(2) 0x%02x\n", 
          crc_ccitt & 0xff, 
          (crc_ccitt>>8) & 0xff);
   printf(" Ether32 CRC  byte(1) 0x%02lx, byte(2) 0x%02lx, byte(3) 0x%02lx, byte(4) 0x%02lx\n", 
          crc_ether32 & 0xff, 
          (crc_ether32>>8) & 0xff, 
          (crc_ether32>>16) & 0xff, 
          (crc_ether32>>24) & 0xff);
   exit(0);
   return 0; /* To keep compiler happy */
}
