/*
 *                           Protability Header
 *
 *    The purpose of this file is to provide a facility for writing C
 * source code that will easily port over various environments.
 *
 */

/*
 * These switches are used to turn on/off machine specific logic inclusion.
 * Turn on switch (set to 1) that reflects the machine type you are
 * targeting for. All other machine types should be turned off (set to 0).
 * Any machine specific logic should use these switches to insure its
 * inclusion/exclusion as appropriate. Critical logic should have a generic
 * version as well as any machine specific versions.
 * Example:
 * #if IBMPC
 *
 *     (PC specific logic)
 *
 * #else
 *
 *     (Generic logic)
 *
 * #endif
 */

#ifndef TANDEM
#define TANDEM       0
#define IBMPC        1
#define DATAGEN      0
#endif

/*
 * These macros are to help in making the scope of variables obvious.
 * They should be used on variables defined outside of functions
 * (in other words, global type variables) or on functions themselves.
 */

#ifndef PUBLIC
#define PUBLIC                /* Nothing needed as this is the default case. */
#endif

#ifndef PRIVATE
#define PRIVATE   static     /* 'static' used on global variables means    */
                             /* they are only visible in this module. This */
                             /* is a different (and ambigious) use of      */
                             /* 'static', which is normally used on auto   */
                             /* (local) variables to indicate that they    */
                             /* retain their value between invocations of  */
                             /* their associated function.                 */
#endif

/*
 * These are some general typedefs and constants to enhance the readability
 * of the C source code by making the use of variables more obvious.
 */

#ifndef BOOLEAN
typedef int        BOOLEAN;    /* Used for conditional switches which can  */
                               /* take on a value of TRUE or FALSE (1 or 0) */
#endif

#ifndef POINTER_TO_INT
typedef int *      POINTER_TO_INT;
#endif

#ifndef POINTER_TO_CHAR
typedef char *      POINTER_TO_CHAR;
#endif

#ifndef ARRAY
#define ARRAY                  /* This one doesn't do anything, but may be */
                               /* included to enhance readability.         */
                               /* Example: int ARRAY itab[10];             */
#endif

#ifndef TRUE

#define TRUE       1
#define FALSE      0

#endif

#ifndef SUCCEED

#define SUCCEED     0
#define FAIL       -1

#endif

#ifndef NULL
#define NULL ((void *) 0)
#endif

#ifndef BITS_IN_BYTE

/*
 * These macros address data handling issues that may arise from machine
 * to machine.
 */

#define BITS_IN_BYTE     8       /* Used to calc the bit sizes of the rest. */

#define BITS_IN_SHORT    (sizeof(short)  * BITS_IN_BYTE)
#define BITS_IN_INT      (sizeof(int)    * BITS_IN_BYTE)
#define BITS_IN_LONG     (sizeof(long)   * BITS_IN_BYTE)
#define BITS_IN_FLOAT    (sizeof(float)  * BITS_IN_BYTE)
#define BITS_IN_DOUBLE   (sizeof(double) * BITS_IN_BYTE)

#endif

#ifndef BYTE
#define BYTE(c) ((c) & 0xff)     /* Returns an unsigned 8bit value */
#endif

#ifndef IURSHIFT
                       /* Returns an int value logically shifted b bits */

#define IURSHIFT(n, b) (((int)(n) >> (b)) & ((((unsigned int)(-1)) / 2) >> ((b) - 1)))

#endif

#ifndef LURSHIFT
                        /* Returns a long value logically shifted b bits */

#define LURSHIFT(n, b) (((long)(n) >> (b)) & ((((unsigned long)(-1)) / 2L) >> ((b) - 1)))

#endif

#ifndef GETBIT
#define GETBIT(b, x)  (((x) >> (b)) & 0x1)      /* Return specified bit.   */
                                                /* Bit numbers start at 0. */
#endif

#ifndef GET8BIT
#define GET8BIT(b, x) (((x) >> (b * 8)) & 0xff) /* Return specified 8 bits.   */
                                                /* Numbers start a 0.    */
#endif

#ifndef GET16BIT
#define GET16BIT(w, x) (((x) >> (w * 16)) & 0xffff) /* Return the specified  */
                                                    /* 16bits.               */
#endif

#ifndef SETBIT
#define SETBIT(b, x) ((x) | (0x1 << (b))) /* Return modified value of x with */
                                          /* bit b set.                      */
                                          /* Note: this does not modify x.   */
#endif

#ifndef NUM_OF_ELEM
#define NUM_OF_ELEM(a) (sizeof(a)/sizeof(*(a)))   /* Returns the number of   */
                                                  /* elements in an array.   */
#endif


#ifndef DIGIT_VAL

/*
 * Returns the int value of a numeric char. If the char is not numeric a 0 is
 * returned.
 */

#define DIGIT_VAL(x) ((int) (((x) - '0') * ((x) >= '0' & (x) <= '9')))

#endif

