/*************************************************************************
 *                          COMLIBMD                                     *
 *                                                                       *
 * This module contains Compiler specific logic.                         *
 *                                                                       *
 *************************************************************************/

#include <dos.h>
#include <stdlib.h>
#include "comlib.h"
#include "comlibmd.h"

extern COM_INFO COMPORT[];

#if TURBOC

void far              cominp (int port);

/*
 * This routine replaces the origional COM1 interrupt. It doesn't do much,
 * except call the real ISR (which isn't an ISR) passing the port number.
 */

void interrupt com1int (void)
  {
    disable ();
    if ((inportb (COMPORT[COM1_PORT].iir) & 0x07) == 4)
      {
        cominp (COM1_PORT);
      }
    if ((inportb (COMPORT[COM3_PORT].iir) & 0x07) == 4)
      {
        cominp (COM3_PORT);
      }
    enable ();
  }

/*
 * This routine replaces the origional COM2 interrupt. It doesn't do much,
 * except call the real ISR (which isn't an ISR) passing the port number.
 */

void interrupt com2int (void)
  {
    disable ();
    if ((inportb (COMPORT[COM2_PORT].iir) & 0x07) == 4)
      {
        cominp (COM2_PORT);
      }
    if ((inportb (COMPORT[COM4_PORT].iir) & 0x07) == 4)
      {
        cominp (COM4_PORT);
      }
    enable ();
  }

/*
 *  This routine sets the COM port for the right baud rate, parity,
 *  stop bits, data length, and xon/xoff flow control. It also allocates
 *  memory for the circular input buffer
 */

void comopen (int port,
             long baud_rate,
             int  parity,
             int  stopbits,
             int  data_len,
             int  flw_flg)
  {
    unsigned ch_persec,
                  temp;

    COM_INFO * cport;

    cport = &COMPORT[port];

    /*
     * Set up new interrupt service routine saving old ISR for later.
     */

    cport->old_com_intr = getvect ((cport->com_int_vec));

    switch (port)
      {
        case COM1_PORT:
        case COM3_PORT:
                                                   /* Hard coded COM1 ISR, */
          setvect (cport->com_int_vec, com1int);   /* because I couldn't   */
                                                   /* think of a way to    */
                                                   /* figure it out on the */
                                                   /* fly.                 */
          break;
        case COM2_PORT:
        case COM4_PORT:
          setvect (cport->com_int_vec, com2int);  /* Same note as above. */
          break;
      }

    cport->baud_divisor = (int) (MAXBAUD / baud_rate);
    cport->data_len     = data_len;
    cport->parity       = parity;
    cport->stopbits     = stopbits;
    cport->flw_ctrl_flg = flw_flg;

    ch_persec = (int) (baud_rate / 10L);          /* Approx Char/sec.        */

    if (cport->combuf == (char *) NULL)           /* If no mem has been      */
      {                                           /* allocated for the buff. */
        if (ch_persec < 960)                      /* If baud rate is slow    */
          {                                       /* give it a 2K buffer to  */
            cport->buf_siz = 2048;                /* help throughput and be  */
          }                                       /* memory effecient.       */
        else
          {
            cport->buf_siz = ch_persec * 3;       /* Make buff big enough  */
                                                  /* to hold 3 sec. worth  */
                                                  /* of data.              */
          }

        cport->combuf = malloc ((cport->buf_siz));
      }
    else                                          /* If mem has already been  */
      {                                           /* allocated ty to reuse it.*/
        if (cport->buf_siz < ch_persec * 2)       /* If its not big enough to */
          {                                       /* hold 2 sec or more       */
            cport->buf_siz = ch_persec * 3;       /* reallocate enough for 3. */

            cport->combuf  = realloc (cport->combuf,
                                      cport->buf_siz);
          }
      }

    cport->head = 0;
    cport->tail = 0;

    if (flw_flg == 1)
      {
        /*
         * Send XOFF if buff has 1 sec worth of char space left. This gives
         * enough time for the other end to receive and process the XOFF
         * so we don't get choked at this end.
         */

        cport->flw_hi_lim = (cport->buf_siz) - ch_persec;

        /*
         * Send XON when there are about .6 sec worth of chars left in buffer.
         * This should allow the other side to start sending again before we
         * run out of chars.
         */

        cport->flw_lo_lim = (int) ((ch_persec * 6) / 10);
      }

    /*
     * Set the baud rate for this com port.
     */

    temp = inportb((cport->lcr)) | 0x80;          /* Turn on hi bit of   */
    outportb((cport->lcr), temp);                 /* LCR (also known as DLAB) */
                                                  /* for baud rate setup. */

    temp = ((cport->baud_divisor) & 0xff);        /* LSB of baud divisor */
    outportb((cport->ior), temp);                 /* goes in IOR.        */

    temp = ((cport->baud_divisor) >> 8) & 0xff;   /* MSB of baud         */
    outportb((cport->ier), temp);                 /* divisor goes in IER.*/

    temp = (parity << 3) + (stopbits << 2) + data_len;   /* Set LCR for parity, */
    outportb((cport->lcr), temp);                 /* stopbits, & data len.*/

    /*
     * Set up the 8250 UART to generate an interrupt to the system
     * when a character is sensed on the COM port.
     */

    disable();             /* Critical code, so disable interrupts */

    outportb((cport->mcr), 0x0b);        /* Set MCR to allow interrupts. */
    inportb((cport->msr));               /* Clear Modem Status Register. */
    inportb((cport->lsr));               /* Clear Line Status Register.  */
    inportb((cport->ior));               /* Clear I/O Register.          */
    temp  = inportb((cport->lcr));       /* Get Line Control Register.   */
    temp &= 0x7f;                        /* Mask off DLAB bit.           */
    outportb((cport->lcr), temp);        /* Set DLAB to 0.               */
    temp  = inportb(IMR);
    temp &= (cport->com_irq_mask);       /* set for IQR4 or 3.           */
    outportb(IMR, temp);                 /* Allow com interrupts.        */
    outportb((cport->ier), 0x01);        /* Enable receive interrupt.    */

    enable();             /* Re-enable interrupts. */
  }

#endif

#if MSC

#ifndef NULL
#define NULL  0
#endif

void far interrupt com1int (void);
void far interrupt com2int (void);

extern void far       cominp (int port);

/*
 * This routine replaces the origional COM1 interrupt. It doesn't do much,
 * except call the real ISR (which isn't an ISR) passing the port number.
 */

void far interrupt com1int (void)
  {
    disable ();
    if ((inportb (COMPORT[COM1_PORT].iir) & 0x07) == 4)
      {
        cominp (COM1_PORT);
      }
    if ((inportb (COMPORT[COM3_PORT].iir) & 0x07) == 4)
      {
        cominp (COM3_PORT);
      }
    enable ();
  }

/*
 * This routine replaces the origional COM2 interrupt. It doesn't do much,
 * except call the real ISR (which isn't an ISR) passing the port number.
 */

void far interrupt com2int (void)
  {
    disable ();
    if ((inportb (COMPORT[COM2_PORT].iir) & 0x07) == 4)
      {
        cominp (COM2_PORT);
      }
    if ((inportb (COMPORT[COM4_PORT].iir) & 0x07) == 4)
      {
        cominp (COM4_PORT);
      }
    enable ();
  }

/*
 *  This routine sets the COM port for the right baud rate, parity,
 *  stop bits, data length, and xon/xoff flow control. It also allocates
 *  memory for the circular input buffer
 */

void comopen (int port,
             long baud_rate,
             int  parity,
             int  stopbits,
             int  data_len,
             int  flw_flg)
  {
    unsigned ch_persec,
                  temp;

    COM_INFO * cport;

    cport = &COMPORT[port];

    /*
     * Set up new interrupt service routine saving old ISR for later.
     */

        stkchkoff();

    cport->old_com_intr = getvect ((cport->com_int_vec));

    switch (port)
      {
        case COM1_PORT:
        case COM3_PORT:
                                                   /* Hard coded COM1 ISR, */
          setvect (cport->com_int_vec, com1int);   /* because I couldn't   */
                                                   /* think of a way to    */
                                                   /* figure it out on the */
                                                   /* fly.                 */
          break;
        case COM2_PORT:
        case COM4_PORT:
          setvect (cport->com_int_vec, com2int);  /* Same note as above. */
          break;
      }

    cport->baud_divisor = (int) (MAXBAUD / baud_rate);
    cport->data_len     = data_len;
    cport->parity       = parity;
    cport->stopbits     = stopbits;
    cport->flw_ctrl_flg = flw_flg;

    ch_persec = (int) (baud_rate / 10L);          /* Approx Char/sec.        */

    if (cport->combuf == (char *) NULL)           /* If no mem has been      */
      {                                           /* allocated for the buff. */
        if (ch_persec < 960)                      /* If baud rate is slow    */
          {                                       /* give it a 2K buffer to  */
            cport->buf_siz = 2048;                /* help throughput and be  */
          }                                       /* memory effecient.       */
        else
          {
            cport->buf_siz = ch_persec * 3;       /* Make buff big enough  */
                                                  /* to hold 3 sec. worth  */
                                                  /* of data.              */
          }

        cport->combuf = malloc ((cport->buf_siz));
      }
    else                                          /* If mem has already been  */
      {                                           /* allocated ty to reuse it.*/
        if (cport->buf_siz < ch_persec * 2)       /* If its not big enough to */
          {                                       /* hold 2 sec or more       */
            cport->buf_siz = ch_persec * 3;       /* reallocate enough for 3. */

            cport->combuf  = realloc (cport->combuf,
                                      cport->buf_siz);
          }
      }

    cport->head = 0;
    cport->tail = 0;

    if (flw_flg == 1)
      {
        /*
         * Send XOFF if buff has 1 sec worth of char space left. This gives
         * enough time for the other end to receive and process the XOFF
         * so we don't get choked at this end.
         */

        cport->flw_hi_lim = (cport->buf_siz) - ch_persec;

        /*
         * Send XON when there are about .6 sec worth of chars left in buffer.
         * This should allow the other side to start sending again before we
         * run out of chars.
         */

        cport->flw_lo_lim = (int) ((ch_persec * 6) / 10);
      }

    /*
     * Set the baud rate for this com port.
     */

    temp = inportb((cport->lcr)) | 0x80;          /* Turn on hi bit of   */
    outportb((cport->lcr), temp);                 /* LCR (also known as DLAB) */
                                                  /* for baud rate setup. */

    temp = ((cport->baud_divisor) & 0xff);        /* LSB of baud divisor */
    outportb((cport->ior), temp);                 /* goes in IOR.        */

    temp = ((cport->baud_divisor) >> 8) & 0xff;   /* MSB of baud         */
    outportb((cport->ier), temp);                 /* divisor goes in IER.*/

    temp = (parity << 3) + (stopbits << 2) + data_len;   /* Set LCR for parity, */
    outportb((cport->lcr), temp);                 /* stopbits, & data len.*/

    /*
     * Set up the 8250 UART to generate an interrupt to the system
     * when a character is sensed on the COM port.
     */

    disable();             /* Critical code, so disable interrupts */

    outportb((cport->mcr), 0x0b);        /* Set MCR to allow interrupts. */
    inportb((cport->msr));               /* Clear Modem Status Register. */
    inportb((cport->lsr));               /* Clear Line Status Register.  */
    inportb((cport->ior));               /* Clear I/O Register.          */
    temp  = inportb((cport->lcr));       /* Get Line Control Register.   */
    temp &= 0x7f;                        /* Mask off DLAB bit.           */
    outportb((cport->lcr), temp);        /* Set DLAB to 0.               */
    temp  = inportb(IMR);
    temp &= (cport->com_irq_mask);       /* set for IQR4 or 3.           */
    outportb(IMR, temp);                 /* Allow com interrupts.        */
    outportb((cport->ier), 0x01);        /* Enable receive interrupt.    */

    enable();             /* Re-enable interrupts. */
  }

#endif
