/*
	Ŀ
	                   CRITICAL ERROR HANDLER                   
	                                                            
	 The Critical Error Handler or CEH as it is also called is  
	 a software routine that attempts to recover from a variety 
	 of internal and external errors that would prevent program 
	 continuance.                                               
	                                                            
	 Consider this example.  A program wants to open a file and 
	 issues the DOS services interrupt (interrupt 21h) with the 
	 "open file" function code in AH (3Dh).  This file is found 
	 on a floppy disk in the A: drive.  However, the drive door 
	 handle has been left open.  What happens?                  
	                                                            
	 Once interrupt 21h is issued, the DOS kernel takes over.   
	 The kernel calls a device driver.  The device driver calls 
	 the BIOS.  The BIOS communicates with the disk controller  
	 via the IN and OUT machine language instructions.  Because 
	 the controller cannot read the disk, it sends error infor- 
	 mation to the BIOS.  The BIOS sends error information to   
	 the device driver.  The device driver sends error infor-   
	 mation to the kernel.  At this point, the DOS designers    
	 could have done one of two things.  The kernel could have  
	 always sent the error information to the program causing   
	 it to deal with the error (the FAIL option) or the kernel  
	 could have made an attempt to solve the problem with user  
	 help (the ABORT, RETRY, IGNORE, FAIL prompt).  This latter 
	 approach was chosen and is more flexible than the previous 
	 approach.  The user has the opportunity of closing the     
	 door handle and selecting RETRY.  This time, the operation 
	 should succeed.                                            
	                                                            
	 The ABORT option causes DOS to return to the COMMAND.COM   
	 drive prompt.  This option could be disasterous if a file  
	 has been opened by the program and information written to  
	 the file but still in a DOS holding buffer.  When ABORT is 
	 chosen, the information in the buffer is not written to    
	 this file and the file is not properly closed (with its    
	 directory entry updated).                                  
	                                                            
	 RETRY allows the DOS kernel to attempt the operation again 
	 but this option is generally only useful if the problem is 
	 due to leaving a drive handle open or the printer offline. 
	 Simply closing the handle or placing the printer online is 
	 all that is needed to get the program running again.       
	                                                            
	 The IGNORE option is a form of gambling.  The DOS kernel   
	 is told to ignore the error and let the program continue.  
	 If the kernel does not feel that the error is that serious 
	 then it will allow the program to continue.                
	                                                            
	 DOS 3.3 added the FAIL option (better than IGNORE).  FAIL  
	 tells the kernel to allow the program to continue but pass 
	 back an error code via interrupt 21h to the program allow- 
	 ing it to make the decision as to continuing or not.       
	                                                            
	 Upon entry to interrupt 24h (the critical error handler),  
	 certain registers contain values indicating the nature of  
	 the error and what course of action should be taken.  Reg- 
	 isters AH, AL, BP:SI, and DI contain this information.     
	                                                            
	 If bit 7 of AH is 0 then the error was caused by a disk    
	 operation and is known as a hard error.  The failing drive 
	 number is placed in AL (0 = A, 1 = B, 2 = C, etc).  If bit 
	 5 is 1 then the IGNORE option is allowed.  If bit 4 is 1   
	 then the RETRY option is allowed.  If bit 3 is 1 then the  
	 FAIL option is allowed.  FAIL is not allowed on versions   
	 prior to 3.3 and IGNORE may not be allowed on versions 3.3 
	 and higher because FAIL is present.  RETRY is allowed in   
	 versions 3.3 and higher.  Bits 2 and 1 contain a code that 
	 indicates the affected disk area.  A "0 0" indicates DOS,  
	 a "0 1" indicates the file allocation table (FAT), a "1 0" 
	 indicates the directory, and a "1 1" indicates the data    
	 area.  If bit 0 is 1 then the operation that was taking    
	 place when the error occured was a write otherwise it was  
	 a read.                                                    
	                                                            
	 If bit 7 of AH is 1 then the error was caused by either a  
	 character device such as the printer or a bad memory image 
	 of the file allocation table (FAT).  BP:SI points to the   
	 device header of the failing device.  If the high bit of   
	 the byte at BP:[SI+4] is 1 then the error is due to a bad  
	 FAT image otherwise it is due to a character device.  AL   
	 contains no useful information if bit 7 of AH is 1.        
	                                                            
	 Regardless of what caused the error (disk or device or bad 
	 FAT image), the lower-half of DI contains an error code.   
	                                                            
	 00h - write-protect violation                              
	 01h - unknown drive number                                 
	 02h - drive not ready                                      
	 03h - unknown command (to controller)                      
	 04h - CRC data error                                       
	 05h - bad request structure length                         
	 06h - seek error                                           
	 07h - unknown media (disk) type                            
	 08h - sector not found                                     
	 09h - printer out of paper                                 
	 0ah - error while writing                                  
	 0bh - error while reading                                  
	 0ch - general failure                                      
	                                                            
	 Upon exit, the CEH returns a code in AL to let the kernel  
	 know what action to take.                                  
	                                                            
	 0 - IGNORE                                                 
	 1 - RETRY                                                  
	 2 - ABORT                                                  
	 3 - FAIL                                                   
	                                                            
	 Since the CEH is called by the DOS kernel and DOS is not   
	 reentrant, only the following interrupt 21h services can   
	 be used from inside the handler.  The services are 01h to  
	 0ch inclusive (the console I/O functions) and 59h (get the 
	 extended error information).  Any other service will cause 
	 the computer to lock up.                                   
	                                                            
	 The default critical error handler is actually located in  
	 the resident portion of the COMMAND.COM TSR.  There are a  
	 few problems with this handler.  First of all, if ABORT is 
	 chosen then the program is aborted as control returns to   
	 the DOS prompt (and any data waiting to be written to disk 
	 never gets there).  Secondly, when the handler writes its  
	 messages to the screen, these messages can overwrite what- 
	 ever is presently on the screen without first saving these 
	 contents.  The screen is not restored when the handler is  
	 finished.  These problems are solved by CEH.C.  CEH.C con- 
	 tains source code for a handler that is more user-friendly 
	 that the default.  Two options are allowed: RETRY or ABORT 
	 (ABORT means FAIL in this case allowing a program to con-  
	 tinue and decide if it should exit).  DOS 3.3 is required  
	 to use this handler because of the FAIL option.            
	                                                            
	 This handler was written in BORLAND C++ 2.0 using the C    
	 compiler.  It has been tested successfully in all memory   
	 models.  It will undoubtedly work on all BORLAND C compil- 
	 ers starting with Turbo C 1.5.  Some modifications will be 
	 necessary to port this code to other C compilers, notably  
	 the console video functions.  Here is a quick rundown on   
	 these functions.                                           
	                                                            
	 window (int left, int top, int right, int bottom);         
	                                                            
	 The console video functions work within a simple windowing 
	 environment.  The window () function defines the rectangle 
	 on the screen where console I/O is performed.  The screen  
	 coordinates are relative to the upper-left corner of this  
	 window and have origin (1, 1).  The default window is the  
	 size of the screen (1, 1, 40, 25) in 40-column text modes  
	 and (1, 1, 80, 25) in 80-column text modes.  The CEH pro-  
	 gram assumes an 80-column mode but could be converted to   
	 support either 80 or 40.                                   
	                                                            
	 gotoxy (int x, int y);                                     
	                                                            
	 This function positions the cursor within the active win-  
	 dow to column x and row y.  It fails if the coordinates    
	 are outside of the window range.                           
	                                                            
	 wherex (void);                                             
	                                                            
	 This function returns the x-coordinate of the cursor.      
	                                                            
	 wherey (void);                                             
	                                                            
	 This function returns the y-coordinate of the cursor.      
	                                                            
	 cputs (const char *string);                                
	                                                            
	 This function writes a string to the window beginning at   
	 the current cursor location within the window.  Scrolling  
	 (vertical) occurs if the string wraps past the lower-right 
	 boundary.  The current attribute defined by textattr () is 
	 used.                                                      
	                                                            
	 putch (int character);                                     
	                                                            
	 This function writes a character to the window at the cur- 
	 rent cursor location within the window.  The cursor loca-  
	 tion is updated and the current attribute defined by text- 
	 attr () is used.                                           
	                                                            
	 gettext (int left, int top, int right, int bottom, char *  
	          buffer);                                          
	                                                            
	 This function uses screen coordinates, not window.  It is  
	 not restricted to a window.  The screen contents bounded   
	 by (left, top) and (right, bottom) are read into an array  
	 pointed to by buffer.  Two bytes are read for each screen  
	 position: one for the character and one for the attribute. 
	 The puttext () function is similar except that it writes   
	 the buffer contents to the screen.                         
	                                                            
	 textattr (int attribute);                                  
	                                                            
	 Set current drawing attribute used by cputs ()/putch () to 
	 attribute.                                                 
								     
	 clrscr (void);                                             
	                                                            
	 Clear the contents of the current window.  Blanks are used 
	 for clearing along with the attribute most recently speci- 
	 fied by textattr ().                                       
	                                                            
	 gettextinfo (struct text_info *t);                         
	                                                            
	 The current windowing state including window boundaries,   
	 cursor location, and drawing attribute is read into the t  
	 structure.  This allows the windowing state to be restored 
	 later.                                                     
	                                                            
	 There are a few additional modifications to be made.       
	                                                            
	 The peekb() macro takes two arguments: segment and offset. 
	 It returns the byte found at this location.                
	                                                            
	 The setvect () function takes two arguments: an integer    
	 identifying the interrupt to be taken over and a pointer   
	 to the new interrupt function.                             
	                                                            
	 The geninterrupt() macro and pseudo-registers (_AH, _ES,   
	 etc.) can be replaced via an int86 () function.            
	                                                            
	 The _osmajor and _osminor global variables contain the     
	 major and minor portions of the DOS version number respec- 
	 tively.  A call to interrupt 21h with function 30h placed  
	 in AH will return the major portion in AL and the minor in 
	 AH.                                                        
	                                                            
	 Written by: Geoff Friesen, November 1991                   
	
*/

#include <conio.H>
#include <dos.H>
#include <string.H>

#include "ceh.H"

#define	CR	0x000d
#define	LEFT	0x4b00
#define	RIGHT	0x4d00

#define	RETRY	1
#define	FAIL	3

#define	HIDDEN	0x2000

#define	SINGLE	0
#define	DOUBLE	1

#define	NCOLS	52
#define	NROWS	12

#define	COLUMN	(80-NCOLS)/2
#define	ROW	(25-NROWS)/2

static char *errmsgs [] =
{
   "Disk is write-protected.",
   "Unknown drive number.",
   "Drive not ready.",
   "Disk controller does not recognize command.",
   "Data integrity error.",
   "Poorly organized data passed to disk controller.",
   "Seek error.",
   "Disk type is unknown.",
   "Sector not found.",
   "Printer out of paper.",
   "Could not write.",
   "Could not read.",
   "General failure - disk might need formatting.",
   "Internal error."
};

static char buffer [NCOLS*NROWS*2];

static struct text_info t;
static unsigned key, mode, shape;

#define	NATTR	5

static int attr [NATTR];

static int bw [] =
{
   (LIGHTGRAY << 4) | BLINK,		/* "Critical Error" header */
   WHITE,				/* border */
   LIGHTGRAY << 4,			/* foreground/background */
   LIGHTGRAY,				/* button */
   BLACK				/* shadow */
};

static int color [] =
{
   (RED << 4) | WHITE | BLINK,		/* "Critical Error" header */
   (BLUE << 4) | WHITE,			/* border */
   (BLUE << 4) | WHITE,			/* foreground/background */
   CYAN,				/* button */
   BLACK				/* shadow */
};

void		border		(int type, int ncols, int nrows);
void		clear		(int ncols, int nrows);
unsigned	getcshape	(void);
unsigned	getkey		(void);
void		setcshape	(unsigned shape);
void		shadow		(int x1, int y1, int x2, int y2, int attr);

static void interrupt ceh (unsigned bp, unsigned di, unsigned si,
			   unsigned ds, unsigned es, unsigned dx,
			   unsigned cx, unsigned bx, unsigned ax)
{
   unsigned i;

   _ES = _DS;

   _AH = 15;
   geninterrupt(0x10);

   mode = (unsigned) _AL;

   for (i = 0; i < NATTR; i++)
	attr [i] = (mode == C80) ? color [i] : bw [i];

   gettextinfo (&t);

   shape = getcshape ();

   setcshape (HIDDEN);
   window (1, 1, 80, 25);
   gettext (COLUMN, ROW, COLUMN+NCOLS-1, ROW+NROWS-1, buffer);

   textattr (attr [2]);

   gotoxy (COLUMN, ROW);
   clear (NCOLS, NROWS);

   gotoxy (COLUMN+2, ROW+2);

   if ((ax & 0x8000) && (peekb(bp, si+4) & 0x80))	/* fat */
       cputs ("Bad file allocation table image.");
   else
   {
       if (ax & 0x8000)
       {
	   cputs ("Device ");
	   for (i = 10; i < 18; i++)
		putch (peekb(bp, si+i));
       }
       else
       {
	   cputs ("Drive ");
	   putch ((ax & 255)+'A');
	   putch (':');
       }

       cputs (" reports a ");

       if (ax & 0x100)
	   cputs ("write");
       else
	   cputs ("read");

       cputs (" problem.");
   }

   i = (di > 12) ? 13 : di;
   gotoxy (COLUMN+(NCOLS-strlen (errmsgs [i]))/2, ROW+4);
   cputs (errmsgs [i]);

   gotoxy (COLUMN+NCOLS/2-5, ROW+8);
   cputs ("\x1b RETURN \x1a");

   textattr (attr [1]);

   gotoxy (COLUMN, ROW);
   border (DOUBLE, NCOLS, NROWS);

   textattr (attr [0]);

   gotoxy (COLUMN+(NCOLS-15)/2, ROW);
   cputs (" Critical Error ");

   for (i = 0; i < 2; i++)
   {
	textattr (attr [3]);

	gotoxy (COLUMN+((!i) ? 7 : NCOLS-14), ROW+6);
	clear (7, 3);

	gotoxy (COLUMN+((!i) ? 8 : NCOLS-13), ROW+7);
	cputs ((!i) ? "ABORT": "RETRY");

	textattr (attr [1]);

	gotoxy (COLUMN+((!i) ? 7 : NCOLS-14), ROW+6);
	border ((!i) ? DOUBLE : SINGLE, 7, 3);

	if (!i)
	    shadow (COLUMN+7, ROW+6, COLUMN+13, ROW+8, attr [4]);
	else
	    shadow (COLUMN+NCOLS-14, ROW+6, COLUMN+NCOLS-8, ROW+8,
		    attr [4]);
   }

   i = 0;

   do
   {
       key = getkey ();

       if (key == LEFT || key == RIGHT)
       {
	   if (i == 1 && key == LEFT)
	   {
	       gotoxy (COLUMN+NCOLS-14, ROW+6);
	       border (SINGLE, 7, 3);
	       gotoxy (COLUMN+7, ROW+6);
	       border (DOUBLE, 7, 3);
	       i = 0;
	   }
	   else
	   if (i == 0 && key == RIGHT)
	   {
	       gotoxy (COLUMN+7, ROW+6);
	       border (SINGLE, 7, 3);
	       gotoxy (COLUMN+NCOLS-14, ROW+6);
	       border (DOUBLE, 7, 3);
	       i = 1;
	   }
       }
   }
   while (key != CR);

   puttext (COLUMN, ROW, COLUMN+NCOLS-1, ROW+NROWS-1, buffer);
   window (t.winleft, t.wintop, t.winright, t.winbottom);
   gotoxy (t.curx, t.cury);
   setcshape (shape);
   textattr (t.attribute);

   ax = (i == 0) ? FAIL : RETRY;
}

int installceh (void)
{
   if (_osmajor < 3 || (_osmajor == 3 && _osminor < 3))
       return ERROR;

   setvect (0x24, ceh);
   return OK;
}

static void border (int type, int ncols, int nrows)
{
   int i;
   int x, y;

   x = wherex ();
   y = wherey ();

   putch ((type == SINGLE) ? '' : '');
    for (i = 0; i < ncols-2; i++)
	putch ((type == SINGLE) ? '' : '');
   putch ((type == SINGLE) ? '' : '');

   for (i = 0; i < nrows-2; i++)
   {
	gotoxy (x, y+i+1);
	putch ((type == SINGLE) ? '' : '');
	gotoxy (x+ncols-1, y+i+1);
	putch ((type == SINGLE) ? '' : '');
   }

   gotoxy (x, y+nrows-1);
   putch ((type == SINGLE) ? '' : '');
   for (i = 0; i < ncols-2; i++)
	putch ((type == SINGLE) ? '' : '');
   putch ((type == SINGLE) ? '' : '');

   gotoxy (x, y);
}

static void clear (int ncols, int nrows)
{
   struct text_info t;

   gettextinfo (&t);

   window (t.curx, t.cury, t.curx+ncols-1, t.cury+nrows-1);
   clrscr ();
   window (t.winleft, t.wintop, t.winright, t.winbottom);
   gotoxy (t.curx, t.cury);
}

static unsigned getcshape (void)
{
   _AH = 3;
   geninterrupt(0x10);

   return _CX;
}

static unsigned getkey (void)
{
   unsigned ax;

   _AH = 0;
   geninterrupt(0x16);

   ax = _AX;

   if (ax & 0x00ff)
       ax &= 0x00ff;

   return ax;
}

static void setcshape (unsigned shape)
{
   _AH = 1;
   _CX = shape;
   geninterrupt(0x10);
}

static void shadow (int x1, int y1, int x2, int y2, int attr)
{
   int i;
   char buffer [2];

   for (i = x1+1; i <= x2+1; i++)
   {
	gettext (i, y2+1, i, y2+1, buffer);
	buffer [1] = attr;
	puttext (i, y2+1, i, y2+1, buffer);
   }

   for (i = y1+1; i <= y2+1; i++)
   {
	gettext (x2+1, i, x2+1, i, buffer);
	buffer [1] = attr;
	puttext (x2+1, i, x2+1, i, buffer);
   }
}