/*======================================================================*
 *																		*
 *		CCT.C	-	a command line pre-processor for Turbo C v2.0		*
 *																		*
 *				  Copyright (c) 1990 by Keith Ledbetter					*
 *																		*
 *----------------------------------------------------------------------*
 *  Note:  Tab size is 4.												*
 *----------------------------------------------------------------------*
 *	This command line pre-processor allows you to embed Pascal-type		*
 *	compiler directives directly in your C source code.  It opens		*
 *	the input source file and looks for lines starting with "#cct" in	*
 *	the first four characters of the line. All switches found in these	*
 *	lines will be parsed to build the appropiate TCC command line.  We	*
 *	then spawn off (exec, actually) TCC, passing it the command line	*
 *	that we have built.  The available switches are listed in this		*
 *	source code below in the "Compiler Switches" block.  This is a good	*
 *	block to save to disk to be read in every time you start new code.	*
 *	Note that not all of the switches have to be present; those that	*
 *	aren't specified will simply be defaulted to by the TCC compiler.	*
 *	See the associated documentation file for each switch definition.	*
 *---------------------------[ Change Log ]-----------------------------*
 *																		*
 *    Date         Programmer             Description of changes.       *
 *	---------	--------------------	------------------------------- *
 *  19-Jan-90   Keith Ledbetter       	Initial release.				*
 *======================================================================*/

/*
			Compiler Switches for CCT pre-processor:
			----------------------------------------

#cct	Model: Small, Ansi: No, Align: Byte, Nested_comments: On
#cct	Merge_duplicate_strings: on, floating_point: none
#cct	optimize: size, include: d:\tc\include, lib: d:\tc\lib
#cct	char_type: unsigned, standard_stack_frame: Off
#cct	test_stack_overflow: off, calling_convention: c, use_registers:on
#cct	register_optimization: off, code_type:8086
#cct	map_file: off, jump_optimization: on
#cct	source_debugging: no, Line_Numbers: no
#cct	end: cct
*/

#ifndef __SMALL__
	#error Should use SMALL compilation model
#endif

#include	<stdio.h>
#include	<stdlib.h>
#include	<io.h>
#include	<dos.h>
#include	<dir.h>
#include	<string.h>
#include	<process.h>

#define		MAX_DEPTH	100				/* max number of lines to read	*/
										/*  looking for '#cct' lines..	*/
int		end_of_options = 0;				/* if set, forces us to exit	*/
int		current_line   = 0;				/* line number we're on now		*/
char	in_line [255];					/* line read from source file	*/
char	link_line [255];				/* holds extra .OBJ names		*/
char	compiler_name [80];				/* holds name of the compiler	*/
char	compiler_line [255];			/* final command line is here	*/
char	command_line [255];				/* switches are built here		*/
char	full_name [80];					/* full name of the source file	*/
char	input_file [24];				/* filename only of the source	*/
char	main_file [80];					/* main .C file to compile		*/
char	*arg_ptr [64];					/* used to pass args to TCC		*/

char	drive [MAXDRIVE];				/* parts of the filename that	*/
char	dir   [MAXDIR];					/*  the user specified..		*/
char	file  [MAXFILE];
char	ext   [MAXEXT];
int		flag;							/* holds flags from fnsplit		*/
char	file_buffer [8192];				/* large input buffer for speed	*/


/*======================================================================*
 *	This table contains all of the switches available, along with a		*
 *	mask defining the possible settings and the corresponding switch.	*
 *	A switch with a setting of "default" will be ignored if specified.	*
 *	Layout of the table is as follows:									*
 *		<switch_name>,<option>,<cl switch>,<option>,<cl switch> ...		*
 *	NOTE: the "switches" mask MUST NOT contain any spaces in it, and	*
 *		  each switch MUST be in pairs (ie: option,cl_switch).			*
 *======================================================================*/

char *switch_masks [] = {
	"model,tiny,-mt,small,default,medium,-mm,compact,-mc,large,-ml,huge,-mh",
	"ansi,on,-A,off,default",
	"align,byte,default,word,-a",
	"nested_comments,on,-C,off,default",
	"merge_duplicate_strings,on,-d,off,default",
	"floating_point,none,-f-,emulation,default,8087,-f87",
	"optimize,speed,-G,size,default",
	"char_type,signed,default,unsigned,-K",
	"standard_stack_frame,on,default,off,-k-",
	"test_stack_overflow,on,-N,off,default",
	"calling_convention,c,default,pascal,-p",
	"use_registers,on,default,off,-r-",
	"register_optimization,on,-Z,off,default",
	"code_type,8086,default,8088,default,80286,-1,80386,-1",
	"map_file,on,-M,off,default",
	"jump_optimization,on,-O,off,default",
	"generate_underbars,on,default,off,-u-",
	"source_debugging,on,-v,off,default",
	"line_numbers,on,-y,off,default",
	NULL
	};



/*======================================================================*
 *	Open the input source file, and read up to MAX_DEPTH lines looking	*
 *	for '#cct' in the first four positions of a line.  If found, set 	*
 *	the corresponding command line switches.  After we're done, set up 	*
 *	the command line and spawn off "tcc".								*
 *======================================================================*/

main (int argc, char *argv[])
{
	FILE *fp;
	int	 x, y;
	char *cp;
	strcpy (command_line, "");
	strcpy (link_line, "");
	strcpy (compiler_name, "tcc");
	if ( argc < 2 ) {
		if (isatty(fileno(stdout)))
			printf("\nTCC Preprocessor by Keith Ledbetter  v1.0  19-Jan-90\n"
				   "\nUsage: CCT filename[.c]\n");
		else
			printf("Error unknown.c 1: Filename must be specified\n");
		exit(1);
	}
	flag = fnsplit (argv [1], drive, dir, file, ext);
	if (! (flag & FILENAME)) {
		printf("Error unknown.c 1: Filename must be specified\n");
		exit(1);
	}
	if (! (flag & EXTENSION)) strcpy (ext, ".C");
    strcpy (input_file, file);
	strcat (input_file, ext);
	strlwr (input_file);
	fnmerge (full_name, drive, dir, file, ext);
	if ( (fp = fopen(full_name, "r")) == NULL ) {
		printf("Error %s 1: unable to open file\n", full_name);
		exit(1);
	}
	setvbuf (fp, file_buffer, _IOFBF, sizeof(file_buffer));
	for (x = 0; x < MAX_DEPTH; ++x) {
		if ( (fgets (in_line, 150, fp)) == NULL )
			break;
		++current_line;
		in_line [strlen(in_line)-1] = 0;
		if (strnicmp(in_line, "#cct", 4) == 0) {
			strlwr (in_line);
			parse_switches (in_line + 4);
		}
		if (end_of_options)
			break;
	}
	fclose(fp);
	strcpy (compiler_line, compiler_name);
	append (compiler_line, command_line);
	append (compiler_line, full_name);
	append (compiler_line, link_line);
	strcat (compiler_line, "  ");
	if (strlen(compiler_line) > 127) {
		printf("Error %s 1: resulting command line is > 127 bytes long\n",
				input_file);
		exit(1);
	}
	printf("\n%s\n", compiler_line);
	for (x = 0; compiler_line [x]; ++x)
		if (compiler_line [x] == ' ')
			compiler_line [x] = 0;
	x = y = 0;
	cp = compiler_line;
	while ( *cp ) {
		arg_ptr [y++] = cp;
		cp += (strlen(cp) + 1);
	}
    arg_ptr [y] = NULL;
	if (execvp (compiler_name, &arg_ptr[0]) < 0) {
		printf("Error %s 1: unable to spawn '%s'\n",input_file,compiler_name);
		exit(1);
	}
}



/*======================================================================*
 *	Parse down the input line, removing each 'option:setting' pair.		*
 *======================================================================*/

parse_switches(char *cl)
{
	char argument [80], parm [80];
	char *cp;
	while ( *cl ) {
		while (is_whitespace (*cl)) ++cl;
		if (*cl == 0) break;
		cp = argument;
		while (*cl && *cl != ':')
			*cp++ = *cl++;
		*cp = 0;
		if (*cl != ':') {
			printf("Error %s %d: argument is missing ending ':'\n",
					input_file, current_line);
			exit(1);
		 }
		else
			++cl;
		while (is_whitespace (*cl)) ++cl;
		if (*cl == 0) {
			printf("Error %s %d: argument is missing parameter\n",
					input_file, current_line);
			exit(1);
		 }
		cp = parm;
		while (*cl && is_whitespace(*cl) == 0)
			*cp++ = *cl++;
		*cp = 0;
		set_switch (argument, parm);
	}
}



/*======================================================================*
 *	Set a corresponding command line switch depending on "arg".			*
 *======================================================================*/

set_switch (char *arg, char *parm)
{
	char local_switch [80];
	int x;
	char *switch_name, *option, *cl_switch;
	if (strcmp (parm, "yes") == 0)				/* allow 'yes' and 'no'	*/
		parm = "on";							/*  as aliases..		*/
	else if (strcmp (parm, "no") == 0)
		parm = "off";
	for (x = 0; switch_masks [x] != NULL; ++x) {
		strcpy (local_switch, switch_masks [x]);
		switch_name = strtok (local_switch, ",");
		if (strcmp (arg, switch_name) == 0) {
			while ( (option = strtok(NULL, ",")) != NULL ) {
				cl_switch = strtok (NULL, ",");
				if (strcmp(parm, option) == 0) {
					if (strcmp(cl_switch, "default") != 0)
						append (command_line, cl_switch);
					return(1);
				}
			}
			printf("Error %s %d: unknown option '%s' for switch '%s'\n",
					input_file, current_line, parm, arg);
			exit(1);
		}
	}
	if (strcmp (arg, "compiler") == 0)
		strcpy (compiler_name, parm);
	else if (strcmp (arg, "main_file") == 0)
		strcpy (full_name, parm);
	else if (strcmp (arg, "link") == 0)
		append (link_line, parm);
	else if (strcmp (arg, "include") == 0) {
		append (command_line, "-I");
		strcat (command_line, parm);
	 }
	else if (strcmp (arg, "lib") == 0) {
		append (command_line, "-L");
		strcat (command_line, parm);
	 }
	else if (strcmp (arg, "output") == 0) {
		append (command_line, "-e");
		strcat (command_line, parm);
	 }
	else if (strcmp (arg, "end") == 0)
		end_of_options = 1;
	else {
		printf("Error %s %d: unknown switch '%s'\n",
					input_file, current_line, arg);
		exit(1);
	}
}



/*======================================================================*
 *		Our specialized "strcat" routine to concatenate strings.		*
 *======================================================================*/

append (char *dest, char *source)
{
	if ( strlen (dest) == 0 )
		strcpy (dest, source);
	else {
		if ( dest [strlen(dest)-1] != ' ' )
			strcat (dest, " ");
		strcat (dest, source);
	}
}



/*======================================================================*
 *			Return true if character is whitespace.						*
 *======================================================================*/

is_whitespace (char c)
{
	return (c == 9 || c == ' ' || c == ',');
}
