/************************************************************************
 *									*
 *  Copyright (c) 1987							*
 *  by CompuServe Incorporated, Columbus, Ohio				*
 *									*
 *  The information in ths software is subject to change without	*
 *  notice and should not be construed as a commitment by CompuServe	*
 *  Incorporated							*
 *									*
 ************************************************************************/

/**
 * File:  BPPACKET.C
 *
 * Facility:  B protocol packet driver
 *
 * Abstract:
 *	'C' header for B protocol packet driver
 *
 * Environment:  None specified
 *
 * Author:  John Pampuch, November 5, 1987
 *
 * Modified by:
 *
 * 1	John Pampuch, November 5, 1987
 *	- Original
 *
 * 2	John Pampuch, January 5, 1988
 * 	- Modified to remove unneeded structure nesting
 *
 * 3	John Pampuch, March 17, 1988
 * 	- changed return values to avoid overloading when possible
 *
 * 4	John Pampuch, August 3, 1988
 * 	- Changed behavior of Wait Acknowledge to update screen
 * 	  display; provide minimal user feedback.
 *
 * 	- Also changed BP_ENQ_Seen routine; it automatically
 * 	  initializes the protocol descriptor block to 'classic' B
 *
 * 5	Randy Raynak, February 17, 1989
 *	- Changed BP_Create_PDB to initialize the Packet_Size to 0.
 *	- Changed BP_Init_PDB to look to see if the Packet_Size is 0.  If
 *	  it is 0 then use the Default_Packet_Size otherwise use the callers
 *	  packet size
 **/

#include "comdef.h"
#include "bppacket.h"

/*
 * Contents:
 */
private WORD NULL_Function (void);
private void Discard_ACKed_Packets (struct PDBstruct *);
private void Init_Check (struct PDBstruct *);
private void Do_Checksum (WORD, struct PDBstruct *);
private void Send_Byte (WORD, struct PDBstruct *);
private void Send_Masked_Byte (WORD, struct PDBstruct *);
private WORD Read_Byte (BOOL, struct PDBstruct *);
private WORD Read_Masked_Byte (struct PDBstruct *);
private void Xmit_Packet (WORD, WORD, UBYTE *, struct PDBstruct *);
private WORD Wait_For_ACK (BOOL, BOOL, BOOL, BOOL, struct PDBstruct *);

#define ETX	0x03
#define ENQ	0x05
#define DLE	0x10
#define XON	0x11
#define XOFF	0x13
#define NAK	0x15

/* defines for improved readability and less typing */
#define Free		(*PDB->s_free)
#define Ch		PDB->Lchar
#define Not_Masked	PDB->NotMasked
#define Checksum	PDB->Check_sum
#define CRC		PDB->Check_sum
#define R_Seq_Num	PDB->Read_Seq_Number
#define S_Seq_Num	( *((PDB->Actual_Xport == 1) ? \
			(&PDB->Send_Seq_Number) : (&PDB->Read_Seq_Number)) )

#define S_Buffer	PDB->Send_Buffer
#define R_Buffer	PDB->Read_Buffer
#define Quoting		PDB->Quoting_Level
#define Last_ACK	PDB->Last_ACK_Seen
#define Sent_ENQ	PDB->Sent_ENQ_To_Remote

/* protocol defined parameters */
#define Max_Time	30		/* seconds */
#define WACK		';'		/* wait acknowledge */
#define	B_Lead2		'B'

typedef enum
    {
    S_Get_DLE,
    S_DLE_Seen,
    S_DLE_B_Seen,
    S_Get_Data,
    S_Get_Check,
    S_Get_CRC,
    S_Verify_CRC,
    S_Verify_CKS,
    S_Verify_Packet,
    S_Send_NAK,
    S_Send_ACK,
    S_Send_ENQ,
    S_Resend_Packets,
    } Sender_Action;

/*$page*/
static UWORD near crc_xmodem_tab[] =
    {
    0x0000, 0x1021, 0x2042, 0x3063, 0x4084, 0x50A5, 0x60C6, 0x70E7,
    0x8108, 0x9129, 0xA14A, 0xB16B, 0xC18C, 0xD1AD, 0xE1CE, 0xF1EF,
    0x1231, 0x0210, 0x3273, 0x2252, 0x52B5, 0x4294, 0x72F7, 0x62D6,
    0x9339, 0x8318, 0xB37B, 0xA35A, 0xD3BD, 0xC39C, 0xF3FF, 0xE3DE,
    0x2462, 0x3443, 0x0420, 0x1401, 0x64E6, 0x74C7, 0x44A4, 0x5485,
    0xA56A, 0xB54B, 0x8528, 0x9509, 0xE5EE, 0xF5CF, 0xC5AC, 0xD58D,
    0x3653, 0x2672, 0x1611, 0x0630, 0x76D7, 0x66F6, 0x5695, 0x46B4,
    0xB75B, 0xA77A, 0x9719, 0x8738, 0xF7DF, 0xE7FE, 0xD79D, 0xC7BC,
    0x48C4, 0x58E5, 0x6886, 0x78A7, 0x0840, 0x1861, 0x2802, 0x3823,
    0xC9CC, 0xD9ED, 0xE98E, 0xF9AF, 0x8948, 0x9969, 0xA90A, 0xB92B,
    0x5AF5, 0x4AD4, 0x7AB7, 0x6A96, 0x1A71, 0x0A50, 0x3A33, 0x2A12,
    0xDBFD, 0xCBDC, 0xFBBF, 0xEB9E, 0x9B79, 0x8B58, 0xBB3B, 0xAB1A,
    0x6CA6, 0x7C87, 0x4CE4, 0x5CC5, 0x2C22, 0x3C03, 0x0C60, 0x1C41,
    0xEDAE, 0xFD8F, 0xCDEC, 0xDDCD, 0xAD2A, 0xBD0B, 0x8D68, 0x9D49,
    0x7E97, 0x6EB6, 0x5ED5, 0x4EF4, 0x3E13, 0x2E32, 0x1E51, 0x0E70,
    0xFF9F, 0xEFBE, 0xDFDD, 0xCFFC, 0xBF1B, 0xAF3A, 0x9F59, 0x8F78,
    0x9188, 0x81A9, 0xB1CA, 0xA1EB, 0xD10C, 0xC12D, 0xF14E, 0xE16F,
    0x1080, 0x00A1, 0x30C2, 0x20E3, 0x5004, 0x4025, 0x7046, 0x6067,
    0x83B9, 0x9398, 0xA3FB, 0xB3DA, 0xC33D, 0xD31C, 0xE37F, 0xF35E,
    0x02B1, 0x1290, 0x22F3, 0x32D2, 0x4235, 0x5214, 0x6277, 0x7256,
    0xB5EA, 0xA5CB, 0x95A8, 0x8589, 0xF56E, 0xE54F, 0xD52C, 0xC50D,
    0x34E2, 0x24C3, 0x14A0, 0x0481, 0x7466, 0x6447, 0x5424, 0x4405,
    0xA7DB, 0xB7FA, 0x8799, 0x97B8, 0xE75F, 0xF77E, 0xC71D, 0xD73C,
    0x26D3, 0x36F2, 0x0691, 0x16B0, 0x6657, 0x7676, 0x4615, 0x5634,
    0xD94C, 0xC96D, 0xF90E, 0xE92F, 0x99C8, 0x89E9, 0xB98A, 0xA9AB,
    0x5844, 0x4865, 0x7806, 0x6827, 0x18C0, 0x08E1, 0x3882, 0x28A3,
    0xCB7D, 0xDB5C, 0xEB3F, 0xFB1E, 0x8BF9, 0x9BD8, 0xABBB, 0xBB9A,
    0x4A75, 0x5A54, 0x6A37, 0x7A16, 0x0AF1, 0x1AD0, 0x2AB3, 0x3A92,
    0xFD2E, 0xED0F, 0xDD6C, 0xCD4D, 0xBDAA, 0xAD8B, 0x9DE8, 0x8DC9,
    0x7C26, 0x6C07, 0x5C64, 0x4C45, 0x3CA2, 0x2C83, 0x1CE0, 0x0CC1,
    0xEF1F, 0xFF3E, 0xCF5D, 0xDF7C, 0xAF9B, 0xBFBA, 0x8FD9, 0x9FF8,
    0x6E17, 0x7E36, 0x4E55, 0x5E74, 0x2E93, 0x3EB2, 0x0ED1, 0x1EF0
    };

/*$page*/
/* Return new CRC value given accumulating CRC, octet, and table.	*/
/* WARNING: This is NOT the same as that used by normal bit-ordered CRCs. */

private void Init_Check (PDB)
    Protocol_Desc_Block *PDB;
    {

    if (PDB->Actual_Check == Check_CRC)
    then
	CRC = 0xffff;
    else
	Checksum = 0;
    }

private void Do_Checksum (ch, PDB)
    WORD ch;
    Protocol_Desc_Block *PDB;
    {
    if (PDB->Actual_Check == Check_B)
    then
	{
	Checksum <<= 1;
	if (Checksum > 255) then Checksum = (Checksum & 0xFF) + 1;
	Checksum += ch & 0xFF;
	if (Checksum > 255) then Checksum = (Checksum & 0xFF) + 1;
	}
    else				    /* CRC */
	BP_Do_CRC (ch, &CRC);
    }

public void BP_Do_CRC (ch, crc)
    WORD ch;
    WORD *crc;
    {
    UWORD tmp;

    tmp = (*crc >> 8 xor ch) & 0xff;
    *crc = crc_xmodem_tab [tmp] xor (*crc << 8);
    
    }

private WORD NULL_Function ()
    {
    return Failure;
    }

private void Discard_ACKed_Packets (PDB)
    Protocol_Desc_Block *PDB;
    {
    WORD n;
    WORD i;
    BOOL Packet_Acked = FALSE;

    Last_ACK = Ch;

    n = (PDB->Next_Packet + PDB->Pending_Count) mod Send_Ahead_Buffers;

    for (i = PDB->Pending_Count; i > 0; i--)
	{
	n--;
	if (n < 0) n += 5;

	if (PDB->Pending[n].Seq_Num == Ch - '0')
	then
	    {
	    Packet_Acked = TRUE;
	    (*PDB->Ch_Read) (PDB->R_Bytes);
	    (*PDB->Ch_Sent) (PDB->S_Bytes);
	    PDB->S_Bytes = PDB->R_Bytes = 0;
	    PDB->Next_Packet = (n + 1) mod Send_Ahead_Buffers;
	    }

	if (Packet_Acked == TRUE)
	then
	    {
	    Free (PDB->Pending [n].packet);
	    PDB->Pending [n].packet = NULL;
	    PDB->Pending_Count--;
	    }
	}
    }

private void Send_Byte (ch, PDB)
    WORD ch;
    Protocol_Desc_Block *PDB;
    {
    /*
     * send character to remote, and update status information
     * as needed.
     */

    while ( ((*PDB->Put_Ch) ((UBYTE) ch)) == 0)
	;

    if (++PDB->S_Bytes == PDB->Update_Freq)
    then 
	{
	(*PDB->Ch_Sent) (PDB->S_Bytes);
	PDB->S_Bytes = 0;
	}
    }

private void Send_Masked_Byte (ch, PDB)
    WORD ch;
    Protocol_Desc_Block *PDB;
    {
    /* Mask any protocol or flow characters and send it to the remote */
    /* per the quote level value in the PDB */

    if (ch < 0x20)
    then
	{
	if (Quoting == Quote_Full or (PDB->Mask [ch] & MaskLowRange))
	then
	    {
	    Send_Byte (DLE, PDB);
	    ch += '@';
	    }
	}
    else if (ch >= 0x80 and ch < 0xA0)
    then
	{
	if (Quoting == Quote_Full or 
	    ((PDB->Mask [ch - 0x80] & MaskHiRange) and PDB->Actual_Plus) )
	then
	    {
	    Send_Byte (DLE, PDB);
	    ch = ch + '`' - 0x80;
	    }
	}

    Send_Byte (ch, PDB);
    }

/*$page*/
private WORD Read_Byte (Wait, PDB)
    BOOL Wait;
    Protocol_Desc_Block *PDB;
    {
    if (Wait)
    then
	Ch = (*PDB->Get_Ch_Timed) (Max_Time);
    else
	Ch = (*PDB->Get_Ch_Timed) (0);
    
    if (Ch == Remote_Timeout)
    then
	{
	PDB->Error_Code = Timeout;
	return Failure;
	}

    if (++PDB->R_Bytes == PDB->Update_Freq)
    then 
	{
	(*PDB->Ch_Read) (PDB->R_Bytes);
	PDB->R_Bytes = 0;
	}

    return Success;
    }

private WORD Read_Masked_Byte (PDB)
    Protocol_Desc_Block *PDB;
    {
    Not_Masked = TRUE;

    if (Read_Byte (TRUE, PDB) == Failure)
    then
	return Failure;

    if (Ch == DLE) 
    then
	{
	if (Read_Byte (TRUE, PDB) == (UWORD) Failure) 
	then
	    return Failure;

	Not_Masked = FALSE;

	if (Ch >= '`')
	then
	    Ch += 0x80;

	Ch &= 0x9F;
	}

    return Success;
    }

/*$page*/
public void BP_Send_ACK (PDB)
/*
 * Function:
 *	Send valid ACK for last successfully received packet
 * 
 * Inputs:
 *      PDB - Parameter block containing linkages to remote and other 
 * 	elements of the application, and some space for housekeeping
 *	matters.
 *
 * Outputs:
 *	none
 *
 * Returns:
 * 	nothing
 */
    Protocol_Desc_Block *PDB;
    {
    Send_Byte (DLE, PDB);
    Send_Byte (R_Seq_Num + '0', PDB);
    }

/*$page*/
public Protocol_Desc_Block *BP_Create_PDB (alloc, free)
/*
 * Function:
 *	Allocate a PDB, and setup the s_alloc, s_free, and initializes
 * 	buffer pointers as required.  Also, it sets up a default
 * 	Mask array, and initializes function pointers to null routines.
 * 
 * Inputs:
 * 	alloc	- function pointer to allocate space
 * 	free	- function pointer to release space
 *
 * Outputs:
 *	none
 *
 * Returns:
 * 	Failure - memory allocation problem, unable to continue, otherwise
 * 	pointer to PDB;
 */
    BYTE *(*alloc) (UWORD);
    void (*free) (BYTE *);
    {
    Protocol_Desc_Block *PDB;

    PDB = (Protocol_Desc_Block *) (*alloc) (sizeof (Protocol_Desc_Block));

    if (PDB == NULL)
    then
	return Failure;

    PDB->s_alloc = alloc;
    PDB->s_free = free;
    S_Buffer = R_Buffer = NULL;

    PDB->Ch_Sent =	(void (*) (UWORD) ) NULL_Function;
    PDB->Ch_Read =	(void (*) (UWORD) ) NULL_Function;
    PDB->Send_Error =	(void (*) (void) ) NULL_Function;
    PDB->Read_Error =	(void (*) (void) ) NULL_Function;

    PDB->Get_Ch_Timed =	(WORD (*) (WORD)) NULL_Function;
    PDB->Put_Ch = 	(WORD (*) (UBYTE)) NULL_Function;
    PDB->Delay = 	(void (*) (UWORD)) NULL_Function;

    PDB->Packet_Size = 0;

    return (PDB);
    }

/*$page*/
public WORD BP_Set_To_Quote (Chr, PDB)
/*
 * Function:
 * 	Force specified character to be quoted when using the
 * 	B Plus quoting mask.  This procedure can be used to set
 * 	any chars to be quoted.
 * 
 * 	Note that each time the PDB is initialized, the quoting
 * 	chars are reset to default values.
 * 
 * Inputs:
 *	Chr	- char to quote
 *	PDB	- Parameter block for linkage to remote, etc.
 * 
 * Outputs:
 *	none
 *
 * Returns:
 * 	Success if possible to Quote,
 * 	Failure if not
 */
    UBYTE Chr;
    Protocol_Desc_Block *PDB;
    {
    if (Chr < 0x20)
    then
	{
	PDB->Mask [Chr] |= MaskLowRange;
	return Success;
	}
    else if (Chr >= 0x80 and Chr < 0xA0)
	{
	PDB->Mask [Chr - 0x80] |= MaskHiRange;
	return Success;
	}
    return Failure;
    }

public WORD BP_Unset_To_Quote (Chr, PDB)
/*
 * Function:
 * 	Force specified character to be not quoted when using the
 * 	B Plus quoting mask.  This procedure can be used to set
 * 	Default Quoting chars off; but should be used with care.
 * 	Note that each time the PDB is initialized, the quoting
 * 	chars are reset to default values.
 * 
 * Inputs:
 *	Chr	- char to un-quote
 *	PDB	- Parameter block for linkage to remote, etc.
 * 
 * Outputs:
 *	none
 *
 * Returns:
 * 	Success if possible to Un-Quote,
 * 	Failure if not
 */
    UBYTE Chr;
    Protocol_Desc_Block *PDB;
    {
    if (Chr < 0x20)
    then
	{
	PDB->Mask [Chr] &= ~MaskLowRange;
	return Success;
	}
    else if (Chr >= 0x80 and Chr < 0xA0)
	{
	PDB->Mask [Chr - 0x80] &= ~MaskHiRange;
	return Success;
	}
    return Failure;
    }

public BOOL BP_Is_Chr_Quoted (Chr, PDB)
/*
 * Function:
 * 	Determine if characters are presently being quoted.
 * 	After processing a 'Plus' packet, this routine can
 * 	be called to see what characters are being quoted.
 * 
 * Inputs:
 *	Chr	- char to quote
 *	PDB	- Parameter block for linkage to remote, etc.
 * 
 * Outputs:
 *	none
 *
 * Returns:
 * 	TRUE if character in question is being quoted
 * 	FALSE if not.
 */
    UBYTE Chr;
    Protocol_Desc_Block *PDB;
    {
    if (Chr < 0x20)
    then
	if (PDB->Mask [Chr] & MaskLowRange)
	then
	    return TRUE;
	else
	    return FALSE;

    else if (Chr >= 0x80 and Chr < 0xA0)
    then
	if (PDB->Mask [Chr - 0x80] & MaskHiRange)
	then
	    return TRUE;
	else
	    return FALSE;

    return FALSE;
    }

/*$page*/
public WORD BP_Init_PDB (Use_B_Plus, Update_Freq, PDB)
/*
 * Function:
 *	Initialize the protocol descriptor block for normal usage.
 * 	It prepares all elements of the protocol descriptor block,
 * 	except for the pointers to required functions.  s_alloc and
 * 	s_free must be prepared prior to calling this routine.  This
 * 	routine also initializes the quoting mask to quote only
 * 	ETX, ENQ, DLE, NAK, XON, XOFF.  Any other characters that
 * 	must be quoted must be reset.  When the 'Plus' packet
 * 	is exchanged, the quoting mask will be updated to match
 * 	the remote quoting requirements.
 * 
 * Inputs:
 *	Use_B_Plus - Flag determing use of B_Plus vs. old style B
 * 
 *	Update_Freq - Maximum Character count between status updates
 * 
 *	Quoting_Level - Level of data quoting inside packets
 * 
 *      PDB - Parameter block containing linkages to remote and other 
 * 	elements of the application, and some space for housekeeping
 *	matters.
 *
 * Outputs:
 *	none
 *
 * Returns:
 * 	Success, or
 * 	Failure - memory allocation problem.
 */
    BOOL Use_B_Plus;
    WORD Update_Freq;
    Protocol_Desc_Block *PDB;
    {
    WORD i;

    PDB->S_Buffer_Len = 0;
    PDB->R_Buffer_Len = 0;

    PDB->Window_Size = 0;

    PDB->Use_B_Plus = Use_B_Plus;
#if 0
    PDB->Packet_Size = Default_Packet_Size;
#endif
    if (PDB->Packet_Size == 0)
	    PDB->Packet_Size = Default_Packet_Size;

    PDB->Max_Errors = Default_Max_Errors;


    if (Update_Freq != 0)
	PDB->Update_Freq = Update_Freq;

    PDB->Quoting_Level = Quote_Mask;

    PDB->Pending_Count = 0;
    PDB->Next_Packet = 0;

    PDB->R_Bytes = PDB->S_Bytes = 0;

    PDB->Send_Seq_Number = PDB->Read_Seq_Number = 0;

    for (i = 0; i < Send_Ahead_Buffers; i++)
	PDB->Pending [i].packet = NULL;

    PDB->Actual_Check = Check_B;
    PDB->Actual_Xport = 0;
    PDB->Packets_Btwn_ACKs = 0;
    Sent_ENQ = FALSE;
    Last_ACK = 0;

    for (i = 0; i < 32; i++)
	    PDB->Mask[i] = 0;

    PDB->Mask [ETX] = MaskLowRange;
    PDB->Mask [ENQ] = MaskLowRange;
    PDB->Mask [DLE] = MaskLowRange;
    PDB->Mask [NAK] = MaskLowRange;
    PDB->Mask [XON] = MaskLowRange;
    PDB->Mask [XOFF] = MaskLowRange;

    if (Update_Freq != 0)
	return BP_Alloc_Buffers (PDB);
    else
	return Success;
    }

/*$page*/

public WORD BP_Alloc_Buffers (PDB)
/*
 * Function:
 * 	Allocate space necessary for buffers based upon required space
 * 	listed in packet size. s_alloc and s_free must be prepared
 *	prior to calling this routine.
 * 
 * Inputs:
 *      PDB - Parameter block containing linkages to remote and other 
 * 	elements of the application, and some space for housekeeping
 *	matters.
 *
 * Outputs:
 *	none
 *
 * Returns:
 * 	Success - able to continue with protocol
 * 	Failure - memory allocation problem, unable to continue
 */
    Protocol_Desc_Block *PDB;
    {
    BP_Free_Buffers (PDB);

    PDB->Error_Code = Memory;		    /* assume allocation error */
    PDB->Send_Buffer = (*PDB->s_alloc) (PDB->Packet_Size);

    if (PDB->Send_Buffer == NULL)
    then
    	return Failure;

    PDB->Read_Buffer = (*PDB->s_alloc) (PDB->Packet_Size);

    if (PDB->Read_Buffer == NULL)
    then
	return Failure;

    PDB->Error_Code = Normal;		    /* no error, so normal */
    return Success;
    }

/*$page*/

public void BP_Free_Buffers (PDB)
/*
 * Function:
 *	Discards the space allocated for buffers, and discards
 * 	any pending packets that have gone un-ACKed.
 * 
 * Inputs:
 *      PDB - Parameter block containing linkages to remote and other 
 * 	elements of the application, and some space for housekeeping
 *	matters.
 *
 * Outputs:
 *	none
 *
 * Returns:
 * 	nothing
 */
    Protocol_Desc_Block *PDB;
    {
    WORD i;

    for (i = 0; i < Send_Ahead_Buffers; i++)
	if (PDB->Pending [i].packet != NULL)
	then
	    Free (PDB->Pending [i].packet);

    if (S_Buffer != NULL)
    then
	Free (S_Buffer);

    if (R_Buffer != NULL)
    then
	Free (R_Buffer);

    R_Buffer = S_Buffer = NULL;
    }

/*$page*/

public void BP_Send_Wait (PDB)
/*
 * Function:
 *	Send valid Wait for last successfully received packet
 * 
 * Inputs:
 *      PDB - Parameter block containing linkages to remote and other 
 * 	elements of the application, and some space for housekeeping
 *	matters.
 *
 * Outputs:
 *	none
 *
 * Returns:
 * 	nothing
 */
    Protocol_Desc_Block *PDB;
    {
    Send_Byte (DLE, PDB);
    Send_Byte (';', PDB);
    }

/*$page*/

public void BP_Seen_ENQ (PDB)
/*
 * Function:
 *	Send response to out-of-protocol ENQ.  Response depends upon
 * 	the intended use of B+;  if B+ is in use, the response is
 * 	DLE '+' '+' DLE digit.  If not, it is simply an ACK.
 *
 * 	Also, we want to reinitialize the protocol to 'classic' B.
 * 
 * Inputs:
 *      PDB - Parameter block containing linkages to remote and other 
 * 	elements of the application, and some space for housekeeping
 *	matters.
 *
 * Outputs:
 *	none
 *
 * Returns:
 * 	nothing
 */
    Protocol_Desc_Block *PDB;
    {
    /*
     * reset to classic B
     */
    BP_Init_PDB (PDB->Use_B_Plus, 0, PDB);
    /*
     * send the response
     */
    if (PDB->Use_B_Plus == B_Plus)
    then
	{
	Send_Byte (DLE, PDB);
	Send_Byte ('+', PDB);
	Send_Byte ('+', PDB);
	}
    BP_Send_ACK (PDB);

    }

/*$page*/

private void Xmit_Packet (Size, Seq, Packet, PDB)
/*
 * Does not wait for any incoming characters, including ACK!
 */
    WORD Size;
    WORD Seq;
    UBYTE *Packet;
    Protocol_Desc_Block *PDB;
    {
    WORD
	I;

    Init_Check (PDB);
    Send_Byte (DLE, PDB);
    Send_Byte (B_Lead2, PDB);
    Send_Byte (Seq + '0', PDB);
    Do_Checksum (Seq + '0', PDB);

    for (I = 0; I < Size; I++)
	{
	Send_Masked_Byte (Packet [I], PDB);
	Do_Checksum (Packet [I], PDB);
	}

    Send_Byte (ETX, PDB);
    Do_Checksum (ETX, PDB);

    if (PDB->Actual_Check == Check_B)
    then
	Send_Masked_Byte (Checksum, PDB);
    else
	{
	Send_Masked_Byte (CRC >> 8, PDB);
	Send_Masked_Byte (CRC & 0xff, PDB);
	}
    }
		
/*$page*/

public WORD BP_Read_Packet (Have_DLE_B, Wait, Send_ACK, PDB)
/*
 * Function:
 *	Receive a packet from the remote.
 * 
 * Inputs:
 *	Have_DLE_B - flag denoting that the DLE B has already been read.
 *
 * 	Wait - flag indicating whether to wait for packet or not.  TRUE
 * 	indicates that Read_Packet should wait for a timeout, FALSE
 * 	indicates that it should return if the packet does not start
 * 	immediately.  Have_DLE_B implies that the packet has already
 * 	started, thus wait is ignored.
 *
 *	Send_ACK - flag specifying that ACKs should be automatically
 * 	sent upon the successful receipt of a packet.  If FALSE, the
 * 	application is required to send its own ACKs.
 *
 *      PDB - Parameter block containing linkages to remote and other 
 * 	elements of the application, and some space for housekeeping
 *	matters.
 *
 * Outputs:
 * 	PDB->Read_Buffer -- contains the packet just received
 *
 * Returns:
 *	Success if packet arrived (length recorded in PDB->S_Buffer_Len)
 * 	Failure if timeout or protocol failure, or ACK arrived
 */
    BOOL Have_DLE_B;
    BOOL Wait;
    BOOL Send_ACK;
    Protocol_Desc_Block *PDB;
    {
    /* Wait_For_ACK, by definition, accomplishes the same thing. */

    Wait_For_ACK (Wait, Have_DLE_B, Send_ACK, TRUE, PDB);

    /* if we received an ACK, then we did not receive a packet */

    if (PDB->Error_Code == Packet_Arrived)
	return Success;
    else
	return Failure;
    
    }

/*$page*/

public void BP_Send_Failure (Code, Text, PDB)
/**
 * Function:
 *	Send a failure packet to the host, ignoring pending packets.
 *
 * Inputs:
 *	Code -- failure code
 *	Text -- error message text to be display at the remote
 *
 * Outputs:
 *
 * Returns:
 *	nothing
 **/
    UBYTE Code;
    UBYTE *Text;
    Protocol_Desc_Block *PDB;
    {
    WORD Len;
    WORD Seq_Num;

    S_Buffer[0] = 'F';
    S_Buffer[1] = Code;
    Len = 2;
    while (*Text) S_Buffer [Len++] = *Text++;

    Seq_Num = (S_Seq_Num + 1) mod 10;

    /* discard pending packets, then, send failure packet. */

    while (PDB->Pending_Count > 0 and
	Wait_For_ACK (TRUE, FALSE, FALSE, FALSE, PDB) != Failure)
	;

    Xmit_Packet (Len, Seq_Num, S_Buffer, PDB);

    do
	Wait_For_ACK (TRUE, FALSE, FALSE, FALSE, PDB);
    while (PDB->Error_Code == Packet_Arrived);

    }

/*$page*/

public WORD BP_Flush_Pending (PDB)
/**
 * Function:
 *	Wait for ACKs on remaining packets in outgoing window
 *
 * Inputs:
 *	PDB - Parameter block for linkage to remote, etc.
 *
 * Outputs:
 *
 * Returns:
 *	Success	- all ACKs received
 *	Failure - error: Packet recieved, Protocol error, etc.
 * 		  See PDB->Error_Code for details
 **/
    Protocol_Desc_Block *PDB;
    {

    /*
     * first, wait for all the remaining ACKs, and resend as
     * needed.  (If any resend fails, return Failure)
     */
    while (PDB->Pending_Count > 0)
	if (Wait_For_ACK (TRUE, FALSE, TRUE, TRUE, PDB) == Failure)
	then
	    return Failure;

    return Success;			
    }

/*$page*/

private WORD Wait_For_ACK (Wait, Have_DLE_B, Send_ACK, Resend, PDB)
/*
 * Function:
 *	Wait for an incoming data, and either accept packet,
 * 	accept ACK and adjust window count, or accept NAK and
 * 	resync and resend packets.
 *
 * Inputs:
 * 	Wait - flag indicating whether to wait for packet or not.  TRUE
 * 	indicates that Read_Packet should wait for a timeout, FALSE
 * 	indicates that it should return if the packet does not start
 * 	immediately.  Have_DLE_B implies that the packet has already
 * 	started, thus wait is ignored.
 *
 *	Have_DLE_B - flag denoting that the DLE B has already been read.
 *
 *	Send_ACK - flag specifying that ACKs should be automatically
 * 	sent upon the successful receipt of a packet.  If FALSE, the
 * 	application is required to send its own ACKs.
 *
 *	PDB - contains send-ahead information
 *
 * Outputs:
 *	PDB->Read_Buffer -- the packet received in response to the send
 *
 * Returns:
 *	Success	- ACK received
 *	Failure - error, packet arrived or other error... see PDB->Error_Code
 */
    BOOL Wait;
    BOOL Have_DLE_B;
    BOOL Send_ACK;
    BOOL Resend;
    Protocol_Desc_Block *PDB;
    {
    Sender_Action
	Action;

    WORD
	i,
	n,
	RCV_Num,
	Errors = 0;


    PDB->Error_Code = Normal;		    /* default error */
    PDB->R_Buffer_Len = 0;
    i = 0;

    if (Have_DLE_B)
    then
	Action = S_DLE_B_Seen;
    else
	Action = S_Get_DLE;

    while (Errors < PDB->Max_Errors)
	switch (Action)
	    {
	    case S_Get_Data:
		if (Read_Masked_Byte (PDB) == Failure)
		then
#if DEBUG
{
			Action = S_Send_NAK;
printf("Sending NAK, S_Get_Data read char failure\n");
}
#else
			Action = S_Send_NAK;
#endif
		else if (Not_Masked && (Ch == ETX))
		then
		    Action = S_Get_Check;
		else if (Not_Masked && Ch == ENQ)
		then
		    Action = S_Send_ACK;
		else if (i == PDB->Packet_Size)
		then
		    {
		    PDB->Error_Code = Buffer_Overflow;
#if DEBUG
printf("Sending NAK, S_Get_Data buffer overflow\n");
#endif
		    Action = S_Send_NAK;
		    }
		else
		    {
		    R_Buffer[i++] = (UBYTE) Ch;
		    Do_Checksum (Ch, PDB);
		    }

		break;

	    case S_Get_DLE:
		/* since Window_Size is one less than actual size... */
		if ((PDB->Packets_Btwn_ACKs > (PDB->Window_Size + 2)) and
		    (PDB->Pending_Count > 0)) 
		then
		    {

		    PDB->Error_Code = Logical_Timeout;
		    PDB->Packets_Btwn_ACKs = 0;
		    Action = S_Send_ENQ;
		    continue;
		    }
		if (Read_Byte (Wait, PDB) == Failure)
		then
		    {
		    /*
		     * If we are not to wait, then just return.  If we are,
		     * and there is a failure, then we need to ENQ the
		     * remote so we can find out where our ACKs are.
		     * If we are waiting for a packet to arrive, we 
		     * will end up waiting for ACKs from the remote
		     * first if no packet is on its way.  This is probably
		     * an error condition anyway.
		     */
		    if (not Wait) 
		    then
			/* change... now considered to be a timeout */
			return Failure;	    
		    else
			Action = S_Send_ENQ;
		    }

		else if (Ch == DLE)
		then
		    Action = S_DLE_Seen;
		else if (Ch == NAK) 
		then
		    Action = S_Send_ENQ;
		else if (Ch == ENQ) 
		then
		    Action = S_Send_ACK;
		else if (Ch == ETX) 
		then
		    {
		    PDB->Error_Code = Early_ETX;
		    Action = S_Send_NAK;
#if DEBUG
printf("Sending NAK, S_Get_DLE...ETX seen early\n");
#endif
		    }

		break;

	    case S_DLE_Seen:
		if (Read_Byte (TRUE, PDB) == Failure)
		then
		    Action = S_Send_ENQ;
		else if (Ch >= '0' and Ch <= '9') 
		then
		    {
		    /*
		     * if we have sent ENQs, then we should be seeing
		     * two matching ACKs in a row.  If we do, then
		     * check and see if we have ACK'd everything in
		     * the pipe.  If so, then we can go on... If not, then
		     * we need to wait for more ACKs.  If one of the ACKs
		     * get lost, then eventually we will time out, and
		     * we will then resend the ENQs.  If there are still
		     * packets pending after receiving two matching ACKs
		     * in a row, then we need to resend packets.
		     */

		    if (Sent_ENQ and (Ch == Last_ACK))
		    then
			{
			Sent_ENQ = FALSE;

			if (PDB->Pending_Count == 0)
			then
			    return (WORD) Success;
			else
			    Action = S_Resend_Packets;
			}
		    else
			{
			Discard_ACKed_Packets (PDB);
			/*
			 * If we have sent an ENQ out, then we still
			 * don't know exactly where the other end is
			 * yet.  So we will continue, until things have
			 * settled down.  Otherwise, we need to return
			 * immediately.
			 */
			if (Sent_ENQ)
			then
			    Action = S_Get_DLE;
			else
			    return (WORD) Success;
			}
		    }

		else if (Ch == WACK) 		/* code may be removeable */
		then
		    {
		    /*
		     * update display, so people don't think we have
		     * given up...
		     */
		    (*PDB->Ch_Read) (PDB->R_Bytes);
		    (*PDB->Ch_Sent) (PDB->S_Bytes);
		    PDB->S_Bytes = PDB->R_Bytes = 0;

		    Action = S_Get_DLE;	/* allows us to wait */
		    }
		else if (Ch == B_Lead2)
		then
		    Action = S_DLE_B_Seen;
		else if (Ch == ENQ)
		then
		    Action = S_Send_ACK;
		else
		    Action = S_Get_DLE;

		break;

	    case S_DLE_B_Seen:
		/*
		 * Start of an arriving "B" protocol packet.
		 */

		if (Read_Byte (TRUE, PDB) == Failure) 
		then
#if DEBUG
{
printf("Sending NAK, S_DLE_B_Seen...read failure");
		    Action = S_Send_NAK;
}
#else
		    Action = S_Send_NAK;
#endif
		else if (Ch == ENQ) 
		then
		    Action = S_Send_ACK;
		else
		    {
		    Init_Check (PDB);
		    RCV_Num = Ch - '0';
		    Do_Checksum (Ch, PDB);
		    i = 0;
		    Action = S_Get_Data;
		    }

		break;

	    case S_Get_Check:
		Do_Checksum (ETX, PDB);

		if (Read_Masked_Byte (PDB) == Failure)
		then
#if DEBUG
{
printf("Sending NAK, S_Get_Check read char failure\n");
		    Action = S_Send_NAK;
}
#else
		    Action = S_Send_NAK;
#endif
		else if (Not_Masked && Ch == ENQ)
		then
		    Action = S_Send_ACK;
		else if (PDB->Actual_Check == Check_CRC)
		then
		    Action = S_Get_CRC;
		else
		    Action = S_Verify_CKS;
		break;

	    case S_Get_CRC:
	    	Do_Checksum (Ch, PDB);

		if (Read_Masked_Byte (PDB) == Failure)
		then
#if DEBUG
{
printf("Sending NAK, S_Get_CRC read char failure\n");
		    Action = S_Send_NAK;
}
#else
		    Action = S_Send_NAK;
#endif
		else if (Not_Masked && Ch == ENQ)
		then
		    Action = S_Send_ACK;
		else
		    Action = S_Verify_CRC;

		break;

	    case S_Verify_CRC:
		Do_Checksum (Ch, PDB);
		if (CRC == 0)
		then
		    Action = S_Verify_Packet;
		else
		    {
		    PDB->Error_Code = CRC_Failure;
#if DEBUG
printf("Sending NAK, S_Verify_CRC failure\n");
#endif
		    Action = S_Send_NAK;
		    }

		break;

	    case S_Verify_CKS:
		if (Checksum == Ch)
		then
		    Action = S_Verify_Packet;
		else
		    {
		    PDB->Error_Code = Check_Failure;
		    Action = S_Send_NAK;
#if DEBUG
printf("Sending NAK, S_Verify_CKS\n");
#endif
		    }

		break;

	    case S_Verify_Packet:
		if (RCV_Num == ((R_Seq_Num + 1) mod 10) or R_Buffer[0] == 'F')
		then
		    {
		    PDB->Packets_Btwn_ACKs++;
		    R_Seq_Num = RCV_Num;
		    (*PDB->Ch_Read) (PDB->R_Bytes);
		    (*PDB->Ch_Sent) (PDB->S_Bytes);
		    PDB->R_Bytes = PDB->S_Bytes = 0;
		    if (Send_ACK)
		    then
			BP_Send_ACK (PDB);

		    PDB->R_Buffer_Len = i;
		    PDB->Error_Code = Packet_Arrived;

		    /* recieved packet is considered a failure, since */
		    /* technically, we are awaiting an ACK.  */
		    
		    return Failure;
		    }
		else if (RCV_Num == R_Seq_Num) 
		then
		    /* Ignore (but accept) duplicate packets */
		    Action = S_Send_ACK;
		else
		    {
#if 1
		    PDB->Error_Code = Pkt_Out_of_Seq;
		    Action = S_Send_NAK;
#if DEBUG
printf("Sending NAK, S_Verify_Packet...packet out of sequence\n");
#endif
#else
		    Action = S_Get_DLE;
#endif
		    }

		break;

	    case S_Send_NAK:
		(*PDB->Read_Error) ();
		Errors++;
		Send_Byte (NAK,PDB);
		Action = S_Get_DLE;
		break;

	    case S_Send_ACK:
		BP_Send_ACK (PDB);
		Action = S_Get_DLE;
		break;

	    case S_Send_ENQ:
		if (PDB->Pending_Count > 0) 	   /* Any outstanding packets? */
		    {
		    (*PDB->Send_Error) ();
		    Errors++;
		    Sent_ENQ = TRUE;
		    Send_Byte (ENQ, PDB);
		    Send_Byte (ENQ, PDB);
		    Last_ACK = -1;             /* 8/28/89 RWR */
		    Action = S_Get_DLE;
		    }
		else
		    Action = S_Send_NAK;       /* Send NAK if nothing outstanding */

		break;

	    case S_Resend_Packets:
		if (Resend)
		    {
		    for (i = 0; i < PDB->Pending_Count; i++)
			{
			n = (PDB->Next_Packet + i) mod Send_Ahead_Buffers;

			Xmit_Packet (
			    PDB->Pending[n].Packet_Size, 
			    PDB->Pending[n].Seq_Num,
			    PDB->Pending[n].packet,
			    PDB);
			}
		    }
		else
		    return Failure;

		Action = S_Get_DLE;
		break;
	    }

    return Failure;
    }

/*page*/

private WORD BP_Send_Packet (Size, PDB)
/*
 * Function:
 *	Send the specified packet to the host.
 *
 * Inputs:
 *	Size -- length of the packet
 * 	Sent -- BOOL, TRUE = packet sent, but not yet acknowledged
 *	PDB->Send_Buffer -- the packet to send
 *
 * Outputs:
 *	PDB->Read_Buffer -- the packet received in response to the send
 *
 * Returns:
 *	Success	- Transmitted
 *	Failure - Protocol error, or packet arrived before sending,
 * 		  length is stored in PDB->S_Buffer_Len
 * 		  If a packet is received, (presumably running
 * 		  under transport layer) the sent packet is NOT
 * 		  implied to be sent, therefore, this routine
 * 		  must be called again.
 */
    WORD Size;
    Protocol_Desc_Block *PDB;
    {
    UWORD Next;
    UWORD Next_Seq;
    UBYTE *ptr;
    UWORD i;

    /*
     * first, wait for an available slot in the window.  If a packet
     * arrives before a slot becomes available, return that status
     * immediately without sending the packet (crude mechanism!)
     */

    while ((PDB->Pending_Count > PDB->Window_Size) )
	if (Wait_For_ACK (TRUE, FALSE, TRUE, TRUE, PDB) == Failure) 
	then
	    return Failure;

    if ((ptr = (*PDB->s_alloc) (Size)) == NULL)
    then
	{
	PDB->Error_Code = Memory;
	return Failure;
	}

    Next = (PDB->Next_Packet + PDB->Pending_Count) mod Send_Ahead_Buffers;
    PDB->Pending_Count++;

    Next_Seq = S_Seq_Num = (S_Seq_Num + 1) mod 10;
    PDB->Pending[Next].Seq_Num = Next_Seq;
    PDB->Pending[Next].packet = ptr;
    PDB->Pending[Next].Packet_Size = Size;
    PDB->Packets_Btwn_ACKs = 0;

    Xmit_Packet (Size, Next_Seq, S_Buffer, PDB);

    for (i = 0; i < Size; i++) ptr[i] = S_Buffer[i];

    return Success;
    }
