/*
    file.c

    segment and metafile disk storage and retrieval routines
*/

#include    "defs.h"
#undef      NULL
#include    <stdio.h>

static  int         key_size = sizeof(disk_key_t);
static  disk_key_t  dkey;    /* key buffer for file i/o */

/* key primitive read buffers */
static  point_t   *pt;
static  line_t    *ln;
static  rect_t    *rt;
static  poly_t    *py;
static  oval_t    *ov;
static  text_t    *tx;
static  ref_t     *rf;

extern  seg_t   *top_seg;
extern  seg_t   *cur_seg;

seg_t   *rd_seg(name)
char    *name;
{
seg_t   *seg, *cl_seg(), *inst_seg();
key_t   *read_list();
rect_t  *get_bbox(), *copy_rect();
FILE    *fp;

    /* Does segment already exist? */
    seg = top_seg;
    while(seg != NULL)
        {
        if(seg->name != NULL)
            if(strcmp(seg->name,name) == 0)
                return(seg);

        seg = seg->next;
        }

    fp = fopen(name,"r");
    if(fp == NULL)
        {
        error("Can't open metafile, from read segment");
        return(NULL);
        }

    /* get an empty segment */
    seg = inst_seg(name);
    if(top_seg == NULL)
        {
        top_seg = seg;
        }
    else
        {
        seg->next = top_seg;
        top_seg = seg;
        }

    /* read the metafile */

    read_key(fp);    /* rt now contains bbox */
    seg->bbox = rt;

    seg->data = read_list(fp);

    /* set segment's initial stuff */
    seg->name = name;
    seg->visible = TRUE;
    seg->locked = FALSE;

    fclose(fp);
    return(seg);
}

key_t   *read_metafile(name)
char    *name;
{
key_t   *k;
key_t   *read_list();
FILE    *fp;
char    str[128];

    fp = fopen(name,"r");

    if(fp == NULL)
        {
        sprintf(str,"Can't open metafile %s",name);
        error(str);
        return(NULL);
        }

    read_key(fp);    /* rt now contains bbox */

    k = read_list(fp);
    fclose(fp);

    return(k);
}

rect_t  *mf_bbox(name)
char    *name;
{
FILE    *fp;
char    str[128];

    fp = fopen(name,"r");

    if(fp == NULL)
        {
        sprintf(str,"Can't open metafile %s",name);
        error(str);
        return(NULL);
        }

    read_key(fp);    /* rt now contains bbox */
    fclose(fp);
    return(rt);
}

key_t   *read_list(fp)
FILE    *fp;
{
key_t   *k;
seg_t   seg;

    seg.data = NULL;
    op_seg(&seg);

    while(read_key(fp))
        switch(dkey.type)
            {
            case    POINT    :
                add_point(pt);
                break;
            case    LINE    :
                add_line(ln);
                break;
            case    RECT    :
                add_rect(rt);
                break;
            case    POLY    :
                add_poly(py);
                break;
            case    OVAL    :
                add_oval(ov);
                break;
            case    TEXT    :
                add_text(tx);
                break;
            case    REF    :
                add_ref(rf);
                break;
            default    :
                add_attr(dkey.type);
            }

    cl_seg();
    return(seg.data);
}

static  int    read_key(fp)
FILE    *fp;
{
int     read_val, i;
line_t  *myline, *end_line();
poly_t  *mypoly, *end_poly();
key_t   *read_list();
point_t vpt;

    if((read_val = fread(&dkey,key_size,1,fp)) == NULL)
        return(NULL);
    else
        {
        switch(dkey.type)
            {
            case    POINT    :
                pt = inst_point();
                fread(pt,sizeof(point_t),1,fp);
                break;

            case    LINE    :
                ln = CALLOC(1, line_t);
                if(ln == NULL)
                    error("Can't instantiate line");
                fread(ln,sizeof(line_t),1,fp);
                i = ln->num_points;
                begin_line();
                while(i--)
                    {
                    fread(&vpt, sizeof(point_t), 1, fp);
                    con_point(vpt.x, vpt.y);
                    }
                myline = end_line();
                ln->point = myline->point;
                break;

            case    RECT    :
                rt = inst_rect();
                fread(rt,sizeof(rect_t),1,fp);
                break;

            case    POLY    :
                py = CALLOC(1, poly_t);
                if(py == NULL)
                    error("Can't instantiate poly");
                fread(py,sizeof(poly_t),1,fp);
                i = py->num_points;
                begin_poly();
                while(i--)
                    {
                    fread(&vpt,sizeof(point_t),1,fp);
                    con_point(vpt.x, vpt.y);
                    }
                mypoly = end_poly();
                py->point = mypoly->point;
                break;

            case    OVAL    :
                ov = inst_oval();
                fread(ov, sizeof(oval_t), 1, fp);
                break;

            case    TEXT    :
                tx = inst_text("");
                fread(tx, sizeof(text_t), 1, fp);
                break;

            case    REF        :
                rf = inst_ref("",0,0,NULL);
                fread(rf,sizeof(ref_t),1,fp);
                /*
                    referenced segments are
                        read from disk when needed
                */
                break;

            default    :
                {
                /* must have been an attribute */
                }
            }
        return(read_val);
        }
}

void    wr_seg(seg)
seg_t   *seg;
{
void    write_metafile();

    if(seg->name == NULL)
        {
        error("Can't write an unamed segment");
        return;
        }

    write_metafile(seg->name, seg->data, seg->bbox);
}

void    write_metafile(name,key,bbox)
char    *name;
key_t   *key;
rect_t  *bbox;
{
key_t   *k;
FILE    *fp;
char    str[128];

    fp = fopen(name,"w");

    if(fp == NULL)
        {
        sprintf(str,"Can't create metafile %s",name);
        error(str);
        return;
        }

    /* write bbox */
    dkey.type = RECT;
    dkey.size =    sizeof(rect_t);
    fwrite(&dkey,key_size,1,fp);
    fwrite(bbox,dkey.size,1,fp);

    k = key;
    while(k != NULL)
        {
        write_key(k,fp);
        k = k->next;
        }
    fclose(fp);
}

static  int    write_key(key,fp)
key_t   *key;
FILE    *fp;
{
int    i,j;

    dkey.type = key->type;
    switch(key->type)
        {
        case    POINT    :
            {
            point_t    *p;

            dkey.size = sizeof(point_t);
            fwrite(&dkey,key_size,1,fp);

            p = key->key.point;
            fwrite(p,dkey.size,1,fp);
            break;
            }
        case    LINE    :
            {
            line_t    *l;
            cpoint_t  *cpt;
            point_t   p;

            l = key->key.line;
            dkey.size = sizeof(line_t) + l->num_points * sizeof(point_t);
            fwrite(&dkey,key_size,1,fp);
            fwrite(l,sizeof(line_t),1,fp);
            cpt = l->point;
            while(cpt != NULL)
                {
                p.x = cpt->point.x;
                p.y = cpt->point.y;
                fwrite(&p,sizeof(point_t),1,fp);
                cpt = cpt->next;
                }
            break;
            }
        case    RECT    :
            {
            rect_t    *r;

            dkey.size =    sizeof(rect_t);
            fwrite(&dkey,key_size,1,fp);
            r = key->key.rect;
            fwrite(r,dkey.size,1,fp);
            break;
            }
        case    POLY    :
            {
            poly_t    *pg;
            cpoint_t  *cpt;
            point_t   p;
            pg = key->key.poly;
            dkey.size = sizeof(poly_t) + pg->num_points * sizeof(point_t);
            fwrite(&dkey,key_size,1,fp);
            fwrite(pg,sizeof(poly_t),1,fp);
            cpt = pg->point;
            while(cpt != NULL)
                {
                p.x = cpt->point.x;
                p.y = cpt->point.y;
                fwrite(&p,sizeof(point_t),1,fp);
                cpt = cpt->next;
                }
            break;
            }
        case    OVAL    :
            {
            oval_t    *o;

            dkey.size = sizeof(oval_t);
            fwrite(&dkey,key_size,1,fp);
            o = key->key.oval;
            fwrite(o,dkey.size,1,fp);
            break;
            }
        case    TEXT    :
            {
            text_t    *t;

            dkey.size =    sizeof(text_t);
            fwrite(&dkey,key_size,1,fp);
            t = key->key.text;
            t->bbox = NULL;
            fwrite(t,dkey.size,1,fp);
            break;
            }
        case    REF        :
            {
            ref_t    *r;
            dkey.size = sizeof(ref_t);
            fwrite(&dkey,key_size,1,fp);
            r = key->key.ref;
            r->instance = NULL;
            fwrite(r,dkey.size,1,fp);
            break;
            }
        default    :
            {
            dkey.size = 0;
            fwrite(&dkey,key_size,1,fp);
            }
        }
}
