
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Zap.prg                                                                 *
 * Version 1.0                                                             *
 * Copyright: (c) 1993 by SSG Inc.                                         *
 *                                                                         *
 * by Glenn D. LeCroy, SSG Inc.                                            *
 *                                                                         *
 * Feel free to distribute unaltered program.                              *
 *                                                                         *
 * If you find this program useful, please send $5 U.S. to:                *
 *                                                                         *
 *                   SSG Inc.                                              *
 *                   P.O.Box 814                                           *
 *                   Blairsville, GA     30512                             *
 *                                                                         *
 * Program Function: Zaps dbase files and reindexes                        *
 *                      related indexes files from                         *
 *                      Clipper '.vew' files for clearing                  *
 *                      files after development testing.                   *
 *                   ZAP can also use a script file                        *
 *                      to zap more than one 'view'                        *
 *                      or DBase file at once                              *
 *                                                                         *
 * This code was written and tested in Clipper 5.01                        *
 *                                                                         *
 * Compile: /m/n/w/l                                                       *
 *                                                                         *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

 ** modes for zap
 #define ZAP_NOTHING     0
 #define ZAP_VEW         1
 #define ZAP_DBF         2

 ** error codes
 #define ERR_NOFILE      1
 #define ERR_OPEN        2
 #define ERR_READ        3
 #define ERR_BADFILE     4
 #define ERR_OTHER     100
 #define CAN_CONTINUE   99
 
 ** file open mode
 #define READONLY        0

 ** file status
 #define BAD_FILE        0
 #define GOOD_FILE       1

 memv scripthandle, fromscript
 memv verify

 ** fields in a '.vew' file
 fiel item_name, contents

 **************************
 proc zzap( fname, option )
 **************************

 loca struct, zapwhat, commandstr, gotcommand

 publ scripthandle, fromscript := .f.
 publ verify := .t.

 ** must have filename from command line
 if fname == nil
     ** tell em'
     showhelp()
 endif

 ** check command line option
 if option != nil
     if at( "/A", upper( option ) ) > 0
          ** user does not want to verify 'zap'
          verify := .f.
     endif
 endif

 ** using a script file?
 if left( fname, 1 ) == "@"

     fname := substr( fname, 2 )
     fromscript := .t.

     ** add extension if none
     if at( ".", fname ) < 1
          fname += ".zap"
     endif

     ** low-level open procedure that handles error
     scripthandle := openfile( fname )

 endif

 ** if not reading a script file, this loop will only execute once
 do while .not. fromscript .or. ;
 ( fromscript .and. getline( scripthandle, @fname ) )

	 ** commands can be specified in script file
      if fromscript .and. upper( left( fname, 4 ) ) == "ZAP "

          ** rest of command string
          commandstr := upper( substr( fname, 5 ) )

          ** don't know if it's a command, assume file name
          gotcommand := .f.

          if at( "/A", commandstr ) > 0
               gotcommand := .t.
               ** user does not want to verify 'zap'
               verify := .f.
          endif

		if upper( left( commandstr, 5 ) ) == "BEGIN"
               gotcommand := .t.
			** perform individual commands on file
			docommands()
          endif               

		if upper( left( commandstr, 3 ) ) == "END"
			** ignore 'END' command without 'BEGIN' command
			** 'END' command will be processed by 'docommands()'
			gotcommand := .t.
		endif

          ** if this was a command, dont process as a file name
          if gotcommand
               loop
          endif

      endif

      ** 'filetype()' sets 'zapwhat' to type of file, handles errors
      ** ( i.e. file not found ), overcomes extension problems
      zapwhat := filetype( @fname )

      ** zapwhat == ZAP_NOTHING will do nothing, skip errors
      do case
      case zapwhat == ZAP_DBF
          zapdbf( fname )
      case zapwhat == ZAP_VEW
          zapvew( fname )
      endcase

      if .not. fromscript
          ** one time through
          exit
      endif

 enddo

 endprog()

 **************
 proc endprog()
 **************
 * clean up
 **************

 if fromscript
     fclose( scripthandle )
 endif

 close all

 qout( "" )

 quit

 **********************
 func filetype( fname )
 ***************************************************************
 * determines if <fname> exists and what type of file it is.
 * <fname> is passed by reference ( '@' ) and may be modified.
 * also handles extension problems.
 *
 * 'filetype()' returns:
 *   0 - invalid file or not found
 *   1 - file is a '.vew' type file
 *   2 - file is a '.dbf' type file
 ***************************************************************

 loca struct, retval

 ** has extension?
 if at( ".", fname ) < 1

     ** no extension, assume '.vew'
     fname := fname + ".vew"

     if !file( fname )

          ** no '.vew', assumne '.dbf'
          fname := left( fname, len( fname ) - 4 ) + ".dbf"

          if !file( fname )

               ** assume no extension
               fname := left( fname, len( fname ) - 4 )

          endif

     endif

 endif

 ** see if file is a valid Clipper data file
 if validfile( fname ) == BAD_FILE
     return ZAP_NOTHING
 endif

 ** file OK if we got here, as far as we can tell

 ** if no extension, add a '.' to file name so 'use'
 ** command wont assume '.dbf'
 fname := fname + if( at( ".", fname ) > 0, "", "." )

 ** determine type of file
 ** compare structure with structure of a '.vew' file
 ** ( only way to tell if we allow non-standard extensions )
 close all
 use ( fname )
 struct := dbstruct()
 close all

 if struct_compare( struct, ;
                    {    {    "ITEM_NAME", "C", 10, 0  }, ;
                         {    "CONTENTS",  "C", 10, 0  }    }    )

     ** yes, its a view file
     retval := ZAP_VEW

 else

     ** assume a dbase file
     retval := ZAP_DBF

 endif

 return retval

 **************************************
 proc error( code, filename, OkToCont )
 **************************************

 do case
 case code == ERR_NOFILE
     qout( "File Not Found!" + ": " + filename )

 case code == ERR_OPEN
     qout( "Cannot Open: " + filename )

 case code == ERR_READ
     qout( "File Read Error" )

 case code == ERR_BADFILE
	qout( "Incompatible File: " + filename )

 endcase

 ** can we continue with execution?
 if OkToCont == nil
      ** No!
      endprog()
 endif

 *********************************
 func struct_compare( arr1, arr2 )
 *********************************
 * used to determine if
 * two database structures
 * are identical. Params
 * are MD array standard for
 * Clipper file structures.
 *********************************

 loca sub1, sub2, equal := .t.

 ** we know they aren't equal if different number of fields
 if len( arr1 ) != len( arr2 )
     return .f.
 endif

 ** for each field
 for sub1 := 1 to len( arr1 )

     ** for name, type, length and decimals of field
     for sub2 := 1 to 4
          if arr1[ sub1 ][ sub2 ] != arr2[ sub1 ][ sub2 ]
               ** not alike!
               equal := .f.
               exit
          endif
     next sub2

     ** don't look anymore if we've found a difference
     if !equal
          exit
     endif

 next sub1

 return equal


 *****************************
 proc zapdbf( fname, indexes )
 **************************************
 * zaps clipper database file <fname>
 * and reindexes <indexes> if an array
 * of index file names are passed
 **************************************

 loca count, key

 if verify
     qout( "Zap " + alltrim( fname ) + "? (Y/N): " )
     key := inkey( 0 )
     ** is it 'Y' or 'y'?
     if key != 121 .and. key != 89
          return
     endif
 endif

 qout( "Zapping " + fname )
 close all
 use ( fname )
 zap

 ** do we have indexes?
 if indexes != nil

     for count := 1 to len( indexes )

          ** check for existence of as-is file name
          if ! file( indexes[ count ] )

               ** didn't find it

               ** if no extension, assume '.ntx'
               if at( ".", indexes[ count ] ) < 1
                    indexes[ count ] += ".ntx"
               endif

          endif

          ** if it's not there now, it's an error!
          if ! file( indexes[ count ] )

               error( ERR_NOFILE, indexes[ count ], CAN_CONTINUE )

          else

               ** open the index
               dbsetindex( indexes[ count ] )

               qout( "   Indexing " + indexes[ count ] )

               ** reindex it
               dbreindex()

               ** close it
               close indexes

          endif

     next count

 endif

 close all

 *********************
 proc zapvew( vewnam )
 *****************************************************
 * reads the filenames from the view file 'vewnam' and
 * calls zapdbf for each set of '.dbf'-'.ntx' files
 *****************************************************

 loca dbfs[ 0 ], ntxs[ 0 ][ 0 ]
 loca dbfnumber

 ** read the view file into the arrays
 readvew( vewnam, dbfs, ntxs )

 ** zap em'
 for dbfnumber := 1 to len( dbfs )
	zapdbf( dbfs[ dbfnumber ], ntxs[ dbfnumber ] )
 next dbfnumber


 ***********************************
 proc readvew( vewname, dbfs, ntxs )
 ************************************************************************
 * reads the contents of a '.vew' type file ( as created by Clipper's
 * dbu.exe ).
 *
 * A '.vew' file is simply a standard Dbase type file with the structure:
 *
 *   field name     type       length   decimals
 *             
 *   ITEM_NAME      Character      10          0
 *   CONTENTS       Character      10          0
 *
 * The first few records in the file contain directory information, number
 * of files, and filter conditions, which do not concern this program.
 *
 * As soon as we find the first record which meets the condition:
 * left( ITEM_NAME, 3 ) == "dbf", each '.dbf' file in the view is
 * stored in the following manner:
 *
 *   ITEM_NAME = "dbf#      " where '#' is a number ( 1 for the first,
 *        2 for the second, and so on )
 *   CONTENTS  = the file name - NOTE: if any file name is longer than
 *        10 characters, the remaining characters are stored in the
 *        CONTENTS field into the following record(s) with the ITEM_NAME
 *        field empty.
 *
 * This method continues in the file until all the view's '.dbf's are
 * read.
 *
 * Then each '.ntx' filename is stored in a record:
 *   ITEM_NAME = "ntx#      " where # is which '.dbf' this file 'goes with'.
 *   CONTENTS  = the file name - the convention for an oversize file name
 *        is the same as above for the '.dbf' file name.
 *
 * Example:
 *   Say you had a view setup with the following database-indexes
 *   and you saved the view as 'my.vew':
 *
 *   DBFs:       PROD.DBF          CUSTOMER.DBF      C:\SOMEDIR\INVOICE.DBF
 *                                                           
 *   NTXs:       CODE.NTX          CUSTCODE.NTX      C:\SOMEDIR\INVNUMB.NTX
 *               DESC.NTX          CUSTNAME.NTX
 *               DEPT.NTX
 *
 *   If you 'browsed' the database file 'my.vew', you would see
 *
 *          ITEM_NAME     CONTENTS
 *        
 *                     .
 *                     .
 *              first few records
 *                     .
 *                     .
 *          dbf1         PROD.DBF
 *          dbf2         CUSTOMER.D
 *                       BF
 *          dbf3         C:\SOMEDIR
 *                       \INVOICE.D
 *                       BF
 *          ntx1         CODE.NTX
 *          ntx1         DESC.NTX
 *          ntx1         DEPT.NTX
 *          ntx2         CUSTCODE.N
 *                       TX
 *          ntx2         CUSTNAME.N
 *                       TX
 *          ntx3         C:\SOMEDIR
 *                       \INVNUMB.N
 *                       TX
 *
 *  ( then other records might follow, which doesn't concern us )
 *
 ************************************************************************

 loca dbfnumber, wholename

 qout( "Reading View File: " + vewname )

 close all
 use ( vewname )

 ** do I have to do this? - makes me feel better
 go top

 ** find the first '.dbf' file name
 do while left( item_name, 3 ) != "dbf" .and. !eof()
     skip 1
 enddo

 ** load array 'dbfs' with dbase file names from '.vew' file
 do while left( item_name, 3 ) == "dbf" .and. !eof()

     wholename := alltrim( contents )

     skip 1

     ** append parts of file name that 'overflowed' into
     ** following records, if any
     do while empty( item_name ) .and. !eof()

          wholename += alltrim( contents )
          skip 1

     enddo

     ** add the file name to the '.dbf' array
     aadd( dbfs, wholename )

 enddo

 ** I don't really know if this is necessary, but it's safe!
 ** skip records until "ntx" is found
 do while left( item_name, 3 ) != "ntx" .and. !eof()
     skip 1
 enddo

 ** load array 'ntxs' with corresponding '.ntx' names
 dbfnumber := 1

 ** for each '.dbf' read above
 do while dbfnumber <= len( dbfs )

     ** start each index sub-array empty
     aadd( ntxs, {} )

     ** as long as indexes go with database file #dbfnumber
     do while item_name == ;
          padr( "ntx" + alltrim( str( dbfnumber ) ), 10 ) .and. !eof()

          wholename := alltrim( contents )
          skip 1

          ** append parts of file name that 'overflowed' into
          ** following records, if any
          do while empty( item_name ) .and. !eof()

               wholename += alltrim( contents )
               skip 1

          enddo

          ** add to sub array
          aadd( ntxs[ dbfnumber ], wholename )

     enddo

     ** do for the next '.dbf'
     dbfnumber ++

 enddo

 *******************
 func openfile( fn )
 ************************************
 * opens lowlevel file, handles error
 * returns DOS file handle
 ************************************

 loca lhandle

 lhandle := fopen( fn, READONLY )

 if lhandle < 0
     ** can't go on!
     error( ERR_OPEN, fn )
 endif

 return lhandle

 ****************************
 func getline( lhandle, str )
 ********************************************************
 * used to read the script file ( if used ).
 * skips over the CR and/or LF at the current position in
 * the low-level file, then reads characters into 'str'
 * until a CR and/or LF or EOF is encountered
 * NOTE: str is passed by reference ( '@' ).
 *
 * returns .t. if a valid string has been read
 * returns .f. if not, or when EOF reached
 ********************************************************

 loca retval := .t.
 loca buff := space( 1 )
 loca result
 loca crlf := chr( 13 ) + chr( 10 )

 ** skip over the CR's and LF's
 do while .t.

     ** read a byte
     result := fread( lhandle, @buff, 1 )

     if result != 1
          ** something's wrong
          str := ""
          return .f.
     endif

     ** not a CR or LF?
     if at( left( buff, 1 ), crlf ) < 1
          ** Ok, we're to the string data
          exit
     endif

 enddo
 str := ""

 ** now, read data
 do while .t.

     ** append the lastest read character to 'str'
     str += left( buff, 1 )

     ** read another
     result := fread( lhandle, @buff, 1 )

     if result != 1

          ** something's wrong

          ** do we have any string data so far?
          if str == ""
               ** no, we're through reading the script file
               retval := .f.
          endif

          exit

     endif

     ** did we hit a CR or LF?
     if at( left( buff, 1 ), crlf ) > 0
          ** yes!, don't read anymore now
          exit
     endif

 enddo

 return retval


 **********************
 func validfile( name )
 ******************************************************
 * returns .t. if <name> exists and is probably a valid
 * Clipper data file, returns .f. otherwise
 ******************************************************

 loca lhandle, buff := " ", result

 ** file must exist
 if !file( name )
     error( ERR_NOFILE, name, CAN_CONTINUE )
     return BAD_FILE
 endif

 ** a '.dbf' or a '.vew' will have either an 03 hex or an 83 hex
 ** as the first byte in the file
 lhandle := openfile( name )

 ** read the first byte
 result := fread( lhandle, @buff, 1 )

 fclose( lhandle )

 if result != 1
      ** read error, bad file
      error( ERR_READ,, CAN_CONTINUE )
      return BAD_FILE
 endif

 ** read OK                                    83 hex Ŀ
 if left( buff, 1 ) == chr( 3 ) .or. left( buff, 1 ) == ""
     ** file OK
     return GOOD_FILE
 endif

 ** else, bad file
 error( ERR_BADFILE, name, CAN_CONTINUE )

 return BAD_FILE


 *****************
 proc docommands()
 ****************************************************************
 * Called if a 'ZAP BEGIN' command is read from the script file
 * The next line read from the script file is expected to be
 * a '.vew' or '.dbf' file name.
 * If the file is a view file, this function reads commands from
 * the script file and processes them on the FIRST '.dbf' file
 * listed in the view file. Otherwise, fname is assumed to be a
 * '.dbf' file and commands are performed on that file. Commands
 * are executed until the 'ZAP END' command or the end of the
 * script file is found.
 ****************************************************************

 loca command, ftype, fname, dbfs[ 0 ], ntxs[ 0 ][ 0 ], ntxnum
 loca fldname, value, struct

 ** 'getline()' returns .f. at end-of-file
 do while getline( scripthandle, @command )

	command := alltrim( command )

	if upper( left( command, 7 ) ) == "ZAP END"
		** done processing commands
		exit
	endif

	if fname == nil

		** don't have a filename yet
		fname := command

		** what type?
		ftype := filetype( @fname )

		if ftype == ZAP_NOTHING

			** invalid file
			** ignore commands until script-EOF or 'ZAP END' command
			do while getline( scripthandle, @command )
				if upper( left( command, 7 ) ) == "ZAP END"
					exit
				endif
			enddo

			return

		endif

		close all

		** is it a '.dbf' type file
		if ftype == ZAP_DBF
			** yes
			use ( fname )
		else

			** no, assume '.vew' type file

			** fill arrays with file names from '.vew'
			readvew( fname, dbfs, ntxs )

			** view has at least one '.dbf' file?
			if len( dbfs ) < 1

				** no '.dbf'
				** ignore commands until script-EOF or 'ZAP END' command
				do while getline( scripthandle, @command )
					if upper( left( command, 7 ) ) == "ZAP END"
						exit
					endif
				enddo

				**
				return

			endif

			** open the database file
			use ( dbfs[ 1 ] )

			** open indexes, if any
			for ntxnum := 1 to len( ntxs[ 1 ] )
				dbsetindex( ntxs[ 1 ][ ntxnum ] )
			next ntxnum

		endif

		** store the structure
		struct := dbstruct()

	else
		** we already have a file name
		** attempt to perform commands
		** ( about any command could be added here )

		do case
		case upper( left( command, 12 ) ) == "APPEND BLANK"
			append blank

		case upper( left( command, 7 ) ) == "REPLACE"

			** get the elements from the command
			fldname := extract( command, 2 )
			value := extract( command, 4 )

			** checks for nils, existence - force correct type
			if replaceOK( fldname, @value, struct )

				** everythings OK
				replace &fldname. with value

			endif

		endcase

	endif

 enddo

 close all


 ****************************************
 func replaceOK( fldname, value, struct )
 ***********************************************
 * checks that parameters have been provided,
 * then that the field name is in the structure,
 * then matches the type for a replace
 ***********************************************

 loca vtype, count, lfound

 if fldname == nil
	qout( "No field name specified for replace." )
	return .f.
 endif

 lfound := .f.
 ** search structure for field name
 for count := 1 to len( struct )
	if upper( struct[ count ][ 1 ] ) == upper( fldname )
		lfound := .t.
		exit
	endif
 next count

 if ! lfound
	qout( "Field: " + fldname + " does not exist!" )
	return .f.
 endif

 if value == nil
	qout( "No value specified for replace." )
	return .f.
 endif

 vtype := valtype( &fldname. )

 if vtype != "C"
	** convert to correct type
	** data from script will be type="C"
	do case
	case vtype == "D"
		value := ctod( value )
	case vtype == "N"
		value := val( value )
	case vtype == "L"
		value := if( upper( value ) == ".F.", .f., .t. )
	case vtype == "M"
		** don't change, but avoid 'otherwise' below.
	otherwise
		value := ""
		** cannot do replace
		qout( "Cannot replace: " + fldname )
		return .f.
	endcase
 endif

 return .t.


 ********************************
 func extract( string, position )
 ***********************************
 * returns the <position>'th word
 * from <string>, or nil on failure.
 ***********************************

 loca retval, slen, count
 loca word := 1, pos := 1

 slen := len( string )

 do while word < position

	** go to next space
	do while substr( string, pos, 1 ) != " " .and. pos <= slen
		pos ++
	enddo

	** skip spaces
	do while substr( string, pos, 1 ) == " " .and. pos <= slen
		pos ++
	enddo

	** specified word is not there
	if pos > slen
		exit
	endif

	** now we're at the next word
	word ++

 enddo

 if pos <= slen
	retval := substr( string, pos )
	** trunicate
	if left( retval, 1 ) == chr( 34 )
		** starts with quote
		retval := substr( retval, 2 )
		count := at( chr( 34 ), retval )
	else
		** normal
		count := at( " ", retval )
	endif
	if count > 0
		retval := left( retval, count - 1 )
	endif
 endif

 return retval

 ***************
 proc showhelp()
 *****************************************
 * called if filename not specified to the
 * program from DOS
 *****************************************

 loca s2 := space( 2 )

 qout( "ZAP - Zaps Clipper Data Files and Updates Indexes after Development Testing" )
 qout( replicate( "", 75 ) )
 qout( "Usage: ZAP <filename> [option...]" )
 qout( s2 + "where <filename> can be a '.vew' file," )
 qout( s2 + "a '.dbf' file, or a Script file" )
 qout( "" )
 qout( "To use a script file, precede <filename> with '@'. If no extension " )
 qout( "is supplied, '.zap' is assumed. The script file may contain names of" )
 qout( "'.vew' and '.dbf' files with or without extensions." )
 qout( "Commands may be included in the script file ( see below )." )
 qout( "" )
 qout( "For each filename in the script or the filename on the command line:" )
 qout( "If no extension is provided, <filename> is assumed to be a '.vew' file. If" )
 qout( "a '.vew' file cannot be found, then the file is assumed to be a '.dbf' file." )
 qout( "If a '.dbf' file cannot be found, then the filename is used as-is. If this" )
 qout( "file cannot be found, an error is reported. If the file is determined to be" )
 qout( "a '.dbf' file, the file is simply zapped. If the file is a '.vew' file," )
 qout( "'.dbfs' are zapped and '.ntx' files are reindexed to reflect each one's" )
 qout( "corresponding '.dbf'." )
 qout( "" )
 qout( "[option] can be:" )
 qout( s2 + "/a - automatic ( don't confirm before 'Zap'ping )" )
 qout( "" )
 qout( "Press any key for more..." )
 inkey( 0 )
 qout( "" )
 qout( "Commands can be included in the script file, for example:" )
 qout( "ZAP /A - will not ask for verification for all zapping that follows" )
 qout( "this command." )
 qout( "" )
 qout( "A few other commands can be performed on a data-index set as follows:" )
 qout( "ZAP BEGIN" )
 qout( "  MYFILE[.EXT]   -    ( the '.dbf' or '.vew' file to act upon )" )
 qout( "  ( Command )" )
 qout( "  ( Command )" )
 qout( "ZAP END" )
 qout( "" )
 qout( "Command above could be:" )
 qout( "  APPEND BLANK - appends a blank record" )
 qout( "  REPLACE <fieldname> WITH <value>" )
 qout( "" )
 qout( "The ZAP BEGIN-END command structure is provided because you might not" )
 qout( "wan't a file totally cleared before distributing to clients. Files sent" )
 qout( "to clients may need to have one or more default records." )
 qout( "" )
 qout( "This utility allows you to test your code, using data files as necessary," )
 qout( "and then, at any stage of development, prepare your data files for" )
 qout( "distribution in a snap!" )
 qout( "" )
 quit

 *** zap.prg EOF ***
