*-------------------------------------------------------------------------
* xms.prg -- Routines for an XMS run-time library.
*-------------------------------------------------------------------------
#include xms.hdr
#include defs.hdr
#include undoc.hdr
#include common.hdr

#pragma W_FUNC_PROC-
*-------------------------------------------------------------------------
VARDEF PRIVATE
   CHAR            RCSid = "$Header: C:\TECH\FAQ\XMS\RCS\xms.prg 0.12 1992/05/20 07:10:30 holmesda Exp holmesda $"
   LONG            driver_address = &NULL

   *---
   * The following variables fake a structure needed by xms_copy()
   *---
   LONG            Copy_Length      && 32-bit number of bytes to transfer
   UINT            SourceHandle     && Handle of source block
   LONG            SourceOffset     && 32-bit offset into source
   UINT            DestHandle       && Handle of destination block
   LONG            DestOffset       && 32-bit offset into destination block

ENDDEF
*-------------------------------------------------------------------------
* function: xms_installed
*
* description: xms_installed() returns TRUE if an XMS driver is present,
*     and FALSE if otherwise.
*-------------------------------------------------------------------------
FUNCTION LOGICAL xms_installed

   VARDEF
      UINT  ax,bx,cx,dx,ds,si,es,di,bp
   ENDDEF

   ax = 0x4300       && Service to get the driver from INT 2Fh
   interrupt( 0x2f, ax, bx, cx, dx, si, di, bp, ds, es )

   if and( ax, 0x80) = 0x80
      return &TRUE
   else
      return &FALSE
   endif

ENDPRO
*-------------------------------------------------------------------------
* function: xms_get_driver
*
* description: get_xms_driver() returns the address of the XMS driver.
*     If the private variable driver_address has not been initalized,
*     it calls the undocumented function INTERRUPT to get it.
*-------------------------------------------------------------------------
FUNCTION LONG xms_get_driver

   VARDEF
      UINT  ax,bx,cx,dx,ds,si,es,di,bp
      UINT  x_d_a[2] based driver_address
   ENDDEF

   *---
   * If we don't already have the driver's address, go fetch it from
   * INT 2fh.  It'll come back as ES:BX (a double word).  We'll stuff
   * that into the private variable ``driver_address'' by using an
   * array called x_d_a of two UINTS based at the same address.
   *---
   if driver_address = &NULL
      if .not. xms_installed()
         return &X_NO_DRIVER
      else
         ax = &XMS_GETDRIVER
         interrupt( 0x2f, ax, bx, cx, dx, si, di, bp, ds, es)
         x_d_a[0] = bx
         x_d_a[1] = es
      endif
   endif

   return( driver_address )

ENDPRO
*-------------------------------------------------------------------------
* function: xms_get_version()
*
* description:  This function returns the version of the XMS driver.  Note
*    that the driver will return its version in binary-coded-decimal (BCD)
*    so we should convert it to straight decimal to return it.  The
*    easiest way to do that is to chop the returned value into two bytes
*    (via the vers[2] array), so that 0x0200 (512 in decimal) because
*    02 and 00.  Then we can return them in an orderly fashion.
*
* Note: Since most of these functions only work with drivers that meet
*     the XMS 2.0 specification, you should check this function and
*     abort if the version is < 200.
*-------------------------------------------------------------------------
FUNCTION UINT xms_get_version

   VARDEF
      UINT  ax,bx,cx,dx,ds,si,es,di,bp
   ENDDEF

   ax = &XMS_GETVERSION       && Subfunction to get version information
   call_driver( xms_get_driver(), ax, bx, cx, dx, si, di, bp, ds, es)
   xms_set_error( ax, bx )

   return hibyte(ax) * 100 + lobyte(ax)

ENDPRO
*-------------------------------------------------------------------------
* function: hma_alloc()
*
* description:  Have the driver assign the High Memory Area to us.
*    Returns TRUE if the driver do so, FALSE otherwise.
*
* Note:  If you use this and your program is a TSR or device driver,
*    add a parameter so that you may pass DX as the space needed in bytes
*    instead of 0xFFFF.
*-------------------------------------------------------------------------
FUNCTION LOGICAL hma_alloc

   VARDEF
      UINT  ax,bx,cx,dx,ds,si,es,di,bp
   ENDDEF

   ax = &XMS_GET_HMA
   dx = 0xFFFF
   call_driver( xms_get_driver(), ax, bx, cx, dx, si, di, bp, ds, es)
   xms_set_error( ax, bx )

   if ax = 1
      return &TRUE
   endif

   return &FALSE

ENDPRO
*-------------------------------------------------------------------------
* function: hma_free()
*
* description: Release control of the HMA.
*-------------------------------------------------------------------------
PROCEDURE hma_free

   VARDEF
      UINT  ax,bx,cx,dx,ds,si,es,di,bp
   ENDDEF

   ax = &XMS_FREE_HMA
   call_driver( xms_get_driver(), ax, bx, cx, dx, si, di, bp, ds, es)
   xms_set_error( ax, bx )

ENDPRO
*-------------------------------------------------------------------------
* function: xms_avail()
*
* description: xms_avail() returns the number of kBytes available in
*     extended memory.  Note that since we return it as an int, this
*     function will return a negative value if you have between 32M
*     and 64M of memory.  It will fail altogether if you have more than
*     64M.  However, most 386 & 486's can access only 16M.
*-------------------------------------------------------------------------
FUNCTION INT xms_avail

   VARDEF
      UINT  ax,bx,cx,dx,ds,si,es,di,bp
   ENDDEF

   ax = &XMS_MEM_AVAIL
   call_driver( xms_get_driver(), ax, bx, cx, dx, si, di, bp, ds, es)
   xms_set_error( ax, bx )

   return dx

ENDPRO
*-------------------------------------------------------------------------
* function: xms_alloc()
*
* description: xms_alloc() allocates a block of XMS memory, of the given
*     length.  xms_alloc() will return TRUE if the driver was able to
*     allocate the block, and the block's handle will be returned through
*     the referenced first parameter.  If the driver was NOT able to
*     allocate the block, the error handler is notified, and FALSE is
*     returned.
*-------------------------------------------------------------------------
FUNCTION LOGICAL xms_alloc
   PARAMETERS INT handle, VALUE INT k_bytes

   VARDEF
      UINT  ax,bx,cx,dx,ds,si,es,di,bp
   ENDDEF

   ax = &XMS_ALLOC_MEM
   dx = k_bytes

   call_driver( xms_get_driver(), ax, bx, cx, dx, si, di, bp, ds, es)
   xms_set_error( ax, bx )

   if xms_get_error() <> 0
      handle = &NULL
      return &FALSE
   else
      handle = dx
   endif

   return &TRUE

ENDPRO
*-------------------------------------------------------------------------
* function: xms_free()
*
* description: xms_free() free's an allocated handle.  The memory is
*     returned to the XMS pool.
*-------------------------------------------------------------------------
PROCEDURE xms_free
   PARAMETERS INT handle

   VARDEF
      UINT  ax,bx,cx,dx,ds,si,es,di,bp
   ENDDEF

   ax = &XMS_FREE_MEM
   dx = handle
   call_driver( xms_get_driver(), ax, bx, cx, dx, si, di, bp, ds, es)
   xms_set_error( ax, bx )

ENDPRO
*-------------------------------------------------------------------------
* function: xms_copy()
*
* description: xms_copy() is the most complex of these XMS wrapper
*     functions, because the driver wants a pointer to a structure of
*     information.  What we'll have to do is fake that structure, and
*     return a pointer to the beginning of it.  Since we only need
*     the structure for the driver call, we'll just use some private
*     variables.  Note that we use the undocumented function make_ptr()
*     to get the segment:offset pointer of the beginning of the structure.
*-------------------------------------------------------------------------
FUNCTION LOGICAL xms_copy
   PARAMETERS VALUE INT  s_handle,  VALUE LONG s_offset, ;
              VALUE INT  d_handle,  VALUE LONG d_offset, ;
              VALUE LONG cp_len

   VARDEF
      UINT  ax,bx,cx,dx,ds,si,es,di,bp
   ENDDEF

   *---
   * Quick note: xms_copy() will fail if the length is not even!
   *---
   if cp_len % 2 = 1
      cp_len = cp_len + 1
   endif

   Copy_Length    = cp_len
   SourceHandle   = s_handle
   SourceOffset   = s_offset
   DestHandle     = d_handle
   DestOffset     = d_offset

   make_ptr( ds, si, Copy_Length )

   ax = &XMS_MOVE_MEM
   call_driver( xms_get_driver(), ax, bx, cx, dx, si, di, bp, ds, es)
   xms_set_error( ax, bx )

   if xms_get_error() <> 0
      return &FALSE
   endif

   return &TRUE

ENDPRO
*-------------------------------------------------------------------------
* function: xms_lock()
*
* description: xms_lock() locks an XMS handle so that the user can treat
*     the XMS block as a regular memory address.  Thus, xms_lock()
*     returns a pointer, er, I mean a LONG value.  If xms_lock() fails,
*     the valued returned is 0000:0000, or &NULL, or just plain 0.
*-------------------------------------------------------------------------
FUNCTION &POINTER xms_lock
   PARAMETERS INT handle

   VARDEF
      UINT       ax,bx,cx,dx,ds,si,es,di,bp
      &POINTER   ptr
      UINT       hilo[2] BASED ptr
   ENDDEF

   ax = &XMS_LOCK_HND
   dx = handle
   call_driver( xms_get_driver(), ax, bx, cx, dx, si, di, bp, ds, es)
   xms_set_error( ax,bx )
   if xms_get_error() <> 0
      ptr = 0
   else
      hilo[0] = dx
      hilo[1] = bx
   endif

   return ptr

ENDPRO
*-------------------------------------------------------------------------
* function: xms_unlock()
*
* description: xms_unlock() tells the driver to unlock the handle.  Once
*     the handle is unlocked, it's address is no longer necessarily valid,
*     since unlocked blocks can move around.  The driver keeps a count of
*     how many times the block has been locked, so xms_unlock() returns
*     the lock count for the handle.
*-------------------------------------------------------------------------
FUNCTION INT xms_unlock
   PARAMETERS INT handle

   VARDEF
      UINT  ax,bx,cx,dx,ds,si,es,di,bp
   ENDDEF

   ax = &XMS_UNLOCK_HND
   dx = handle
   call_driver( xms_get_driver(), ax, bx, cx, dx, si, di, bp, ds, es)
   xms_set_error( ax, bx )
   if xms_get_error() = 0
      if ax = 1               && Is it unlocked?
         return 0
      else
         ax = &XMS_HND_INFO
         dx = handle
         call_driver( xms_get_driver(), ax, bx, cx, dx, si, di, bp, ds, es)
         return loword( bx )
      endif
   endif

   return -1

ENDPRO
*-- EOF: xms.prg ---------------------------------------------------------
