/******************************************************************************
*
* FUNCTION:   WC_Cmp -- Wild Card String Compare.
*
* DESCRIPTION:  Given a string containing wild cards interspersed with 
* literal text, and a literal string, this function will compare the wild
* card augmented "Model" string against the literal string, and return True
* if the match succeeds.  If the Model String is Empty, then by convention
* this function will return True.  
*
* PARAMETERS  I/O T Purpose
* ----------  --- - -------
* Model_str    I  C String of wild cards interspersed with literal text to
*                   compare against the contents of "Compare_str".  The 
*                   following conventions for wild cards are followed:
*                          
*                   "*"     stands for any group of characters.  For example:
*                   W*r     gives fields beginning with "W", ending with "r"
*                   *w*r    gives fields having "w" before last character "r"
*
*                   "?"     stands for any single character.  For example:
*                   J?n?s   gives Janes, Janus, Jones, and so on.
*
* Compare_str  I  C Literal String to compare against "Model_str".  
*
*
* EXAMPLE USE IN CLIPPER                         RESULT
* ----------------------                         ------
* _wc_cmp( "Fi?sewa*rma?", "Fitswatterman" )     True
*
* LOCATE FOR _wc_cmp( "Jo*Lennon", DBF->Name )   Searches for record with
*                                                "Name" field containing a
*                                                name starting with "Jo" and
*                                                ending with "Lennon".  You
*                                                would probably want to use
*                                                TRIM( DBF->Name ).
*
* MODIFICATION HISTORY
* --------------------
* 07/24/89  PAT WEISSER    Converted from Clipper to C.
*
******************************************************************************/

#include <stdio.h>
#include <D:\Clipper\Extend\Nandef.H>
#include <D:\Clipper\Extend\Extend.H>

#define TRUE  1
#define FALSE 0

extern char *_parc();

void wc_cmp()

{
   char              *model_str, *compare_str;

   char              *ndx_model_str, *ndx_compare_str;
   char              *char_ptr_after_last_wild_card;


   model_str   = _parc( 1 );
   compare_str = _parc( 2 );

   if( ( *model_str == 0 ) && ( *compare_str == 0 ) )
   {
     /*  Both strings are empty, so they match:  */

      _retl( TRUE );

      return;
   }


   ndx_model_str   = model_str;
   ndx_compare_str = compare_str;

   char_ptr_after_last_wild_card = model_str;


   while( *ndx_model_str )
   {
      
      switch( *ndx_model_str )
      {                     
         case '?':

            /* Single character wild-card match, Advance one character in each buffer: */

            ++ndx_model_str;
            ++ndx_compare_str;

            break;


         case '*':

            /* Must find out what the next non-wild card character is in "Model_str"
             * and advance the "ndx_model_str" to that character position:  */

            while( ( ( *ndx_model_str == '*' ) ||
                     ( *ndx_model_str == '?' ) ) &&
                     ( ndx_model_str )
                 )
            {
               ++ndx_model_str;
            }


            if( !( *ndx_model_str ) )
            {

               /* Then we've exhausted the model string, and it ended with a wild
                * card sequence allowing for universal matching.  Because of this we
                * don't care what characters remain in the Comparison String, the
                * match has succeeded.  */

               _retl( TRUE );

               return;
            }


            /* At this point we have found the next non-wild card character in the
             * Model String.  We must search forward in the Comparison string until
             * we find that character.  If we reach the end of the Comparison String
             * before finding the character, the match will fail.  */

            char_ptr_after_last_wild_card = ndx_model_str;

            if( !find_next_matching_compare_char( *ndx_model_str, &ndx_model_str, &ndx_compare_str ) )
            {
               _retl( FALSE );

               return;
            }

            break;


         default:

            /* We have encountered a literal (non-Wild Card) character in the Model
             * string.  We must be sure that it matches the current character in
             * the Comparison String, and advance both pointers if they do:  */

            if( *ndx_model_str == *ndx_compare_str )
            {
               ++ndx_model_str;
               ++ndx_compare_str;
            }
            else if( char_ptr_after_last_wild_card == model_str )
            {
               /* Match has failed since two corresponding literal characters 
                  did not match.  */

               _retl( 0 );

               return;
            }
            else
            {
               /* Reset Model Ptr back to the first character after the last 
                  Wild Card and look for the Next matching character in the 
                  Comparison String:  */

               ndx_model_str = char_ptr_after_last_wild_card;

               if( !find_next_matching_compare_char( *ndx_model_str, &ndx_model_str, &ndx_compare_str ) )
               {
                  _retl( 0 );

                  return;
               }

            }

      }

   }

   /* At this point, all characters from the Model String have compared favorably
      against the Compare String, and the Model String ended on a NON-Wild Card
      character.  Because of this, we must have also reached the end of the 
      Compare String if the entire compare is to succeed.  If we are not at 
      the end of the Compare String, the compare has failed. */

   if( !( *ndx_compare_str ) )
   {  
      /* Then we also exhausted the Comparison String.  The Compare succeeded.  */

      _retl( 1 );

      return;
   }
   else
   {
      _retl( 0 );

      return;
   }

}





/******************************************************************************
*
* FUNCTION:   Find_Next_Matching_Compare_Char
*
* DESCRIPTION:  This function is not autonomous, it is subservient to the
* "WC_str_Cmp()"  function.  It returns true if the specified character
* is found in what remains to be traversed of the Comparison String.  It also
* has the side effect of incrementing the Model string and Compare String
* pointers.
*
* PARAMETERS     I/O T Purpose
* ----------     --- - -------
* Model_Char      I  C Model String character to look for the next occurence
*                      of in the Comparison string.
* Ndx_Model_Str   B  S Index into the model string.
* Ndx_Compare_Str B  S Index into the compare string.
*
* MODIFICATION HISTORY
* 07/15/89     PAT WEISSER    Created.
*
******************************************************************************/
find_next_matching_compare_char( model_char, ndx_model_str, ndx_compare_str )

char    model_char;
char    **ndx_model_str;
char    **ndx_compare_str;

{

   while( ( **ndx_compare_str ) && ( ( **ndx_compare_str ) != model_char ) )
   {
      ++( *ndx_compare_str );
   }


   if( !( **ndx_compare_str ) )
   {
      /* Then we've exhausted the Comparison String without finding the
       * required match for the current character of the Model String.  The
       * match has failed. */

      return( FALSE );
   }

   /* At this point we have updated the Comparison String pointer, and the
    * calling environment is ready to continue the comparison starting with the 
    * next characters of each buffer:  */

   ++( *ndx_model_str );
   ++( *ndx_compare_str );

   return( TRUE );

}


