/*
  Report to Array Demo
  Shows how to print to an array, that can then be directed to a 
  user selected output device - Printer, Screen, or File.
  Saved Report files can be read in from disk and view/edited/printed/saved.
  
  Based on the Clipper Advisor article for July-Aug 94.
  
  Modified to add a menu for the print preview screen, and using the
  Frankie Mouse LIB functions to allow navigation and editing of the
  print array.

  David L. Smith CIS 72351,225
  Compile with MakeDemo.Bat
  Uses:      
             Clipper 5.2d
             Blinker
             Frankie Mouse
*/

//Include Files
#include "Inkey.ch"
#include "common.ch"
#include "Setcurs.ch"
#include "Frankie.ch"

// Report Definitions:
#define LEFT_MARG   Space(1)             //Report Left Margin
#define PAGE_LEN    60					 //Report Page Length
#define PAGE_WID    78					 //Report Page Width
#define HEAD_SKIP    1					 //Report Header blank lines
#define PRINT_2_PRINTER  1				 //Print to Printer
#define PRINT_2_SCREEN   2				 //Print to Screen
#define PRINT_2_FILE     3				 //Print to File

FUNCTION Main()
LOCAL lOldBlink:=SetBlink(.f.)
LOCAL nSel:=0
LOCAL aScn := ADcls( "", "R/W")[2]
ADm_limits(0,0,24,79)
Setcolor("N/BG","N/W*",,,"N/BG")
WHILE nSel <> 3
nSel:=ADvermenu( , ,;  
                {  "Print Report" ,;        
                   "View Saved Report"   ,;
                   "Exit Demo"  ;
                 },;
                {1,1,1}  ,; 
                               ,;		                      
                {|| ADvm_header( "Main Demo Menu","Ĵ"), ;
                    ADvm_boxattr( { "Ŀ ",;             
                                    NIL,;
                                    NIL,;
                                    2;                                        
                                   };
                                 ),;
                    ADvm_colors( {  "W+/B",;
                                    "N/W*",;		     
                                    "W+/B",;		     
                                    "R" ;		         
                                   };
                                 ) ;				   
                 } ;
               )
IF nSel == 1
  DemoReport()
ELSEIF nSel == 2
  xSaveRep()
ELSE
  ADMessage({"Goodbye"},,,.F.,,{"GR+/RB","GR+/RB","GR+/RB"})
ENDIF
ENDDO
INKEY(2)
SetBlink(lOldBlink)
AdRestScn(aScn)
RETURN NIL

/*
   This function walks through the Repdemo database and creates the report
   array.  It shows how to call all of the array printing functions, the user
   would typically replace this with their own data/report layout.
   
   The variables you need to set up are:
   aHeader  An array of Header strings.  As many lines as you wish, they
            will be deducted from the total page available.
            
   aFooter  An array of Footer strings.
   aRptArry A blank array that will be filled as the report is generated.
            It will be a multi-dimensional array, the first dimension
            is each page, the second dimension is each line of the page.
   nLine    Line counter, initialize to a number greater that the page
            length to force createion of the first page when the report
            starts generating.
   nPgNum   Page counter, intialize to 0.
   i        The loop counter used for stuffing the Page counts in the header
   nReptLen Used when calculating and stuffing the page counts
*/
FUNCTION DemoReport()
LOCAL nLine:=66,nPgNum:=0,i,nReptLen,aRptArry:={}
LOCAL aFooter:={}
LOCAL aHeader:={ PADC("Array Printing Report Demo",80)      , ;
                 REPLICATE("_",80) ;
                 }
LOCAL aScn:=ADMessage({"Preparing Report"},,,.F.,,{"GR+/RB","GR+/RB","GR+/RB"})
USE REPDEMO NEW
WHILE REPDEMO->(!EOF()) .AND. INKEY() # K_ESC
   //Start a new page if the line counter gets beyond the page length
   //minus the length of the footer, minus 4 lines in this case
   //because the report is 3 lines per record and I don't want a
   //runtime error accessing an array element beyond the limits.
   IF nLine > PAGE_LEN-LEN(aFooter)-4
      // Print the Header
      AADD( aRptArry, NewPage( @nPgNum, aHeader, aFooter, PAGE_LEN , ;
         PAGE_WID, HEAD_SKIP, .t. ) )
      nLine := LEN( aHeader ) + HEAD_SKIP + 2
   ENDIF
   //Stuff the data into the line elements
   aRptArry[ nPgNum , nLine  ] += PADR(PROPER(RTRIM(REPDEMO->FIRST) + " " + ;
                                              RTRIM(REPDEMO->LAST)),		;
                                        PAGE_WID/2)                     +   ;
                                  PADL("Age:" + STR(REPDEMO->AGE,3),        ;
                                        PAGE_WID/2)
   ++nLine
   aRptArry[ nPgNum , nLine  ] += PADR(RTRIM(REPDEMO->STREET),PAGE_WID/2) + ;
                                  PADL("Hiredate: " + DTOC(REPDEMO->HIREDATE),;
                                       PAGE_WID/2)
   ++nLine
   aRptArry[ nPgNum , nLine  ] += PADR(RTRIM(REPDEMO->CITY)  + ", " +       ;
                                       REPDEMO->STATE + " " +				;
                                       REPDEMO->ZIP,                        ;
                                       PAGE_WID/2) +						;
                                  PADL("Salary:" +                          ;
                                       DollarFormat(REPDEMO->SALARY),       ;
                                       PAGE_WID/2)
   nLine+=2
   REPDEMO->( DBSkip() )
ENDDO
// Close File
DbCloseArea()

// Stuff in "Page XX of XX" into all the headers
// Stringed out to 4 positions because of Clipper limit
// on array elements == 4096.
nReptLen:=LEN(aRptArry)
FOR i = 1 TO nReptLen
   aRptArry[ i, 2 ] +=PADL("Page "+ALLTRIM(STR(i,4,0)) + ;
                           " of "+ALLTRIM(STR(nReptLen,4,0)),17)
NEXT i
WhichDevice(aRptArry)
AdRestScn(aScn)
RETURN NIL


/*
    This function prepares a new page array for the report
    Each array represents a complete page, with date and
    page numbering, and optional headers and footers.
      nPg    Page Number variable, passed by reference
      aHdr   Array of Header text
      aFtr   Array of Footer Text
      nLn    Number of lines per page, defaults to 60
      nWid   Number of columns, defaults to 74
      nSkip  Skipping non-printing areas
      lFF    Logical flag if formfeed required at end of page
*/
Function NewPage(nPg,aHdr,aFtr,nLn,nWid,nSkip,lFF)
LOCAL i, aPage := {}           //using i as generic counter

// Default parameters if they are not passed
DEFAULT nPg TO 0          , ;
   aHdr TO {}             , ;
   aFtr TO {}             , ;
   nLn  TO  PAGE_LEN      , ;
   nWid TO  PAGE_WID       , ;
   nSkip TO HEAD_SKIP     , ;
   lFF   TO .t.			  

++nPg                          // increment page counter
aPage := ARRAY(nLn)			   // fill the page with lines
aFill( aPage, LEFT_MARG)

/* 
   Allow nSkip=0 to bypass the header/footer, I use this when 
   printing labels.
*/
IF nSkip > 0
   
   //Place date in the first line
   aPage[ nSkip + 1] += "Date:" +Dtoc(DATE()) + SPACE(nWid - 30) 
   
   // Put the Header in the Report
   FOR i = 1 TO LEN( aHdr )
      aPage[ i + nSkip + 1 ] ;
         += Padc( ALLTRIM(aHdr[ i ]), nWid, " ")
   NEXT
   
   // Put the Footer in the Report
   IF LEN( aFtr ) > 0
      FOR i = 1 TO LEN( aFtr )
         aPage[ nLn - LEN( aFtr) + i - 1 ] ;
            += Padc( LTRIM(RTRIM(aFtr[ i ])),nWid," ")
      NEXT
   ENDIF
   
ENDIF

// Put FormFeed at end of page
IF lFF
   aPage[ nLn ] := FORMFEED
ENDIF

RETURN aPage

// Function to allow user to direct the report array output.
FUNCTION WhichDevice( aRept )
LOCAL nWhere := 0
nWhere :=  ADpushmenu( { "Please specify the output device"   ,;
                         "for this report...              " } ,;
                       { "Printer", "Preview", "File" }       ,;
                       {1,2,1} )
IF nWhere > 0
   lPrint( aRept, nWhere)
ENDIF
RETURN nWhere

// Function to process the report output
FUNCTION lPrint( aRept, n2Print )
LOCAL cFileName:=SPACE(8)
LOCAL nOldRow:= ROW()
LOCAL nOldCol:= COL()

DEFAULT aRept TO {} , n2Print TO 2

DO CASE
CASE n2Print = PRINT_2_PRINTER
   PrintRpt(aRept)
CASE n2Print = PRINT_2_SCREEN
   ScrPrt (aRept)
CASE n2Print = PRINT_2_FILE
   ADg_oneb( ,,, "Enter File Name", @cFileName )
   IF !EMPTY( cFileName )
      WriteFile(aRept,cFileName)
   ENDIF
ENDCASE
DEVPOS( nOldRow, nOldCol)
RETURN NIL

/* 
   Function to write the report array to disk with a nested loop
   to handle the 2 dimensions of the array
*/
FUNCTION WriteFile(aRept,cFileName)
LOCAL i:=0, j:=0
LOCAL nHandle:=0
LOCAL aScn:=ADMessage({"Saving Report"},,,.F.,,{"GR+/RB","GR+/RB","GR+/RB"})
cFilename:=LTRIM(RTRIM(cFilename)) + ".RPT"
nHandle := FCreate( cFileName )
IF nHandle > 0
   FOR i := 1 TO LEN( aRept )
      FOR j := 1 TO LEN( aRept[ i ] )
         FWrite( nHandle, aRept[ i, j ] + LINEFEED )
      NEXT
   NEXT
   FClose( nHandle )
ENDIF
ADRestScn(aScn)
RETURN NIL

// Function to print the report array, once again with a nested loop
FUNCTION PrintRpt(aRept,nStart,nEnd)
LOCAL i,j
LOCAL aScn:=ADMessage({"Printing Report"},,,.F.,,{"GR+/RB","GR+/RB","GR+/RB"})

DEFAULT nStart TO 1, nEnd TO LEN(aRept)

IF nStart > LEN(aRept) .OR. nEnd > LEN(aRept)
   ADMessage({"Page number exceeds report length!"},,,.T.,,{"GR+/RB","GR+/RB","GR+/RB"})
ELSE
   SET PRINTER ON
   SET CONSOLE OFF
   FOR i := nStart TO nEnd
      FOR j := 1 TO LEN( aRept[ i ] )
         Qout( aRept[ i, j ] )
      NEXT
   NEXT
   SET PRINTER OFF
   SET CONSOLE ON
ENDIF
ADRestScn(aScn)
RETURN NIL

// Function to review report array on screen
FUNCTION ScrPrt( aRept )
LOCAL aCols:= {}, aCsr, aScn, nPtr := 1, nPage := 1
aCsr := ADSavecsr(SC_NONE)
aScn := ADSavescn(0,0, Maxrow(), MaxCol() )
aCols:={ { "", {|| aRept[ nPage, nPtr ] }, 80 } }
@ Maxrow(),0 SAY PADC("[]    []    [PgDn]    [PgUp]   [Next Page]   [Prev Page]   [ESC=Menu]",80," ") ;
   COLOR "W+/B"
TBPageArray( 0, 0, Maxrow() -1, Maxcol(), aCols, aRept, @nPtr, @nPage)
ADRestcsr(aCsr)
ADRestscn(aScn)
RETURN NIL

// Function to set up a Tbrowse object for view an array
FUNCTION TbPageArray( nTop, nLeft, nBot, nRight, ;
                      aCols, aRept, nPtr, nPage )
LOCAL oArrayBrow , oColumn, i := 0 , lBrowsing := .t.

// Create the Tbrowse object
oArrayBrow := tBrowseNew( nTop, nLeft, nBot, nRight )

// Custom Browse skippers for the array
oArrayBrow:GoTopBlock := { || nPtr := 1 }
oArrayBrow:GoBottomBlock := { || nPtr := Len( aRept[ nPage ] ) }
oArrayBrow:SkipBlock := { |n| PageArSkip( n, aRept[ nPage ], @nPtr ) }

// Generic column adder for Tbrowse object
FOR i = 1 to Len( aCols )
   oColumn:=TBColumnNew( aCols[ i , 1 ], aCols[ i, 2 ] )
   IF Len( aCols[ i ] ) > 2
      oColumn:width := aCols[ i, 3 ]
   ENDIF
   oArrayBrow:AddColumn( oColumn )
NEXT

// Stabilize
oArrayBrow:ForceStable()

// Wait for keystrokes to process
WHILE lBrowsing
   oArrayBrow:ForceStable()
   lBrowsing := PageArKey( oArrayBrow, aRept, @nPtr, @nPage)
END
RETURN NIL

/*
   This is where all the review on screen work gets done.
   The Inkey() function is replaced with Frankie ADWait().
   Passed the browse object, Report Array, Pointer, page number.
*/
FUNCTION PageArKey( oBrow, aRept, nPtr, nPage)
LOCAL lRetVal := .t. , oCol, bSavIns
LOCAL getlist :={}, cGet, nEngine, nSel
LOCAL bConfig := {|| ADr_keys( {K_ENTER} , ;
                               {|| PutArry(aRept,nPtr,nPage), ADr_save()} ;
                              ) ;
                    }
LOCAL cFileName:=SPACE(8),nGoPage:=1,nReptLen,x
LOCAL aGets:={1,2}
LOCAL bGetConfig := {||ADgm_labels( {"From Page:","To Page:"} ) , ;
                       ADgm_Pictures( {"9999","9999"}         ) , ;
                       ADgm_Header( "[ Enter Print Range ]"     , ;
                                    "[ ESC to abort ]" )        , ;
                       ADgm_pads({1,4,1,4} )                    , ;
                       ADgm_Color( {"GR+/B",NIL,NIL,NIL}  )       ;
                      }
LOCAL aKey := ADwait() 

// Process the array returned by ADWait()
IF AKey[2] = 1 .AND. AKey[3]!=24
   // User left clicked in the report area of the screen
   IF ADm_dblclicked(1,.2) == 2
      IF AKey[3] == 0
         OBrow:GoTop()
         OBrow:RefreshAll()
      ELSE
         WHILE AKey[3] <> OBrow:RowPos
            IF AKey[3] > OBrow:RowPos
               OBrow:Down()
            ELSEIF AKey[3] < OBrow:RowPos
               OBrow:Up()
            ELSE
               //shouldn't enter here because aKey[3] would be == oBrow:RowPos
            ENDIF
            OBrow:ForceStable()
         END
         // One more bump to get it aligned
         OBrow:Down()
         OBrow:ForceStable()
      ENDIF
      // Force the GET with a double click
      AKey[1]:=K_ENTER
   ENDIF
ELSEIF (AKey[2] == 1 .AND. (AKey[3]=24 .AND.   ;
      !(     (AKey[4]>3 .AND. AKey[4]<7) .OR.  ;
             (AKey[4]>10 .AND. AKey[4]<14)     ;
        )	                                   ;
                             )          	   ;
         ) .OR. AKey[2] == 2
   // User left clicked on the bottom line or right clicked
   // wait for release of mouse button
   Adm_rwait()
ENDIF

// Process request
DO CASE
CASE aKey[1] == K_ENTER
   // Get the current column object from the browse
   oCol := oBrow:getColumn( oBrow:colPos )

   // Create a Get variable for ADget
   cGet:=EVAL(oCol:block)

   // Call ADGet to add var to local getlist
   @ ROW(), COL() ADGet cGet
   
   // Set insert key to toggle insert mode and cursor shape
   bSavIns := setkey( K_INS, { || InsToggle() } )
   
   // Set initial cursor shape
   setcursor( iif( ReadInsert(), SC_INSERT, SC_NORMAL ) )
   ADREAD(GetList,bConfig)
   setcursor( SC_NONE )
   setkey( K_INS, bSavIns )
   oBrow:Refreshcurrent()


// Bring up the report menu
CASE aKey[1] == K_ESC .OR. aKey[2] == 2 .OR. ;
      (aKey[2] == 1 .AND. aKey[3] == 24 .AND. (aKey[4] > 64 .AND. aKey[4] < 75 ) )
   nSel:=ADvermenu( 8,30     ,;                        // left corner of box
   {  "Print Report" ,;                                // menu options
      "Print Page"   ,;
      "Print Pages"  ,;
      "Delete Page"  ,;
      "Go To Page "  ,;
      "Save Report"  ,;
      "Exit Report"  ;
      },;
      {1,2,3,1,1,1}  ,;                                // trigger key positions
   ,;		                                           // Process codeblock
   {|| ADvm_header( "Report Options"+";"	 ;		   // header
   +LTRIM(STR(LEN(aRept),4))+" Pages","Ĵ"), ;
      ADvm_boxattr( { "Ŀ ",;                   // box frame
   NIL,;
      NIL,;
      2;                                              // padding
   };
      ),;
      ADvm_colors( {  "W+/B",;		                  // std color
                      "N/W*",;		                  // enh color
                      "W+/B",;		                  // Box & header color
                      "R" ;		                      // trigger color
                     };
      ) ;				   
      } ;
      )
   
   // Process report menu options
   DO CASE
   CASE nSel==1
      //Print Report
      PrintRpt(aRept)
      lRetval:=.t.
   CASE nSel==2
      //Print Page
      PrintRpt(aRept,nPage,nPage)
      lRetval:=.t.
   CASE nSel==3
      //Print Pages
      IF ADg_many( aGets, bGetConfig )
         PrintRpt(aRept,aGets[1],aGets[2])
      ENDIF
      lRetval:=.t.
   CASE nSel==4
      //Delete Page
      IF LEN(aRept) > 1
         ADEL(aRept,nPage)
         ASIZE(aRept,LEN(aRept)-1)
         IF nPage > 1
            --nPage
         ENDIF
         //Renumber Pages
         nReptLen:=LEN(aRept)
         FOR x = 1 TO nReptLen
            aRept[ x, 2 ] :=SUBSTR(aRept[ x, 2],1,62) + ;
                            PADL("Page "+ALLTRIM(STR(x,4,0)) + ;
                                 " of "+ALLTRIM(STR(nReptLen,4,0)),17)
         NEXT
         oBrow:RefreshAll()
      ELSE
         * can't delete last page
         ADMessage({"Can't delete last page in a report!"},,,.T.,,{"GR+/RB","GR+/RB","GR+/RB"})
      ENDIF
      lRetval:=.t.
   CASE nSel==5
      //Go To Page
      ADg_oneb( ,,, "Go To Page:", @nGoPage, "9999" )
      IF LEN(aRept) >= nGoPage .AND. nGoPage > 1
         nPage:=nGoPage
         oBrow:RefreshAll()
      ELSE
         * can't go past last page
         ADMessage({"Can't go past last page in a report!"},,,.T.,,{"GR+/RB","GR+/RB","GR+/RB"})
      ENDIF
      lRetval:=.t.
   CASE nSel == 6
      ADg_oneb( ,,, "Enter File Name", @cFileName )
      IF !EMPTY( cFileName )
         WriteFile(aRept,cFileName)
      ENDIF
      lRetVal:=.t.
   CASE nSel == 7
      //Exit
      lRetVal := .f.
   OTHERWISE
      lRetVal := .t.
   END CASE
CASE aKey[1] == K_DOWN .OR. ;
      (aKey[2] == 1 .AND.	aKey[3] == 24 .AND. (aKey[4]>3 .AND. aKey[4]<7) )
   oBrow:Down()
   IF oBrow:Hitbottom()
      ADMessage({"End of Page!"},,,.T.,,{"GR+/RB","GR+/RB","GR+/RB"})
   ENDIF
   oBrow:Refreshcurrent()
CASE aKey[1] == K_UP  .OR. ;
      (aKey[2] == 1 .AND.	aKey[3] == 24 .AND. (aKey[4]>10 .AND. aKey[4]<14) )
   oBrow:Up()
   IF oBrow:Hittop()
      ADMessage({"Top of Page!"},,,.T.,,{"GR+/RB","GR+/RB","GR+/RB"})
   ENDIF
   oBrow:Refreshcurrent()
CASE aKey[1] == K_PGDN ;
      .OR.(aKey[2] == 1 .AND. aKey[3] == 24 .AND. (aKey[4] > 17 .AND. aKey[4] < 24 ) )
   oBrow:PageDown()
   oBrow:RefreshAll()
CASE aKey[2] == 1 .AND. aKey[3] >= 0 .AND. aKey[3] <> 24
   IF aKey[3] == 0
      oBrow:GoTop()
      oBrow:RefreshAll()
   ELSE
      WHILE aKey[3] <> oBrow:RowPos
         IF aKey[3] > oBrow:RowPos
            oBrow:Down()
         ELSEIF aKey[3] < oBrow:RowPos
            oBrow:Up()
         ELSE
            //shouldn't enter here because aKey[3] would be == oBrow:RowPos
         ENDIF
         oBrow:ForceStable()
      END
      // One more bump to get it aligned
      oBrow:Down()
      oBrow:Refreshcurrent()
   ENDIF
CASE aKey[1] == K_PGUP ;
      .OR.(aKey[2] == 1 .AND. aKey[3] == 24 .AND. (aKey[4] > 27 .AND. aKey[4] < 34 ) )
   oBrow:PageUp()
   oBrow:RefreshAll()
CASE aKey[1] == K_CTRL_PGDN .OR. ;
      (aKey[2] == 1 .AND. aKey[3] == 24 .AND. (aKey[4] > 36 .AND. aKey[4] < 48 ) )
   IF nPage < Len( aRept )
      ++nPage
      nPtr:=1
   ELSE
      TONE(900,2)
      ADMessage({"Last page in report!"},,,.T.,,{"GR+/RB","GR+/RB","GR+/RB"})
   ENDIF
   oBrow:RowPos := 1
   oBrow:RefreshAll()
CASE aKey[1] == K_CTRL_PGUP	.OR. ;
      (aKey[2] == 1 .AND. aKey[3] == 24 .AND. (aKey[4] > 50 .AND. aKey[4] < 62 ) )
   IF nPage > 1
      --nPage
      nPtr:=1
   ELSE
      TONE(900,2)
      ADMessage({"First page in report!"},,,.T.,,{"GR+/RB","GR+/RB","GR+/RB"})
   ENDIF
   oBrow:RowPos := 1
   oBrow:RefreshAll()
END CASE
RETURN lRetVal

// This Function replaces the array element with the current GET value
FUNCTION PutArry(aRept, nPtr, nPage)
aRept[nPage,nPtr] := ADr_varget(1)
RETURN NIL

// Tbrowse skipper for Array Browsing
FUNCTION PageArSkip( nReq, aRept, naPtr)
LOCAL nSkipped := 0
IF nReq > 0
   While ( nSkipped < nReq ) .and. ( naPtr < Len( aRept ) )
      nSkipped++
      naPtr++
   End
ELSEIF nReq < 0
   While ( nSkipped > nReq ) .and. (naPtr > 1 )
      nSkipped--
      naPtr--
   End
END IF
RETURN nSkipped

// This is right from CA-Clipper
/***
*
*   InsToggle()
*
*   Toggle the global insert mode and the cursor shape.
*
*/
*!*****************************************************************************
FUNCTION InsToggle()
if readinsert()
   readinsert( FALSE )
   setcursor( SC_NORMAL )
else
   readinsert( TRUE )
   setcursor( SC_INSERT )
endif
RETURN NIL

// Format Dollar values
FUNCTION DollarFormat(nAmount)
RETURN "$"+LTRIM(TRANSFORM(nAmount, "99,999,999"))

// Format proper names
FUNCTION PROPER(cString)
LOCAL n,nLenStr,lFirstLtr:=.T.,cChar

//Trim the String
cString:=LTRIM(RTRIM(cString))

//Get the trimmed length
nLenStr:=LEN(cString)

//Parse each character
FOR n = 1 TO nLenStr
   
   // Strip out the current character
   cChar:=Substr(cString,n,1)
   
   IF ISUPPER(cChar) .AND. !lFirstLtr
      // Replace with Lower
      cString:=STUFF(cString,n,1,LOWER(cChar))
      
   ELSEIF cChar $ ".,;:/\"
      //Skip over punctuation, add any others you want
      //Don't Reset the First Letter Flag
      lFirstLtr:=.F.
      
   ELSEIF cChar $ " -'"
      //Use spaces as the delimiters, add any others you want
      //Added Hypen to delimiters for Hyphen Last Names!
      //Reset the First Letter Flag
      lFirstLtr:=.T.
      
   ELSEIF lFirstLtr
      // First letters of Proper words fall in here
      // Replace with Upper if not already
      IF ISLOWER(cChar)
         cString:=STUFF(cString,n,1,UPPER(cChar))
      ENDIF
      lFirstLtr:=.F.
      
   ELSE
      // It must be lower case, not a first letter, and not punctuation
      // or a space. Do Nothing
      
   ENDIF
   
NEXT
RETURN cString

// My Frankie Push Button Menu
FUNCTION ADPushMenu(APrompts,AButtons,ATriggers,CCol,NTop,NLeft)
LOCAL NEngine,AScn,CBackGrnd,NButtonlen,NRetVal:=0
LOCAL NHeight,NWidth
LOCAL NCurs:=SETCURSOR(SC_NONE)
LOCAL NRow:=ROW(),NCol:=COL()

// Check Parameters and Assign defaults
DEFAULT APrompts TO {"Push a Button"}
DEFAULT AButtons TO {"Accept","Cancel"}
DEFAULT ATriggers TO AFILL(ARRAY(LEN(AButtons)),1)
DEFAULT CCol TO "GR+/RB"

//Set the background Color for the shadow button
CBackgrnd:=RIGHT(CCol,LEN(CCol)-RAT("/",CCol))

//Calc length of buttons with interspacing
NButtonlen:=LEN(ADarr2str(AButtons,""))+(3*(LEN(AButtons)-1))

//Calc width of box by comparing longer of Buttons or Text
NWidth:=IIF(NButtonlen > ADaMaxlen(APrompts),NButtonlen,ADaMaxlen(APrompts))+8

// Set box height
NHeight := LEN(APrompts)+3

// Default the box corner coords
DEFAULT NLeft TO ADscn_hcenter( NWidth )
DEFAULT NTop TO ADscn_vcenter( NHeight )

//Create button engine
NEngine := ADpb_horizontal( NTop + NHeight , ;
   NLeft + NWidth/2 - NButtonlen/2 , ;
   AButtons, ;
   ATriggers )

//Increment the top line counter before painting the prompts
NTop++

// Buffer all the screen painting
DISPBEGIN()
Ascn:=ADbox( NTop, NLeft, NTop + NHeight + 1, NLeft + NWidth + 1 , CCol)
AEVAL(APrompts,{|e| ADcsay(++NTop,NLeft,NLeft+NWidth+1,E,CCol)} )
Adpb_show(NEngine,CBackgrnd)
DISPEND()

//Wait for user selection
NRetVal:=Adpb_activate(NEngine)

//Cleanup and return
IF NRetVal > 0      // made a selection
   //Clean out LASTKEY() value
   KEYBOARD CHR(0)
   INKEY(0.1)
   ADpb_push( NEngine, NRetval )
ENDIF
ADpb_kill(NEngine)
ADRestscn(AScn)
SETCURSOR(NCurs)
SETPOS(NRow,NCol)
RETURN NRetVal


//Function to view/print saved files
FUNCTION xSaveRep()
LOCAL cFile,aRptArry:={},nPgNum:=0,nLine:=PAGE_LEN+1
LOCAL nHandle,nLastPg,aScn:=ADSaveScn() 
IF FILE("*.RPT")
  cFile:=ALLTRIM(SUBSTR(ADpl_files( ,,, "*.RPT" , ,  ;             	 
                          {|| ADvm_header( "Select Saved Report" )},;
                          {|| ADpl_filformat( "DT" )} )		, ;
                          1,12))
  IF FILE( cFile )
     // Read the File into a Report Array
     nHandle := fopen( cFile )
     IF Ferror() == 0
       ADmessage({ "Reading File..."},,,.F.,.F.,{"GR+/RB","GR+/RB","GR+/RB"})
       do while !ADfeof( nHandle )
         IF nLine == PAGE_LEN+1
           // Add a new page
           AADD( aRptArry, ArrPage( @nPgNum ) )
           nLine:=1
         ENDIF
         // Write the line into the report array
         aRptArry [ nPgNum, nLine ] := ADfgetl( nHandle )
         nLine++
       enddo
       WhichDevice(aRptArry)
     ELSE
       ADmessage({ "Unable to Open Report File!" , "Press any key"},,,.T.,.F.,{"GR+/RB","GR+/RB","GR+/RB"})
     ENDIF
  ENDIF
ELSE
  ADmessage({ "No Report Files Found!" , "Press any key"},,,.T.,.F.,{"GR+/RB","GR+/RB","GR+/RB"})
ENDIF 
ADRestScn(aScn)
RETURN NIL

// Create a new page when reading report from disk
Function ArrPage(nPg,nLn)
LOCAL aPage
DEFAULT nLn TO PAGE_LEN
++nPg     // increment page counter
aPage := ARRAY(nLn)
aFill( aPage, LEFT_MARG)
RETURN aPage

