
/* ====================================================================== *

  PROGRAM:  USPS Bar Code printing routine for Laserjet printers

  AUTHOR:   Brian E. Hoffman
            CIS ID 77307,566

  COPYRIGHT (c) 1991, Behoff Systems, Inc.
            212 East 47th Street
            Suite 21-A
            New York, New York 10017-2128
            (212) 750-5324

  PURPOSE   Print Postal service barcodes at either the current baseline cursor
            location, or the location specified in the invocation.

  SYNTAX    USPS_Bar( cZip, Xloc, Yloc )
            PostNet(  cZip )

  PARAMETER cZip is the Zip+4 code, or Zip+4 + last two digits
                 of the street address for Advanced Postnet
            Xloc is the X location to print the barcode, in inches
            Yloc is the Y location to print the barcode, in inches
                 If you want to print at the current cursor
                 position, just call PostNet directly.


  ASSUMES   Printer is ON, or you will wind up in your ErrorSys.
            Leaves the cursor at the baseline at end of last bar.

  EXPLANATION
            Zip+4 barcoding uses framing bars at the beginning and end
            with ten digits between (9 + a checksum), each coded as a
            series of five bars.  This makes 52 bars total.

            Bars represent 1 (long) and zero (short).

            Bars have the values 7, 4, 2, 1, and 0 from left to right.

            Each series of five bars (excluding the framing bars) must
            have two long bars and 3 short bars.

            The digit zero is a special case - in order to have two long
            bars in the series of five, zero is coded as eleven.  Thus

                74210
            0 = 11000
            1 = 00011
            2 = 00101
            3 = 00110
            4 = 01001
            5 = 01010
            6 = 01100
            7 = 10001
            8 = 10010
            9 = 10100

   PLACEMENT
            Older equipment requires the barcodes to be put in a strict
            location on the outside of the envelope.  Newer equipment is
            allowing the USPS to find the barcodes in less rigidly defined
            locations.  The new standard seems to be above the first line of
            the address (look at your Citibank credit card bills).

   The code between #ifdef/#endif is simply a test to call the
   actual routines.  This module would normally be called with the
   appropriate parameters passed.

   NOTE:  This module relies on the existince of a printer reset variable
          called PR_RESET (see the first code line of PostNet()) to determine
          if it is printing to an HP-PCL printer or a printer that emulates
          Epson FX codes.  These are the only two I need to support, so I never
          implemented beyond that.  You will have to adapt that section of the
          code to your purposes.  The easiest way would be to just pull out
          that IF sequence, perhaps even making the STATICS below into @defines.
          If you have any questions about exactly what the codes are doing,
          drop me a line.

   This module is distributed as shareware - no warranty is provided and all
   that legal jazz.  If you find it useful a small contribution of $10 is
   requested.  Or at least a message with your suggestions, modifications, etc.

*/


#define NTRIM(x) LTRIM(STR(x,15,0))

STATIC Pr_BarPre, Pr_BarSuf, Pr_Bar1, Pr_Bar0

#ifdef DEBUG
  PROCEDURE main ( cZip, nX, nY )
#else
  PROCEDURE USPS_Bar ( cZip, nX, nY )
#endif

     nX := IF(nX = NIL, 0, nX)
     nY := IF(nY = NIL, 0, nY)

#ifdef DEBUG
     CLEAR
     cZip := REPLICATE("0", 9)
     nX := nY := 1.00
     @ 1,  5 SAY    "Enter zip code to print:" GET cZip ;
                                           PICTURE "@R #####-####"
     @ 2,  2 SAY "   At baseline coordinates:"
     @ 3, 27 SAY                          "X:" GET nX PICTURE "##.####"
     @ 4, 27 SAY                          "Y:" GET nY PICTURE "##.####"
     READ
     IF LASTKEY() != 27
        SET PRINTER ON
        SET DEVICE TO PRINT
        SET CONSOLE OFF
#endif

        ?? HP_Goto(nX, nY)
        PostNet(TRIM(cZip))

#ifdef DEBUG
        EJECT
        SET CONSOLE ON
        SET DEVICE TO SCREEN
        SET PRINTER OFF
     ENDIF
#endif

  RETURN

* ====================================================================== *
  FUNCTION   HP_Goto (x, y)
  NOTE       Returns the appropriate codes to position the HP cursor
  *          at a particular row, column expressed in inches.
* ====================================================================== *
  RETURN IF(x == 0 .AND. y == 0, "", ;
            CHR(27) + "*p" + NTRIM(x*300) + ;
            IF(y != 0, "x"+NTRIM(y*300)+"Y", "X"))

* ====================================================================== *
  PROCEDURE PostNet(Zip4)
  LOCAL     x, Digit, ZipSum
* ====================================================================== *
     IF LEFT(Pr_Reset, 2) = CHR(27) + "E"
        * Laserjet -
        Pr_BarPre = CHR(27) + "&f0S" + CHR(27) + "*p-39Y"
        Pr_BarSuf = CHR(27) + "&f1S"
        Pr_Bar1   = CHR(27) + "*c4a39b0P" + CHR(27) + "*p+14X"
        Pr_Bar0   = CHR(27) + "*c4a39b0P" + CHR(27) + "*c4a23b1P" + CHR(27) + "*p+14X"
     ELSE
        * Epson - single density, 60 dpi
        Pr_BarPre = CHR(27) + "K" + CHR(1) + CHR(141)
        Pr_BarSuf = REPLICATE(CHR(8), 23) && restore starting position
        Pr_Bar1   = REPLICATE(CHR(255), 2) + REPLICATE(CHR(0), 2)
        Pr_Bar0   = REPLICATE(CHR(  7), 2) + REPLICATE(CHR(0), 2)
     ENDIF

     IF Between(LEN(Zip4), 9, 11)   &&  Only nine digit (or 11 for Advanced Postnet)
        ZipSum = 0
        ?? Pr_BarPre
        PrintBar(1, 0)  && Framing bar
        FOR x = 1 TO LEN(Zip4)
           Digit = VAL(SUBSTR(Zip4, x, 1))
           ZipSum = ZipSum + Digit
           CodeNumber( Digit )
        NEXT
        CodeNumber( 10 - (ZipSum % 10) )
        PrintBar(1, 0)
        ?? Pr_BarSuf
     ENDIF
  RETURN

* ====================================================================== *
  PROCEDURE CodeNumber ( Number )
  LOCAL     BarCount
* ====================================================================== *
     IF Number = 0
        Number := 11  && special case
     ENDIF
     BarCount = PrintBar( @Number, 7 ) + ;
                PrintBar( @Number, 4 ) + ;
                PrintBar( @Number, 2 ) + ;
                PrintBar( @Number, 1 )
     PrintBar( IF(BarCount = 1, 1, 0), 1)
  RETURN

* ====================================================================== *
  FUNCTION  PrintBar ( digit, limit )
* PURPOSE   Print a single digit and advance the cursor
  LOCAL     result
* ====================================================================== *
     IF digit < limit
        ?? Pr_Bar0
        result := 0
     ELSE
        ?? Pr_Bar1
        digit := digit - limit
        result := 1
     ENDIF
  RETURN (result)
