/*
 * File......: TTS.PRG
 * Author....: Glenn Scott
 * CIS ID....: 71620,1521
 * Date......: $Date$
 * Revision..: $Revision$
 * Log file..: $Logfile$
 * 
 * This is an original work by Glenn Scott and is placed in the
 * public domain.
 *
 * Modification history:
 * ---------------------
 *
 * $Log$
 *
 */

/*  $DOC$
 *  $FUNCNAME$
 *      OVERVIEW
 *  $CATEGORY$
 *      TTS
 *  $ONELINER$
 *      TTS Overview
 *  $SYNTAX$
 *
 *  $ARGUMENTS$
 *
 *  $RETURNS$
 *
 *  $DESCRIPTION$
 *
 *       [ Put the Overview here ]
 *
 *  $EXAMPLES$
 *
 *  $SEEALSO$
 *
 *  $INCLUDE$
 *
 *  $END$
 */

#include "ftint86.ch"
#include "netto.ch"

#define    TT_BEGIN       0  /* Never change these defines! */
#define    TT_END         1
#define    TT_AVAIL       2
#define    TT_ABORT       3
#define    TT_STATUS      4
#define    TT_GET_AT      5
#define    TT_SET_AT      6
#define    TT_GET_WT      7
#define    TT_SET_WT      8

/*  $DOC$
 *  $FUNCNAME$
 *     FN_TTAVAIL()
 *  $CATEGORY$
 *     TTS
 *  $ONELINER$
 *     Determine TTS availability
 *  $SYNTAX$
 *
 *     fn_ttAvail() -> nAvail
 *
 *  $ARGUMENTS$
 *
 *     None
 *
 *  $RETURNS$
 *
 *    <nAvail>, a numeric as follows:
 *
 *             0     Transaction tracking not available
 *             1     Transaction tracking is available
 *           253     Transaction tracking disabled
 *
 *  $DESCRIPTION$
 *
 *    Use this call to determine whether or not TTS has been
 *    installed on the network.  Note that you can use TTS
 *    calls even if the NetWare network doesn't have TTS
 *    installed, and you'll get 0 return codes from all the 
 *    TTS API functions.
 *
 *    A simple wrapper around this function is FN_HASTTS(),
 *    which allows a simple true/false return.  See the 
 *    FN_HASTTS() documentation.
 *
 *  $EXAMPLES$
 *
 *    if fn_ttAvail() == 1
 *       qout( "We have transaction tracking!" )
 *    endif
 * 
 *  $SEEALSO$
 *    FN_HASTTS()
 *  $INCLUDE$
 *
 *  $END$
 */


function fn_ttAvail()
  local aRet := _fnTTreq( TT_AVAIL )
  return lowbyte( aRet[ AX ] )

/*  $DOC$
 *  $FUNCNAME$
 *     FN_HASTTS()
 *  $CATEGORY$
 *     TTS
 *  $ONELINER$
 *     Is TTS available?  Yes or No?
 *  $SYNTAX$
 *     
 *     fn_hasTTS() -> lHasTTS
 *
 *  $ARGUMENTS$
 *
 *     None
 *
 *  $RETURNS$
 *
 *     Logical TRUE (.t.) if TTS is available, FALSE (.f.) if it 
 *     isn't.
 *
 *  $DESCRIPTION$
 *
 *     See the documentation for FN_TTAVAIL().  
 *
 *  $EXAMPLES$
 *
 *    if fn_hasTTS()
 *       qout( "We have TTS!" )
 *    endif
 *
 *  $SEEALSO$
 *    FN_TTAVAIL()
 *  $INCLUDE$
 *
 *  $END$
 */


function fn_hasTTS()
  return ( fn_ttAvail() == 1 )

/*  $DOC$
 *  $FUNCNAME$
 *     FN_TTBEGIN()
 *  $CATEGORY$
 *     TTS
 *  $ONELINER$
 *     Begin transaction
 *  $SYNTAX$
 *
 *     fn_ttBegin() -> nRes
 *
 *  $ARGUMENTS$
 *
 *     None
 *
 *  $RETURNS$
 *
 *     <nRes>, a numeric, as follows:
 *
 *          0       Success!
 *        150       Out of dynamic workspace
 *        254       There's an implicit transaction already 
 *                  active, but now it has been converted to
 *                  an explicit transaction
 *        255       There's an explicit transaction already 
 *                  active
 *
 *  $DESCRIPTION$
 *
 *     Begin an explicit transaction on the current file server.
 *
 *  $EXAMPLES$
 *
 *    if fn_hasTTS()
 *       if fn_ttBegin() == 0
 *          qout( "Transaction started!" )
 *          nTrans := fn_ttEnd()
 *          qout( "Transaction ended, trans # = ", nTrans )
 *       else
 *          qout( "Can't begin the transaction" )
 *       endif
 *    endif
 *
 *  $SEEALSO$
 *    FN_TTEND()
 *  $INCLUDE$
 *
 *  $END$
 */


function fn_ttBegin()
  local aRet := _fnTTreq( TT_BEGIN )
  return lowbyte( aRet[ AX ] )

/*  $DOC$
 *  $FUNCNAME$
 *     FN_TTEND()
 *  $CATEGORY$
 *     TTS 
 *  $ONELINER$
 *     End transaction
 *  $SYNTAX$
 *
 *     fn_ttEnd( @nTran ) -> nRes
 *
 *  $ARGUMENTS$
 *
 *     <nTran>, a numeric, which will be replaced with the 
 *     transaction reference number.  THIS MUST BE PASSED
 *     BY REFERENCE!
 *
 *  $RETURNS$
 *
 *    <nTran>, the transaction's "transaction reference number."
 *    You can use this number with FN_TTSTAT().
 *
 *    <nRes>, a numeric, as follows:
 *
 *          0     Success
 *        253     Transaction tracking is disabled
 *        254     Transaction ended, records locked
 *        255     No explicit transaction active
 *
 *  $DESCRIPTION$
 *
 *  $EXAMPLES$
 *
 *  $SEEALSO$
 *     FN_TTBEGIN() FN_TTSTAT()
 *  $INCLUDE$
 *
 *  $END$
 */


function fn_ttEnd( nTran )
  local aRegs := _fnTTreq( TT_END )
  default nTran to 0

  nTran := _fnReg2L( aRegs[ CX ], aRegs[ DX ] )

  return lowbyte( aRegs[ AX ] ) 

/*  $DOC$
 *  $FUNCNAME$
 *     FN_TTABORT()
 *  $CATEGORY$
 *     TTS
 *  $ONELINER$
 *     Abort transaction
 *  $SYNTAX$
 *
 *     fn_ttAbort() -> nRes
 *
 *  $ARGUMENTS$
 *
 *     None
 *
 *  $RETURNS$
 *
 *     <nRes>, a numeric, as follows:
 *
 *          0     Success!
 *        253     TTS was disabled, nothing done
 *        254     Transaction ended; records locked
 *        255     No explicit transaction was active
 *
 *  $DESCRIPTION$
 *
 *  $EXAMPLES$
 *
 *  $SEEALSO$
 *
 *  $INCLUDE$
 *
 *  $END$
 */


function fn_ttAbort()
  local aRet := _fnTTreq( TT_ABORT )
  return lowbyte( aRet[ AX ] )

/*  $DOC$
 *  $FUNCNAME$
 *     FN_TTSTAT()
 *  $CATEGORY$
 *     TTS
 *  $ONELINER$
 *     Get TTS Transaction Status
 *  $SYNTAX$
 * 
 *     fn_ttStat( nTran ) -> lWritten
 *
 *  $ARGUMENTS$
 *
 *     <nTran>, a numeric, which is the transaction reference 
 *     number returned from fn_ttEnd().
 *
 *  $RETURNS$
 *
 *    <lWritten>, logical TRUE (.t.) if the transaction has been
 *    written to disk, FALSE (.f.) if it hasn't yet been 
 *    written.
 *
 *  $DESCRIPTION$
 *
 *  $EXAMPLES$
 *
 *  $SEEALSO$
 *
 *  $INCLUDE$
 *
 *  $END$
 */


function fn_ttStat( nNum )
  local aTmp := _fnL2Reg( nNum ), aRet
  aRet := _fnTTreq( TT_STATUS, aTmp[1], aTmp[2] )
  return lowbyte( aRet[ AX ] ) == 0 

/*  $DOC$
 *  $FUNCNAME$
 *      FN_TTGAPT()
 *  $CATEGORY$
 *      TTS
 *  $ONELINER$
 *      Get TTS Application Thresholds
 *  $SYNTAX$
 *
 *      fn_ttGApT() -> aRes
 *
 *  $ARGUMENTS$
 *
 *  $RETURNS$
 *
 *      <aRes>, an array, as follows:
 *
 *             aRes[1]     Return code.  Should be 0.
 *             aRes[2]     Logical lock threshold
 *             aRes[3]     Physical lock threshold
 *
 *  $DESCRIPTION$
 *
 *  $EXAMPLES$
 *
 *  $SEEALSO$
 *
 *  $INCLUDE$
 *
 *  $END$
 */


function fn_ttGapT()
  return _fnGThresh( TT_GET_AT )

/*  $DOC$
 *  $FUNCNAME$
 *     FN_TTSAPT()
 *  $CATEGORY$
 *     TTS
 *  $ONELINER$
 *     Set TTS Application Thresholds
 *  $SYNTAX$
 *
 *     fn_ttSApT( nLog, nPhy ) -> lSuccess
 *
 *  $ARGUMENTS$
 *
 *     <nLog>, a numeric
 *
 *     <nPhy>, a numeric
 *
 *  $RETURNS$
 *
 *     
 *  $DESCRIPTION$
 *
 *  $EXAMPLES$
 *
 *  $SEEALSO$
 *
 *  $INCLUDE$
 *
 *  $END$
 */


function fn_ttSapT( nLog, nPhy )
  return ( _fnSThresh( TT_SET_AT, nLog, nPhy ) == 0 )

/*  $DOC$
 *  $FUNCNAME$
 *      FN_TTGWST()
 *  $CATEGORY$
 *      TTS
 *  $ONELINER$
 *      Get TTS workstation thresholds
 *  $SYNTAX$
 *
 *      fn_ttGWsT() -> aRes
 *
 *  $ARGUMENTS$
 *
 *      None
 *
 *  $RETURNS$
 *
 *      <aRes>, an array, as follows
 *
 *             aRes[1]     Return code.  Should be 0.
 *             aRes[2]     Logical lock threshold
 *             aRes[3]     Physical lock threshold
 *
 *  $DESCRIPTION$
 *
 *  $EXAMPLES$
 *
 *  $SEEALSO$
 *
 *  $INCLUDE$
 *
 *  $END$
 */


function fn_ttGwsT()
  return _fnGThresh( TT_GET_WT )     

/*  $DOC$
 *  $FUNCNAME$
 *     FN_TTSWST()
 *  $CATEGORY$
 *     TTS
 *  $ONELINER$
 *     Set TTS Workstation Thresholds
 *  $SYNTAX$
 *
 *     fn_ttSWsT( nLog, nPhy ) -> lSuccess
 *
 *  $ARGUMENTS$
 *
 *    <nLog>, a numeric
 *
 *    <nPhy>, a numeric
 *
 *  $RETURNS$
 *
 *    <lSuccess>
 *
 *  $DESCRIPTION$
 *
 *  $EXAMPLES$
 *
 *  $SEEALSO$
 *
 *  $INCLUDE$
 *
 *  $END$
 */


function fn_ttSwsT( nLog, nPhy )
  return ( _fnSThresh( TT_SET_WT, nLog, nPhy ) == 0 )

/* ------------------------------------------------------------------------ */

static function _fnGThresh( xFunc )
  local aRegs := _fnTTReq( xFunc )
  return { ;
     lowbyte(  aRegs[ AX ] ), ;    // Completion code
     lowbyte(  aRegs[ CX ] ), ;    // Logical lock threshold
     highbyte( aRegs[ CX ] )  ;    // Physical lock threshold
     }

static function _fnSThresh( xFunc, nLog, nPhy )
  local aRet
  default nLog to 0, nPhy to 0
 
  aRet := _fnTTReq( xFunc, makehi( nPhy ) + nLog )
  return lowbyte( aRet[ AX ] )

static function _fnTTreq( xFunc, cx, dx )
  local aRegs[ INT86_MAX_REGS ]

  default cx to 0, dx to 0

  aRegs[ AX ] := makehi( 199 ) + xFunc      // C7h

  if ( xFunc == TT_SET_AT ) .or. ( xFunc == TT_SET_WT )
     aRegs[ CX ] := cx
  endif

  if ( xFunc == TT_STATUS )
     aRegs[ CX ] := cx
     aRegs[ DX ] := dx
  endif

  if !ft_int86( INT21, aRegs )
     _fnSetErr( EINT86 )
  else
     _fnSetErr( lowbyte( aRegs[ AX ] ) )
  endif

  return aRegs
     
/* ------------------------------------------------------------------------ */
