/*
 * File......: LOGIN.PRG
 * Author....: Steve Larsen
 * CIS ID....: 76370,1532
 * Date......: $Date$
 * Revision..: $Revision$
 * Log file..: $Logfile$
 *
 * This is an original work by Steve Larsen and is placed in the
 * public domain.
 *
 * Modification history:
 * ---------------------
 *
 * $Log$
 *
 */

#include "ftint86.ch"
#include "netto.ch"

#define E3h	227


/*  $DOC$
 *  $FUNCNAME$
 *      fn_entrLog()
 *  $CATEGORY$
 *      Connection/Workstation
 *  $ONELINER$
 *      Enter Login Area
 *  $SYNTAX$
 *
 *      fn_entrLog( [ <cSubDirectoryName> ][, <nLocalDrives> ] )
 *
 *  $ARGUMENTS$
 *
 *      <cSubDirectoryName> is a string containing the name of the sub-
 *      directory below SYS:LOGIN where the file LOGIN.EXE can be executed.
 *      If omitted, the directory SYS:LOGIN is assumed.
 *
 *      <nLocalDrives> are the number of drives local to the calling
 *      workstation.  This number is used to determine the workstation drive
 *      ID to assign to the server's SYS: volume.  If omitted <nLocalDrives>
 *      defaults to the value returned by fn_numLDrv().
 *
 *  $RETURNS$
 *
 *      Nothing
 *
 *  $DESCRIPTION$
 *
 *      For applications where a custom LOGIN.EXE utility exists, it is
 *      sometimes useful to place the login utility in a subdirectory of
 *      SYS:LOGIN.  Other executable files could be placed here as well, as
 *      the SYS:LOGIN directory is the only directory where one may execute
 *      files without having first logged into the file server.
 *
 *  $EXAMPLES$
 *
 *      fn_entrLog( "RBOOT" )		// enter the directory "\LOGIN\RBOOT"
 *											// on the default server
 *
 *  $SEEALSO$
 *      fn_loginFS()
 *  $INCLUDE$
 *
 *  $END$
 */

FUNCTION fn_entrLog( cDir, nDrives )
	LOCAL cReqBuffer, cRepBuffer := I2BYTE( 0 ) + I2BYTE( 0 ) + I2BYTE( 0 )
	DEFAULT cDir TO "", nDrives TO fn_numLDrv()

	cReqBuffer := I2BYTE( 10 ) 			+;
					  I2BYTE( nDrives )		+;
					  fn_nameL( UPPER( cDir ) )

	nDrives :=_fnReq( E3h, cReqBuffer, @cRepBuffer )

RETURN(NIL)


/*  $DOC$
 *  $FUNCNAME$
 *      fn_loginFS()
 *  $CATEGORY$
 *      Connection/Workstation
 *  $ONELINER$
 *      LoginToFileServer
 *  $SYNTAX$
 *
 *      fn_loginFS( <cName> [, nType ] [, <cPassword> ] ) -> lSuccess
 *
 *  $ARGUMENTS$
 *
 *      <cName> is the name of the object to login.  It can be from 1 to 47
 *      characters in length.
 *
 *      <nType> is an optional object type.  For a list of Netware Object
 *      types, refer to NETTO.CH, or the "Information" section of this
 *      Norton Guide.  If omitted, <nType> defaults to OT_USER.
 *
 *      <cPassword> is an optional password, up to 127 characters.  If a
 *      password is not supplied but required, the system will prompt for
 *      one.
 *
 *  $RETURNS$
 *
 *      A logical indicating whether the object was successfully logged in.
 *
 *  $DESCRIPTION$
 *
 *      This function attempts to log in to the default file server.  If
 *      successful, the server places the Bindery object's ID in the server's
 *      Password Table in the same location as the object occupies in the
 *      server's Connection ID table.
 *
 *      For example, if the workstation occupies postition 97 in the File
 *      Server Connection Table, then the Bindery object's ID number is
 *      placed in the 97th position in the Password Table.  Therefore, if
 *      a Bindery object logs into a specific server from three workstations,
 *      its Binery object ID number is placed in the Password Table three
 *      times, in the positions reserved for each workstation.
 *
 *      NOTE!  This function does _not_ support encrypted passwords.
 *      Due to licensing restrictions from Novell, we could not include
 *      encrypted support in this public domain release.  However, it
 *      is available.  See elsewhere in the Norton Guide for details
 *      on password encryption support.
 *
 *      NOTE:  THIS FUNCTION DOES NOT EMULATE THE NETWARE LOGIN COMMAND!
 *      It does not run the login script, nor does it map any drives!
 *      It simply logs you in.  
 *
 *  $EXAMPLES$
 *
 *      IF !fn_attchFS( "SALES" )          // attach to server
 *         <... unable to attach ...>
 *      ELSE
 *
 *         aFSNameTable := fn_FSName()     // get list of attached servers
 *                                         //  to find ConnectionID
 *
 *         nConnID := ASCAN( aFSNameTable, {|e| e == "SALES" })
 *
 *         fn_SPFCID( nConnID )            // set preferred server
 *                                         //  and log in to it
 *
 *         fn_loginFS( "SLARSEN", OT_USER, "dork" )
 *
 *      ENDIF
 *
 *  $SEEALSO$
 *      FN_LOGOUT() FN_CONNID() FN_CONNINF() FN_FSNAME()
 *  $END$
 */

FUNCTION fn_loginFS( cObjectName, nObjectType, cObjectPassword )
	LOCAL cReqBuffer, cRepBuffer := I2BYTE( 0 ) + I2BYTE( 0 ), nRet
	DEFAULT cObjectName TO "", nObjectType TO OT_USER, cObjectPassword TO ""

#ifdef NOENCRYPTION

	/* this uses the old style unencrypted NetWare API call */

	cReqBuffer := I2BYTE( FT_HEX2DEC( "14" ) )			+;
					  W2HILO( nObjectType )       		+;
					  fn_nameL( UPPER( cObjectName ), 48 )	+;
					  fn_nameL( UPPER( cObjectPassword ) )

	RETURN ( _fnReq( E3h, cReqBuffer, @cRepBuffer ) == 0 )

#else

	/* this provides encryption compatibility with 3.xx and up OS's */

   nRet := _fnLogFS3( cObjectName, nObjectType, cObjectPassword )
   _fnSetErr( nRet )
   return ( nRet == 0 )

#endif

/*  $DOC$
 *  $FUNCNAME$
 *      fn_logout()
 *  $CATEGORY$
 *      Connection/Workstation
 *  $ONELINER$
 *      LogoutFromNetwork
 *  $SYNTAX$
 *
 *      fn_logout() -> NIL
 *
 *  $ARGUMENTS$
 *
 *      None
 *
 *  $RETURNS$
 *
 *      Nothing.  Any errors can be detected with fn_error().
 *
 *  $DESCRIPTION$
 *
 *      This function closes all open files belonging to the object at the
 *      current workstation, logs the workstation out of all file servers,
 *      detaches the workstation from all file servers except the default
 *      file server, and maps a drive to the default file server's
 *      SYS:LOGIN directory.
 *
 *  $EXAMPLES$
 *
 *      fn_logout()								// logs out from all servers
 *
 *  $SEEALSO$
 *      FN_LOGOUTFS() FN_LOGINFS()
 *  $INCLUDE$
 *
 *  $END$
 */

FUNCTION fn_logout()
	LOCAL aRegs[ INT86_MAX_REGS ]

	aRegs[ AX ] := ft_HEX2DEC( "D700" )   // Netware Service D7h

	IF !ft_int86( INT21, aRegs )          // logout
		_fnSetErr( EINT86 )
	ENDIF
RETURN(NIL)


/*  $DOC$
 *  $FUNCNAME$
 *      fn_logoutFS()
 *  $CATEGORY$
 *      Connection/Workstation
 *  $ONELINER$
 *      LogoutFromFileServer
 *  $SYNTAX$
 *
 *      fn_logoutFS( <nConnectionID> ) -> NIL
 *
 *  $ARGUMENTS$
 *
 *      <nConnectionID> is the position that the server occupies in the
 *      workstation shell's Connection ID Table.
 *
 *  $RETURNS$
 *
 *      Nothing.  Any errors can be detected with fn_error().
 *
 *  $DESCRIPTION$
 *
 *      This function logs the current workstation out from a specific
 *      file server.  Logging out from a file server is not the same as
 *      detaching from a file server.  Logging out from a file server does
 *      not relinquish the connection number, but retains the connection
 *      for continued service.  The shell automatically removes all drives
 *      mapped to the specified file server and maps one drive to the
 *      SYS:LOGIN directory on that file server.
 *
 *  $EXAMPLES$
 *
 *      aFSNameTable := fn_FSName()     // get list of attached servers
 *                                      //  to find ConnectionID
 *
 *      nConnID := ASCAN( aFSNameTable, {|e| e == "SALES" })
 *
 *      fn_logoutFS( nConnID )
 *
 *  $SEEALSO$
 *      fn_detchFS() fn_logout() fn_loginfs()
 *  $INCLUDE$
 *
 *  $END$
 */

FUNCTION fn_logoutFS( nConnID )
	LOCAL aRegs[ INT86_MAX_REGS ]
	DEFAULT nConnID TO fn_defCID()

	aRegs[ AX ] := ft_HEX2DEC( "F102" )   // Netware Service F1h, function 2h
	aRegs[ DX ] := nConnID

	IF !ft_int86( INT21, aRegs )          // logout
		_fnSetErr( EINT86 )
	ENDIF
RETURN(NIL)
