/***********************************************************
 * File......: HGlass.prg
 * Author....: Bill Johnson  CIS 73577,465
 * Date......: 12/26/90
 * Revision..: 1.00
 * 
 * Demo of an hour glass progress display with examples of using the
 *  hglass() function to display the progress of an indexing operation
 *  and the progress of a database operation
 *
 * The concept for this function was adapted from a freeware progress
 *  chart by TJS LAB, Orlando, Florida 32858-5366
 *
 *  Note: compile with options /W /A /N
 *
 * Modification history:
 * ---------------------
 *
 */

// constants for reference to row/col array elements
#define DISP_ROW 1
#define DISP_COL 2

// number of records to put in test database
#define NUM_TEST 182

// this FUNCTION for demo purposes only
FUNCTION demohrg

   LOCAL aTestDb    := { { "FIELD1","C",10,0 } }
   LOCAL cScreen    := SAVESCREEN(0,0,maxrow(),maxcol())
   LOCAL nOldRow    := row()
   LOCAL nOldCol    := col()
   LOCAL cOldCursor := SET(_SET_CURSOR,0)
   LOCAL cOldColor  := iIF(ISCOLOR(),SETcolor("W/B"),SETcolor("W/N"))
   LOCAL n

   FIELD FIELD1

   // create a test database - TSTHRG.DBF

   DBCREATE("TSTHRG",aTestDb)
   USE TSTHRG
   FOR n := 1 to NUM_TEST
      APPEND BLANK
      // put in some junk data
      REPLACE FIELD1 with "SOMEJUNK" + ALLTRIM(str(NUM_TEST-n))
   NEXT
   GO TOP

   CLS

   FOR n = 0 to 24
      @ n,0 say REPLICATE("X",80)
   NEXT

   // ok - display the hour glass at default location - centered
   HGlass()

   // now build an INDEX and update the hour glass to show progress
   INDEX on HGlass(FIELD1) to TSTHRGIX

   // ok - pause, clean up, and continue
   CLOSE databases

   // now step through database and update the hour glass to show progress

   USE TSTHRG INDEX TSTHRGIX

   // display the hour glass
   HGlass(0,55,90)

   FOR n := 1 to LASTREC()
      HGlass(n)
      //
      // any data manipulation would be done here
      //
      SKIP
   NEXT

   // leave things like we found them and get out
   ERASE TSTHRG.DBF
   ERASE TSTHRGIX.NTX
   SETCOLOR(cOldColor)
   RESTSCREEN(0,0,maxrow(),maxcol(),cScreen)
   SETPOS(nOldRow,nOldCol)
   SET(_SET_CURSOR,cOldCursor)
   RETURN NIL

/* END of demo code */


/***********************************************************
 * FUNCTION: HGlass() -> xExp
 * purpose : draw and update a progress hour glass
 * syntax  : HGlass([xExp], [nExp1, nExp2])
 * params  : xExp   = NIL or 0 FOR SETup or
 *                  = count IF numeric or
 *                  = char, date, or logical FIELD to INDEX on
 *           nExp1  = top row of hour glass
 *           nExp2  = left column of hour glass
 * RETURNs : unaltered xExp
 *
 * NOTE: Support FOR INDEXing on numeric keys has been intentionally
 *        left out of this FUNCTION until all Clipper 5.0 bugs have
 *        been resolved!
 *
 * usage examples:
 *     HGlass() - SETup hour glass in middle of screen
 *     HGlass(0,2,10) - SETup hour glass with top left corner @ 2,10
 *     HGlass( ,1,50) - SETup hour glass with top left corner @ 1,50
 *     HGlass(nCount)- update glass using numeric count
 *     HGlass(cFIELD)- update glass using FIELD being
 *                       USEd FOR INDEX ON
 *     ( SET test code above)
 */
FUNCTION HGlass(xParm,nStartTop,nStartLeft)
STATIC nTopIx, nBottomIx, nCharHalf
STATIC aBotPos, aTopPos, nTop, nLeft, nHighBot
STATIC cBackColor, cGlassColor, cBaseColor, cSand, cOldColor, cWindow

LOCAL nPercent

// setup the hour glass?

IF ( valtype(xParm) == "N" .and. xParm == 0) .or. xParm == NIL

   // SET STATIC variables to intial values in case this is
   // not the first time we USE HGlass()

   nTopIx    := 1
   nBottomIx := 1
   nCharHalf := 0

   // make sure screen coordinates are valid

   DO CASE
      CASE nStartTop == NIL
         nTop := (maxrow()/2)-9
      CASE nStartTop < 1
         nTop := 1
      CASE nStartTop > (maxrow()-19)   // center vertically
         nTop := maxrow()-19
      OTHERWISE
         nTop := nStartTop
   ENDCASE
   DO CASE
      CASE nStartLeft == NIL
         nLeft := (maxcol()/2)-9
      CASE nStartLeft < 4
         nLeft := 4
      CASE nStartLeft > (maxcol()-16)  // center horizontally
         nLeft := maxcol()-16
      OTHERWISE
         nLeft := nStartLeft
   ENDCASE

   // SET bottom limit FOR falling sand

   nHighBot := nTop + 13

   // SET screen colors

   IF ISCOLOR()
      cBackColor  := "B+/BG+"   // window background
      cGlassColor := "GR+/BG+"  // hour glass
      cBaseColor  := "GR+/GR"   // base of hour glass
      cSand       := "W+/BG+"   // sand
    ELSE
      cBackColor  := cGlassColor := cBaseColor := "W/N"
      cSand       := "W+/N"
   ENDIF

   cOldColor := SETCOLOR(cBackColor)

   // clear the window and draw the hour glass

   cWindow := SAVESCREEN(nTop-1,nLeft-4,nTop+19,nLeft+16)
   @ nTop-1,nLeft-4 clear to nTop+18,nLeft+15
   @ nTop-1,nLeft-4 to nTop+18,nLeft+15 double
   BoxShadow(nTop-1,nLeft-4,nTop+18,nLeft+15)

   SETCOLOR(cGlassColor)
   @ nTop   ,nLeft   say   "ķ"
   @ nTop+1 ,nLeft   say   "          "
   @ nTop+2 ,nLeft   say   "          "
   @ nTop+3 ,nLeft   say   "          "
   @ nTop+4 ,nLeft   say   "          "
   @ nTop+5 ,nLeft   say   "          "
   @ nTop+6 ,nLeft   say   "ķ  Ľ"
   @ nTop+7 ,nLeft+4 say       "  "
   @ nTop+8 ,nLeft+4 say       "  "
   @ nTop+9 ,nLeft   say   "Ľ  ķ"
   @ nTop+10,nLeft   say   "          "
   @ nTop+11,nLeft   say   "          "
   @ nTop+12,nLeft   say   "          "
   @ nTop+13,nLeft   say   "          "
   @ nTop+14,nLeft   say   "          "
   SETCOLOR(cBaseColor)
   IF ISCOLOR()
      @ nTop+15,nLeft-3   say "                  "
    ELSE
      @ nTop+15,nLeft-3   say "   Ľ   "
   ENDIF
   @ nTop+16,nLeft-3 say "     Working!     "
   @ nTop+17,nLeft-3 say "                  "
   SETCOLOR(cSand)
   @ nTop+1 ,nLeft+1 say   ""
   @ nTop+2 ,nLeft+1 say   ""
   @ nTop+3 ,nLeft+1 say   ""
   @ nTop+4 ,nLeft+1 say   ""
   @ nTop+5 ,nLeft+1 say   ""

   // create the arrays FOR the top and bottom of the hour glass
   // each array contains 50 SETs of row/column coordinates

   // each SET of row/column coordinates is accessed twice - once
   //  FOR the bottom half of the block  and once FOR the top half
   //  of the block. This double access allows 100 movements from the
   //  top of the hour glass to the bottom of the hour glass - one FOR
   //  each percentage point of operation competion.

   // each block moves in two parts, first the top half is sucked down 
   //  on each side of the hour glass, then the bottom half is sucked down
   //  on each side.

   aBotPos := { {nTop+14,nLeft+5} , {nTop+14,nLeft+6} , {nTop+14,nLeft+4} ,;
                {nTop+14,nLeft+7} , {nTop+13,nLeft+5} , {nTop+13,nLeft+6} ,;
                {nTop+14,nLeft+3} , {nTop+14,nLeft+8} , {nTop+13,nLeft+4} ,;
                {nTop+13,nLeft+7} , {nTop+12,nLeft+5} , {nTop+12,nLeft+6} ,;
                {nTop+14,nLeft+2} , {nTop+14,nLeft+9} , {nTop+13,nLeft+3} ,;
                {nTop+13,nLeft+8} , {nTop+12,nLeft+4} , {nTop+12,nLeft+7} ,;
                {nTop+11,nLeft+5} , {nTop+11,nLeft+6} , {nTop+14,nLeft+1} ,;
                {nTop+14,nLeft+10}, {nTop+13,nLeft+2} , {nTop+13,nLeft+9} ,;
                {nTop+12,nLeft+3} , {nTop+12,nLeft+8} , {nTop+11,nLeft+4} ,;
                {nTop+11,nLeft+7} , {nTop+10,nLeft+5} , {nTop+10,nLeft+6} ,;
                {nTop+13,nLeft+1} , {nTop+13,nLeft+10}, {nTop+12,nLeft+2} ,;
                {nTop+12,nLeft+9} , {nTop+11,nLeft+3} , {nTop+11,nLeft+8} ,;
                {nTop+10,nLeft+4} , {nTop+10,nLeft+7} , {nTop+12,nLeft+1} ,;
                {nTop+12,nLeft+10}, {nTop+11,nLeft+2} , {nTop+11,nLeft+9} ,;
                {nTop+10,nLeft+3} , {nTop+10,nLeft+8} , {nTop+11,nLeft+1} ,;
                {nTop+11,nLeft+10}, {nTop+10,nLeft+2} , {nTop+10,nLeft+9} ,;
                {nTop+10,nLeft+1} , {nTop+10,nLeft+10} }

   aTopPos := { {nTop+1,nLeft+6} , {nTop+1,nLeft+5} , {nTop+1,nLeft+7} ,;
                {nTop+1,nLeft+4} , {nTop+2,nLeft+6} , {nTop+2,nLeft+5} ,;
                {nTop+1,nLeft+8} , {nTop+1,nLeft+3} , {nTop+2,nLeft+7} ,;
                {nTop+2,nLeft+4} , {nTop+3,nLeft+6} , {nTop+3,nLeft+5} ,;
                {nTop+1,nLeft+9} , {nTop+1,nLeft+2} , {nTop+2,nLeft+8} ,;
                {nTop+2,nLeft+3} , {nTop+3,nLeft+7} , {nTop+3,nLeft+4} ,;
                {nTop+4,nLeft+6} , {nTop+4,nLeft+5} , {nTop+1,nLeft+10},;
                {nTop+1,nLeft+1} , {nTop+2,nLeft+9} , {nTop+2,nLeft+2} ,;
                {nTop+3,nLeft+8} , {nTop+3,nLeft+3} , {nTop+4,nLeft+7} ,;
                {nTop+4,nLeft+4} , {nTop+2,nLeft+10}, {nTop+2,nLeft+1} ,;
                {nTop+3,nLeft+9} , {nTop+3,nLeft+2} , {nTop+4,nLeft+8} ,;
                {nTop+4,nLeft+3} , {nTop+3,nLeft+10}, {nTop+3,nLeft+1} ,;
                {nTop+4,nLeft+9} , {nTop+4,nLeft+2} , {nTop+4,nLeft+10},;
                {nTop+4,nLeft+1} , {nTop+5,nLeft+10}, {nTop+5,nLeft+1} ,;
                {nTop+5,nLeft+9} , {nTop+5,nLeft+2} , {nTop+5,nLeft+8} ,;
                {nTop+5,nLeft+3} , {nTop+5,nLeft+7} , {nTop+5,nLeft+4} ,;
                {nTop+5,nLeft+6} , {nTop+5,nLeft+5} }

ELSEIF valtype(xParm) == "N" .or.;
       (empty(INDEXkey(INDEXord()))) .and. (empty(INDEXkey(1)))

   // IF xParm is numeric then we are being passed a record count. USE
   // the record count to calculate the percent complete.

   // IF xParm is not numeric then it must be a FIELD. Check to see IF the
   // current INDEX key expression is empty and no other INDEXes are active
   // FOR the current DBF. IF both these conditions are TRUE then we are
   // INDEXing and want to display progress.

   // calculate the percent complete

   IF valtype(xParm) == "N"
      nPercent := int( (xParm/LASTREC()) * 100 )
    ELSE
      nPercent := int( (recno()/LASTREC()) * 100 )
   ENDIF

   // make sure the color is right

   SETCOLOR(cSand)

   // allow the sand to flow until the percentage of sand moved is
   // >= the percentage of operation completion

   WHILE ( (nBottomIx * 2) <= nPercent ) .and. nBottomIx < 51

      // is the top half of a character falling ?

      IF nCharHalf < 2

         // let the top half of the character fall

         @ aTopPos[nTopIx][DISP_ROW],aTopPos[nTopIx][DISP_COL] say ''

         // let the sand fall

         SandFall(nTop,nLeft,nHighBot)

         // move the sand to the bottom of the hour glass
         //  and increment the top sand array INDEX pointer

         @ aBotPos[nTopIx][DISP_ROW],aBotPos[nTopIx++][DISP_COL] say ''

         // increment how many top halfs have fallen

         nCharHalf++

       ELSE

         // let the bottom half of a character fall

         @ aTopPos[nBottomIx][DISP_ROW],aTopPos[nBottomIx][DISP_COL] say ' '

         // let the sand fall

         SandFall(nTop,nLeft,nHighBot)

         // move the sand to the bottom of the hour glass
         //  and increment the bottom sand array INDEX pointer

         @ aBotPos[nBottomIx][DISP_ROW],aBotPos[nBottomIx++][DISP_COL] say ''

         // IF not at 100 percent complete, update the
         //  the upper bottom limit FOR falling sand and
         //  increment how many bottom halfs have fallen
         //  ELSE restore screen and prepare to get out

         IF nBottomIx < 51
            nHighBot := min(nHighBot,aBotPos[nBottomIx][DISP_ROW])
            nCharHalf := iIF(++nCharHalf>3,0,nCharHalf)
          ELSE
            RESTSCREEN(nTop-1,nLeft-4,nTop+19,nLeft+16,cWindow)
         ENDIF


      ENDIF

   END

ENDIF

// restore the original color and get out

SETCOLOR(cOldColor)
RETURN (xParm)


/***********************************************************
 * FUNCTION: SandFall()
 * syntax  : SandFall(nExp1, nExp2)
 *     nExp1 == top row of hour glass
 *     nExp2 == left column of hour glass
 * purpose : let the sand fall in the hour glass
 *
 * declared STATIC since only HGlass needs to call
 */
STATIC FUNCTION SandFall(nTop,nLeft,nHighBot)
LOCAL nWorkIx
   @ nTop+6,nLeft+5 say "."
   FOR nWorkIx := nTop+6 to nHighBot-2
       IF (nWorkIx % 2 == 0)
          @ nWorkIx+1,nLeft+5 say "."
        ELSE
          @ nWorkIx+1,nLeft+5 say "."
       ENDIF
       inkey(.01)
       @ nWorkIx,nLeft+5   say "  "
   NEXT
   @ nHighBot-1,nLeft+5 say "  "
   RETURN NIL

/*
*  This FUNCTION comes directly from the Nantucket sample code
*   BoxMenu.prg provided with 5.0 in CLIPPER5\SOURCE\SAMPLE
*
*
*  BoxMenu.prg
*  Sample USEr-defined FUNCTIONs defining menus
*
*  Copyright, Nantucket Corporation, 1990
*  Jake Jacob
*
*  NOTE: compile with /n/w/a/m
*  
*
*  BoxShadow( <nTop>, <nLeft>, <nBottom>, <nRight> ) --> NIL
*  Draw a box shadow with see through
*
*/
FUNCTION BoxShadow( nTop, nLeft, nBottom, nRight )
   LOCAL nShadTop, nShadLeft, nShadBottom, nShadRight
   
   nShadTop   := nShadBottom := MIN(nBottom + 1, MAXROW())
   nShadLeft  := nLeft + 1
   nShadRight := MIN(nRight + 1, MAXCOL())

   RESTSCREEN( nShadTop, nShadLeft, nShadBottom, nShadRight,;
       TRANSFORM( SAVESCREEN(nShadTop, nShadLeft, nShadBottom, nShadRight),;
       REPLICATE("X", nShadRight - nShadLeft + 1 ) ) )

   nShadTop    := nTop + 1
   nShadLeft   := nShadRight := MIN(nRight + 1, MAXCOL())
   nShadBottom := nBottom

   RESTSCREEN( nShadTop, nShadLeft, nShadBottom, nShadRight,;
       TRANSFORM( SAVESCREEN(nShadTop,  nShadLeft , nShadBottom,  nShadRight),;
       REPLICATE("X", nShadBottom - nShadTop + 1 ) ) )

   RETURN NIL
