/////////////////////////////////////////////////////////////////////////////
// Name:     GetIt.Prg
// Copyright:1993 Communication Horizons
// Author:   Neil Weicher
// Platform: Clipper 5.01 or Clipper 5.2
// Notes:    This file contains the implementation of a number of the GetIt 
//           functions.  These implementations do not require that any 
//           changes be made to GETSYS.PRG, and thus should work with any 
//           reasonable variation of GETSYS.PRG.  Note that many functions 
//           can be implemented a lot easier (and quicker) if you are 
//           willing to change GETSYS.PRG.
// 
//           The 'event trapping' functions of GetIt (N_ONTICK, N_NOKEY, 
//           N_ONTIME, etc.), have been migrated to NetLib 6.0.  If you own 
//           GetIt, you can "crossgrade" to NetLib 6.0 at the same cost as 
//           if you were upgrading from an earlier version of NetLib, even 
//           though NetLib is a more expensive product.  Contact Pinnacle 
//           Publishing for more information at 800-788-1900 or 206-251-
//           1900.
// 
//           Compile with option '/n'.
//
//           This file may be freely distributed as source code, only, with 
//           all notices in tact.
////////////////////////////////////////////////////////////////////////////

#include "error.ch"
#include "getexit.ch"
#include "inkey.ch"

memvar GetList

#define ROW_FUDGE	32768	// offset added to field row to "delete" field
#define MAX_COL		80	// number  of columns on screen

static aGETSTACK := nil		// preserve Get Stack info during GotoGet
static nFORCEGET := nil		// this is the target field number
static cNEWFLD := nil		// N_NEWFLD function
static lGOTOACTIVE := .F.	// N_GOTOGET currently active


////////////////////////////////////////////////////////////////////////////
// Syntax:   nGetNo :: N_GOTOGET([GetID])
// Parms:    GetID is the name or number (base 1) of a Get field.
// Action:   "Stuffs" keyboard with down-or up arrows to move the cursor to 
//           the appropriate field.  Temporarily neutralizes preBlock 
//           (WHEN) and postBlock (VALID) methods for fields being 
//           traversed.
// Returns:  The new get number, if successful, 0 otherwise
// Notes:    Requires use of NetLib function N_NEXTKEY.
//
FUNCTION N_GOTOGET(gn)
local k
local old			// current Getno
local new			// new Getno
local pb			// temporary preBlock of target Get
if lGOTOACTIVE			// N_GOTGET currently active: error
	pb := errornew()
	pb:severity := ES_ERROR
	pb:gencode := 0 
	pb:subsystem := "N_GOTOGET"
	pb:description := "Cannot stack N_GOTOGET"
	eval(errorblock(), pb)
	return 0
endif
old := N__GETNO() 		// current Get number
new := N__GETNO(gn)		// target get number
pb := "{ || N__GOTOGET("+str(old,3)+","+str(new,3)+") }"
if new > 0 .and. old > 0 .and. new != old
	k := len(GetList)+3
	aGETSTACK := ARRAY(k)	// create array to store get stack
	aGETSTACK[k] := GetList[old]:postBlock 	// save&disable "verify"
	GetList[old]:postBlock = nil
	aGETSTACK[k-1] := Setkey(K_UP)		// save&disable key, if set
	Setkey(K_UP,nil)
	aGETSTACK[k-2] := Setkey(K_DOWN)	// save&disable key, if set
	Setkey(K_DOWN,nil)
	if new > old				// moving down
		do while old <= new		// save & disable "WHEN"s
			aGETSTACK[old] := GetList[old]:preBlock
			GetList[old]:preBlock := { || .F. }
			old++
		enddo
		keyboard chr(K_DOWN) + N_NEXTKEY()	// prepend to buffer
	else					// moving up
		do while old >= new		// save & disable "WHEN"s
			aGETSTACK[old] := GetList[old]:preBlock
			GetList[old]:preBlock := { || .F. }
			old--
		enddo
		keyboard chr(K_UP) + N_NEXTKEY()	// prepend to buffer
	endif
	GetList[new]:preBlock = ( &pb )		// transitory "WHEN"
	lGOTOACTIVE := .T.			// N_GOTOGET is active
endif
return new


////////////////////////////////////////////////////////////////////////////
// Notes:    Called internally by N_GOTOGET.
// Action:   Acts as the temporary preBlock method of the target field.  
//           Replaces all preBlock and postBlock methods of the traversed 
//           fields.
//
FUNCTION N__GOTOGET(old,new)
local k,pb
lGOTOACTIVE := .F.
k := len(aGETSTACK)
GetList[old]:postBlock = aGETSTACK[k]		// restore prior get stack
if new > old
	do while old <= new
		GetList[old]:preBlock := aGETSTACK[old]
		old++
	enddo
else
	do while old >= new
		GetList[old]:preBlock := aGETSTACK[old] 
		old--
	enddo
endif
SetKey(K_UP,aGETSTACK[k-1])			// restore set keys
SetKey(K_DOWN,aGETSTACK[k-2])
aGETSTACK := nil
pb = GetList[new]:preBlock			// evaluate original "WHEN"
return iif(pb == nil , .T., eval(pb))


////////////////////////////////////////////////////////////////////////////
// Syntax:   nRow :: N_GETROW([GetID])
// Parms:    GetID is the name or number (base 1) of a Get field.
//           If omitted, the current input field is assumed.
// Returns:  Starting row of field on screen.
//
FUNCTION N_GETROW(gn)
local r := iif(gn == nil, getactive():row, GetList[N__GETNO(gn)]:row)
return iif(r < ROW_FUDGE, r , r - ROW_FUDGE)



////////////////////////////////////////////////////////////////////////////
// Syntax:   nCol :: N_GETCOL([GetID])
// Parms:    GetID is the name or number (base 1) of a Get field.
//           If omitted, the current input field is assumed.
// Returns:  Starting column of field on screen.
//
FUNCTION N_GETCOL(gn)
return iif(gn == nil, getactive():Col, GetList[N__GETNO(gn)]:Col)



////////////////////////////////////////////////////////////////////////////
// Syntax:   cAlias :: N_GETALIAS([GetID])
// Parms:    GetID is the name or number (base 1) of a Get field.
//           If omitted, the current input field is assumed.
// Returns:  Alias (including '->') of specified field.
//
FUNCTION N_GETALIAS(get)
local k, gn
local g := iif(get == nil , getactive(), GetList[N__GETNO(get)])
if g:subscript != nil
	return ""
endif
gn := upper(G:name)
if '->' $ gn
	return substr(gn,1,at('->',gn)+1)
endif
if !empty(alias())
	if CH_FLDNUM(gn) > 0
		return alias()+"->"
	endif
endif
return "M->"


////////////////////////////////////////////////////////////////////////////
// Syntax:   nName :: N_GETVAR([GetID])
// Parms:    GetID is the name or number (base 1) of a Get field.
//           If omitted, the current input field is assumed.
// Returns:  Full name of specified input field, including 'alias->' or 
//           [subscript] if applicable.
//
FUNCTION N_GETVAR(gn)
local n, i, k
if gn == nil
	n := upper(getactive():name)
	k := N__GETITEM()
else
	i := N__GETNO(gn)
	n := upper(GetList[i]:Name)
	k := N__GETITEM(i)
endif
if '->' $ n
	n := substr(n,at('->',n)+2)
endif
for i = 1 to len(k) step 2
	n := n + "[" + ltrim(str(ch_asc2(k,i))) + ']'
next i
return n


////////////////////////////////////////////////////////////////////////////
// Syntax:   nSub :: N_GETSUB1([GetID])
// Parms:    GetID is the name or number (base 1) of a Get field.
//           If omitted, the current input field is assumed.
// Returns:  Subcript #1 of the specified field.  Zero is returned if field 
//           is not part of an array, or if an invalid field is specified.
//
FUNCTION N_GETSUB1(gn)
local i
local k := iif(gn == nil, N__GETITEM() , N__GETITEM(N__GETNO(gn)))
if len(k) > 0
	return ch_asc2(k,1)
endif
return 0


////////////////////////////////////////////////////////////////////////////
// Syntax:   nSub :: N_GETSUB2([GetID])
// Parms:    GetID is the name or number (base 1) of a Get field.
//           If omitted, the current input field is assumed.
// Returns:  Subcript #2 of the specified field.  Zero is returned if field 
//           is not part of a multi-demensional array or if an invalid 
//           field is specified.
//
FUNCTION N_GETSUB2(gn)
local i
local k := iif(gn == nil, N__GETITEM() , N__GETITEM(N__GETNO(gn)))
if len(k) > 2
	return ch_asc2(k,3)
endif
return 0


////////////////////////////////////////////////////////////////////////////
// Syntax:   nLastGet :: N_GETLAST()
// Returns:  Number of Get fields in currently active READ.
//
FUNCTION N_LASTGET
return len(GetList)


////////////////////////////////////////////////////////////////////////////
// Syntax:   cPic :: N_GETPIC([GetID [,cNewPic]])
// Parms:    GetID is the name or number (base 1) of a Get field.
//           If omitted, the current input field is assumed.
// Action:   If 'cNewPic' is specified, it becomes the new picture for the 
//           field.
// Returns:  Current picture.
//
FUNCTION N_GETPIC(gn,pic)
local k
if gn == nil
	return getactive():picture
endif
k := N__GETNO(gn)
if valtype(pic) = 'C'
	GetList[k]:Picture = pic
endif
return GetList[k]:Picture


////////////////////////////////////////////////////////////////////////////
// Syntax:   bValid :: N_GETVALID([GetID [,bNewValid]])
// Parms:    GetID is the name or number (base 1) of a Get field.  If 
//           omitted, the current input field is assumed.
// Action:   If 'bNewValid' is specified, it becomes the new VALID 
//           postBlock of the specified field.
// Returns:  Current postBlock of field.
//
FUNCTION N_GETVALID(gn)
return iif(gn == nil, getactive():postblock,GetList[N__GETNO(gn)]:postBlock)


////////////////////////////////////////////////////////////////////////////
// Syntax:   nLen :: N_GETLEN([GetID])
// Parms:    GetID is the name or number (base 1) of a Get field.  If 
//           omitted, the current input field is assumed.
// Returns:  Display length of field.
//
FUNCTION N_GETLEN(gn)
return iif(gn == nil, N__GETLEN() , N__GETLEN(N__GETNO(gn)))


////////////////////////////////////////////////////////////////////////////
// Syntax:   N_REGET([nFrom [,nTo]))
// Parms:    2 parameters- start and end Get number to refresh
//           1 parameter- Get number to refresh
//           0 parameters- refresh all Gets
// Action:   Redisplays all or specified Get fields on screen.  Useful for 
//           when returning from a nested read.  Clipper automatically 
//           refreshes current Get, but no others.
//
FUNCTION N_REGET(get1,get2)
local k
if valtype(get1) = 'U'
	get1 := 1
	get2 := len(GetList)
elseif valtype(get2) = 'U'
	get2 := get1
endif
for k = get1 to get2
	if GetList[k]:Row < ROW_FUDGE
		GetList[k]:display()
	endif
next k
if getactive():preblock != nil
	eval(getactive():preblock)
endif
return NIL


////////////////////////////////////////////////////////////////////////////
// Syntax:   cProc :: N_NEWFLD([cProc])
// Parms:    Name of procedure to be called before entering an input field.  
//           Null string ("") to disable function. 
// Action:   Executes specified procedure before entering into any Get 
//           field.  We recommend using the WHEN clause of the GET command 
//           since it is far more flexible.
// Returns:  Name of procedure.
//
FUNCTION N_NEWFLD(fn)
local pb,last,k
if pcount() == 0
	return cNEWFLD
endif
if fn == nil .or. empty(fn)
	cNEWFLD := nil
	return cNEWFLD
endif
last := len(GetList) 
cNEWFLD := fn
for k = 1 to last
	if GetList[k]:Row < ROW_FUDGE
		pb := " { || N__newfld('"+cNEWFLD+"'," + str(k,3) + ") }"
		GetList[k]:preBlock := ( &pb )
	endif
next k
return cNEWFLD


////////////////////////////////////////////////////////////////////////////
// Notes:    Used internally by N_NEWFLD
//
FUNCTION N__NEWFLD(name,num)
nFORCEGET := num
DO &name
nFORCEGET := nil
return .T.


////////////////////////////////////////////////////////////////////////////
// Syntax:   N_DELGET([GetID])
// Parms:    GetID is the name or number (base 1) of a Get field.
//           If omitted, the current input field is assumed.
// Action:   Disables Get (i.e., does not allow input).  It does not 
//           actually remove it from the Get list.
//
FUNCTION N_DELGET(gn)
local k := N__GETNO(gn)
if GetList[k]:Row < ROW_FUDGE
	GetList[k]:Row := GetList[k]:Row + ROW_FUDGE
	GetList[k]:preBlock := { || .F. }
endif
return NIL


////////////////////////////////////////////////////////////////////////////
// Syntax:   N_ISDELGET([GetID])
// Parms:    GetID is the name or number (base 1) of a Get field.
//           If omitted, the current input field is assumed.
// Returns:  .T. if Get has been "deleted" with N_DELGET, .F. if the Get is 
//           active or if an invalid Get field has been specified.
//
FUNCTION N_ISDELGET(gn)
return (GetList[N__GETNO(gn)]:Row >= ROW_FUDGE)


////////////////////////////////////////////////////////////////////////////
// Syntax:   N_RECGET([GetID])
// Parms:    GetID is the name or number (base 1) of a Get field.
//           If omitted, the current input field is assumed.
// Action:   "Recalls" Get (i.e., re-enables Get after N_DELGET).
//
FUNCTION N_RECGET(gn)
local pb
local k := N__GETNO(gn)
if GetList[k]:Row >= ROW_FUDGE
	GetList[k]:Row = GetList[k]:Row - ROW_FUDGE 
	if cNEWFLD != nil
		pb := " { || N__newfld('"+cNEWFLD+"'," + str(k,3) + ") }"
		GetList[k]:preBlock := ( &pb )
	endif
endif
return .T.


////////////////////////////////////////////////////////////////////////////
// Syntax:   getList :: N_SAVEGETS()
// Returns:  Current value of global GetList.  
// Notes:    Use with N_RESTGETS.  This implementation only works with the 
//           global GetList.
//
FUNCTION N_SAVEGETS
local saveblock := GetList
GetList := {} 
return (saveblock)


////////////////////////////////////////////////////////////////////////////
// Syntax:   N_RESTGETS(getList)
// Action:   Restores value of global GetList.
// Notes:    Use with N_SAVEGETS.  This implementation only works with the 
//           global GetList.
//
FUNCTION N_RESTGETS(saveblock)
GetList := saveblock
return NIL


////////////////////////////////////////////////////////////////////////////
// Syntax:   N_GETNO([GetID])
// Parms:    GetID is the name or number (base 1) of a Get field.
//           If omitted, the current input field is assumed.
// Returns:  Specified Get number (base 1).
//
FUNCTION N_GETNO(gn)
return N__GETNO(gn)


////////////////////////////////////////////////////////////////////////////
// Syntax:   nGetNo :: N_SFIELD(nRow,nCol)
// Parms:    Screen row and column
// Returns:  Get number of field that lies under specified cursor 
//           coordinate.
// Notes:    Useful for use with "mouse" libraries.
//
FUNCTION N_SFIELD(r,c)
local k, start
local cursor := iif(pcount() = 0 , row() * MAX_COL + col() , ;
				r * MAX_COL + c)
for k = len(GetList) to 1 step -1
	start := GetList[k]:Row * MAX_COL + GetList[k]:Col
	if cursor >= start .and. cursor < start + N__GETLEN(k)
		exit
	endif
next k
return k


////////////////////////////////////////////////////////////////////////////
// Syntax:   N_MOVEGET([nFrom [,nTo]))
// Action:   Repositions 'from' to be in front of 'to' Get.
//
FUNCTION N_MOVEGET(from,to)
local frgetp, togetp, k
local varlist := {}
local frget := iif(from == nil , len(GetList), N__GETNO(from))
local toget := iif(to   == nil , N__GETNO() , N__GETNO(to))
if frget > 0 .and. toget > 0 .and. toget != frget
	varlist := array(len(GetList))
	frgetp := togetp := 1
	if frget > toget
		for k = 1 to len(varlist)
			if (togetp == toget)
				varlist[togetp++] := GetList[frget]
			elseif (frgetp == frget)
				varlist[togetp++] = GetList[++frgetp]
			else
				varlist[togetp++] := GetList[frgetp++]
			endif
		next k
	else
		for k = 1 to len(varlist)
			if (togetp == toget-1)
				varlist[togetp++] := GetList[frget]
			elseif (frgetp == frget)
				varlist[togetp++] = GetList[++frgetp]
				frgetp++
			else
				varlist[togetp++] := GetList[frgetp++]
			endif
		next k
	endif
	for k = 1 to len(varlist)
		GetList[k] := varlist[k]
	next k
	N_NEWFLD(N_NEWFLD())
else
	frget := -1
endif
return frget >= 0


////////////////////////////////////////////////////////////////////////////
// Notes:    Used internally.  
// Returns:  If parameter is numeric, function returns the same number if 
//           it is a valid get field, zero otherwise.  If parameter is 
//           character, function assumes it is a field name and returns 
//           corresponding number.  If no parameter, function returns 
//           current get number.
//
static FUNCTION N__GETNO(gn)
local k,name, item, alias, kname, kitem, kalias
do case
case valtype(gn) = 'N'
	return iif(gn > 0 .and. gn <= len(GetList), gn, 0)
case valtype(gn) = 'U'
	if nFORCEGET != nil
		return nFORCEGET
	endif
	for k = len(GetList) to 1 step -1
		if GetList[k]:HasFocus 
			exit
		endif
	next k
	return k
case valtype(gn) = 'C'
	name := N__GETVAR(gn)
	item := N__GETSUB(gn)
	alias := N__GETALIAS(gn)
	for k = len(GetList) to 1 step -1
		kname := upper(GetList[k]:Name)
		kalias := N__GETALIAS(kname)
		kname := N__GETVAR(kname)
		if name == kname .and. (empty(alias) .or. alias == kalias)
			if item == N__GETITEM(k)
				exit
			endif
		endif
	next k
	return k
endcase
return 0


////////////////////////////////////////////////////////////////////////////
// Notes:    Used internally.
// Parm:     Name or number of Get field.
// Returns:  Display length of specified field.
//
static FUNCTION N__GETLEN(k)
local p, n
if k == nil
	n := getactive():name
	p := getactive():picture
else
	n := GetList[k]:Name
	p := GetList[k]:Picture
endif
if p == nil
	p := ""
endif
return len(transform( (&n) ,p))


////////////////////////////////////////////////////////////////////////////
// Notes:    Used internally.
// Parm:     Name of Get Field
// Returns:  Base field name (i.e., strips off array '[]' or 'alias->' 
//           references from field name.
//
static FUNCTION N__GETVAR(name)
local n
n = trim(ltrim(upper(name)))
if '[' $ n
	n = trim(substr(n,1,at('[',n)-1))
elseif '->' $ n
	n = substr(n,at('->',n)+2)
endif
return n


////////////////////////////////////////////////////////////////////////////
// Notes:    Used internally.
// Parm:     Name of Get Field
// Returns:  'alias->' portion of get field name, if one is specified.
//
static FUNCTION N__GETALIAS(name)
local n
n = trim(ltrim(upper(name)))
if '[' $ n
	n = ""
elseif '->' $ n
	n = trim(substr(n,1,at('->',n)+1))
elseif !empty(alias())
	if CH_FLDNUM(n) > 0
		n = alias() + "->"
	else
		n = ""
	endif
else
	n = ""
endif
return n


////////////////////////////////////////////////////////////////////////////
// Notes:    Used internally.
// Parm:     Name of Get Field
// Returns:  Ascii value of subscript reference in field name.
//
static FUNCTION N__GETSUB(name)
local k
local i := ""
local n := trim(ltrim(upper(name)))
do while (k := at('[',n)) > 0
	n := substr(n,k+1)
	i := i + ch_chr2(val(n))
enddo
return i


////////////////////////////////////////////////////////////////////////////
// Notes:    Used internally.
//
static FUNCTION N__GETITEM(k)
local i := ""
local m := iif(k == nil, getactive():subscript, GetList[k]:subscript)
if m != nil
	for k = 1 to len(m)
		i = i + ch_chr2(m[k])
	next k
endif
return i

// end of GETIT.PRG
