/*-------------------------------------------------------------------*
 | DBF.C  -  Displays Info on dBase III and Clipper data/index files |
 |                                                                   |
 |        -  R.Trevithick - Sun  11-13-1988                          |
 |                                                                   |
 |        -  Public Domain                                           |
 |                                                                   |
 | -  Turbo C 2.0 ...... tcc -w -p -lt -mt -f- -G -N- -K -O -r -Z -d |
 *-------------------------------------------------------------------*/

char *usage =
        "\ndbase file information \xfe R.Trevithick " __DATE__ "\n\n"
        "    DBF filespec1 [filespec2...] [/v] [/b] [/t]\n\n\t"
        "/v   Verbose; header information\n\t"
        "/b   Brief; single line summary mode\n\t"
        "/t   Total number of files and bytes\n";

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <dir.h>
#include <dos.h>
#include <fcntl.h>
#include <io.h>
#include <process.h>
#include "dbf.h"

enum { OK = 0, BOMB, ARGS };            /* General use return codes */

int             handle;                 /* General-use file handle  */
char            verbose = 0,            /* Extra information mode   */
                brief = 0,              /* Single line summary mode */
                disp_total = 0,         /* Display # files found    */
                rigid;                  /* rigid testing if != .N?X */
unsigned        dbf_count = 0,          /* Total - data files       */
                dbt_count = 0,          /* Total - memo files       */
                ndx_count = 0;          /* Total - index files      */
unsigned long   byte_count = 0L;        /* Total - number of bytes  */

struct ffblk    f;                      /* global access filenames  */

void    display(char *dfn),             /* function prototypes      */
        display_dbf(char *dfn),
        display_ndx(char *dfn),
        end_prog(int code);




/**********************************************************
 * MAIN()                                                 *
 *                                                        *
 * --Parse command line to get filespec(s) and switches.  *
 * --If no extent entered, try all of the usual ones.     *
 * --Call display() routine with the filespec(s).         *
 * --Call end_prog() routine with exit status code.       *
 **********************************************************/
void cdecl main(int argc, char *argv[])
{
    char    *ext_ptr, dfn[MAXPATH];
    int     i, found;

    for (i = 1, found = 0; i < argc; ++i) {
        if (argv[i][0] == '/')
            switch(tolower(argv[i][1])) {
                case 'v': verbose = 1; break;
                case 'b': brief = 1; break;
                case 't': disp_total = 1;
            } 
        else found = i;
    }
    if (argc < 2 || !found) end_prog(ARGS);

    for (i = 1; i < argc; i++) {
        if (argv[i][0] == '/') continue;
        strcpy(dfn, strupr(argv[i]));

        if (!strchr(dfn, '.')) {
            strcat(dfn, ".DBF");
            ext_ptr = strchr(dfn, '.');
            display(dfn);
            strcpy(ext_ptr, ".N?X");
        }

        display(dfn);
    }
    end_prog(byte_count ? OK : BOMB);
}




/*********************************************************************
 * FUNCTION DISPLAY()                                                *
 * This routine is a simple dispatcher.  It accepts a filespec from  *
 * main(), then calls display_dbf() with a full constructed pathname *
 * corresponding to each matching directory entry found.  If extent  *
 * is .N?X, sets 'relaxed' mode for less testing of headers.         *
 *********************************************************************/
void display(char *dfn)
{
    char    *name_ptr, fn[MAXPATH];
    int     found;

    name_ptr = strcpy(fn, dfn);
    for (found = strlen(fn); found > -1; found--)
        if (fn[found] == '\\' || fn[found] == ':') {
            name_ptr = fn+found+1;
            break;
        }

    if (0 != (found = findfirst(dfn, &f, DBF_ATTR))) return;
    while (!found) {
        rigid = !(strstr(f.ff_name, ".NDX") || strstr(f.ff_name, ".NTX"));
        strcpy(name_ptr, f.ff_name);
        display_dbf(fn);
        found = findnext(&f);
    }
}




/********************************************************************
 * FUNCTION DISPLAY_DBF()                                           *
 * This opens the file and trys to interpret it as a DBF structure. *
 * If the file can't be opened, it simply returns.                  *
 * If opened but wrong format, calls display_ndx() which then trys  *
 * to interpret the file as an index structure.                     *
 ********************************************************************/
void display_dbf(char *dfn)
{
    struct  dbf_header  dh;                 /* defined in dbf.h */
    struct  dbf_record  dr;
    struct ftime        tm;
    char                *cp, mfn[MAXPATH];
    int                 f_num = 0, memo_status = -1;
    unsigned long       size, m_size;


    if (-1 == (handle = _open(dfn, O_BINARY | O_RDONLY))) {
        _close(handle);
        return;
    }

/*-------------------------------------------------------------------*
 | Now the fun starts.  Apparently a data (DBF) file must have:      |
 |                                                                   |
 | - 1st byte == 0x03 or 0x83 (high bit means .dbt file needed)      |
 | - valid year, month and day values (i.e. within normal ranges)    |
 | - non-zero header length and record length                        |
 |                                                                   |
 | We'll cheat a bit for speed.  Since the year and day bytes are in |
 | the same position as the two critical bytes for index file tests, |
 | we can check them now - and avoid a call to display_ndx() if they |
 | are non-zero.  I'll re-write all of this when I get some docs...  |
 *-------------------------------------------------------------------*/
    if ( sizeof(dh) != (_read(handle, (char *) &dh, sizeof(dh))) ) {
        _close(handle);
        return;
    }

    if ( !(dh.id_byte == NORM_DBF || dh.id_byte == MEMO_DBF)
      || !(dh.year > -1 && dh.year < 100)
      || !(dh.month > 0 && dh.month < 13)
      || !(dh.day > 0 && dh.day < 32)
      || !(dh.header_len && dh.record_len) )
          if (rigid && (dh.year || dh.day)) {
              _close(handle);
              return;
          } else {
              display_ndx(dfn);
              return;
          }

    ++dbf_count;
    byte_count += (size = filelength(handle));

    if (brief) {
        printf("\n%s%*c%.2d/%.2d/%.2d%10lu Recs%12ld bytes",
                   f.ff_name, 21 - strlen(f.ff_name), ' ',
                   dh.month, dh.day, dh.year, dh.num_recs, size);
    } else {
        printf("\n\n%s %s\n\n%13s%.2d/%.2d/%.2d%16s%lu\n",
                "***File:", dfn, 
                "Updated...", dh.month, dh.day, dh.year,
                "Records...", dh.num_recs);

        while (sizeof(dr) == (_read(handle, (char *) &dr, sizeof(dr)))) {

            if (dr.field_name[0] == 0x0d) break;

            printf("\n%4d.......%s%*c",
                ++f_num, &dr.field_name,
                16 - strlen((char *) &dr.field_name), ' ');

            switch(dr.data_type) {
                case 'C': cp = "char   "; break;
                case 'N': cp = "numeric"; break;
                case 'L': cp = "Logical"; break;
                case 'D': cp = "Date   "; break;
                case 'M': cp = "Memo   ";
            }
            printf("%s%8u", cp, dr.field_length);
            if (dr.data_type == 'N') printf("%5u", dr.decimals);
        }
        printf("\n");
    }

    /* Get the memo (.dbt) file data if we'll be needing it */
    if (dh.id_byte == MEMO_DBF && (disp_total || verbose)) {
        strcpy(mfn, dfn);
        if (NULL != (cp = strchr(mfn, '.')))
            strcpy(cp, ".DBT");
        else
            strcat(mfn, ".DBT");
        _close(handle);
        memo_status = (handle = _open(mfn, O_BINARY | O_RDONLY));
        if (memo_status != -1) {
            ++dbt_count;
            byte_count += (m_size = filelength(handle));
        }
    }

    if (verbose) {
        printf("\n - File ID byte............%.2Xh"
               "\n - Header size.............%u"
               "\n - Record size.............%u"
               "\n - Actual file size........%lu",
                   dh.id_byte, dh.header_len, dh.record_len, size);

        if (dh.id_byte == MEMO_DBF) {
            printf("\n - Memo file");
            if (memo_status != -1) {
                getftime(handle, &tm);
                tm.ft_year = (tm.ft_year + 1980) % 100;
                printf(" size..........%lu\n"
                       " - Memo file updated.......%.2d/%.2d/%.2d",
                           m_size, tm.ft_month, tm.ft_day, tm.ft_year);
            } else
                printf("...............cannot be opened!");
        }
        printf("\n");
    }
    _close(handle);
}




/********************************************************************
 * FUNCTION DISPLAY_NDX()                                           *
 * This is called whenever display_dbf() can't interpret the file.  *
 * We do a seek back to the start, then try to interpret it as an   *
 * index file (allowing for both dBase III and Clipper formats.)    *
 * -This module is called only by display_dbf().                    *
 ********************************************************************/
void display_ndx(char *dfn)
{
    struct  ndx_header  xh;                  /* defined in dbf.h */
    struct  ftime       tm;
    int     clipper;
    long    size;


    if (0L != (lseek(handle, 0L, SEEK_SET))) {
        _close(handle);
        return;
    }

/*------------------------------------------------------------------------*
 | This next part is very questionable.  It _appears_ that the following  |
 | rules allow determining what sort of index file we have:               |
 |                                                                        |
 | * - if 2nd or 4th byte of id_bytes != 0 it's not a valid index         |
 |   - if key_length field == 0 it's not valid                            |
 |   - if there isn't an alpha char in 1st byte of key it's not valid     |
 |   - if 1st byte of id_bytes == 0x06 it's probably a Clipper file       |
 |                                                                        |
 | * These two bytes were ok'd by display_dbf(), so no need to test them. |
 |                                                                        |
 | Note that the asciiz key expression field begins 2 bytes earlier, and  |
 | the key length field 1 byte later, in the Clipper index structure than |
 | in the dBase structure.  Just to keep us on our toes, no doubt.        |
 *------------------------------------------------------------------------*/
    if ( sizeof(xh) != (_read(handle, (char *) &xh, sizeof(xh))) ) {
        _close(handle);
        return;
    }

    clipper = (xh.id_bytes[0] == CLIP_NXF);

    if ( rigid
      && ((clipper && !(xh.clip_klen && isalpha(xh.clip_key[0])))
      ||  (!clipper && !(xh.db_klen && isalpha(xh.db_key[0])))) ) {
        _close(handle);
        return;
    }

    ++ndx_count;
    byte_count += (size = filelength(handle));

    getftime(handle, &tm);
    tm.ft_year = (tm.ft_year + 1980) % 100;

    if (brief) {
        printf("\n%s%*c", f.ff_name, 14 - strlen(f.ff_name), ' ');
    } else
        printf("\n\n%s %s\n\n%13s%.2d/%.2d/%.2d%22s%lu\n\n   ",
                "***File:", dfn, "Updated...", 
                            tm.ft_month, tm.ft_day, tm.ft_year,
                            "Size in bytes...", size);

    printf("Index: %s", clipper ? &xh.clip_key[0] : &xh.db_key[0]);

    if (!brief) printf("\n");

    if (verbose) {
        printf("\n - Key length.........%d"
               "\n - Probable type......%s"
               "\n - Reserved bytes.....%.2Xh %.2Xh %.2Xh %.2Xh\n",
                   clipper ? xh.clip_klen : xh.db_klen,
                   clipper ? "Clipper" : "dBase",
                   xh.id_bytes[0], xh.id_bytes[1], 
                   xh.id_bytes[2], xh.id_bytes[3]);
    }
    _close(handle);
}




/************************************************************
 * FUNCTION END_PROG()                                      *
 * Display an exit message, set DOS errorlevel return code. *
 * -Uses exit() instead of _exit() so that stdout stream    *
 *  will be properly flushed when doing i/o redirection.    *
 ************************************************************/
void end_prog(int code)
{
    unsigned long   total;

    if (code == ARGS) 
        fprintf(stderr, usage);
    else
        if (code == BOMB || !byte_count)
            fprintf(stderr, "\n---no valid files found\n");
    else 
        if (brief && !verbose) printf("\n");

    if (disp_total) {
        total = (unsigned long) dbf_count + dbt_count + ndx_count;
        printf("\n---%lu bytes in %d file", byte_count, total);
        if (total) {
            if (total > 1) printf("s");
            printf("   (");
            if (dbf_count) printf("  %d data", dbf_count);
            if (dbt_count) printf("  %d memo", dbt_count);
            if (ndx_count) printf("  %d index", ndx_count);
            printf("  )");
        } else printf("s");
        printf("\n");
    }
    exit(code);
}
