 /*
   DBFDOC.PRG
   
   Author     : Phil Barnett
   
   Credits    : Looks like Scan-A-Lyzer .DBF output.
   
   Written    : 07/27/93
   
   Function   : Document .DBF structures.
   
   Purpose    : Creates File Structure Appendix of User Manual.
   
   Syntax     : DBFDOC FILENAME  (Must be .DBF)
   
   Parameters : FILENAME is the exact name of the file to be documented.
   
   Returns    : NIL
   
   Compile    : CLIPPER DBFDOC /N
   
   Link       : RTLINK FI DBFDOC
   
   Example    : DBFDOC ANY.DBF
   
   Warnings   : Creates a file with same name and .STU extension.
   
   Released to Public Domain by Author
   
*/

#include "directry.ch"

#define dbfsize() ( ( RECSIZE() * lastrec() ) + HEADER() + 1 )

static nHandle

FUNCTION dbfdoc( cFileName )

local nOutputWidth := 79
local cFiller := SPACE(6)
local aFieldList
local cSepLine
local nOffset
local nDotAt
local cRootName
local aDirData
local dFilDate
local cFilTime
local x
local cOutput
local cDataType

cFileName := UPPER( alltrim( cFileName ) )

IF !( '.DBF' $ cFileName )
  ? 'Usage:'
  ?
  ? 'DBFDOC FILESPEC       Use a file name, must end with .DBF'
  ?
  QUIT
ENDIF

IF !FILE(cFileName)
  ? 'File Not Found'
  ?
  QUIT
ENDIF

USE (cFileName) ALIAS TempFile

aFieldList := dbstruct()

cSepLine := replicate( '-', ncOutputWidth )

nOffset := 0

nDotAt := AT('.',cFileName)
cRootName := LEFT(cFileName,nDotAt-1)

nHandle := fcreate( cRootName + '.STU' )

aDirData := directory(cFileName)

dFilDate := aDirData[1,F_DATE]
cFilTime := aDirData[1,F_TIME]

send( '' )
send( '  DBFDOC' + padl( 'The .DBF Structure Analyzer/Documentor!', ncOutputWidth - 10 ) )
send( cSepLine )
send( pad('  '+cFileName,30)+padl('File Date:   '+fulldate(dFilDate)+' '+LTRIM(ampm(cFilTime))+'  ',ncOutputWidth-32) )
send( cSepLine )
send( '' )
send( pad('  Record Count: ' + LTRIM(STR(lastrec())),22) + padl('Table Size: ' + LTRIM(TRANSFORM(dbfsize(),'999,999,999,999')) + '  ',ncOutputWidth-22) )
send( '' )
send( '  Field    Field Name  Type           Width       Dec.    Begin      End' )
send( '' )

FOR x := 1 TO fcount()
  
  DO CASE
  CASE aFieldList[ x, 2 ] == 'C'
    cDataType := "Character"
  CASE aFieldList[ x, 2 ] == 'N'
    cDataType := "Numeric  "
  CASE aFieldList[ x, 2 ] == 'L'
    cDataType := "Logical  "
  CASE aFieldList[ x, 2 ] == 'M'
    cDataType := "Memo     "
  CASE aFieldList[ x, 2 ] == 'D'
    cDataType := "Date     "
  OTHERWISE
    cDataType := "Undefined"
  ENDCASE
  
  IF aFieldList[ x, 4 ] == 0
    cOutput := SPACE(4)
  ELSE
    cOutput := STR( aFieldList[ x, 4 ], 4 )
  ENDIF
  
  send( STR(x,5) + cFiller + pad( aFieldList[ x, 1 ], 12 ) + cDataType +;
  cFiller + STR( aFieldList[ x, 3 ], 4 ) +;
  cFiller + cOutput +;
  cFiller + STR( nOffset, 4 ) +;
  cFiller + STR( nOffset + aFieldList[ x, 3 ] - 1, 4 ) )
  
  nOffset += aFieldList[ x, 3 ]
  
NEXT

send( '' )
send( '           Total                '+STR( nOffset ) )

fclose(nHandle)

USE

RETURN nil

*!*****************************************************************************
*!
*!       Function: SEND()
*!
*!*****************************************************************************
static FUNCTION send(cTxt,lNewLine)

lNewLine := IF(lNewLine == nil,.T.,lNewLine)

IF lNewLine
  cTxt += CHR(13)+CHR(10)
ENDIF

fwrite(nHandle,cTxt)

return NIL

*!*****************************************************************************
*!
*!       Function: FULLDATE()
*!
*!*****************************************************************************
static FUNCTION fulldate(dAnyDate)

dAnyDate := IIF(valtype(dAnyDate) <> 'D',DATE(),dAnyDate)

RETURN CMONTH(dAnyDate) + ' ' + LTRIM(STR(DAY(dAnyDate),2)) + ', ' + STR(YEAR(dAnyDate),4)

