#include <nandef.h>
#include <extend.h>

#define	numelems 32		/* max number of parms passed at one time */
#define statbuf 1		/* non-zero if using a static buffer */

static unsigned nextparm = 0, numparms = 0;

static struct {			/* array of parm types and positions */
  char parmtype;
  unsigned short parmpos;
} parmtable[numelems];

#if statbuf

/* declare static parm buffer and initialize buffer size variable */

static char parmbuffer[numelems * sizeof(double)];
static unsigned parmbufsize = (numelems * sizeof(double));

#else 

/* otherwise, parm buffer is created by the parminit(n) function, called from
   clipper, where n is the size of the buffer to create. parmfree() is used to
   release the buffer. This method has the disadvantage of fragmenting the 
   free pool if not used with care.
*/

static char *parmbuffer;
static unsigned parmbufsize = 0;


/* parminit(n) - initializes the parm buffer to n bytes. Return codes are:
      0 - success 
     -1 - unable to allocate parm buffer
*/

CLIPPER parminit(void) {
	int bufsize, rcode;

	bufsize = _parni(1);
	rcode = 0;

	/* If a previous buffer exists, release it */
	if (parmbufsize)
		_exmback(parmbuffer, parmbufsize);

	/* Allocate new buffer */
	if ((parmbuffer = _exmgrab(bufsize)) == NULL) {
		rcode = -1;
		parmbufsize = 0;
	}
	else {
		parmbufsize = bufsize;
		numparms = nextparm = 0;
		parmtable[0].parmtype = ' ';
		parmtable[0].parmpos = 0;
	}
	_retni(rcode);
}

/* parmfree() - takes no parameters. Return codes are:
      0 - success 
*/

CLIPPER parmfree(void) {
	if (parmbufsize) {
		_exmback(parmbuffer, parmbufsize);
		parmbufsize = 0;
	}
	_retni(0);
}

#endif

/* cparm(n) - returns parameter n to Clipper. If n < 1 or n > num parms passed
   then cparm() returns a (.f.). Parameter count and types can be gotten by
   calling cparmtype(n) and cparmcount()
*/

CLIPPER cparm(void) {
	unsigned parmno;
	void *retbuf;

	parmno = _parni(1) - 1;

	if (numparms == 0) {
		numparms = nextparm;
		nextparm = 0;
	}

	/* Check to see that a valid parameter is requested */

	if ((0 <= parmno) && (parmno < numparms)) {
		retbuf = parmbuffer + parmtable[parmno].parmpos;
		switch (parmtable[parmno].parmtype) {
			case 'C' :	_retc(* ((char **) retbuf));
					break;
			case 'I' :	_retni(* ((int *) retbuf));
					break;
			case 'L' :	_retnl(* ((long *) retbuf));
					break;
			case 'D' :	_retnd(* ((double *) retbuf));
					break;
			case 'l' :	_retl(* ((int *) retbuf));
					break;
			case 'd' :	_retds(* ((char **) retbuf));
					break;
		}
	}
	else

		/* Otherwise, return .f. */
		_retl(0);
}

/* cparmtype(n) - returns the type of parameter n to Clipper. If n is not a
   valid parameter then the type returned is "U".
*/

CLIPPER cparmtype(void) {
	unsigned parmno;

	parmno = _parni(1) - 1;

	if (numparms == 0) {
		numparms = nextparm;
		nextparm = 0;
	}

	/* Check to see that a valid parameter is requested */
	
	if ((0 <= parmno) && (parmno < numparms))
		switch (parmtable[parmno].parmtype) {
			case 'C' :	_retc("C");
					break;
			case 'I' :
			case 'L' :
			case 'D' :	_retc("N");
					break;
			case 'l' :	_retc("L");
					break;
			case 'd' :	_retc("D");
					break;
		}
	else

		/* Otherwise, return "U" */
		_retc("U");
}

/* cparmcount() - returns the number of parameters passed to Clipper. */

CLIPPER cparmcount(void) {

	if (numparms == 0) {
		numparms = nextparm;
		nextparm = 0;
	}

	_retni(numparms);
}

/* maxparmcount() - returns the max number of parms that can be passed from C.
   called from C, maxparmcount() returns the number of doubles which could be
   passed, if smaller parms are passed then the max number goes up.
*/

maxparmcount(void) {
	return(numelems);
}

/* updateparmtable() - internal function used to store parameters passed from
   C in the static buffer. Return codes are:
   	0 - success
       -1 - max number of parms exceeded
       -2 - size of parm buffer exceeded
*/

updateparmtable(char newtype, void *parmptr) {
	char *bufptr;
	unsigned bufstart, parmsize;
	int rcode;

	rcode = 0;

	if (numparms > 0) {
		numparms = 0;
		nextparm = 0;
	}

	/* Check number of elements remaining in parm table */

	if (nextparm >= numelems)
		rcode = -1;
	else {
		if (!nextparm)
			parmtable[0].parmpos = 0;

		bufstart = parmtable[nextparm].parmpos;
		bufptr = parmbuffer + bufstart;

		/* get size of parm to be stored */

		switch (newtype) {
			case 'I' :
			case 'l' : parmsize = sizeof(int);
				   break;
			case 'C' :
			case 'd' : parmsize = sizeof(char *);
				   break;
			case 'L' : parmsize = sizeof(long);
				   break;
			case 'D' : parmsize = sizeof(double);
				   break;
		}

		/* if size of parm is too large, return an error code */

		if (bufstart + parmsize > parmbufsize)
			rcode = -2;
		else {
			/* otherwise, place parm into buffer */
			switch (newtype) {
				case 'I' :
				case 'l' :
					* (int *) bufptr = * (int *) parmptr;
					break;
				case 'C' :
				case 'd' :
					* (char **) bufptr = (char *) parmptr;
					break;
				case 'L' :
				       * (long *) bufptr = * (long *) parmptr;
					break;
				case 'D' :
					* (double *) bufptr = * (double *) parmptr;
					break;
			}
			parmtable[nextparm++].parmtype = newtype;
			if (nextparm < numelems)
				parmtable[nextparm].parmpos = bufstart + parmsize;
		}
	}
	return (rcode);
}

/* the following functions are called from C to store a parmeter in the buffer
   waiting to be retrieved by Clipper. Return codes are:
   	0 - success
       -1 - max number of parms exceeded
       -2 - size of parm buffer exceeded
*/

storeparmc(char *clparm) {
	return(updateparmtable('C', clparm));
}

storeparmni(int clparm) {
	return(updateparmtable('I', &clparm));
}

storeparmnl(long clparm) {
	return(updateparmtable('L', &clparm));
}

storeparmnd(double clparm) {
	return(updateparmtable('D', &clparm));
}

storeparml(int clparm) {
	return(updateparmtable('l', &clparm));
}

storeparmds(char *clparm) {
	return(updateparmtable('d', clparm));
}
