/*
==============================================================================
Function:        INDEXCOND.PRG
FileName:        CONDKEY.PRG
Version:         1.1
Author:          Sammy Smith of Data Management Systems
Notice:          Copyright 1992, Data Management Systems
Creation Date:   08/11/92
Creation Time:   02:50:pm
Complete Date:   08/11/92
Release:
Compile:         Clipper CONDKEY -l (clipper 50x)
Notes:           Returns the FOR Conditional Expression contained in index
                 file header.
                 You must be using the DBFNTX.LIB driver, Version 2.0.  The
                 originally released DBFNTX.LIB will not allow conditional
                 index expressions.
Revision Date:
Revision Author:
Revision Notes:

==============================================================================
*/


/*
  Function IndexCond(cNtxName)

  ==============
  NETWORK NOTES:  The IndexCond() function takes a character expression, being
  ==============  the path + index file name, or a MemVar may be substituted.
                  This function works without a hitch on a single-user
                  environment, and needs no housekeeping.

                  If this function is used in a network environment, it also
                  works without a hitch, if the data base the index file is
                  associated with, is opened for SHARED use.

                  If this function is used in a network environment, it will
                  work, provided you do one little housekeeping chore.  If
                  you intend on opening the data base the index file is
                  associated with, for EXCLUSIVE use, you may use one of the
                  two(2) following methods:

                  (1) Open the data base in SHARED MODE first, with the index
                      file.  Call the IndexCond() function with the index
                      name for which you want the FOR Conditional expression.

                                           -- OR --

                  (2) Before any opening of data bases with their index files,
                      call the IndexCond() function once for each index file
                      which will be used.


                  See the following examples, demonstrating each method:

METHOD #1: INTITIAL MODE - SHARED
---------------------------------
     Use Customer Index CustName NEW SHARED
     IndexCond("custname.ntx")  // Use the same path + filename as in the
                                // 'USE <dbf> INDEX <indexfile> ...' statement.


METHOD #2: NO INITIAL DataBase or Index file USE
------------------------------------------------
     // Let's say you are going to use three(3) index files in your
        application.

     IndexCond("custname.ntx")
     IndexCond("custcode.ntx")
     IndexCond("c:\factory\factcode.ntx")

IT'S THAT SIMPLE!

     After using one of the two fore mentioned methods, the IndexCond()
function may be called at anytime, even if the data base and its' associated
index file is opened for EXCLUSIVE use.

     The way this works is, when a call is made to the IndexCond() function,
it first tries to open the index file, and read in the index file header.  If
the data base is opened for EXCLUSIVE use, it cannot open the index file.
When this fails, an area of Clipper memory is searched, for the name of the
index file.  If it finds this patch in memory, it then knows where to locate
the offset in memory for the conditional expression.

     The Clipper memory search will ONLY work, (remember, this memory search
will only be invoked if the file is opened in EXCLUSIVE, otherwise, the index
file header is read directly) if a call was made PRIOR to USE EXCLUSIVE.  This
means you MUST use one of the two fore mentioned methods, as described.

--Sammy Smith
*/

#include "dbfntx.ch"
#include "fileio.ch"

Function IndexCond(cNtxName)
LOCAL nNtxHandle                       // Dos file handle number of index file
LOCAL cExp, cRetExp                    // File buffer & RETURNed expression
LOCAL nExpOffSet                       // File Offset position of conditional
                                       // expression in index file
LOCAL nFoundAt                         // Result of ArScan()
STATIC aCondExp := {}                  // Static array holding index filename &
                                       // condional exp. info.

 
 cNtxName   := Upper(cNtxName)         // Index File name
 nNtxHandle := 0                       // Index file handle number
 cExp       := Space(255)              // Buffer for reading index file header
 cRetExp    := Chr(0)                  // Returned expression
 nExpOffSet := 282                     // Offset position of cond. exp.
 nFoundAt   := 0

 If ValType(cNtxName)#"C"              // Allow only a character string param
    Return("")                         // Return NULL
 Endif

 nNtxHandle := Fopen(cNtxName,0+FO_SHARED)  // Attempt to open index file, in
                                            // SHARED Mode.

 If nNtxHandle # -1                      // Data Base is opened for SHARED
    fseek(nNtxHandle,nExpOffSet,FS_SET)  // Move file ptr to condtional exp.
    fread(nNtxHandle,@cExp,250)        // Read 250 chars (max len of index exp)
    fclose(nNtxHandle)                 // Close the file
    cRetExp := SubStr(cExp,1, (At(Chr(0),cExp) - 1) )  // Assign all the
                                       // characters up to the first null, to
                                       // the MemVar cRetExp.

    If len(aCondExp) == 0       // Array is empty, add the expression to array.
      Aadd( aCondExp,{cNtxName, cRetExp} )
    Else                        // Scan array for index name
      nFoundAt := ArScan(aCondExp, cNtxName)  // Scan for index name in array
      If nFoundAt == 0          // Expression not found
        Aadd( aCondExp,{cNtxName, cRetExp} )  // Add to the array
      Endif
    Endif

 Else                           // File is opened for EXCLUSIVE
    If Len(aCondExp) >0         // Make sure array has elements
       nFoundAt := ArScan(aCondExp, cNtxName)  // Scan array for index name
       If nFoundAt # 0          // Expression found, assign to cRetExp
          cRetExp := aCondExp[nFoundAt, 2]  // Assign 2nd elements contents
       Endif                                // to MemVar cRetExp.
    Endif
 Endif

Return( cRetExp )                      // Return the Expression


STATIC Function ArScan(aArytoScan,uSearchExp)
// Scan array for index filename
// Return element position if found,
//        0 if not found.
LOCAL nPosition, nElements, lDoMore
 nPosition := 0
 nElements := Len(aAryToScan)
 lDoMore   := .T.
 While lDoMore .and. nPosition <=nElements
    nPosition++
    lDoMore := If(aAryToScan[nPosition,1] == uSearchExp,.F.,.T.) // A match
 Enddo
Return( If(lDoMore,0,nPosition) )