/*

  Program Ŀ
                                                                        
  File Name...: NETIO.PRG                                               
  Author......: Vernon E. Six, Jr.                                      
  Date created: 02-20-94              Date updated: 09-28-94           
  Time created: 01:26:28pm            Time updated: 05:43:58pm         
  CopyRight...: (c) 1994 by FrontLine Software                          
                                                                        
 
  

*/

announce netio

#ifndef CLIP52
   #define CLIP52
#endif

#include "BAS_VERN.CH"
#include "cmx52.ch"
#include "flexrdd.ch"
#include "dbstruct.ch"
#include "flexfile.ch"
#include "setcurs.ch"

#define __WAIT_TIME  10
#define __DBFNAME    saStructs[nPos,1]
#define __STRUCT     saStructs[nPos,2]
#define __TAGS       saStructs[nPos,3]

static slProgress := .t.
static saStructs  := {}      // { {cDbfName,aStruct,aTags}, ... }
//                          aTags := {{cTagName,cKeyExpr,cForClause},...}


init procedure NetIo()

   cmxAutoShare(0)
   v_superRdd("Comix")
   dbSetDriver("FlexFile")
   v_SetCmp( CMP_SIZE )      // maximum compression

   return



/* HYPERTEXT START
!short: basStruct()     "Post" (a/k/a define) a structure for i/o functions
basStruct()     "Post" (a/k/a define) a structure for i/o functions

^BDescription: ^B

   basStruct() is a very crucial piece of the the i/o functions
   (basOpenDbf(), basAddRec(), etc).  basStruct() allows your functions
   to "post" (a/k/a define) a database structure and its associated
   index tags.


^BSyntax:^B

   nDbfCount := basStruct( cDbfName, aStruct, aTags )


^BPass:^B

   ^BcDbfName^B is a character expression that should contain the name of
   the database you are "posting".  No path or extension is allowed.  ^UIn ^U
   ^Uother words... your *.DBF, *.DBV and *.CDX files will all be in the same^U
   ^Udirectory as your *.EXE.^U

   ^BaStruct^B is an array suitable for passing to Clipper's dbStruct()
   function.  ^BaStruct^B has the following structure...

      { {cFldName,cFldType,nFldLen,nFldDec}, ... }

      ^BcFldName^B is a character expression that should contain the
      name of the field.

      ^BcFldType^B is a character expression that should contain the
      type of the field.  Valid values are "C" for character, "N" for
      numeric, "L" for logical and "M" for memo

      ^BnFldLen^N is a numeric expression that should contain the desired
      length of this field.

      ^BnFldDec^B is a numeric expression that should contain the desired
      number of decimal places for numeric fields.  For non-numeric fields
      this value should be zero (0).

   ^BaTags^B is an array that defines the index tags for this database.
   ^BaTags^B has the following structure...

      { { cTagName, cKeyExpr, cForClause }, ... }

      ^BcTagName^B is a character expression that should contain the name
      of the tag.  The max length for this value is ten characters.

      ^BcKeyExpr^B is a character expression that should contain the key
      expression for this tag.  See Comix manual for valid values.

      ^BcForClause^B is a character expression that should contain the
      FOR clause for this tag.  See Comix manual for valid values.

      ^RNote: basStruct() will automatically add the following element to^R
      ^RaTags for you...                                                 ^R
      ^R                                                                 ^R
      ^R   { "_DELETED_ ", "RECNO()", "DELETED()" }                      ^R
      ^R                                                                 ^R
      ^RThis step is essential in order for basAddRec() to operate       ^R
      ^Rproperly.                                                        ^R

^BReturns:^B

   ^BnDbfCount^B is a numeric expression that will contain the total number
   of databases that have been "posted" via basStruct() thusfar.


^BSource:^B

   NETIO.PRG

HYPERTEXT END */
function basStruct(pcDbfName,paStruct,paTags)

   local nPos := 0

   pcDbfName := lower(alltrim(pcDbfName))

   nPos := aScan( saStructs, {|x| x[1] == pcDbfName })

   if nPos == 0
      aAdd( saStructs, { pcDbfName, paStruct, paTags } )
      nPos := len(saStructs)
   endif

   __DBFNAME := pcDbfName
   __STRUCT  := paStruct
   __TAGS    := paTags

   aAdd( __TAGS, { "_DELETED_ ", "RECNO()", "DELETED()" })

   return saStructs[nPos]



/* HYPERTEXT START
!short: basProgress()   Get/Set progress indicator status
basProgress()   Get/Set progress indicator status

^BDescription: ^B

   basProgress() allows you to get/set the progress indicator status.  In
   other words... you can now control whether the i/o functions will display
   a progress indicator "gas gauge" while re-indexing, re-structuring, etc.


^BSyntax:^B

   lOrigProg := basProgress( [lNewProg] )


^BPass:^B

   ^BlNewProg^B is an optional logical expression that should be set to TRUE
   if you want the i/o functions to display a progress indicator "gas gauge"
   while re-indexing, re-structuring, etc, otherwise it should be set to
   FALSE.


^BReturns:^B

   ^BlOrigProg^B is a logical expression that will be the current setting
   for the whether or not the i/o functions will display a progress indicator
   "gas gauge".


^BSource:^B

   NETIO.PRG

HYPERTEXT END */
function basProgress(plNew)

   local lOld := slProgress

   if pcount() > 0
      slProgress := plNew
   endif

   return lOld


/* HYPERTEXT START
!short: basAppend()     Add a record to the currently selected database
basAppend()     Add a record to the currently selected database

^BDescription: ^B

   basAppend() allows you to add a record to the currently selected database

   ^RNote: basAddRec() is a much better method of adding records as it^R
   ^Rrecycles deleted records automatically thus eliminating "file bloat".^R

^BSyntax:^B

   lSuccess := basAppend()


^BPass:^B

   Nothing


^BReturns:^B

   ^BlSuccess^B is a logical expression that will be set to TRUE if
   basAppend() succeeds, otherwise it will be to FALSE.


^BSource:^B

   NETIO.PRG

HYPERTEXT END */
function basAppend()

   local lSuccess := .f.
   local nStart   := _TIME_NOW

   while .t.

      append blank

      if .not. NetErr()
         lSuccess := .t.
         exit
      endif

      Inkey(.5)              // Delay eases network traffic

      // Have we timed out yet?
      if basElapSecs( nStart, _TIME_NOW ) > __WAIT_TIME
         exit
      endif

   enddo

   return lSuccess


/* HYPERTEXT START
!short: basRLock()      Lock the current record in the currently selected dbf
basRLock()      Lock the current record in the currently selected dbf

^BDescription: ^B

   basRLock() allows your functions to lock the current record in the
   currently selected database.


^BSyntax:^B

   lSuccess := basRLock()


^BPass:^B

   Nothing


^BReturns:^B

   ^BlSuccess^B is a logical expression that will be set to TRUE if
   basRLock() succeeds, otherwise it will be to FALSE.


^BSource:^B

   NETIO.PRG

HYPERTEXT END */
function basRLock()

   local lSuccess  := .f.
   local nStart    := _TIME_NOW

   while .t.

      if rlock()
         lSuccess := .t.
         exit
      endif

      Inkey(.5)              // Delay eases network traffic

      // Have we timed out yet?
      if basElapSecs( nStart, _TIME_NOW ) > __WAIT_TIME
         exit
      endif

   enddo

   return lSuccess


/* HYPERTEXT START
!short: basUse()        Open a database file
basUse()        Open a database file

^BDescription: ^B

   basUse() allows your functions to open a database file in a new file area.

   ^RNote: basOpenDbf() is a better choice than basUse() since basOpenDbf()^R
   ^Rwill ensure the structure is correct and re-create index tags, etc.   ^R
   ^RbasUse() is designed to open temporary databases, database with       ^R
   ^Runknown structures, etc.                                              ^R


^BSyntax:^B

   lSuccess := basUse( cDbfName, [cAlias], [lExcl] )


^BPass:^B

   ^BcDbfName^B is a character expression that should contain the name of
   the database you want to open.

   ^BcAlias^B is an optional character expression that should contain the
   desired "alias" name for ^BcDbfName^B.

   ^BlExcl^B is an optional logical expression that should be set to TRUE
   if you want ^BcDbfName^B to be opened in exclusive mode.  FALSE is the
   default.


^BReturns:^B

   ^BlSuccess^B is a logical expression that will be set to TRUE if
   basUse() succeeds, otherwise it will be to FALSE.


^BNotes:^B

   You must include the following statement somewhere in your application
   in order to use this function...

         REQUEST NETIO



   Requires Comix v3.x from...

          LoadStone Inc.
          215 Barmount Drive
          Columbia, SC 29210

          Fax:    803/731-9798
          Sales:  803/731-9128


^BSource:^B

   NETIO.PRG


HYPERTEXT END */
function basUse( pcDbfName, pcAlias, plExclusive )

   local lSuccess  := .f.
   local nStart    := _TIME_NOW

   pcDbfName := alltrim(pcDbfName)

   assume plExclusive is .f. if missing
   assume pcAlias     is ""  if missing

   // check to see if it's already open
   if (!empty(pcAlias)) .and. (select(pcAlias) > 0)

      select (pcAlias)
      ordSetFocus(1)

      return .t.

   endif

   // Make sure the file exists.  Avoid "file Not Found"
   if .not. file( pcDbfName + ".DBF" )
      return .f.
   endif

   while .t.

      if empty(pcAlias)

         if plExclusive
            use (pcDbfName) new exclusive
         else
            use (pcDbfName) new shared
         endif

      else

         if plExclusive
            use (pcDbfName) alias (pcAlias) new exclusive
         else
            use (pcDbfName) alias (pcAlias) new shared
         endif

      endif

      if .not. NetErr()
         lSuccess := .t.
         exit
      endif

      Inkey(.5)              // Delay eases network traffic

      // Have we timed out yet?
      if basElapSecs( nStart, _TIME_NOW ) > __WAIT_TIME
         exit
      endif

   enddo

   if lSuccess
      ordSetFocus(1)         // always set first tag as controlling
   endif

   return lSuccess



/* HYPERTEXT START
!short: basDelRec()     Delete the current record in the current database
basDelRec()     Delete the current record in the current database

^BDescription: ^B

   basDelRec() allows you to delete the current record in the currently
   selected database.


^BSyntax:^B

   lSuccess := basDelRec()


^BPass:^B

   Nothing


^BReturns:^B

   ^BlSuccess^B is a logical expression that will be set to TRUE if
   basDelRec() succeeds, otherwise it will be to FALSE.


^BSource:^B

   NETIO.PRG

HYPERTEXT END */
function basDelRec()

   if basRlock()
      dbDelete()
      dbUnlock()
      return .t.
   endif

   return .f.






/* HYPERTEXT START
!short: basAddRec()     Add a record to the currently selected database
basAddRec()     Add a record to the currently selected database

^BDescription: ^B

   basAddRec() allows you to add a record to the currently selected database


^BSyntax:^B

   lSuccess := basAddRec()


^BPass:^B

   Nothing


^BReturns:^B

   ^BlSuccess^B is a logical expression that will be set to TRUE if
   basAddRec() succeeds, otherwise it will be to FALSE.


^BNotes:^B

   You must include the following statement somewhere in your application
   in order to use this function...

         REQUEST NETIO



   Requires Comix v3.x from...

          LoadStone Inc.
          215 Barmount Drive
          Columbia, SC 29210

          Fax:    803/731-9798
          Sales:  803/731-9128


^BSource:^B

   NETIO.PRG


HYPERTEXT END */
function basAddRec()

   local nCntr    := 0
   local lDeleted := set( _SET_DELETED, .f. )
   local cOrigTag := ordSetFocus("_DELETED_")
   local lSuccess := .t.

   dbGoTop()

   if .not. eof()

      if basRlock()
         dbRecall()
         basClrRec()
         lSuccess := .t.
      endif

      ordSetFocus( cOrigTag )

   else

      ordSetFocus( cOrigTag )

      if basAppend()
         lSuccess := .t.
      endif

   endif

   set( _SET_DELETED, lDeleted )

   return lSuccess



/* HYPERTEXT START
!short: basClrRec()     Reset all field values to "default" state
basClrRec()     Reset all field values to "default" state

^BDescription: ^B

   basClrRec() allows you to set all field values for the currently
   selected record in the current database to their default state.

   Note: You must lock the record with basRLock() before calling basClrRec()

^BSyntax:^B

   basClrRec()


^BPass:^B

   Nothing


^BReturns:^B

   Always nil


^BSource:^B

   NETIO.PRG

HYPERTEXT END */
function basClrRec()

   local nCntr    := 0
   local cFldType := ""

   // Set everything to the state of a new record (blank out)
   for nCntr := 1 to fcount()

      cFldType := ValType( FieldGet(nCntr) )

      do case
         case cFldType == "C" ; FieldPut(nCntr, space(len(FieldGet(nCntr))))
         case cFldType == "L" ; FieldPut(nCntr, .f.)
         case cFldType == "N" ; FieldPut(nCntr, 0  )
         case cFldType == "D" ; FieldPut(nCntr, ctod("  /  /  "))
         case cFldType == "M" ; FieldPut(nCntr, "")
      endcase

   next nCntr

   return nil



/* HYPERTEXT START
!short: basOpenDbf()    Open a dbf that has been "posted" w/basStruct()
basOpenDbf()    Open a dbf file that has been "posted" w/basStruct()

^BDescription: ^B

   basOpenDbf() allows your functions to open a database file that has been
   "posted" with basStruct() in a new file area.  basOpenDbf() will alter
   the database structure to match the structure that has been "posted" as
   well as re-create index tags, if needed.


^BSyntax:^B

   nError := basOpenDbf( cDbfName )


^BPass:^B

   ^BcDbfName^B is a character expression that should contain the name of
   the database you want to open.


^BReturns:^B

   ^BnError^B is a numeric expression that will contain one of the
   following values...

       Ŀ
        Value  Description                                    
       Ĵ
                                                              
          0    No Error.  Database successfully opened        
                                                              
         -1    Couldn't create database                       
                                                              
         -2    Couldn't open database                         
                                                              
         -3    Couldn't update structure                      
                                                              
         -4    Couldn't re-index database                     
                                                              
         -5    Couldn't re-open database                      
                                                              
       


^BNotes:^B

   You must include the following statement somewhere in your application
   in order to use this function...

         REQUEST NETIO



   Requires Comix v3.x from...

          LoadStone Inc.
          215 Barmount Drive
          Columbia, SC 29210

          Fax:    803/731-9798
          Sales:  803/731-9128


^BSource:^B

   NETIO.PRG


HYPERTEXT END */
function basOpenDbf( pcDbfName )

   local nError      := 0
   local aCurrStruct := {}
   local nPos        := 0

   pcDbfName := lower(alltrim(pcDbfName))

   nPos := aScan( saStructs, {|x| x[1] == pcDbfName } )




   begin sequence

      if .not. ( file( pcDbfName + ".DBF" ) .and. file( pcDbfName + ".CDX" ) )

         if .not. sCreateDbf(pcDbfName)
            nError := -1
            break
         endif

      endif

      if .not. basUse(pcDbfName,pcDbfName,.f.)
         nError := -2
         break
      endif

      // make sure structure is EXACTLY correct
      aCurrStruct := (pcDbfName)->( dbStruct() )
      if .not. sIsSame( __STRUCT, aCurrStruct )

         if .not. sUpdStruct(pcDbfName,aCurrStruct,__STRUCT)
            nError := -3
            break
         endif

      endif

      if file(pcDbfName+".CDX")

         if .not. (pcDbfName)->( sChekCdx(__TAGS) )

            (pcDbfName)->( dbCloseArea() )
            fErase(pcDbfName+".CDX")

         endif

      endif

      if .not. file( pcDbfName + ".CDX" )

         if select(pcDbfName) > 0
            (pcDbfName)->( dbCloseArea() )
         endif

         if .not. sReindex(pcDbfName,__TAGS)
            nError := -4
            break
         endif

         if .not. basUse(pcDbfName,pcDbfName,.f.)
            nError := -5
            break
         endif

      endif

   end sequence

   return nError





static function sCreateDbf( pcDbfName )

   local aCurrStruct := {}
   local lSuccess    := .f.
   local nPos        := 0

   pcDbfName := lower(alltrim(pcDbfName))

   nPos := aScan( saStructs, {|x| x[1] == pcDbfName } )

   begin sequence

      // make sure it was a valid database name
      if len(__STRUCT) == 0
         break
      endif

      // if not there, create it
      if .not. file( pcDbfName + ".DBF" )

         dbCreate(pcDbfName,__STRUCT)

         // if we couldn't create it, bomb out
         if .not. file( pcDbfName + ".DBF" )
            break
         endif

      endif

      if file(pcDbfName+".CDX")
         fErase(pcDbfName+".CDX")
      endif

      lSuccess := .t.

   end sequence

   return lSuccess


static function sChekCdx( paTags )

   local nI       := 0
   local nLength  := len(paTags)
   local lSame    := .f.

   begin sequence

      for nI := 1 to nLength

         if .not. ordNumber(paTags[nI,1]) == nI
            break
         endif

         if .not. ordKey(nI) == paTags[nI,2]
            break
         endif

         if .not. ordFor(nI) == paTags[nI,3]
            break
         endif

      next nI

      lSame := .t.

   end sequence

   return lSame




static function sIsSame( paOne, paTwo )

   local lSame    := .f.
   local nI       := 0
   local nLength  := 0

   begin sequence

      if .not. len(paOne) == len(paTwo)
         break
      endif

      nLength := len(paOne)

      for nI := 1 to nLength

         if .not. alltrim(paOne[nI,DBS_NAME]) == alltrim(paTwo[nI,DBS_NAME])
            break
         endif

         if .not. paOne[nI,DBS_TYPE] == paTwo[nI,DBS_TYPE]
            break
         endif


         if .not. paOne[nI,DBS_LEN] == paTwo[nI,DBS_LEN]
            break
         endif


         if .not. paOne[nI,DBS_DEC] == paTwo[nI,DBS_DEC]
            break
         endif

      next nI

      lSame := .t.

   end sequence

   return lSame


static function sUpdStruct( pcDbfName, paOld, paNew )

   local lSuccess       := .f.

   begin sequence

      if file("NEW_FILE.DBF")
         fErase("NEW_FILE.DBF")
      endif

      if file("NEW_FILE.DBV")
         fErase("NEW_FILE.DBV")
      endif

      select 0
      dbCreate( "NEW_FILE", paNew )
      if .not. basUse( "new_file", "new_file", .t. )
         break
      endif

      if slProgress

         basSaveScrn()

         basWind(10,18,14,61,"","",4)

         @ 11,20 say padc("Restructuring " + upper(pcDbfName) + ".DBF...",40)
         @ 12,20 say replicate("",40)

      endif

      // walk thru the old file
      (pcDbfName)->( ordSetFocus(0) )
      (pcDbfName)->( dbGoTop()      )

      while .not. (pcDbfName)->( eof() )

         (pcDbfName)->( sProgress() )

         sConvert( paOld, paNew, pcDbfName )

         (pcDbfName)->( dbSkip() )

      enddo

      if slProgress
         basRestScrn()
      endif

      // delete old_file, rename new_file
      new_file->( dbCloseArea() )
      (pcDbfName)->( dbCloseArea() )

      // erase the BAK files
      if file( pcDbfName + ".DFK" )
         fErase( pcDbfName + ".DFK" )
      endif

      if file( pcDbfName + ".DVK" )
         fErase( pcDbfName + ".DVK" )
      endif

      if file( pcDbfName + ".CDX" )
         fErase( pcDbfName + ".CDX" )
      endif


      fRename( pcDbfName + ".DBF", pcDbfName + ".DFK" )

      if file( pcDbfName + ".DBV" )
         fRename( pcDbfName + ".DBV", pcDbfName + ".DVK" )
      endif

      fRename( "NEW_FILE.DBF", pcDbfName + ".DBF" )

      if file( "NEW_FILE.DBV" )
         fRename( "NEW_FILE.DBV", pcDbfName + ".DBV" )
      endif

      lSuccess := .t.

   end sequence

   return lSuccess


static function sConvert( paOldStru, paNewStru, pcDbfName )

   local nLength  := len(paNewStru)
   local nI       := 0
   local nOldPos  := 0
   local cFldName := ""
   local xOldData := nil

   new_file->( dbAppend() )

   for nI := 1 to nLength

      cFldName := alltrim(paNewStru[nI,DBS_NAME]) // easier to read

      nOldPos := aScan( paOldStru, {|x|alltrim(x[DBS_NAME]) == cFldName } )

      // is this a new field?
      if nOldPos = 0
         loop
      endif

      xOldData := (pcDbfName)->( fieldget(nOldPos) )

      do case

         case paNewStru[nI,DBS_TYPE] = "C"

            sCvtChar( paOldStru[nOldPos], paNewStru[nI], nI, xOldData )

         case paNewStru[nI,DBS_TYPE] = "M"

            sCvtMemo( paOldStru[nOldPos], paNewStru[nI], nI, xOldData )

         case paNewStru[nI,DBS_TYPE] = "L"

            sCvtLog( paOldStru[nOldPos], paNewStru[nI], nI, xOldData )

         case paNewStru[nI,DBS_TYPE] = "N"

            sCvtNmbr( paOldStru[nOldPos], paNewStru[nI], nI, xOldData )

         case paNewStru[nI,DBS_TYPE] = "D"

            sCvtDate( paOldStru[nOldPos], paNewStru[nI], nI, xOldData )

      endcase

   next nI

   return nil


static function sCvtChar( paOld, paNew, pnCntr, pxOldData )

   local cNewData := ""

   do case

      case paOld[DBS_TYPE] = "L"

         if pxOldData = .t.
            cNewData := "T"
         else
            cNewData := "F"
         endif


      case paOld[DBS_TYPE] = "M"

         cNewData := pxOldData


      case paOld[DBS_TYPE] = "C"

         cNewData := pxOldData


      case paOld[DBS_TYPE] = "N"

         cNewData := alltrim( str(pxOldData) )


      case paOld[DBS_TYPE] = "D"

         cNewData := dtoc(pxOldData)

   endcase

   new_file->( fieldPut(pnCntr,left(cNewData,paNew[DBS_LEN])) )

   return nil



static function sCvtMemo( paOld, paNew, pnCntr, pxOldData )

   local cNewData := ""

   do case

      case paOld[DBS_TYPE] = "L"

         if pxOldData = .t.
            cNewData := "T"
         else
            cNewData := "F"
         endif


      case paOld[DBS_TYPE] = "M"

         cNewData := pxOldData


      case paOld[DBS_TYPE] = "C"

         cNewData := pxOldData


      case paOld[DBS_TYPE] = "N"

         cNewData := alltrim( str(pxOldData) )


      case paOld[DBS_TYPE] = "D"

         cNewData := dtoc(pxOldData)

   endcase

   new_file->( fieldPut(pnCntr,cNewData) )

   return nil



static function sCvtLog( paOld, paNew, pnCntr, pxOldData )

   local lNewData := .f.

   do case

      case paOld[DBS_TYPE] = "L"

         lNewData := pxOldData


      case paOld[DBS_TYPE] = "M"

         if left(pxOldData,1) $ "Tt"
            lNewData := .t.
         else
            lNewData := .f.
         endif


      case paOld[DBS_TYPE] = "C"

         if left(pxOldData,1) $ "Tt"
            lNewData := .t.
         else
            lNewData := .f.
         endif


      case paOld[DBS_TYPE] = "N"

         if int(pxOldData) == 0
            lNewData := .f.
         else
            lNewData := .t.
         endif


      case paOld[DBS_TYPE] = "D"

         if pxOldData = ctod("  /  /  ")
            lNewData := .f.
         else
            lNewData := .t.
         endif


   endcase

   new_file->( fieldPut(pnCntr,lNewData) )

   return nil


static function sCvtNmbr( paOld, paNew, pnCntr, pxOldData )

   local nNewData := .f.

   do case

      case paOld[DBS_TYPE] = "L"

         if pxOldData
            nNewData := 1
         else
            nNewData := 0
         endif


      case paOld[DBS_TYPE] = "M"

         nNewData := val(pxOldData)


      case paOld[DBS_TYPE] = "C"

         nNewData := val(pxOldData)


      case paOld[DBS_TYPE] = "N"

         nNewData := pxOldData


      case paOld[DBS_TYPE] = "D"

         nNewData := 0

   endcase

   nNewData := val( left( alltrim(str(nNewData)), paNew[DBS_LEN] ) )

   new_file->( fieldPut(pnCntr,nNewData) )

   return nil


static function sCvtDate( paOld, paNew, pnCntr, pxOldData )

   local dNewData := .f.

   do case

      case paOld[DBS_TYPE] = "L"

         dNewData := ctod("  /  /  ")


      case paOld[DBS_TYPE] = "M"

         dNewData := ctod( left(pxOldData,8) )


      case paOld[DBS_TYPE] = "C"

         dNewData := ctod( left(pxOldData,8) )


      case paOld[DBS_TYPE] = "N"

         dNewData := ctod("  /  /  ")


      case paOld[DBS_TYPE] = "D"

         dNewData := pxOldData

   endcase

   new_file->( fieldPut(pnCntr,dNewData) )

   return nil



static function sReindex(pcDbfName,paTags)

   local lSuccess    := .f.
   local nTags       := len(paTags)
   local nI          := 0
   local nRecCount   := 0

   begin sequence

      if slProgress

         basSaveScrn()

         basWind(09,18,14,61,"","",4)

         @ 10,20 say padc("Indexing " + upper(pcDbfName) + ".DBF",40)
         @ 12,20 say replicate("",40)

      endif

      if .not. basUse(pcDbfName,pcDbfName,.t.)
         break
      endif

      nRecCount := (pcDbfName)->( reccount() )

      for nI := 1 to nTags

         if slProgress
            @ 11,20 say padc("Creating Tag: " + paTags[nI,1],40)
         endif

         if empty( paTags[nI,3] )

            xindex on (paTags[nI,2]) tag (paTags[nI,1]) ;
               eval {||sProgress()} every nRecCount / 20

         else

            xindex on (paTags[nI,2]) tag (paTags[nI,1]) for (paTags[nI,3]) ;
               eval {||sProgress()} every nRecCount / 20

         endif

      next nI

      if slProgress
         basRestScrn()
      endif

      lSuccess := .t.

   end sequence

   if select(pcDbfName) > 0
      (pcDbfName)->( dbCloseArea() )
   endif

   return lSuccess


static function sProgress()

   local nPercent := recno() / recCount()
   local cStr     := padc( str(nPercent * 100, 3, 0) + "% complete", 40 )
   local nReverse := int( 40 * nPercent )

   if slProgress

      DispBegin()
      SetCursor(SC_NONE)
      @ 13,20 say cStr
      @ 13,20 say left(cStr,nReverse) color basUnsColor()
      DispEnd()

   endif

   return .t.



