/*
	tsr.C

	Version 1.1

	Generate TSR .asm file.

	Command-line format: tsr ((/|-)m |
				  [(/|-)hx] [(/|-)sc] [(/|-)sy] tsrspec)

	m  generate map of resident tsrs
	h  defines the default hotkey (x is either 'A'-'Z' or '0'-'9')
	sc indicates that setup/cleanup function calls are included
	s  defines the stack size (y is in bytes - 256 min, 4K max)

	tsrspec defines the .tsr file to be TSRed

	Copyright (C) 1993, Geoff Friesen B.Sc.
	All rights reserved.

	Developed with: Borland C++ 3.1
*/

#if !defined(__LARGE__)
#error Large Memory Model Expected
#endif

#include <ctype.H>
#include <dir.H>
#include <dos.H>
#include <process.H>
#include <stdio.H>
#include <stdlib.H>
#include <string.H>

#define	OK	0
#define	ERROR	!OK

#define	IGNORE	1

#define	FREEMCB	0
#define	BLOCMCB	'M'
#define	LASTMCB	'Z'

#define	KEYOFS	0x103
#define	SIGOFS	0x105

#define	MAXLINE		256

char *listing1 =

"\n\n"
"_TEXT\t\tSEGMENT\tBYTE PUBLIC 'CODE'\n"
"\t\tASSUME\tcs:_TEXT, ds:NOTHING, es:NOTHING, ss:NOTHING\n"
"\t\tORG\t100h\n"
"\n"
"CR\t\tEQU\t13\n"
"LF\t\tEQU\t10\n"
"TAB\t\tEQU\t9\n"
"\n"
"INACTIVE\tEQU\t0\t\t; TSR not active.\n"
"TRIGGERED\tEQU\t1\t\t; TSR triggered.\n"
"ACTIVE\t\tEQU\t2\t\t; TSR active.\n"
"\n"
"tsr:\n"
"\t\tjmp\tNEAR PTR install\n"
"\n"
"key\t\tDB\t'";

char *listing2 =

"'\t\t; Hotkey character.\n"
"scancode\tDB\t";

char *listing3 =

"\t\t; Activation scancode.\n"
"signature\tDB\t";

char *listing4 =

"\t; Our signature in memory.\n"
"SIGLENGTH\tEQU\t$-signature\t; Length of this signature.\n"
"\t\tDB\t0\n"
"\n"
"; --------------------- ;\n"
"; Resident Data Section ;\n"
"; --------------------- ;\n"
"\n"
"critdisk\tDB\t0\t\t; Critical disk flag.\n"
"criterr\t\tDD\t?\t\t; Address of critical error flag.\n"
"critsec\t\tDD\t?\t\t; Address of critical section flag.\n"
"dta\t\tDD\t?\t\t; Saved DTA address.\n"
"errax\t\tDW\t?\t\t; Saved extended error info.\n"
"errbx\t\tDW\t?\n"
"errcx\t\tDW\t?\n"
"oldint8h\tDD\t?\t\t; Old interrupt 8h vector.\n"
"oldint9h\tDD\t?\t\t; Old interrupt 9h vector.\n"
"oldint13h\tDD\t?\t\t; Old interrupt 13h vector.\n"
"oldint1bh\tDD\t?\t\t; Old interrupt 1bh vector.\n"
"oldint23h\tDD\t?\t\t; Old interrupt 23h vector.\n"
"oldint24h\tDD\t?\t\t; Old interrupt 24h vector.\n"
"oldint28h\tDD\t?\t\t; Old interrupt 28h vector.\n"
"port61h\t\tDB\t?\t\t; Current value in port 61h.\n"
"_psp\t\tDW\t?\t\t; Saved PSP.\n"
"savesp\t\tDW\t?\t\t; Saved stack pointer.\n"
"savess\t\tDW\t?\t\t; Saved stack segment.\n"
"tsr_state\tDB\tINACTIVE\t; TSR state.\n"
"\n"
"; --------------------- ;\n"
"; Resident Code Section ;\n"
"; --------------------- ;\n"
"\n"
"; ----------------------- ;\n"
"; New interrupt 8h vector ;\n"
"; ----------------------- ;\n"
"\n"
"newint8h\tPROC\tFAR\n"
"\t\tpushf\n"
"\t\tcall\tDWORD PTR oldint8h\n"
"\n"
"\t\tcmp\tBYTE PTR tsr_state, TRIGGERED\n"
"\t\tjnz\tnewint8h1\n"
"\n"
"\t\tpush\tax\n"
"\t\tpush\tbx\n"
"\t\tpush\tes\n"
"\n"
"\t\tles\tbx, DWORD PTR criterr\n"
"\t\tmov\tal, es:[bx]\n"
"\n"
"\t\tcmp\tal, 0\n"
"\n"
"\t\tpop\tes\n"
"\t\tpop\tbx\n"
"\t\tpop\tax\n"
"\n"
"\t\tjnz\tnewint8h1\n"
"\n"
"\t\tpush\tax\n"
"\t\tpush\tbx\n"
"\t\tpush\tes\n"
"\n"
"\t\tles\tbx, DWORD PTR critsec\n"
"\t\tmov\tal, es:[bx]\n"
"\n"
"\t\tcmp\tal, 0\n"
"\n"
"\t\tpop\tes\n"
"\t\tpop\tbx\n"
"\t\tpop\tax\n"
"\n"
"\t\tjnz\tnewint8h1\n"
"\n"
"\t\tcmp\tBYTE PTR critdisk, 0\n"
"\t\tjnz\tnewint8h1\n"
"\n"
"\t\tmov\tBYTE PTR tsr_state, ACTIVE\n"
"\n"
"\t\tcall\tlaunch\n"
"\n"
"\t\tmov\tBYTE PTR tsr_state, INACTIVE\n"
"newint8h1:\n"
"\t\tiret\n"
"newint8h\tENDP\n"
"\n"
"; ----------------------- ;\n"
"; New interrupt 9h vector ;\n"
"; ----------------------- ;\n"
"\n"
"newint9h\tPROC\tFAR\n"
"\t\tpush\tax\n"
"\n"
"\t\tin\tal, 60h\t\t; Examine scan code.\n"
"\n"
"\t\tcmp\tal, scancode\t; Does it match our scan code?\n"
"\t\tjz\tnewint9h2\t; Yes, branch.\n"
"newint9h1:\n"
"\t\tpop\tax\t\t; Exit to original handler.\n"
"\t\tjmp\tDWORD PTR oldint9h\n"
"newint9h2:\n"
"\t\tpush\tes\t\t; Check for simultaneous ALT press.\n"
"\t\tmov\tax, 0\n"
"\t\tmov\tes, ax\n"
"\n"
"\t\tmov\tal, es:[417h]\t; Get shift status byte.\n"
"\t\tand\tal, 15\t\t; Remove toggle data.\n"
"\t\tcmp\tal, 8\t\t; ALT (alone) pressed?\n"
"\n"
"\t\tpop\tes\n"
"\t\tjnz\tnewint9h1\t; No, branch.\n"
"\n"
"\t\tin\tal, 61h\t\t; Acknowledge scan code.\n"
"\t\tjmp\t$+2\n"
"\t\tmov\tah, al\n"
"\t\tor\tal, 10000000b\n"
"\t\tout\t61h, al\n"
"\t\tjmp\t$+2\n"
"\t\tmov\tal, ah\n"
"\t\tout\t61h, al\n"
"\t\tjmp\t$+2\n"
"\n"
"\t\tcmp\tBYTE PTR tsr_state, INACTIVE\n"
"\t\tjnz\tnewint9h3\t; Branch if TSR not inactive.\n"
"\n"
"\t\tmov\tBYTE PTR tsr_state, TRIGGERED\n"
"newint9h3:\n"
"\t\tmov\tal, 20h\t\t; Reset 8259 PIC.\n"
"\t\tout\t20h, al\n"
"\n"
"\t\tpop\tax\n"
"\t\tiret\n"
"newint9h\tENDP\n"
"\n"
"; ------------------------ ;\n"
"; New interrupt 13h vector ;\n"
"; ------------------------ ;\n"
"\n"
"newint13h\tPROC\tFAR\n"
"\t\tmov\tBYTE PTR critdisk, 1\n"
"\n"
"\t\tpushf\n"
"\t\tcall\tDWORD PTR oldint13h\n"
"\n"
"\t\tmov\tBYTE PTR critdisk, 0\n"
"\n"
"\t\tsti\n"
"\t\tret\t2\t\t; Return without original flags.\n"
"newint13h\tENDP\n"
"\n"
"; ------------------------ ;\n"
"; New interrupt 1bh vector ;\n"
"; ------------------------ ;\n"
"\n"
"newint1bh\tPROC\tFAR\n"
"\t\tiret\n"
"newint1bh\tENDP\n"
"\n"
"; ------------------------ ;\n"
"; New interrupt 23h vector ;\n"
"; ------------------------ ;\n"
"\n"
"newint23h\tPROC\tFAR\n"
"\t\tiret\n"
"newint23h\tENDP\n"
"\n"
"; ------------------------ ;\n"
"; New interrupt 24h vector ;\n"
"; ------------------------ ;\n"
"\n"
"newint24h\tPROC\tFAR\n"
"\t\tmov\tax, 3\t\t; Fail code.\n"
"\t\tiret\n"
"newint24h\tENDP\n"
"\n"
"; ------------------------ ;\n"
"; New interrupt 28h vector ;\n"
"; ------------------------ ;\n"
"\n"
"newint28h\tPROC\tFAR\n"
"\t\tpushf\n"
"\t\tcall\tDWORD PTR oldint28h\n"
"\n"
"\t\tcmp\tBYTE PTR tsr_state, TRIGGERED\n"
"\t\tjnz\tnewint28h1\n"
"\n"
"\t\tpush\tax\n"
"\t\tpush\tbx\n"
"\t\tpush\tes\n"
"\n"
"\t\tles\tbx, DWORD PTR criterr\n"
"\t\tmov\tal, es:[bx]\n"
"\n"
"\t\tcmp\tal, 0\n"
"\n"
"\t\tpop\tes\n"
"\t\tpop\tbx\n"
"\t\tpop\tax\n"
"\n"
"\t\tjnz\tnewint28h1\n"
"\n"
"\t\tpush\tax\n"
"\t\tpush\tbx\n"
"\t\tpush\tes\n"
"\n"
"\t\tles\tbx, DWORD PTR critsec\n"
"\t\tmov\tal, es:[bx]\n"
"\n"
"\t\tcmp\tal, 1\n"
"\n"
"\t\tpop\tes\n"
"\t\tpop\tbx\n"
"\t\tpop\tax\n"
"\n"
"\t\tja\tnewint28h1\n"
"\n"
"\t\tcmp\tBYTE PTR critdisk, 0\n"
"\t\tjnz\tnewint28h1\n"
"\n"
"\t\tmov\tBYTE PTR tsr_state, ACTIVE\n"
"\n"
"\t\tcall\tlaunch\n"
"\n"
"\t\tmov\tBYTE PTR tsr_state, INACTIVE\n"
"newint28h1:\n"
"\t\tiret\n"
"newint28h\tENDP\n"
"\n"
"; ------------------------------ ;\n"
"; Launch: Launch TSR Application ;\n"
"; ------------------------------ ;\n"
"\n"
"\t\tASSUME\tds:_TEXT, es:_TEXT, ss:_TEXT\n"
"\n"
"launch\t\tPROC\tNEAR\n"
"\t\tmov\tcs:savess, ss\t; Switch stacks.\n"
"\t\tmov\tcs:savesp, sp\n"
"\t\tmov\tsp, cs\n"
"\t\tmov\tss, sp\n"
"\t\tlea\tsp, _stack\n"
"\t\tsti\n"
"\n"
"\t\tcld\n"
"\t\tpush\tax\n"
"\t\tpush\tbx\n"
"\t\tpush\tcx\n"
"\t\tpush\tdx\n"
"\t\tpush\tsi\n"
"\t\tpush\tdi\n"
"\t\tpush\tbp\n"
"\t\tpush\tds\n"
"\t\tpush\tes\n"
"\n"
"\t\tpush\tcs\n"
"\t\tpop\tds\n"
"\n"
"\t\tmov\tax, 351bh\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tWORD PTR oldint1bh, bx\n"
"\t\tmov\tWORD PTR oldint1bh [2], es\n"
"\n"
"\t\tmov\tax, 3523h\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tWORD PTR oldint23h, bx\n"
"\t\tmov\tWORD PTR oldint23h [2], es\n"
"\n"
"\t\tmov\tax, 3524h\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tWORD PTR oldint24h, bx\n"
"\t\tmov\tWORD PTR oldint24h [2], es\n"
"\n"
"\t\tlea\tdx, newint1bh\n"
"\t\tmov\tax, 251bh\n"
"\t\tint\t21h\n"
"\n"
"\t\tlea\tdx, newint23h\n"
"\t\tmov\tax, 2523h\n"
"\t\tint\t21h\n"
"\n"
"\t\tlea\tdx, newint24h\n"
"\t\tmov\tax, 2524h\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tah, 62h\t\t; Save current PSP.\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\t_psp, bx\n"
"\n"
"\t\tmov\tbx, cs\t\t; Set PSP to our own.\n"
"\t\tmov\tah, 50h\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tah, 2fh\t\t; Save current DTA.\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tWORD PTR dta, bx\n"
"\t\tmov\tWORD PTR dta [2], es\n"
"\n"
"\t\tmov\tdx, 80h\t\t; Set DTA to our own.\n"
"\t\tmov\tah, 1ah\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tbx, 0\t\t; Must be zero.\n"
"\t\tmov\tah, 59h\t\t; Save extended error info.\n"
"\t\tint\t21h\n"
"\n"
"\t\tpush\tcs\t\t; DS destroyed by function.\n"
"\t\tpop\tds\n"
"\n"
"\t\tmov\terrax, ax\n"
"\t\tmov\terrbx, bx\n"
"\t\tmov\terrcx, cx\n"
"launch1:\n"
"\t\tmov\tah, 1\t\t; Remove unwanted keys.\n"
"\t\tint\t16h\n"
"\t\tjz\tlaunch2\n"
"\n"
"\t\tmov\tah, 0\n"
"\t\tint\t16h\n"
"\t\tjmp\tSHORT launch1\n"
"launch2:\n"
"\t\tin\tal, 61h\t\t; Save port value.\n"
"\t\tmov\tport61h, al\n"
"\n"
"\t\tand\tal, 0fch\t; Disconnect speaker.\n"
"\t\tout\t61h, al\n"
"\n"
"\t\tpush\tcs\n"
"\t\tpop\tes\n"
"\n"
"\t\tcall\t_main\t\t; Invoke app entry point.\n"
"\n"
"\t\tmov\tal, 61h\t\t; Restore port value.\n"
"\t\tout\t61h, al\n"
"\n"
"\t\tpush\tds\n"
"\n"
"\t\tmov\tbx, 0\t\t; Must be zero.\n"
"\t\tlea\tdx, errax\t; Point to info structure.\n"
"\t\tmov\tax, 5d0ah\t; Restore extended error info.\n"
"\t\tint\t21h\n"
"\n"
"\t\tpop\tds\n"
"\n"
"\t\tlds\tdx, dta\t\t; Restore DTA.\n"
"\t\tmov\tah, 1ah\n"
"\t\tint\t21h\n"
"\n"
"\t\tpush\tcs\t\t; Restore DS.\n"
"\t\tpop\tds\n"
"\n"
"\t\tmov\tbx, _psp\t; Restore PSP.\n"
"\t\tmov\tah, 50h\n"
"\t\tint\t21h\n"
"\n"
"\t\tlds\tdx, cs:oldint24h\n"
"\t\tmov\tax, 2524h\n"
"\t\tint\t21h\n"
"\n"
"\t\tlds\tdx, cs:oldint23h\n"
"\t\tmov\tax, 2523h\n"
"\t\tint\t21h\n"
"\n"
"\t\tlds\tdx, cs:oldint1bh\n"
"\t\tmov\tax, 251bh\n"
"\t\tint\t21h\n"
"\n"
"\t\tpop\tes\n"
"\t\tpop\tds\n"
"\t\tpop\tbp\n"
"\t\tpop\tdi\n"
"\t\tpop\tsi\n"
"\t\tpop\tdx\n"
"\t\tpop\tcx\n"
"\t\tpop\tbx\n"
"\t\tpop\tax\n"
"\n"
"\t\tcli\t\t\t; Switch stacks.\n"
"\t\tmov\tss, cs:savess\n"
"\t\tmov\tsp, cs:savesp\n"
"\t\tret\n"
"launch\t\tENDP\n"
"\n";

char *listing5 =

"\n"
"; ------------------------- ;\n"
"; Installation Data Section ;\n"
"; ------------------------- ;\n"
"\n"
"installsec:\n"
"\n"
"ctailaddr\tDW\t81h\t\t; Command tail address.\n"
"psp\t\tDW\t?\t\t; PSP of \"searched\" TSR.\n"
"\n"
"scancodes\tDB\t30,48,46,32,18,33,34,35,23,36,37,38,50 ; A-M\n"
"\t\tDB\t49,24,25,16,19,31,20,22,47,17,45,21,44 ; N-Z\n"
"\t\tDB\t11,02,03,04,05,06,07,08,09,10          ; 0-9\n"
"\n"
"tsr_delmsg\tDB\tCR, LF\n"
"\t\tDB\t\"successfully uninstalled\"\n"
"\t\tDB\tCR, LF\n"
"\t\tDB\t'$'\n"
"\n"
"tsr_keymsg\tDB\tCR, LF\n"
"\t\tDB\t\"hotkey = ALT-\"\n"
"tsr_keymsg1\tDB\t?\n"
"\t\tDB\tCR, LF\n"
"\t\tDB\t'$'\n"
"\n"
"tsr_errmsg1\tDB\tCR, LF\n"
"\t\tDB\t\"invalid DOS version (must be 3.3 or higher)\"\n"
"\t\tDB\tCR, LF\n"
"\t\tDB\t'$'\n"
"\n"
"tsr_errmsg2\tDB\tCR, LF\n"
"\t\tDB\t\"invalid character on command line\"\n"
"\t\tDB\tCR, LF\n"
"\t\tDB\t'$'\n"
"\n"
"tsr_errmsg3\tDB\tCR, LF\n"
"\t\tDB\t\"invalid option\"\n"
"\t\tDB\tCR, LF\n"
"\t\tDB\t'$'\n"
"\n"
"tsr_errmsg4\tDB\tCR, LF\n"
"\t\tDB\t\"not in memory\"\n"
"\t\tDB\tCR, LF\n"
"\t\tDB\t'$'\n"
"\n"
"tsr_errmsg5\tDB\tCR, LF\n"
"\t\tDB\t\"unable to uninstall\"\n"
"\t\tDB\tCR, LF\n"
"\t\tDB\t'$'\n"
"\n"
"tsr_errmsg6\tDB\tCR, LF\n"
"\t\tDB\t\"invalid hotkey character\"\n"
"\t\tDB\tCR, LF\n"
"\t\tDB\t'$'\n"
"\n";

char *listing6 =

"tsr_errmsg7\tDB\tCR, LF\n"
"\t\tDB\t\"unable to setup\"\n"
"\t\tDB\tCR, LF\n"
"\t\tDB\t'$'\n"
"\n";

char *listing7 =

"; ------------------------- ;\n"
"; Installation Code Section ;\n"
"; ------------------------- ;\n"
"\n"
"install:\n"
"\t\tlea\tbx, _title\t; Display suitable title.\n"
"\t\tmov\tdx, [bx]\n"
"\t\tmov\tah, 9\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tah, 30h\t\t; Obtain DOS version number.\n"
"\t\tint\t21h\n"
"\n"
"\t\tcmp\tal, 3\t\t; Test major portion.\n"
"\t\tjb\tinstall1\t; Branch if less that 3.\n"
"\t\tja\tinstall2\t; Branch if greater than 3.\n"
"\n"
"\t\tcmp\tah, 30\t\t; Test minor portion.\n"
"\t\tjae\tinstall2\t; Branch if >= 30.\n"
"install1:\n"
"\t\tlea\tdx, tsr_errmsg1\t; Display error message.\n"
"\t\tmov\tah, 9\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tax, 4c01h\t; Exit with error code 1.\n"
"\t\tint\t21h\n"
"install2:\n"
"\t\tmov\tax, 5d06h\t; Obtain critical error flag.\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tax, ds\n"
"\n"
"\t\tpush\tcs\t\t; Restore DS.\n"
"\t\tpop\tds\n"
"\n"
"\t\tmov\tWORD PTR criterr, si\n"
"\t\tmov\tWORD PTR criterr [2], ax\n"
"\n"
"\t\tmov\tah, 34h\t\t; Obtain critical section flag.\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tWORD PTR critsec, bx\n"
"\t\tmov\tWORD PTR critsec [2], es\n"
"\n"
"\t\tlea\tbx, signature\t; Search for resident copy.\n"
"\t\tnot\tBYTE PTR [bx]\t; Avoid false cache match.\n"
"\t\tmov\tcx, SIGLENGTH\t; Get length of signature.\n"
"\t\tcall\tsearch\t\t; Search for prior copy.\n"
"\t\tmov\tpsp, ax\t\t; Save PSP of prior/current TSR.\n"
"\n"
"\t\tcall\tskipws\t\t; Skip whitespace.\n"
"\t\tcall\tparse\t\t; Parse nonwhitespace character.\n"
"\n"
"\t\tcmp\tal, CR\t\t; Empty command tail?\n"
"\t\tjnz\tinstall3\t\t; No, branch.\n"
"\n"
"\t\tmov\tax, psp\t\t; Point ES to resident or ...\n"
"\t\tmov\tes, ax\t\t; ... current segment.\n"
"\n"
"\t\tmov\tal, es:key\t; Get hotkey character.\n"
"\t\tmov\ttsr_keymsg1, al\n"
"\n"
"\t\tpush\tcs\t\t; Restore ES if necessary.\n"
"\t\tpop\tes\n"
"\n"
"\t\tlea\tdx, tsr_keymsg\t; Display hotkey message.\n"
"\t\tmov\tah, 9\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tax, cs\t\t; First install?\n"
"\t\tcmp\tax, psp\n"
"\t\tjz\tinstall10\t; Yes, branch.\n"
"\n"
"\t\tmov\tax, 4c00h\t; Exit with OK code.\n"
"\t\tint\t21h\n"
"install3:\n"
"\t\tcmp\tal, '/'\t\t; Option?\n"
"\t\tjz\tinstall4\n"
"\n"
"\t\tlea\tdx, tsr_errmsg2\t; Display error message.\n"
"\t\tmov\tah, 9\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tax, 4c01h\t; Exit with error code 1.\n"
"\t\tint\t21h\n"
"install4:\n"
"\t\tcall\tparse\t\t; Extract next character.\n"
"\n"
"\t\tcmp\tal, 'K'\t\t; Hotkey option?\n"
"\t\tjz\tinstall5\t; Yes, branch.\n"
"\n"
"\t\tcmp\tal, 'k'\t\t; Lowercase version?\n"
"\t\tjnz\tinstall6\t; No, branch.\n"
"install5:\n"
"\t\tcall\tparse_key\t; Parse hotkey.\n"
"\n"
"\t\tmov\tax, psp\t\t; Point ES to resident or ...\n"
"\t\tmov\tes, ax\t\t; ... current segment.\n"
"\n"
"\t\tmov\tal, es:key\t; Get hotkey character.\n"
"\t\tmov\ttsr_keymsg1, al\n"
"\n"
"\t\tpush\tcs\t\t; Restore ES if necessary.\n"
"\t\tpop\tes\n"
"\n"
"\t\tlea\tdx, tsr_keymsg\t; Display hotkey message.\n"
"\t\tmov\tah, 9\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tax, cs\t\t; First install?\n"
"\t\tcmp\tax, psp\n"
"\t\tjz\tinstall10\t; Yes, branch.\n"
"\n"
"\t\tmov\tax, 4c00h\t; Exit with OK code.\n"
"\t\tint\t21h\n"
"install6:\n"
"\t\tcmp\tal, 'U'\t\t; Uninstall option?\n"
"\t\tjz\tinstall7\t; Yes, branch.\n"
"\n"
"\t\tcmp\tal, 'u'\t\t; Lowercase version?\n"
"\t\tjnz\tinstall9\t; No, branch.\n"
"install7:\n"
"\t\tmov\tax, cs\t\t; First install?\n"
"\t\tcmp\tax, psp\n"
"\t\tjnz\tinstall8\t; No, branch.\n"
"\n"
"\t\tlea\tdx, tsr_errmsg4\t; Display error message.\n"
"\t\tmov\tah, 9\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tax, 4c01h\t; Exit with error code 1.\n"
"\t\tint\t21h\n"
"install8:\n"
"\t\tjmp\tuninstall\n"
"install9:\n"
"\t\tlea\tdx, tsr_errmsg3\t; Display error message.\n"
"\t\tmov\tah, 9\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tax, 4c01h\t; Exit with error code 1.\n"
"\t\tint\t21h\n"
"install10:\n";

char *listing8 =

"\t\tcall\t_setup\t\t; Attempt additional setup.\n"
"\t\tor\tax, ax\t\t; Setup successful?\n"
"\t\tjz\tinstall10_5\t; Yes, branch.\n"
"\n"
"\t\tlea\tdx, tsr_errmsg7\t; Display error message.\n"
"\t\tmov\tah, 9\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tax, 4c01h\t; Exit with error code 1.\n"
"\t\tint\t21h\n"
"install10_5:\n";

char *listing9 =

"\t\tmov\tax, 3508h\t; Get timer interrupt vector.\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tWORD PTR oldint8h, bx\n"
"\t\tmov\tWORD PTR oldint8h [2], es\n"
"\n"
"\t\tmov\tax, 3509h\t; Get key interrupt vector.\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tWORD PTR oldint9h, bx\n"
"\t\tmov\tWORD PTR oldint9h [2], es\n"
"\n"
"\t\tmov\tax, 3513h\t; Get disk interrupt vector.\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tWORD PTR oldint13h, bx\n"
"\t\tmov\tWORD PTR oldint13h [2], es\n"
"\n"
"\t\tmov\tax, 3528h\t; Get idle interrupt vector.\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tWORD PTR oldint28h, bx\n"
"\t\tmov\tWORD PTR oldint28h [2], es\n"
"\n"
"\t\tmov\tax, ds:[002ch]\t; Attempt to free environment.\n"
"\t\tcmp\tax, 0\t\t; Can we free environment?\n"
"\t\tjz\tinstall11\t; No, branch.\n"
"\n"
"\t\tmov\tes, ax\t	; Free environment block.\n"
"\t\tmov\tah, 49h\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tWORD PTR ds:[002ch], 0\t; Indicate no environment.\n"
"install11:\n"
"\t\tmov\tax, 2508h\t; Set new timer interrupt vector.\n"
"\t\tlea\tdx, newint8h\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tax, 2513h\t; Set new disk interrupt vector.\n"
"\t\tlea\tdx, newint13h\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tax, 2528h\t; Set new idle interrupt vector.\n"
"\t\tlea\tdx, newint28h\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tax, 2509h\t; Set new key interrupt vector.\n"
"\t\tlea\tdx, newint9h\n"
"\t\tint\t21h\n"
"\n"
"\t\tlea\tdx, installsec\t; Calculate number of para- ...\n"
"\t\tadd\tdx, 15\t\t; ... graphs to keep resident.\n"
"\t\tmov\tcl, 4\n"
"\t\tshr\tdx, cl\n"
"\n"
"\t\tmov\tax, 3100h\t; TSR.\n"
"\t\tint\t21h\n"
"\n"
"; ---------------------------------------------- ;\n"
"; Convert: Convert hotkey character to scancode. ;\n"
";                                                ;\n"
"; Entry: AL = character ('A'-'Z' or '0'-'9')     ;\n"
";                                                ;\n"
"; Exit:  AL = scancode                           ;\n"
";                                                ;\n"
"; All other registers preserved.                 ;\n"
"; Requires presence of scancodes array.          ;\n"
"; ---------------------------------------------- ;\n"
"\n"
"convert\t\tPROC\tNEAR\n"
"\t\tpush\tbx\n"
"\n"
"\t\tmov\tbl, al\t\t; Get hotkey character.\n"
"\t\txor\tbh, bh\n"
"\n"
"\t\tcmp\tbl, 'A'\t\t; Alphabetic hotkey?\n"
"\t\tjb\tconvert1\t; No, branch.\n"
"\n"
"\t\tsub\tbl, 'A'\t\t; Get alphabetic scancode.\n"
"\t\tmov\tal, scancodes [bx]\n"
"\n"
"\t\tpop\tbx\n"
"\t\tret\n"
"convert1:\n"
"\t\tsub\tbl, '0'\t\t; Get digit scancode.\n"
"\t\tmov\tal, scancodes [bx+26]\n"
"\n"
"\t\tpop\tbx\n"
"\n"
"\t\tret\n"
"convert\t\tENDP\n"
"\n"
"; ------------------------------------- ;\n"
"; Parse: Parse nonwhitespace character. ;\n"
";                                       ;\n"
"; Entry: none                           ;\n"
";                                       ;\n"
"; Exit: AL = character                  ;\n"
";                                       ;\n"
"; All other registers preserved.        ;\n"
"; ------------------------------------- ;\n"
"\n"
"parse\t\tPROC\tNEAR\n"
"\t\tpush\tbx\n"
"\n"
"\t\tmov\tbx, ctailaddr\t; Get command tail address.\n"
"\t\tmov\tal, [bx]\t; Get character at this address.\n"
"\n"
"\t\tcmp\tal, CR\t\t; End of command tail?\n"
"\t\tjz\tparse1\t\t; Yes, branch.\n"
"\n"
"\t\tinc\tbx\t\t; Point to next character.\n"
"\t\tmov\tctailaddr, bx\t; Update command tail address.\n"
"parse1:\n"
"\t\tpop\tbx\n"
"\n"
"\t\tret\n"
"parse\t\tENDP\n"
"\n"
"; -------------------------- ;\n"
"; Parse_key: Parse K option. ;\n"
";                            ;\n"
"; Entry: none                ;\n"
";                            ;\n"
"; Exit: none                 ;\n"
";                            ;\n"
"; All registers preserved.   ;\n"
"; -------------------------- ;\n"
"\n"
"parse_key\tPROC\tNEAR\n"
"\t\tpush\tax\n"
"\t\tpush\tes\n"
"\n"
"\t\tcall\tparse\t\t; Parse character after K.\n"
"\n"
"\t\tcmp\tal, 'a'\t\t; Convert to uppercase.\n"
"\t\tjb\tparse_key1\n"
"\n"
"\t\tcmp\tal, 'z'\n"
"\t\tja\tparse_key1\n"
"\n"
"\t\tsub\tal, 32\n"
"parse_key1:\n"
"\t\tcmp\tal, '0'\t\t; Validate.\n"
"\t\tjb\tparse_key2\n"
"\n"
"\t\tcmp\tal, 'Z'\n"
"\t\tja\tparse_key2\n"
"\n"
"\t\tcmp\tal, '9'\n"
"\t\tjbe\tparse_key3\n"
"\n"
"\t\tcmp\tal, 'A'\n"
"\t\tjae\tparse_key3\n"
"parse_key2:\n"
"\t\tlea\tdx, tsr_errmsg6\t; Display error message.\n"
"\t\tmov\tah, 9\n"
"\t\tint\t21h\n"
"\n"
"\t\tpop\tes\t\t; Cleanup stack even though ...\n"
"\t\tpop\tax\t\t; ... we don't have to.\n"
"\n"
"\t\tmov\tax, 4c01h\t; Exit with error code 1.\n"
"\t\tint\t21h\n"
"parse_key3:\n"
"\t\tpush\tax\n"
"\n"
"\t\tmov\tax, psp\t\t; Point ES to resident segment.\n"
"\t\tmov\tes, ax\n"
"\n"
"\t\tpop\tax\n"
"\n"
"\t\tmov\tes:key, al\t; Save character.\n"
"\n"
"\t\tcall\tconvert\t\t; Convert character to scancode.\n"
"\n"
"\t\tmov\tes:scancode, al\t; Save scancode.\n"
"\n"
"\t\tpop\tes\n"
"\t\tpop\tax\n"
"\n"
"\t\tret\n"
"parse_key\tENDP\n"
"\n"
"; ------------------------------------------------ ;\n"
"; Search: Search memory for previous TSR copy.     ;\n"
";                                                  ;\n"
"; Entry: BX = address of signature to search       ;\n"
";        CX = length of signature                  ;\n"
";                                                  ;\n"
"; Exit:  AX = PSP address of prior or current copy ;\n"
";                                                  ;\n"
"; All registers preserved (direction flag cleared) ;\n"
"; ------------------------------------------------ ;\n"
"\n"
"search\t\tPROC\tNEAR\n"
"\t\tpush\tbx\n"
"\t\tpush\tcx\n"
"\t\tpush\tsi\n"
"\t\tpush\tdi\n"
"\t\tpush\tes\n"
"\n"
"\t\tmov\tsi, bx\t\t; SI will point to signature.\n"
"\n"
"\t\tmov\tax, 0a000h\t; Initial segment.\n"
"\t\tmov\tbx, cs\t\t; Final segment.\n"
"\t\tcld\t\t\t; Forward string operations.\n"
"search1:\n"
"\t\tinc\tax\t\t; Point to next segment.\n"
"\n"
"\t\tcmp\tax, bx\t\t; At current segment?\n"
"\t\tjz\tsearch2\t\t; Yes, exit search.\n"
"\n"
"\t\tmov\tes, ax\t\t; Target segment.\n"
"\t\tmov\tdi, si\t\t; Same offset.\n"
"\n"
"\t\tpush\tcx\t\t; CMPSB destroys CX and SI.\n"
"\t\tpush\tsi\n"
"\n"
"\t\trep\tcmpsb\t\t; Search for match.\n"
"\n"
"\t\tpop\tsi\n"
"\t\tpop\tcx\n"
"\n"
"\t\tjnz\tsearch1\t\t; Branch if no match.\n"
"search2:\n"
"\t\tpop\tes\n"
"\t\tpop\tdi\n"
"\t\tpop\tsi\n"
"\t\tpop\tcx\n"
"\t\tpop\tbx\n"
"\n"
"\t\tret\n"
"search\t\tENDP\n"
"\n"
"; ---------------------------------------- ;\n"
"; Skipws: Skip whitespace on command tail. ;\n"
";                                          ;\n"
"; Entry: none                              ;\n"
";                                          ;\n"
"; Exit: none                               ;\n"
";                                          ;\n"
"; All registers preserved.                 ;\n"
"; ---------------------------------------- ;\n"
"\n"
"skipws\t\tPROC\tNEAR\n"
"\t\tpush\tax\n"
"\t\tpush\tbx\n"
"\n"
"\t\tmov\tbx, ctailaddr\t; Get command tail address.\n"
"skipws1:\n"
"\t\tmov\tal, [bx]\t; Get command tail character.\n"
"\n"
"\t\tcmp\tal, ' '\t\t; Is character a space?\n"
"\t\tjz\tskipws2\t\t; Yes, skip character.\n"
"\n"
"\t\tcmp\tal, TAB\t\t; Is character a tab?\n"
"\t\tjnz\tskipws3\t\t; Yes, skip character.\n"
"skipws2:\n"
"\t\tinc\tbx\t\t; Point to next character.\n"
"\t\tjmp\tSHORT skipws1\t; Continue.\n"
"skipws3:\n"
"\t\tmov\tctailaddr, bx\t; Update command tail address.\n"
"\n"
"\t\tpop\tbx\n"
"\t\tpop\tax\n"
"\n"
"\t\tret\n"
"skipws\t\tENDP\n"
"\n"
"; ---------------------------------- ;\n"
"; Uninstall: Remove TSR from memory. ;\n"
";                                    ;\n"
"; Entry: none                        ;\n"
";                                    ;\n"
"; Exit: does not return              ;\n"
"; ---------------------------------- ;\n"
"\n"
"uninstall\tPROC\tNEAR\n"
"\t\tmov\tax, 3508h\t; Make sure timer int exists.\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tax, es\n"
"\t\tcmp\tax, psp\n"
"\t\tjnz\tuninstall1\n"
"\n"
"\t\tmov\tax, 3509h\t; Make sure key int exists.\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tax, es\n"
"\t\tcmp\tax, psp\n"
"\t\tjnz\tuninstall1\n"
"\n"
"\t\tmov\tax, 3513h\t; Make sure disk int exists.\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tax, es\n"
"\t\tcmp\tax, psp\n"
"\t\tjnz\tuninstall1\n"
"\n"
"\t\tmov\tax, 3528h\t; Make sure idle int exists.\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tax, es\n"
"\t\tcmp\tax, psp\n"
"\t\tjnz\tuninstall1\n"
"\n"
"\t\tmov\tax, psp\n"
"\t\tmov\tes, ax\n"
"\n"
"\t\tlds\tdx, es:oldint8h\t; Restore timer int.\n"
"\t\tmov\tax, 2508h\n"
"\t\tint\t21h\n"
"\n"
"\t\tlds\tdx, es:oldint9h\t; Restore key int.\n"
"\t\tmov\tax, 2509h\n"
"\t\tint\t21h\n"
"\n"
"\t\tlds\tdx, es:oldint13h ; Restore disk int.\n"
"\t\tmov\tax, 2513h\n"
"\t\tint\t21h\n"
"\n"
"\t\tlds\tdx, es:oldint28h ; Restore idle int.\n"
"\t\tmov\tax, 2528h\n"
"\t\tint\t21h\n"
"\n"
"\t\tpush\tcs\n"
"\t\tpop\tds\n"
"\n"
"\t\tmov\tax, psp\t\t; Get resident PSP segment.\n"
"\t\tmov\tes, ax\n"
"\n";

char *listing10 =

"\t\tcall\t_cleanup\t; Cleanup after setup.\n"
"\n";

char *listing11 =

"\t\tmov\tah, 49h\t\t; Free memory.\n"
"\t\tint\t21h\n"
"\t\tjc\tuninstall1\n"
"\n"
"\t\tlea\tdx, tsr_delmsg\t; Display success message.\n"
"\t\tmov\tah, 9\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tax, 4c00h\t; Exit with OK code.\n"
"\t\tint\t21h\n"
"uninstall1:\n"
"\t\tlea\tdx, tsr_errmsg5\t; Display error message.\n"
"\t\tmov\tah, 9\n"
"\t\tint\t21h\n"
"\n"
"\t\tmov\tax, 4c01h\t; Exit with error code 1.\n"
"\t\tint\t21h\n"
"uninstall\tENDP\n"
"\n"
"_TEXT\t\tENDS\n"
"\t\tEND\ttsr\n";

int	cbreak		(void);
void	error		(char *message);
int	map		(void);
void	usage		(void);

void main (int argc, char **argv)
{
   char drive [MAXDRIVE];
   char dir [MAXDIR];
   char file [MAXFILE];
   char ext [MAXEXT];

   char asmspec [MAXPATH], tsrspec [MAXPATH];

   FILE *f;
   int arg, hotkey = 'T', sc = 0, scancode = 0x14, stack_size = 256;

   char scancodes [] =
   {
      30, 48, 46, 32, 18, 33, 34, 35, 23, 36, 37, 38, 50,	/* A-M */
      49, 24, 25, 16, 19, 31, 20, 22, 47, 17, 45, 21, 44,	/* N-Z */
      11,  2,  3,  4,  5,  6,  7,  8,  9, 10, 0			/* 0-9 */
   };

   ctrlbrk (cbreak);

   fprintf (stderr, "ͻ\n");
   fprintf (stderr, " TSR v1.1: Generate TSR .asm file        \n");
   fprintf (stderr, "                                         \n");
   fprintf (stderr, " Copyright (C) 1993, Geoff Friesen B.Sc. \n");
   fprintf (stderr, " All rights reserved.                    \n");
   fprintf (stderr, "ͼ\n\n");

   if (_osmajor < 3 || _osmajor == 3 && _osminor < 30)
       error ("DOS 3.30 or higher required");

   if (argc < 2 || argc > 5)
       usage ();

   if (argc == 2 &&
       (!stricmp (argv [1], "/m") || !stricmp (argv [1], "-m")))
       exit (map ());

   for (arg = 1; arg < argc; arg++)
	if (*argv [arg] != '/' && *argv [arg] != '-')
	    break;
	else
            if (!strnicmp (argv [arg]+1, "h", 1))
	    {
	        if (!isalnum (argv [arg] [2]))
		    error ("alphanumeric hotkey expected");

		hotkey = toupper (argv [arg] [2]);

                scancode = isalpha (hotkey) ? scancodes [hotkey-'A'] :
		 	   scancodes [hotkey-'0'+26];
	    }
	    else
		if (!strnicmp (argv [arg]+1, "sc", 2))
		    sc = 1;
		else
		    if (!strnicmp (argv [arg]+1, "s", 1))
		    {
			stack_size = atoi (argv [arg]+2);

			if (stack_size < 256 || stack_size > 8192)
			    error ("stack size out of range");
		    }
		    else
			usage ();

   if (arg == argc)
       error ("tsrspec is missing");

   strcpy (tsrspec, argv [argc-1]);

   if (!(fnsplit (tsrspec, drive, dir, file, ext) & EXTENSION))
   {
       strcpy (ext, ".tsr");
       fnmerge (tsrspec, drive, dir, file, ext);
   }
   else
       if (stricmp (ext, ".tsr"))
	   error ("invalid extension (.tsr expected)");

   strcpy (ext, ".asm");
   fnmerge (asmspec, drive, dir, file, ext);

   if ((f = fopen (asmspec, "wt")) == NULL)
       error ("unable to create .asm file");

   if (fprintf (f, "; %s%s", asmspec, listing1) == EOF ||
       fprintf (f, "%c%s", hotkey, listing2) == EOF ||
       fprintf (f, "%02xh%s", scancode, listing3) == EOF ||
       fprintf (f, "\"TSR:%s\"%s", file, listing4) == EOF ||
       fprintf (f, "\t\tDB\t%d DUP (?)\n"
		   "_stack\t\tLABEL\tBYTE\n"
		   "\n"
		   "\t\tINCLUDE %s\n", stack_size, tsrspec) == EOF ||
       fprintf (f, "%s", listing5) == EOF ||
       sc && fprintf (f, "%s", listing6) == EOF ||
       fprintf (f, "%s", listing7) == EOF ||
       sc && fprintf (f, "%s", listing8) == EOF ||
       fprintf (f, "%s", listing9) == EOF ||
       sc && fprintf (f, "%s", listing10) == EOF ||
       fprintf (f, "%s", listing11) == EOF)
       error ("unable to write to .asm file");

   (void) fcloseall ();
   fprintf (stderr, "tsr: %s successfully generated\n", asmspec);
   exit (OK);
}

/* -------------------- */
/* int cbreak (void);   */
/*                      */
/* Disable CTRL-BREAKs. */
/* -------------------- */

int cbreak (void)
{
   return IGNORE;
}

/* --------------------------------------------- */
/* void error (char *message);                   */
/*                                               */
/* Display the specified error message and exit. */
/* --------------------------------------------- */

void error (char *message)
{
   (void) fcloseall ();
   fprintf (stderr, "tsr: %s\n", message);
   exit (ERROR);
}

/* ------------------------------- */
/* int map (void);                 */
/*                                 */
/* Display a map of resident TSRs. */
/* Return OK or ERROR.             */
/* ------------------------------- */

int map (void)
{
   char typemcb;
   unsigned i, ntsrs = 0, segment = 0x40;

   do
   {
       if (peekb(segment, 0) == BLOCMCB)	/* Found first MCB?      */
	   if (peek(segment, 1) == segment+1)	/* First MCB owns ...    */
	       break;				/* ... following segment */

       if (++segment == 0xa000)			/* Goto 640K segment ... */
       {					/* ... even if less mem  */
	   fprintf (stderr, "tsr: bad MCB chain\n");
	   return ERROR;
       }
   }
   while (1);

   puts ("---   ------   ---");
   puts ("PSP   HOTKEY   TSR");
   puts ("---   ------   ---\n");

   do
   {
      if ((~peekb(segment+1, SIGOFS) == 'T') &&
	  (peekb(segment+1, SIGOFS+1) == 'S') &&
	  (peekb(segment+1, SIGOFS+2) == 'R'))
      {
	  printf ("%04X  ", ++segment);		/* Point to PSP          */

	  printf ("ALT-%c    ", peekb(segment, KEYOFS));

	  putchar('T');
	  for (i = SIGOFS+1; putchar(peekb(segment, i++));)
	       ;
	  puts ("");

	  ntsrs++;
	  segment--;				/* Point to MCB          */
       }

      if (peekb(segment, 0) == LASTMCB)
	  break;

      segment += (peek(segment, 3)+1);		/* Point to next MCB     */
      typemcb = peekb(segment, 0);              /* Obtain MCB type       */

      if (typemcb != BLOCMCB && typemcb != LASTMCB)
      {
	  fprintf (stderr, "tsr: bad MCB chain\n");
	  return ERROR;
      }
   }
   while (1);

   if (ntsrs)
       puts ("");

   printf ("tsr: found %d tsr%s", ntsrs, (ntsrs != 1) ? "s.\n" : ".\n");
   return OK;
}

/* --------------------------------------------- */
/* void usage (void);                            */
/*                                               */
/* Display usage information when user enters an */
/* invalid command-line.  Exit program.          */
/* --------------------------------------------- */

void usage (void)
{
   error ("invalid command-line\n\n"
	  "usage: tsr ((/|-)m | [(/|-)hx] [(/|-)sc] [(/|-)sy] tsrspec)"
	  "\n"
	  "       (/|-)m generate map of resident tsrs\n"
	  "       (/|-)h defines the default hotkey "
	  "(x is either 'A'-'Z' or '0'-'9')\n"
	  "       (/|-)sc indicates that setup/cleanup function "
	  "calls are included\n"
	  "       (/|-)s defines the stack size "
	  "(y is in bytes - 256 min, 4K max)"
	  "\n"
	  "       tsrspec defines the .tsr file to be TSRed");
}