%TITLE  "String Procedures -- Copyright 1989 by Tom Swan"

	IDEAL
	DOSSEG
	MODEL	small

	CODESEG

	PUBLIC	MoveLeft, MoveRight, StrNull, StrLength
	PUBLIC	StrUpper, StrCompare,StrDelete,StrInsert
	PUBLIC	StrConcat, StrCopy,StrPos,StrRemove

;--------------------------------------------------------------------
;	For usage see book.
;-------------------------------------------------------------------

ASCNull 	EQU	0

%NEWPAGE
;-------------------------------------------------------------------
; MoveLeft -  move byte-block left (down) in memory
;
;	Input:	si = address of source string (s1)
;		di = address of destination string (s2)
;		bx = index s1 (i1)
;		dx = index s2 (i2)
;		cx = number of bytes to move (count)
;	Output: count bytes from s1[i1] moved to the location starting at s2[i2]
;	Registers:  none
;
;-------------------------------------------------------------------

PROC	MoveLeft
	jcxz	@@99
	push	cx
	push	si
	push	di
	add	si,bx
	add	di,dx
	cld
	rep	movsb
	pop	di
	pop	si
	pop	cx
@@99:
	ret
ENDP	MoveLeft
%NEWPAGE
;-------------------------------------------------------------------
; MoveRight -  move byte-block right (up) in memory
;
;	Input:	si = address of source string (s1)
;		di = address of destination string (s2)
;		bx = index s1 (i1)
;		dx = index s2 (i2)
;		cx = number of bytes to move (count)
;	Output: count bytes from s1(i1) moved to the location starting at s2(i2)
;	Registers:  none
;
;-------------------------------------------------------------------

PROC	MoveRight
	jcxz	@@99
	push	cx
	push	di
	push	si
	add	si,bx
	add	di,dx
	add	si,cx
	dec	si
	add	di,cx
	dec	di
	std
	rep	movsb
	pop	si
	pop	di
	pop	cx
@@99:
	ret
ENDP	MoveRight
%NEWPAGE
;-------------------------------------------------------------------
; StrNull -  Erase all characters in a string
;
;	Input:	di = address of string (s)
;	Output: s[0] <- null character (ASCII 0)
;	Registers:  none
;
;-------------------------------------------------------------------

PROC	StrNull
	mov	[byte ptr di], ASCNull
	ret
ENDP	StrNull
%NEWPAGE
;-------------------------------------------------------------------
; StrLength -  Count non-null characters in a string
;
;	Input:	di = address of string (s)
;	Output: cx = number of non-null characters in (s)
;	Registers:  cx
;
;-------------------------------------------------------------------

PROC	StrLength
	push	ax
	push	di
	xor	al,al
	mov	cx,0ffffh
	cld
	repnz	scasb
	not	cx
	dec	cx
	pop	di
	pop	ax
	ret
ENDP	StrLength
%NEWPAGE
;-------------------------------------------------------------------
; StrUpper  - Convert characters in string to uppercase
;
;	Input:	di = address of string to convert (s)
;	Output: Lowercase chars in string converted to uppercase
;	Registers:  none
;
;-------------------------------------------------------------------

PROC	StrUpper
	push	ax
	push	cx
	push	di
	push	si
	call	StrLength
	jcnxz	@@99
	cld
	mov	si,di
@@10:
	lodsb
	cmp	al, 'a'
	jb	@@20
	cmp	al, 'z'
	ja	@@20
	sub	al, 'a'-'A'
@@20:
	stosb
	loop	@@10
@@99:
	pop	si
	pop	di
	pop	cx
	pop	ax
	ret
ENDP	StrUpper
%NEWPAGE
;-------------------------------------------------------------------
; StrCompare   compare two strings
;
;	Input:	si = address of string 1 (s1)
;		di = address of string 2 (s2)
;	Output: flags set for conditional jump using jb,jbe,je,ja, or jae.
;	Registers:  none
;
;-------------------------------------------------------------------

PROC	StrCompare
	push	ax
	push	di
	push	si
	cld
@@10:
	lodsb
	scasb
	jne	@@20
	or	al,al
	jne	@@10
@@20:
	pop	si
	pop	di
	pop	ax
	ret
ENDP	StrCompare
%NEWPAGE
;-------------------------------------------------------------------
; StrDelete - delete characters anywhere in a string
;
;	Input:	di = address of string (s)
;		dx = index (i) of first char to delete
;		cx = number of characters to delete (n)
;	Output: n characters deleted from string at s[i]
;		NOTE: prevents deleting past end of string
;	Registers:  none
;
;-------------------------------------------------------------------

PROC	StrDelete
	push	bx
	push	cx
	push	di
	push	si
	mov	bx,dx
	add	bx,cx
	call	StrLength
	cmp	cx,bx
	ja	@@10
	add	di,dx
	mov	[byte ptr di],ASCNull
	jmp	short @@99
@@10:
	mov	si,di
	sub	cx,bx
	inc	cx
	call	MoveLeft
@@99:
	pop	si
	pop	di
	pop	cx
	pop	bx
	ret
ENDP	StrDelete
%NEWPAGE
;-------------------------------------------------------------------
; StrInsert - insert one string into another
;
;	Input:	si = address of string 1 (s1)
;		di = address of string 2 (s2)
;		dx = insertion index for s2 (i)
;		NOTE: s2 must be large enough to expand by length (s1)!
;	Output: chars from string s1 inserted at s2[i], s1 not changed
;	Registers:  none
;
;-------------------------------------------------------------------

PROC	StrInsert
	push	ax
	push	bx
	push	cx
	xchg	si,di
	call	StrLength
	xchg	si,di
	mov	ax,cx
	call	StrLength
	sub	cx,dx
	inc	cx
	push	dx
	push	si
	mov	si,di
	mov	bx,dx
	add	dx,ax
	call	MoveRight
	pop	si
	pop	dx
	xor	bx,bx
	mov	cx,ax
	call	MovLeft
	pop	cx
	pop	bx
	pop	ax
	ret
ENDP	StrInsert
%NEWPAGE
;-------------------------------------------------------------------
; StrConcat - concatenate (join) two strings
;
;	Input:	si = address of source string (s1)
;		di = address of destination string (s2)
;		NOTE:  s2 must be large enough to expand by length (s1) !
;	Output: chars from s1 added to end of s2
;	Registers:  none
;
;-------------------------------------------------------------------

PROC	StrConcat
	push	bx
	push	cx
	push	dx
	call	StrLength
	mov	dx,cx
	xchg	si,di
	call	StrLength
	inc	cx
	xchg	si,di
	xor	bx,bx
	call	MoveLeft
	pop	dx
	pop	cx
	pop	bx
	ret
ENDP	StrConcat
%NEWPAGE
;-------------------------------------------------------------------
; StrCopy - copy one string to another
;
;	Input:	si = address of source string (s1)
;		di = address of destination string (s2)
;	Output: chars in s1 copied to s2
;		NOTE: s2 must be at least length (s1) + 1 bytes long
;	Registers:  none
;
;-------------------------------------------------------------------

PROC	StrCopy
	push	bx
	push	cx
	push	dx
	xchg	si,di
	call	StrLength
	inc	cx
	xchg	si,di
	xor	bx,bx
	xor	dx,dx
	call	MoveLeft
	pop	dx
	pop	cx
	pop	bx
	ret
ENDP	StrCopy
%NEWPAGE
;-------------------------------------------------------------------
; StrPos - search for position of a substring in a string
;
;	Input:	si = address of substring to find
;		di = address of target string to scan
;	Output: if zf = 1 then dx = index of substring
;		if zf = 0 then substring was not found
;		NOTE: dx is meaningless if zf = 0
;	Registers:  dx
;
;-------------------------------------------------------------------

PROC	StrPos
	push	ax
	push	bx
	push	cx
	push	di
	call	StrLength
	mov	ax,cx
	xchg	si,di
	call	StrLength
	mov	bx,cx
	xchg	si,di
	sub	ax,bx
	jb	@@20
	mov	dx,0ffffh
@@10:
	inc	dx
	mov	cl,[byte bx + di]
	mov	[byte bx + di],ASCNull
	call	StrCompare
	mov	[byte bx + di],cl
	je	@@20
	inc 	di
	cmp	dx,ax
	jne	@@10
	xor	cx,cx
	inc	cx
@@20:
	pop	di
	pop	cx
	pop	bx
	pop	ax
	ret
ENDP	StrPos
%NEWPAGE
;-------------------------------------------------------------------
; StrRemove - remove substring from a string
;
;	Input:	si = address of substring to delete
;		di = address of string to delete substring from
;	Output: if zf = 1 then substring removed
;		if zf = 0 then substring was not found
;		NOTE: string at si is not changed
;		NOTE: if zf = 0 then string at di is not changed
;	Registers:  none
;
;-------------------------------------------------------------------

PROC	StrRemove
	push	cx
	push	dx
	call	StrPos
	jne	@@99
	pushf
	xchg	si,di
	call	StrLength
	xchg	si,di
	call	StrDelete
	popf
@@99:
	pop	dx
	pop	cx
	ret
ENDP	StrRemove

	END
