%TITLE  "Binary to/from ASCII conversion routines"

	IDEAL
	DOSSEG
	MODEL	small

ASCnull		EQU	0

	DATASEG

	CODESEG

;------ From STRINGS.OBJ

	EXTRN	StrLength:proc, StrUpper:proc

	PUBLIC	HexDigit, ValCh, NumToASCII
	PUBLIC	BinToAscHex, SBinToAscDec, BinToAscDec, BinToAscBin
	PUBLIC	AscToBin

%NEWPAGE
;---------------------------------------------------------------------
;  HexDigit - converts 4-bit value to ASCII digit
;---------------------------------------------------------------------
; 	Input:	dl = value limited to range 0...15
;	Output:	dl = ASCII hex digit equivalent
;	Registers:  dl
;---------------------------------------------------------------------
PROC	HexDigit
	cmp	dl,10
	jb	@@10
	add	dl,'A'-10
	ret
@@10:
	or	dl,'0'
	ret
ENDP	HexDigit
%NEWPAGE
;---------------------------------------------------------------------
;  ValCh - converts ASCII digit char to binary value
;---------------------------------------------------------------------
; 	Input:	dl = ASCII digit '0'...'9'; 'A'..'F'
;		bx = base (2 = binary, 10 = decimal, 16 = hexadecimal)
;	Output:	cf = 0: dx = equivalent binary value
;		cf = 1: bad char for this number base (dx is meaningless)
;	Registers:  dx
;---------------------------------------------------------------------
PROC	ValCh
	cmp	dl, '9'
	jbe	@@10
	sub	dl,7
@@10:
	sub	dl, '0'
	test	dl,0f0h
	jnz	@@99
@@20:
	xor	dh,dh
	cmp	dx,bx
@@99:
	cmc
	ret
ENDP	ValCh
%NEWPAGE
;---------------------------------------------------------------------
;  NumToASCII - converts unsigned binary value to ASCII
;---------------------------------------------------------------------
; 	Input:	ax = 16-bit value to convert
;		bx = base for result (2 = binary, 10 = decimal, 16 = hex)
;		cx = minimum number of digits to output
;		di = address of string to hold result
;		NOTE: assumes string is large enough to hold result
;		NOTE: creates full result if cx is less than the number
;			of digits required to specify the result or cx = 0
;		NOTE: if cx = 0 and ax = 0 then length of string will be 0
;                    	set cx = 1 if you want string to = '0' if ax = 0
;		NOTE: assumes (2 <= bx <= 16)
;	Output:	none
;	Registers:  ax,cx
;---------------------------------------------------------------------
PROC	NumToASCII
	push	dx
	push	di
	push	si
	xor	si,si
	jcxz	@@20
@@10:
	xor	dx,dx
	div	bx
	call	HexDigit
	push	dx
	inc	si
	loop	@@10
@@20:
	inc	cx
	or	ax,ax
	jnz	@@10
	mov	cx,si
	jcxz	@@40
	cld
@@30:
	pop	ax
	stosb
	loop	@@30
@@40:
	mov	[byte di], ASCnull
	pop	si
	pop	di
	pop	dx
	ret
ENDP	NumToASCII
%NEWPAGE
;---------------------------------------------------------------------
;  BinToAscHex - converts binary values to ASCII hex strings
;---------------------------------------------------------------------
; 	Input:	ax = 16-bit value to convert
;		cx = minimum number of digits to output
;		di = address of string to hold result
;		NOTE: assumes string is large enough to hold result
;		NOTE: outputs full result if cx is less than the number
;			of digits required to specify the result
;	Output:	none
;	Registers:  ax,cx
;---------------------------------------------------------------------
PROC	BinToAscHex
	push	bx
	mov	bx,16
	call	NumToAscii
	pop	bx
	ret
ENDP	BinToAscHex
%NEWPAGE
;---------------------------------------------------------------------
;  BinToAscDec - converts binary values to ASCII decimal strings
;---------------------------------------------------------------------
; 	Input:	ax = 16-bit value to convert
;		cx = minimum number of digits to output
;		di = address of string to hold result
;		NOTE: assumes string is large enough to hold result
;		NOTE: outputs full result if cx is less than the number
;			of digits required to specify the result
;	Output:	none
;	Registers:  ax,cx (indirectly)
;---------------------------------------------------------------------
PROC	BinToAscDec
	push	bx
	mov	bx,10
	call	NumToAscii
	pop	bx
	ret
ENDP	BinToAscDec
%NEWPAGE
;---------------------------------------------------------------------
;  SBinToAscDec - converts signed binary values to ASCII decimal strings
;---------------------------------------------------------------------
; 	Input:	ax = signed 16-bit value to convert
;		cx = minimum number of digits to output
;		di = address of string to hold result
;		NOTE: assumes string is large enough to hold result
;		NOTE: outputs full result if cx is less than the number
;			of digits required to specify the result
;	Output:	none
;	Registers:  ax,cx
;---------------------------------------------------------------------
PROC	SBinToAscDec
	push	bx
	push	di
	cmp	ax,0
	jge	@@10
	neg	ax
	mov	[byte di], '-'
	inc	di
@@10:
	mov	bx,10
	call	NumToAscii
	pop	di
	pop	bx
	ret
ENDP	SBinToAscDec
%NEWPAGE
;---------------------------------------------------------------------
;  BinToAscBin - converts binary values to ASCII binary strings
;---------------------------------------------------------------------
; 	Input:	ax = 16-bit value to convert
;		cx = minimum number of digits to output
;		di = address of string to hold result
;		NOTE: assumes string is large enough to hold result
;		NOTE: outputs full result if cx is less than the number
;			of digits required to specify the result
;	Output:	none
;	Registers:  ax,cx (indirectly)
;---------------------------------------------------------------------
PROC	BinToAscBin
	push	bx
	mov	bx,2
	call	NumToAscii
	pop	bx
	ret
ENDP	BinToAscBin
%NEWPAGE
;---------------------------------------------------------------------
;  ChToBase - returns number base for string
;---------------------------------------------------------------------
; NOTE: private subroutine for AscToBin.
;
;	Input:	si = pointer to null terminator at end of string
;		Note: assumes length of string >= 1
;	Output:	bx = 2(binary), 10 (decimal/default), 16 (hexadecimal)
;		si = address of last probable digit character in string
;	Registers:  bx,dl,si
;---------------------------------------------------------------------
PROC	ChgToBase
	mov	dl,[byte si - 1]
	mov	bx,16
	cmp	dl,'H'
	je	@@10
	mov	bx,2
	cmp	dl,'B'
	je	@@10
	mov	bx,10
	cmp	dl,'D'
	jne	@@20
@@10:
	dec	si
@@20:
	ret
ENDP	ChToBase
%NEWPAGE
;---------------------------------------------------------------------
;  AscToNum - converts ASCII characters to binary
;---------------------------------------------------------------------
; NOTE: private subroutine for AscToBin.
; 	Input:	ax = initial value (0)
;		bx = number base ( 2 = binary, 10 = decimal, 16 = hexadecimal)
;		di = address of unsigned string (any format)
;		si = address of last probable digit char in string
;	Output:	cf = 0 : ax = unsigned value
;		cf = 1 : bad character in string (ax is meaningless)
;	Registers:  ax, bx,dx,si
;---------------------------------------------------------------------
PROC	AscToNum
	mov	cx,1
@@10:
	cmp	si,di
	je	@@99
	dec	si
	mov	dl,[byte si]
	call	ValCh
	jc	@@99
	push	cx
	xchg	ax,cx
	mul	dx
	add	cx,ax
	pop	ax
	mul	bx
	xchg	ax,cx
	jmp	@@10
@@99:
	ret
ENDP	AscToNum
%NEWPAGE
;---------------------------------------------------------------------
;  AscToBin - converts ASCII strings to binary values
;---------------------------------------------------------------------
; NOTE: private subroutine for AscToBin.
; 	Input:  di = ASCIIZ string to convert to binary
;		'H' at end of string = hexadecimal
;               'B' at end of string = binary
;		'D' or digit at end of string = decimal
;		'-' at s[0] indicates negative number
;             NOTE: no blanks allowed in string
;	Output:	cf = 0 : ax = value of string
;		cf = 1 : bad character in string (ax is undefined)
;	      NOTE: chars in string converted to uppercase
;             NOTE: null strings set ax = 0
;	Registers:  ax
;---------------------------------------------------------------------
PROC	AscToBin
	push	bx
	push	cx
	push	dx
	push	si
	call	StrUpper
	call	StrLength
	xor	ax,ax
	jcxz	@@99
	mov	si,di
	add	si,cx
	cmp	[byte di], '-'
	pushf
	jne	@@10
	inc	di
@@10:
	call	ChToBase
	call	AscToNum
	rcl	bx,1
	popf
	jne	@@20
	neg	ax
	dec	di
@@20:
	rcr	bx,1
@@99:
	pop	si
	pop	dx
	pop	cx
	pop	bx
	ret
ENDP	AscToBin

	END
