; Program used to speed up timer and handle timer interrupt. While timer
;  is speeded up, program continues to service previous interrupt 8 dependent
;  routines at the normal 18.27 per second rate. This is useful when timeing
;  faster than the normal 18.27 per second rate is required.

sseg    segment  para   stack   'stack'
db      0fffh           dup(?)
sseg    ends

newspeed equ    2                 ; number of times to increase the interrupt 
                                  ;  rate. Can be any number from 1 to 65534,
                                  ;  but "not" 65535, as explained below. Keep
                                  ;  in mind that only certain things can be 
                                  ;  done at an extrememely fast rate. I would
                                  ;  suggest keeping this value under "10" un-
                                  ;  less you do a little research first.

cseg    segment  para   public  'code'
        assume  cs:cseg, ds:cseg, ss:sseg

countdown       dw      newspeed    ; countdown timer
old_int_8       dd      0           ; old pointer saved here
hi      db      "hello",20h,"$"

;***************************************************************
; ------------ START OF PROGRAM --------------------------------
;***************************************************************

; SET VECTOR AND SPEED UP TIMER CHIP
start:                    

        push    cs
        pop     ds

; get the current interrupt 8 vector and save using the "reset vector" function
;  of int 21h supplied by DOS

        push    es
        mov     ax, 3508h
        int     21h                ; get current int 8 pointer
        
; saveing the complete address in one variable makes for a smooth jump to this
;  address as is done later in the program.
        mov word ptr [cs:old_int_8],bx
        mov word ptr [cs:old_int_8+2],es
        pop     es

; install new interrupt 8 routine
        push    cs
        pop     ds
        mov     dx, offset HANDLER
        mov     ax, 2508h
        int     21h     ; install our interrupt 


cli                     ; Do not allow interrupts while fiddling with ports!
mov al, 36h             ; This value is hard to explain here. It essentially
                        ;  sets the bit pattern for 16-bit, mode 3, binary
                        ;  operation.
out 43h, al             ; "Out" this value to port 43h which will be 8253 on
                        ;  an XT and 8254 on an AT
mov ax, 65532/newspeed  ; Reset speed to current rate divided by new rate. The
                        ;  idea is, that the lower the number in this register,
                        ;  the faster it counts down. Upon reaching zero, it 
                        ;  sends a pulse.
;----------------------------------------------------------------------------
; Some systems require a short delay between accessing the port due to fast
;  bus speed. The following "nops" create that delay
nop
nop
nop

out 40h, al
mov al, ah

nop
nop
nop
out 40h, al
sti

push ax
xor ax,ax
int 16h
pop ax

exit:
        push    cs      ; Let everyone know that the data is in the code segment
        pop     ds      ; Data found..... :)
        
        cli
        mov al, 36h
        out 43h, al
        mov ax,0        ; This value will be immediately decremented to 0ffffh
                        ;  (which is the original setting of this register at
                        ;  boot-up by BIOS) so it is the correct value to use 
                        ;  instead of the supected 0fffh.
        
nop
nop
nop
        out 40h, al     ; LSB (least significant byte must go in first, so this
                        ;  is a two-step process to load register correctly
        mov al, ah
        
nop
nop
nop
        out 40h, al
        sti             ; Re-enable interrupts
        
        mov     dx, word ptr [cs:old_int_8]     ; Move offset address of old 
                                                ;  interrupt handler (saved
                                                ;  earlier in the program into 
                                                ;  DX
        mov     ds, word ptr [cs:old_int_8+2]   ; Move segment address of old
                                                ;  interrupt handler into DS.
                                                ;  prior to re-assigning inter-
                                                ;  rupt handler.
        mov     ax, 2508h ; Reset interrupt vector to address of previous inter-
                          ;  rupt handler (now loaded into DS:DX) using the DOS
                          ;  routine designed to do so.

        int     21h       ; Immediately after this line, old interrupt handler
                          ;  will be back in place in interrupt vector.

mov ah,04ch               ; DOS function to end program
int 21h                   ; DOS service to execute function
;*******************************************************************************
;*******************************************************************************
; This is the new handler for interrupt #8. What it does, is it resets the sys-
;  tem timer to a faster rate. The system timer is controlled by the system 
;  clock. The system clock pulses the timer at a rate of 1,1931,180 times per
;  second (1,1931,180/65535=18.27  which  is  the normal speed of the system 
;  timer. BIOS preloads the timer with the value 65535 (0ffffh) at boot-up.
;  With each pulse, the value is decremented by 1. When the value reaches 0,
;  then it issues an interrupt #8. Actually what happens is the following:
;  Because it is a hardware device, it issues an "IRQ 0" which then issues a 
;  software interrupt #8. Anyhow, once the value has reached 0, the whole pro-
;  cess repeats itself. Now, since we are going to reset the timer to a faster
;  rate, it is important to still service all the routines that are dependent
;  on the signal from int 8,,, the time of day as well as the timeinng of cer-
;  tain hardware devices (hard-disk for example in an AT.
;******************************************************************************
;******************************************************************************
                                
PROC HANDLER                    ; Name of the new int 8 handler. Can of course        
                                ;  be any name.
;-----------------------------------------------------------------------------
; This is the "important" step mentioned above. What it does, is it takes the
;  divisorr  of  the "rate  change" and  decrements it by one, each time this
;  routine is re-entered. Once it reaches 0, then we know it's time to service
;  the usual routines that are ordinarily serviced at the original rate.
dec cs:[countdown]
;-----------------------------------------------------------------------------

;-----------------------------------------------------------------------------
; Go back to our routine that is making use of the faster clock-ticks, if the
;  value in "countdown" hasn't yet reached zero.
jnz cs:int_8_exit
;-----------------------------------------------------------------------------

;-----------------------------------------------------------------------------
; The handling of the old interrupt handler dependents are serviced by re-direc-
;  ting the int 8 to address the old interrupt handler who's address has been
;  saved previously in the variable "old_int_8"
jmp cs:[old_int_8]
;------------------------------------------------------------------------------

;----------------------------------------------------------
; Let's leave this place and do something else for a while.
int_8_exit:
;-----------------------------------------------------------

;------------------------------------------------------------------------------
; Prevent interrupts from occuring while we fiddle with the port registers.
cli
;------------------------------------------------------------------------------

;------------------------------------------------------------------------------
; Since we are currently within the new interrupt handler, we can now issue and
;  "end of interrupt" command to the interrupt controller, informing it that 
;  this interrupt has been taken care of and other interrupts can be processed.
; Followed by a command to once again allow all interrupts to occur.
mov al,20h
out 20h,al
sti

;********* This is where your "fast" routine would go *************************
; -> Of course care MUST be taken to preserve flags and registers since your
;     routine "will" be interrupted at currently reset fast rate
;------------------------------------------------------------------------------
; Tell DOS where the data is located for the following routine. This screen-write
;  will occur at a faster than normal rate (the current speeded up timer rate),
;  as DOS seems to use the timer to time itself. This is a very rudmentary way
;  to show the usefulness of a faster timer rate. I used it here only to give
;  visualality to the speed-up of the timer. Do not increase the speed-up rate
;  in this program and expect this screen-write to work properly -- it won't.

;------------------------------------------------------------------------------
; Save all registers that will be modified, returning there value soon after.
push ax
push dx
push ds
;------------------------------------------------------------------------------

push cs
pop ds
mov ah,09h
mov dx, offset hi
int 21h

pop ds
pop dx
pop ax
;------------------------------------------------------------------------------        

        
; An interrupt handler procedure is returned from by using "iret"
        iret

endp    HANDLER

cseg    ends
end     start
