/*******************************************************************/
/* Include MicroSoft's graph facilities for the cursor positioning */
/* and text output capability.                                     */
/*******************************************************************/

#include <graph.h>

#include <stdio.h>
#include <ctype.h>
#include <dos.h>

#include "memlib.h"         /* for function prototypes */

/* Special keys used in keyboard input */

#define LEFT_ARROW  75
#define RIGHT_ARROW 77
#define BACK_SPACE   8
#define FRONT_TAB    9
#define BACK_TAB    15
#define CRETURN     13
#define DEL         83
#define INS         82
#define F9          67
#define F10         68

/* Standard definitions */

#define PASSED 0

#define FALSE  0
#define TRUE   1

#define ACCEPT 1
#define EXIT   0

/* Definitions for setting cursor position */

#define ROW             26
#define COLUMN           0
#define VIDEO_PAGE      0x00
#define SET_CURSOR_FUNC 0x02
#define BIOS_INT        0x10

/* Definitions of starting locations for various */
/* info on the rolodex screen                    */

#define LOGO_START_ROW     3
#define LOGO_START_COL     8

#define INSIDE_START_ROW   2
#define INSIDE_START_COL  20
#define INSIDE_END_ROW    17
#define INSIDE_END_COL    61

#define OUTSIDE_START_ROW  1
#define OUTSIDE_START_COL 16
#define OUTSIDE_END_ROW   25
#define OUTSIDE_END_COL   65

#define TOTAL_EXP_MEM_ROW 16
#define TOTAL_EXP_MEM_COL  9

#define MAX_BLOCK_ROW     17
#define MAX_BLOCK_COL     14

#define MENU_ROW1         22
#define MENU_ROW2         23
#define MENU_ROW3         24
#define MENU_COL           2

#define BLOCK_SIZE_ROW    17
#define BLOCK_SIZE_COL    16

 
/* Define some sizes of the strings used in the rolodex structure */

#define FIRST_NAME_SIZE    10  
#define LAST_NAME_SIZE     20  
#define AREA_CODE_SIZE      3  
#define PHONE_PREFIX_SIZE   3  
#define PHONE_POSTFIX_SIZE  4  
#define ADDR_LINE1_SIZE    30  
#define ADDR_LINE2_SIZE    30  
#define CITY_SIZE          20  
#define STATE_SIZE          2  
#define ZIP_PREFIX_SIZE     5  
#define ZIP_POSTFIX_SIZE    4  
#define NOTES_SIZE         30  

/* total size of strings */

#define TOTAL_SIZE  FIRST_NAME_SIZE   + LAST_NAME_SIZE     + AREA_CODE_SIZE  + \
                    PHONE_PREFIX_SIZE + PHONE_POSTFIX_SIZE + ADDR_LINE1_SIZE + \
                    ADDR_LINE2_SIZE   + CITY_SIZE          + STATE_SIZE      + \
                    ZIP_PREFIX_SIZE   + ZIP_POSTFIX_SIZE   + NOTES_SIZE

/* maximum size of strings */

#define MAX_DATA_SIZE 30

/* Define positions of the strings used in the rolodex structure */

#define FIRST_NAME_POSITION     0    
#define LAST_NAME_POSITION      1    
#define AREA_CODE_POSITION      2    
#define PHONE_PREFIX_POSITION   3    
#define PHONE_POSTFIX_POSITION  4     
#define ADDRESS_LINE1_POSITION  5    
#define ADDRESS_LINE2_POSITION  6    
#define CITY_POSITION           7    
#define STATE_POSITION          8    
#define ZIP_PREFIX_POSITION     9    
#define ZIP_POSTFIX_POSITION   10   
#define NOTES_POSITION         11   

/* Rolodex structure */

typedef struct
   {

   /* strings displayed.  Each string is offset in data */

   char    data[TOTAL_SIZE];

   /* "pointers" to the next and previous nodes in the list */

   int     next_node;
   int     prev_node;

   } NODE;

/* Size and offset table to show where each string starts in data */
/* Offset table gets initialized in initialize_list.              */

int size_table[] =
   { FIRST_NAME_SIZE,    LAST_NAME_SIZE,  AREA_CODE_SIZE,   PHONE_PREFIX_SIZE,
     PHONE_POSTFIX_SIZE, ADDR_LINE1_SIZE, ADDR_LINE2_SIZE,  CITY_SIZE,
     STATE_SIZE,         ZIP_PREFIX_SIZE, ZIP_POSTFIX_SIZE, NOTES_SIZE };

int offset_table[12];

/* "Pointers" to the start of the list and the currently viewed node */

int start_of_list;
int current_view_node;

/* Structure describing options on a rolodex card */

typedef struct
   {
   char *title;         /* Pointer to a title for this option  */
   int  title_length;   /* Length of title string              */
   int  row;            /* The row this option is on           */
   int  column;         /* The column                          */
   int  back_tab;       /* Which option to go to on a back tab */
   int  front_tab;      /* Ditto for forward tab               */
   int  c_return;       /* Ditto for carraige return           */
   int  number_chars;   /* How many characters in this option  */
   } OPTION;

/* Option table used to build the rolodex screen */

OPTION option_list[] =
   {
                    /* first_name option (0)         */
   "NAME: ",        /* title                         */
   6,               /* title_length                  */
   3,               /* row                           */
   12,              /* column                        */
   11,              /* back_tab (to notes)           */
   1,               /* front_tab (to last_name)      */
   2,               /* c_return (to area code)       */
   10,              /* number_chars                  */

                    /* last_name option (1)          */
   "",              /* title                         */
   0,               /* title_length                  */
   3,               /* row                           */
   23,              /* column                        */
   0,               /* back_tab (to first_name)      */
   2,               /* front_tab (to area_code)      */
   2,               /* c_return (to area code)       */
   20,              /* number_chars                  */

                    /* area_code option (2)          */
   "TELEPHONE NUMBER: (",         /* title           */
   19,              /* title_length                  */
   5,               /* row                           */
   25,              /* column                        */
   1,               /* back_tab (to last_name)       */
   3,               /* front_tab (to phone_first_3)  */
   5,               /* c_return (to address line 1)  */
   3,               /* number_chars                  */
                                                
                    /* phone_first_3 option (3)      */
   ") ",            /* title                         */
   2,               /* title_length                  */
   5,               /* row                           */
   30,              /* column                        */
   2,               /* back_tab (to area_code)       */
   4,               /* front_tab (to phone_last_4)   */
   5,               /* c_return (to address line 1)  */
   3,               /* number_chars                  */
                                                
                    /* phone_last_4 option (4)       */
   "-",             /* title                         */
   1,               /* title_length                  */
   5,               /* row                           */
   34,              /* column                        */
   3,               /* back_tab (to phone_first_3)   */
   5,               /* front_tab (to address_line_1) */
   5,               /* c_return (to address line 1)  */
   4,               /* number_chars                  */

                    /* address_line_1 option (5)     */
   "ADDRESS: ",     /* title                         */                  
   9,               /* title_length                  */
   7,               /* row                           */
   15,              /* column                        */
   4,               /* back_tab (to phone_last_4)    */
   6,               /* front_tab (to address_line_2) */
   6,               /* c_return (to address line 2)  */
   30,              /* number_chars                  */

                    /* address_line_2 option (6)     */
   "",              /* title                         */
   0,               /* title_length                  */
   9,               /* row                           */
   15,              /* column                        */
   5,               /* back_tab (to address_line_1)  */
   7,               /* front_tab (to city)           */
   7,               /* c_return (to city)            */
   30,              /* number_chars                  */

                    /* city option (7)               */
   "CITY: ",        /* title                         */
   6,               /* title_length                  */
   11,              /* row                           */
   12,              /* column                        */
   6,               /* back_tab (to address_line_2)  */
   8,               /* front_tab (to state)          */
   9,               /* c_return (to zip)             */
   20,              /* number_chars                  */

                    /* state option (8)              */
   "STATE: ",       /* title                         */
   7,               /* title_length                  */
   11,              /* row                           */
   39,              /* column                        */
   7,               /* back_tab (to city)            */
   9,               /* front_tab (to zip_first_5)    */
   9,               /* c_return (to zip)             */
   2,               /* number_chars                  */

                    /* zip_first_5 option (9)        */
   "ZIP: ",         /* title                         */
   5,               /* title_length                  */
   13,              /* row                           */
   35,              /* column                        */
   8,               /* back_tab (to notes)           */
   10,              /* front_tab (to last_name)      */
   11,              /* c_return (to notes)           */
   5,               /* number_chars                  */

                    /* zip_last_4 option (10)        */
   "-",             /* title                         */
   1,               /* title_length                  */
   13,              /* row                           */
   41,              /* column                        */
   9,               /* back_tab (to zip_first_5)     */
   11,              /* front_tab (to notes)          */
   11,              /* c_return (to notes)           */
   4,               /* number_chars                  */

                    /* notes option (11)             */
   "NOTES: ",       /* title                         */
   7,               /* title_length                  */
   16,              /* row                           */
   13,              /* column                        */
   10,              /* back_tab (to zip_last_4)      */
   0,               /* front_tab (to first_name)     */
   0,               /* c_return (to first_name)      */
   30               /* number_chars                  */
   };

const NUMBER_OF_OPTIONS = sizeof(option_list) / sizeof(OPTION);


void abort(), main();

unsigned int initialize_list(), save_list(), display_total_exp_mem();
unsigned int display_max_block(), display_block_size(), show_node();
unsigned int insert_node(), edit_node(), delete_node();


/* Housekeeping functions for the rolodex */

/*$PAGE*/
/**********************************************************************/
/*                                                                    */
/*     Name:  void hide_cursor (void)                                 */
/*                                                                    */
/*     Description:                                                   */
/*        Hide the cursor by making a call to the set cursor function */
/*     within BIOS.  We put the cursor beyond the boundries of the    */
/*     displayable screen.                                            */
/*                                                                    */
/*     Parameters:  None                                              */
/*                                                                    */
/*     Results returned:  None                                        */
/*                                                                    */
/*     Calls: None                                                    */
/*                                                                    */
/*     Called by: main_menu()                                         */
/*                                                                    */
/*     Globals referenced/modified: None                              */
/*                                                                    */
/**********************************************************************/

void hide_cursor (void)
{
   union REGS inregs, outregs;

   inregs.h.dh = ROW;     
   inregs.h.dl = COLUMN;  
   inregs.h.bh = VIDEO_PAGE;
   inregs.h.ah = SET_CURSOR_FUNC;  /* set cursor position function request */

   int86 (BIOS_INT, &inregs, &outregs);
}


/*$PAGE*/
/**********************************************************************/
/*                                                                    */
/*     Name:  void draw_border (void)                                 */
/*                                                                    */
/*     Description:                                                   */
/*        Draws the border for the rolodex screen display.            */
/*                                                                    */
/*     Parameters:  None                                              */
/*                                                                    */
/*     Results returned:  None                                        */
/*                                                                    */
/*     Calls:  None                                                   */
/*                                                                    */
/*     Called by: main()                                              */
/*                                                                    */
/*     Globals referenced/modified:  None                             */
/*                                                                    */
/**********************************************************************/

void draw_border (void)

   {
    static char *border[] =
       {
       "   Ŀ",
       "                                              ",
       "                                              ",
       "                                              ",
       "                                              ",
       "                                              ",
       "                                              ",
       "                                              ",
       "                                              ",
       "                                              ",
       "                                              ",
       "                                              ",
       "                                              ",
       "                                              ",
       " ڿ                                           ڿ",
       "                                            ",
       "                                            ",
       " ٳ",
       "                                              ",
       "                                              "
       };

   /* number of border lines to draw */

   static int number_of_border_lines = sizeof (border) / sizeof (border[0]);
   int        i;      /* loop                */
   int        row;    /* row to draw line on */

   row = 0;
   for (i = 0; i < number_of_border_lines; i++)
       {

       /* increment row (start at row 1), set position, and draw */

       row++;
       _settextposition (row, 1);
       _outtext (border[i]);
       }
   }  /** end of draw_border **/

/*$PAGE*/
/**********************************************************************/
/*                                                                    */
/*     Name:  void draw_logo (void)                                   */
/*                                                                    */
/*     Description:                                                   */
/*        Draws the Intel logo used when viewing first (dummy) node   */
/*     in the list.                                                   */
/*                                                                    */
/*     Parameters:  None                                              */
/*                                                                    */
/*     Results returned:  None                                        */
/*                                                                    */
/*     Calls:  None                                                   */
/*                                                                    */
/*     Called by: show_node()                                         */
/*                                                                    */
/*     Globals referenced/modified:  None                             */
/*                                                                    */
/**********************************************************************/

void draw_logo (void)

   {
   static char *logo[] =
       {
       "                            (R)",
       "                                 ",
       "                    ",
       "                         ",
       "                     ",
       "                     ",
       "                ",
       "                                  ",
       "                               ",
       "                                     ",
       "                                     ",
       "               ROLODEX               "
       };

   /* Number of lines to draw in the logo */

   static int number_of_logo_lines = sizeof(logo) / sizeof(logo[0]);

   int        i;      /* loop                */
   int        row;    /* row to draw line on */

   row = LOGO_START_ROW;
   for (i = 0; i < number_of_logo_lines; i++)
       {

       /* increment row (start at row 4), set position, and draw */

       row++;
       _settextposition (row, LOGO_START_COL);
       _outtext (logo[i]);
       }
   }  /** end draw_logo **/

/*$PAGE*/
/**********************************************************************/
/*                                                                    */
/*     Name:  void clear_rolodex (void)                               */
/*                                                                    */
/*     Description:                                                   */
/*        Clears the rolodex screen with out erasing the border.      */
/*                                                                    */
/*     Parameters:  None                                              */
/*                                                                    */
/*     Results returned:  None                                        */
/*                                                                    */
/*     Calls:  None                                                   */
/*                                                                    */
/*     Called by:  show_node()                                        */
/*                 edit_fields()                                      */
/*                                                                    */
/*     Globals referenced/modified:  None                             */
/*                                                                    */
/**********************************************************************/

void clear_rolodex (void)

   {

   _settextwindow (INSIDE_START_ROW, INSIDE_START_COL, INSIDE_END_ROW,
                   INSIDE_END_COL);
   _clearscreen (_GWINDOW);
   _settextwindow (OUTSIDE_START_ROW, OUTSIDE_START_COL, OUTSIDE_END_ROW,
                   OUTSIDE_END_COL);

   }  /** end clear_rolodex **/

/*$PAGE*/
/**********************************************************************/
/*                                                                    */
/*     Name:  int compare_last_names(ptr1, ptr2)                      */
/*            NODE far *ptr1, *ptr2;                                  */
/*                                                                    */
/*     Description:                                                   */
/*        Compares two names - strcmpi won't work on far data items.  */
/*     (Bug in MS C).                                                 */
/*                                                                    */
/*     Parameters:                                                    */
/*        input  ptr1   pointer to last name 1                        */
/*        input  ptr2   pointer to last name 2                        */
/*                                                                    */
/*     Results returned:                                              */
/*        <0   ptr1 < ptr2                                            */
/*        0    ptr1 = ptr2                                            */
/*        >0   ptr1 > ptr2                                            */
/*                                                                    */
/*     Calls:  None                                                   */
/*                                                                    */
/*     Called by: insert_node()                                       */
/*                                                                    */
/*     Globals referenced/modified:  size_table                       */
/*                                   offset_table                     */
/*                                                                    */
/**********************************************************************/

int compare_last_names (ptr1, ptr2)
NODE far *ptr1, *ptr2;

    {
    char near_1[LAST_NAME_SIZE + 1];   /* local buffer for last name 1 */
    char near_2[LAST_NAME_SIZE + 1];   /* local buffer for last name 2 */
    int  i;                            /* loop                         */
    int  ret;                          /* return value                 */

   /************************************/
   /* copy last name 1 to local buffer */
   /************************************/

   for (i = 0; i < size_table[1]; i++)
       near_1[i] = ptr1->data[offset_table[LAST_NAME_POSITION] + i];
   near_1[i] = 0;

   /************************************/
   /* copy last name 2 to local buffer */
   /************************************/

   for (i = 0; i < size_table[1]; i++)
       near_2[i] = ptr2->data[offset_table[LAST_NAME_POSITION] + i];
   near_1[i] = 0;

   /***************************************/
   /* Use string compare on local strings */
   /***************************************/

   ret = strcmpi (near_1, near_2);

   return(ret);
   } /** end compare_last_names **/

/*$PAGE*/
/**********************************************************************/
/*                                                                    */
/*     Name: int edit_fields (title_line, this_node)                  */
/*           char *tile_line;                                         */
/*           int this_node;                                           */
/*                                                                    */
/*     Description:                                                   */
/*        To edit the fields in this node.  We must map in the block  */
/*      associated with this node in order to gain access to it.      */
/*                                                                    */
/*     Parameters:                                                    */
/*        input this_node  "pointer" to the node to edit fields in.   */
/*                                                                    */
/*     Results returned:                                              */
/*        ACCEPT ==> accept was picked                                */
/*        EXIT   ==> exit was picked                                  */
/*                                                                    */
/*     Calls:  set1eptr()                                             */
/*             abort()                                                */
/*             clear_rolodex()                                        */
/*             show_node()                                            */
/*                                                                    */
/*     Called by:  insert_node()                                      */
/*                 edit_node()                                        */
/*                                                                    */
/*     Globals referenced/modified:  option_list                      */
/*                                   size_table                       */
/*                                   offset_table                     */
/*                                                                    */
/**********************************************************************/

int edit_fields (title_line, this_node)
char *title_line;
int  this_node;

   {
   int          cur_option;     /* which option we're on               */
   int          i;              /* loop                                */
   int          pos;            /* column position for INS, DEL        */
   int          cur_char;       /* Current character within an option  */
   unsigned int status;         /* error tarcking                      */
   unsigned int ret_status;     /* return status (EXIT/ACCEPT)         */
   NODE far     *ptr;           /* ptr to a tempoary node              */
   short        row;            /* row we're on for screen positioning */
   short        col;            /* column we're on                     */
   short        key;            /* key that was pressed                */
   short        ext_scan;       /* TRUE/FALSE ext_scan code            */
   static char  out[] = " ";    /* string for _outtext() call          */
   static char  *menu_line[] =  /* bottom menu lines                   */
       {
       "TITLE LINE TITLE LINE TITLE LINE TITLE LINE TITLE",
       "    Use Tab, Larrow and Rarrow to move cursor.   ",
       "  F9 = accept changes  F10 = exit without change "
       };

   /******************************************/
   /* Set first menu line to passed in title */
   /******************************************/

   menu_line[0] = title_line;

   /**********************************/
   /* Get access to the node to edit */
   /**********************************/

   status = set1eptr (this_node, &ptr);

   if (status != PASSED)
       abort(status);

   else
       {

       /****************************/
       /* Clear the rolodex screen */
       /****************************/

       clear_rolodex();

       /*****************************************/
       /* Set spaces in data to '_' for display */
       /*****************************************/

       for (i = 0; i < TOTAL_SIZE; i++)
           if (ptr->data[i] == ' ')
               ptr->data[i] = '_';

       /******************/
       /* Show this node */
       /******************/

       show_node(this_node);

       /**********************/
       /* Put up bottom menu */
       /**********************/

       _settextposition(MENU_ROW1, MENU_COL);
       _outtext(menu_line[0]);
       _settextposition(MENU_ROW2, MENU_COL);
       _outtext(menu_line[1]);
       _settextposition(MENU_ROW3, MENU_COL);
       _outtext(menu_line[2]);

       /***************************************************************/
       /* Set current option and current character within that option */
       /***************************************************************/

       cur_option = 0;
       cur_char = 0;

       /*************************************************************/
       /* Get key presses until F9 (accept) or F10 (exit) is picked */
       /*************************************************************/

       do
           {

           /********************************************************/
           /* set the cursor postion to the row,col in option_list */
           /********************************************************/

           row = option_list[cur_option].row;
           col = option_list[cur_option].column + cur_char;
           _settextposition (row, col);

           /***************************************************/
           /* Get a key press and test for extended scan code */
           /***************************************************/

           key = getch();
           if (key == 0)
               {
               ext_scan = TRUE;
               key = getch();
               }
           else
               ext_scan = FALSE;

           /* If normal key then ... */

           if (!ext_scan)
               switch (key)
                   {

                   /***************************************************/
                   /* Carraige return: Set current option to the      */
                   /* option determined by the c_return field of the  */
                   /* option_list.  Set current character to 0.       */
                   /***************************************************/

                   case CRETURN:
                       cur_option = option_list[cur_option].c_return;
                       cur_char = 0;

                       break;

                   /****************************************************/
                   /* Back space: Erase current character in data. Set */
                   /* cursor one space left (if beyond beginning of    */
                   /* option set to previous option).  Set row,col     */
                   /* and output an '-' to screen.                     */
                   /****************************************************/

                   case BACK_SPACE:
                       ptr->data[offset_table[cur_option] + cur_char] = '_';

                       cur_char--;
                       if (cur_char < 0)
                           {
                           cur_option = option_list[cur_option].back_tab;
                           cur_char = size_table[cur_option] - 1;
                           }

                       row = option_list[cur_option].row;
                       col = option_list[cur_option].column + cur_char;
                       _settextposition (row, col);
                       out[0] = '_';
                       _outtext (out);

                       break;

                   /***************************************************/
                   /* Front Tab:  Set current option the the next     */
                   /* option in option_list.  Set the current         */
                   /* character to 0 (start at beginning of option).  */
                   /***************************************************/

                   case FRONT_TAB:
                       cur_option = option_list[cur_option].front_tab;
                       cur_char = 0;

                       break;

                   /***************************************************/
                   /* Default:  If the key pressed is a printable key */
                   /* then set current character in data to the key.  */
                   /* Increment current character (if beyond the end  */
                   /* of the option, set to next option).  Output the */
                   /* key pressed.                                    */
                   /***************************************************/

                   default:
                       if (isprint(key))
                           {
                           ptr->data[offset_table[cur_option] + cur_char] = key;

                           cur_char++;
                           if (cur_char >= size_table[cur_option])
                               {
                               cur_char = 0;
                               cur_option = option_list[cur_option].front_tab;
                               }

                           out[0] = key;
                           _outtext (out);

                           }

                           break;
                   } /** end switch **/

           else             /* The key pressed was an ext scan code */
               switch (key)
                   {

                   /***************************************************/
                   /* Back tab: Set cuurent option to the previous    */
                   /* option in option_list.  Set currrent character  */
                   /* to 0 (beginning of the option).                 */
                   /***************************************************/

                   case BACK_TAB:
                       cur_option = option_list[cur_option].back_tab;
                       cur_char = 0;

                       break;

                   /***************************************************/
                   /* Left arrow: Set current character back one.  If */
                   /* beyond beginning of option set to last character*/
                   /* of previous option.                             */
                   /***************************************************/

                   case LEFT_ARROW:
                       cur_char--;
                       if (cur_char < 0)
                           {
                           cur_option = option_list[cur_option].back_tab;
                           cur_char = size_table[cur_option] - 1;
                           }

                       break;

                   /***************************************************/
                   /* Right arrow:  Set current character up one.  If */
                   /* beyond end of option set to first character of  */
                   /* next option.                                    */
                   /***************************************************/

                   case RIGHT_ARROW:
                       cur_char++;
                       if (cur_char >= size_table[cur_option])
                           {
                           cur_option = option_list[cur_option].front_tab;
                           cur_char = 0;
                           }

                       break;

                   /***************************************************/
                   /* Delete: From the current character to the end   */
                   /* of the current option,  set each character in   */
                   /* data to the next character and output it to the */
                   /* screen.  This effectively deletes the current   */
                   /* character and pulls the rest of the option one  */
                   /* space forward.                                  */
                   /***************************************************/

                   case DEL:
                       for (i = cur_char; i < size_table[cur_option]; i++)
                           {

                           pos = offset_table[cur_option] + i;
                           if (i < size_table[cur_option] - 1)
                               ptr->data[pos] = ptr->data[pos + 1];
                           else
                               ptr->data[pos] = '_';

                           col = option_list[cur_option].column + i;
                           _settextposition (row, col);
                           out[0] = ptr->data[pos];
                           _outtext (out);
                           }

                       break;

                   /***************************************************/
                   /* Insert: Insert a space at the current location  */
                   /* and push the rest of the option back one space. */
                   /***************************************************/

                   case INS:
                       for (i = size_table[cur_option] - 1; i >= cur_char; i--)
                           {

                           pos = offset_table[cur_option] + i;
                           if (i > cur_char)
                               ptr->data[pos] = ptr->data[pos - 1];
                           else
                               ptr->data[pos] = '_';

                           col = option_list[cur_option].column + i;
                           _settextposition (row, col);
                           out[0] = ptr->data[pos];
                           _outtext(out);
                           }

                       break;

                   }  /** end switch **/

           }  /** end do **/
       while (!ext_scan || ((key != F9) && (key != F10)));

       /***********************************************************/
       /* Set status to either FAILED (exit was picked) or PASSED */
       /* (accept was picked).                                    */
       /***********************************************************/

       if (key == F10)
           ret_status = EXIT;
       if (key == F9)
           {
           ret_status = ACCEPT;

           /*********************************/
           /* Set all underscores to spaces */
           /*********************************/

           for (i = 0; i < TOTAL_SIZE; i++)
               if (ptr->data[i] == '_')
                   ptr->data[i] = ' ';
           }

       } /** end if status **/

   return (ret_status);

   } /** end edit_fields **/

/*$PAGE*/
/**********************************************************************/
/*                                                                    */
/*     Name:  void main_menu (void)                                   */
/*                                                                    */
/*     Description:                                                   */
/*        This routine is the executive routine for the rolodex       */
/*     program.  It puts up the main menu, waits for a keypress, and  */
/*     interpets the option chosen.                                   */
/*                                                                    */
/*     Parameters:  None                                              */
/*                                                                    */
/*     Results returned:  None                                        */
/*                                                                    */
/*     Calls:  show_node()                                            */
/*             hide_cursor()                                          */
/*             set1eptr()                                             */
/*             abort()                                                */
/*             insert_node()                                          */
/*             edit_node()                                            */
/*             delete_node()                                          */
/*             save_list()                                            */
/*                                                                    */
/*     Called by:  main()                                             */
/*                                                                    */
/*     Globals referenced/modified:  current_view_node                */
/*                                                                    */
/**********************************************************************/

void main_menu (void)

   {
   char         key;            /* Key pressed                  */
   NODE far     *ptr;           /* Pointer to current view node */
   unsigned int status;         /* Error tracking               */
   unsigned int ext_scan_code;  /* TRUE/FALSE on keypress       */
   unsigned int need_redraw;    /* Do we need to redraw node??  */
   static char  *menu_line[] =  /* Bottom menu lines            */
       {
       "            I N T E L   R O L O D E X            ",
       "     <--- backwards            ---> forwards     ",
       "  I) Insert  D) Delete  E) Edit  S) Save  Q) Quit"
       };

   /*******************************************************/
   /* As long as the key pressed was not a 'Q' (for Quit) */
   /* keep interpeting options.                           */
   /*******************************************************/

   do
       {

       /* Show the current view node */

       show_node (current_view_node);

       /* Put up bottom menu */

       _settextposition (MENU_ROW1, MENU_COL);
       _outtext (menu_line[0]);
       _settextposition (MENU_ROW2, MENU_COL);
       _outtext (menu_line[1]);
       _settextposition (MENU_ROW3, MENU_COL);
       _outtext (menu_line[2]);

       /********************************************************/
       /* We don't want to see the cursor while waiting for a  */
       /* menu option.                                         */
       /********************************************************/

       hide_cursor();

       /***********************************************************/
       /* As long as we don't need to redraw, ask for another key */
       /***********************************************************/

       do
           {

           need_redraw = TRUE;

           /****************************************/
           /* Get key, test for extended scan code */
           /****************************************/

           key = getch();
           if (key == 0)
               {
               ext_scan_code = TRUE;
               key = getch();
               }
           else
               ext_scan_code = FALSE;

           /*******************************/
           /* Based on key, decide option */
           /*******************************/

           switch (toupper (key))
               {

               /******************************************************/
               /* Left arrow: Set current view node to previous node */
               /******************************************************/

               case LEFT_ARROW:
                   if (ext_scan_code)
                       {
                       status = set1eptr (current_view_node, &ptr);
                       if (status == PASSED)
                           current_view_node = ptr->prev_node;
                       else
                           abort (status);
                       }

                   break;

               /**************************************************/
               /* Left arrow: Set current view node to next node */
               /**************************************************/

               case RIGHT_ARROW:
                   if (ext_scan_code)
                       {
                       status = set1eptr (current_view_node, &ptr);
                       if (status == PASSED)
                           current_view_node = ptr->next_node;
                       else
                           abort (status);
                       }

                   break;

               /******************/
               /* I: Insert node */
               /******************/

               case 'I':
                   status = insert_node();
                   if (status != PASSED)
                       abort (status);
                   break;

               /****************/
               /* E: Edit node */
               /****************/

               case 'E':
                   status = edit_node();
                   if (status != PASSED)
                       abort (status);
                   break;

               /******************/
               /* D: Delete node */
               /******************/

               case 'D':
                   status = delete_node();
                   if (status != PASSED)
                       abort (status);
                   break;

               /****************/
               /* S: Save list */
               /****************/

               case 'S':
                   status = save_list();
                   if (status != PASSED)
                       abort (status);
                   break;

               /***********************************/
               /* Q: Quit (don't set need_redraw) */
               /***********************************/

               case 'Q':
                   break;

               /************************************************/
               /* If it's some other key set need_redraw false */
               /* so that we won't redraw.                     */
               /************************************************/

               default:
                   need_redraw = FALSE;
                   break;

               } /** end switch **/

           } /** end do **/
       while (!need_redraw);

       } /** end do **/
   while (toupper (key) != 'Q');

   } /** end main_menu **/

