title   serial port functions

; This file contains functions to directly access the async serial ports with
; buffered, interrupt-driven communications.
;
; Version of 27 May 1991
;
; Special version of 6 June 1991 for release with COMRESET source -- removed
; references to external multitasking functions not needed in this program.
;
; PRELIMINARY VERSION
;
; A future version should have adjustable transmit and receive buffers.
;
; COMMAND LINE ASSEMBLY SWITCHES
;
; By default, this file assembles into a small-model object file containing
; functions prefixed with "_md" (modem commands) and "_mp" (modem port
; initialization and deinitialization commands) which are intended to
; control one serial port at a time.  (Multiple serial ports can still be
; accessed by the program, but each must be closed via <mpclose()> and the
; next opened with <mpopen()> before they can be used.  As a result, only
; one is accessible at a time.)
;
; If the constant AUX is defined via the /D command line switch, this file
; assembles an alternate set of functions, prefixed with "_ax" and "_ap," to
; allow for a second, auxiliary serial port in programs that need to access
; more than one serial device simultaneously.  The auxiliary port also has
; smaller transmit and receive buffers than the primary serial port.
;
; The command line to compile the auxiliary port in small model is:
;
;	masm /mx /dAUX pcserial.asm,pcaux.obj;
;
; Defining the constant MEDMODEL causes the file to be assembled in the medium
; memory model:
;
;	masm /mx /dMEDMODEL pcserial.asm,mpcmodem.obj;
;
; The switches may be combined, to produce a medium model auxiliary port
; library:
;
;	masm /mx /dAUX /dMEDMODEL pcserial.asm,mpcaux.asm;

; -------------------------
; segment setup directives
; -------------------------

dosseg

ifdef MEDMODEL
	.model medium

	PARAM1		equ	<word ptr [bp + 6]>	; first parameter
	PARAM2		equ	<word ptr [bp + 8]>	; second parameter
	PARAM3		equ	<word ptr [bp + 10]>	; third parameter
	PARAM4		equ	<word ptr [bp + 12]>	; fourth parameter
else
	.model small

	PARAM1		equ	<word ptr [bp + 4]>	; first parameter
	PARAM2		equ	<word ptr [bp + 6]>	; second parameter
	PARAM3		equ	<word ptr [bp + 8]>	; third parameter
	PARAM4		equ	<word ptr [bp + 10]>	; fourth parameter
endif

ifdef AUX

	; redefine function names for auxiliary port

	_mdist		equ	<_axist>
	_mdost		equ	<_axost>
	_mdin		equ	<_axin>
	_mdout		equ	<_axout>
        _mddtr          equ     <_axdtr>
        _mdrts          equ     <_axrts>
        _mdcts          equ     <_axcts>
        _mddsr          equ     <_axdsr>
        _mdri           equ     <_axri>
	_mddcd		equ	<_axdcd>
	_mdclr		equ	<_axclr>
	_mdbaud		equ	<_axbaud>
	_mdsend		equ	<_axsend>
	_mdoclr		equ	<_axoclr>
        _mpopen         equ     <_apopen>
	_mpclose	equ	<_apclose>
	_mdiful		equ	<_axiful>
	_mdoful		equ	<_axoful>
	_mdxoff		equ	<_axxoff>
	_mdxon		equ	<_axxon>
	_mdlook		equ	<_axlook>
	_mderrs		equ	<_axerrs>
        _mdoerrs        equ     <_axoerrs>
        _mdperrs        equ     <_axperrs>
        _mdferrs        equ     <_axferrs>
        _mdbreaks       equ     <_axbreaks>
        _mderst         equ     <_axerst>
        _mdparams       equ     <_axparams>
        _mdfifo         equ     <_axfifo>
	_mdxost		equ	<_axxost>
	_mdnoxo		equ	<_axnoxo>
        _mdstxo         equ     <_axstxo>
        _mdnohs         equ     <_axnohs>
        _mdsths         equ     <_axsths>
        _mp_ready       equ     <_ap_ready>
endif

; -----------------------------
; public/external declarations
; -----------------------------

ifdef AUX
        public _axist, _axost, _axin, _axout, _axdtr, _axrts, _axcts, _axdsr
        public _axri, _axdcd, _axclr, _axbaud, _axsend, _axoclr, _apopen
        public _apclose, _axiful, _axoful, _axxoff, _axxon, _axlook, _axerrs
        public _axoerrs, _axperrs, _axferrs, _axbreaks, _axparams, _axfifo
        public _axxost, _axnoxo, _axstxo, _axnohs, _axsths, _ap_ready
else
        public _mdist, _mdost, _mdin, _mdout, _mddtr, _mdrts, _mdcts, _mddsr
        public _mdri, _mddcd, _mdclr, _mdbaud, _mdsend, _mdoclr, _mpopen
        public _mpclose, _mdiful, _mdoful, _mdxoff, _mdxon, _mdlook, _mderrs
        public _mdoerrs, _mdperrs, _mdferrs, _mdbreaks, _mdparams, _mdfifo
        public _mdxost, _mdnoxo, _mdstxo, _mdnohs, _mdsths, _mp_ready
endif

; ---------------------------------
; register & interrupt definitions
; ---------------------------------

	BASE_8259A	equ	20h		; 8259A base port
	IMP_8259A	equ	BASE_8259A + 1	; interrupt control mask port
        XOFF            equ     19              ; XOFF character
        DEFAULT_BASE    equ     02f8h           ; default to COM2

        ; the following are alternative names for some of the port addresses

        in_port         equ     <base_port>     ; data input port
        out_port        equ     <base_port>     ; data output port
        dlm_port        equ     <ier_port>      ; divisor latch MSB
        fcr_port        equ     <iir_port>      ; FIFO control register

; ------------------------
; MS-DOS function EQUates
; ------------------------

	DOSI_FUNC	equ	21h		; DOS interrupt
	DOSF_INTVEC	equ	35h		; get interrupt vector
	DOSF_SIVEC	equ	25h		; set interrupt vector

; -------------------
; internal constants
; -------------------

	TRUE		equ	1
	FALSE		equ	0

ifdef AUX
	I_BUF_SIZE	equ	100			; input buffer size
	O_BUF_SIZE	equ	100 			; output buffer size
else
	I_BUF_SIZE	equ	400			; input buffer size
	O_BUF_SIZE	equ	2000			; output buffer size
endif

	I_BUF_RESV	equ	I_BUF_SIZE * 2 / 10	; input buffer reserve
	O_BUF_RESV	equ	O_BUF_SIZE * 2 / 10	; output buffer reserve

; ------------------------
; main executable section
; ------------------------

.code

; ------------------
; ***** setst *****
; ------------------
;
; Sets up the status words in the data area based on the values found in the
; modem status register.  On entry, AL contains current MSR value.  This
; function cannot be called from a C program.

setst	proc

        test    al, 10000000b                   ; check DCD bit
        jz      setst1                          ; false condition
	mov	word ptr dcd_status, TRUE	; otherwise true
        jmp     setst2

setst1:
        mov     word ptr dcd_status, FALSE      ; set to false

setst2:
        test    al, 01000000b                   ; check RI bit
        jz      setst3
        mov     word ptr ri_status, TRUE        ; RI is true
        jmp     setst4

setst3:
        mov     word ptr ri_status, FALSE       ; RI is false

setst4:
        test    al, 00100000b                   ; check DSR bit
        jz      setst5
        mov     word ptr dsr_status, TRUE       ; DSR is true
        jmp     setst6

setst5:
        mov     word ptr dsr_status, FALSE      ; DSR is false

setst6:
        test    al, 00010000b                   ; check CTS bit
        jz      setst7
        mov     word ptr cts_status, TRUE       ; CTS is true
        mov     word ptr hs_mode, FALSE         ; so free to send
        ret

setst7:
        mov     word ptr cts_status, FALSE      ; CTS is false
        cmp     hs_enable, FALSE                ; if handshake enabled
        je      setst8
        mov     word ptr hs_mode, TRUE          ; ... we should not send

setst8:
	ret

setst	endp

; ------------------------
; ***** int mdist() *****
; ------------------------
;
; Returns whether the buffer has a character waiting to be received.

_mdist	proc

	mov	ax, in_chars			; check count in buffer
	or	ax, ax
	jz	mdifals				; AX already false
	mov	ax, TRUE			; return true

mdifals:
	ret

_mdist	endp

; ------------------------
; ***** int mdost() *****
; ------------------------
;
; Returns whether the transmit buffer is empty.

_mdost	proc

        mov     ax, out_chars                   ; check count in buffer
        or      ax, ax
	jz	mdotru				; if buffer empty then true
	xor	ax, ax				; else return false
	ret

mdotru:
        mov     ax, TRUE                        ; return true
	ret

_mdost	endp

; -------------------------
; ***** int mdiful() *****
; -------------------------
;
; Returns whether the input buffer is full, with a reserve capacity
; specified by constant I_BUF_RESV.

_mdiful	proc

	cmp	in_chars, I_BUF_SIZE - I_BUF_RESV	; buffer full?
	jae	miftru				; yes, return true
	xor	ax, ax				; no, return false
	ret

miftru:
	mov	ax, TRUE			; return TRUE
	ret

_mdiful	endp

; -------------------------
; ***** int mdoful() *****
; -------------------------
;
; Returns whether the output buffer is full, with a reserve capacity
; specified by constant O_BUF_RESV.

_mdoful	proc

	cmp	out_chars, O_BUF_SIZE - O_BUF_RESV	; buffer full?
	jae	moftru				; yes, return true
	xor	ax, ax				; no, return false
	ret

moftru:
	mov	ax, TRUE			; return true
	ret

_mdoful	endp

; -----------------------------------
; ***** void mdout(int letter) *****
; -----------------------------------
;
; Sends the single character LETTER out through the port.

_mdout	proc

	push	bp
	mov	bp, sp				; look at stack
        mov     xoff_mode, FALSE                ; cancel XOFF, if any
        cli                                     ; while in buffer
        mov     bx, last_out                    ; offset into buffer
	inc	out_chars			; increment char count
	inc	last_out			; increment buffer address
	cmp	bx, O_BUF_SIZE - 1		; at end of buffer?
	jb	mdo1				; no
	mov	last_out, 0			; reset to start of buffer

mdo1:
	mov	ax, PARAM1			; get output parameter
	mov	out_buf[bx], al			; store char in buffer

	; if buffer is full, do not accept any more characters until it
	; has at least one space available - this throttles all output
	; to rate of serial port after buffer fills

mdo2:
        sti
        cmp     out_chars, O_BUF_SIZE           ; loop until not full
        jb      mdo6
        jmp     mdo2

mdo6:
        ; if there is exactly one character in the output buffer, turn
        ; on the transmit interrupt

        cmp     out_chars, 1
        jne     mdo3

        ; this procedure for enabling the THRE empty interrupt is as
        ; suggested in National Semiconductor Applications Note AN-493,
        ; reprinted in the 1988 Data Communications / Local Area Networks /
        ; UARTs Handbook, page 4-86

mdo4:
        cli                                     ; don't let IER change now
        mov     bl, current_ier                 ; get interrupt enable reg
        test    bl, 00000010b                   ; don't if already enabled
        jnz     mdo3

        ; do not change the IER until the transmitter holding register is
        ; in fact empty

        mov     dx, lsr_port                    ; check THRE bit

mdo5:
        in      al, dx
        test    al, 00100000b                   ; change IER only if empty
        jz      mdo5
        mov     al, bl                          ; restore value of IER
        or      al, 00000010b                   ; turn on xmit interrupt
        mov     dx, ier_port                    ; get right port address
        out     dx, al                          ; access IER twice before
        out     dx, al                          ;  reading IIR which clears it
        mov     current_ier, al                 ; record for next time

mdo3:
        sti
        pop     bp
	ret					; C cleans up stack

_mdout	endp

; -----------------------
; ***** int mdin() *****
; -----------------------
;
; Returns the current character waiting in the buffer and removes it
; from the buffer.
;
; Function <mdist()> MUST be checked first to make sure the buffer has a
; character waiting.

_mdin	proc

	cli					; don't allow interrupt
	mov	bx, first_in			; get offset in buffer
	dec	in_chars			; decrement available count
	inc	first_in			; increment buffer position
	cmp	bx, I_BUF_SIZE - 1		; if at end of buffer,
	jb	mdin1
	mov	first_in, 0			; reset to start of buffer

mdin1:
	mov	al, in_buf[bx]			; get char
	sti
	xor	ah, ah				; clear AH
	ret

_mdin	endp

; -------------------------
; ***** int mdlook() *****
; -------------------------
;
; Returns the current character waiting in the buffer but DOES NOT
; remove it from the buffer.
;
; Function <mdist()> MUST be checked first to make sure the buffer has a
; character waiting.

_mdlook	proc

	mov	bx, first_in			; get offset in buffer
	xor	ah, ah				; clear AH
	mov	al, in_buf[bx]			; get char
	ret

_mdlook	endp

; -------------------------
; ***** void mdclr() *****
; -------------------------
;
; Clears the input channel prior to attempting to read data from the serial
; port.

_mdclr	proc

	cli					; no interrupts allowed
	mov	in_chars, 0			; clear input buffer counters
	mov	first_in, 0
	mov	last_in, 0
	sti					; restore interrupts
	ret

_mdclr	endp

; --------------------------
; ***** void mdoclr() *****
; --------------------------
;
; Clears the output channel prior to attempting to send data to the port.

_mdoclr	proc

	cli					; don't allow interrupt
	mov	first_out, 0			; clear output buffer counters
	mov	last_out, 0
	mov	out_chars, 0
        sti                                     ; reenable interrupt
	ret

_mdoclr	endp

; --------------------------
; ***** void mdxoff() *****
; --------------------------
;
; Sets XOFF mode flag to TRUE so output interrupt handler will ignore
; subsequent transmitter interrupts.  Called by high level function in
; response to keyboard input or other conditions.

_mdxoff	proc

	cli
	mov	xoff_mode, TRUE			; flag XOFF mode
	sti
	ret

_mdxoff	endp

; -------------------------
; ***** void mdxon() *****
; -------------------------
;
; Sets XOFF mode flag to FALSE and reactivates the transmitter.  Called by
; high level function in response to keyboard input or other conditions.
; Output may also be restarted by simply calling the <mdout()> function.

_mdxon	proc

	cli
	mov	xoff_mode, FALSE		; flag no XOFF mode
	sti
	ret

_mdxon	endp

; -------------------------
; ***** int mdxost() *****
; -------------------------
;
; Returns whether the system is currently in the XOFF state, meaning
; an XOFF character has been received, but no other characters have
; been transmitted since.

_mdxost	proc

	mov	ax, xoff_mode			; retrieve data
	ret

_mdxost	endp

; --------------------------
; ***** void mdnoxo() *****
; --------------------------
;
; Cancels the XOFF recognition mode.  After calling this function, XOFF will
; be returned by the input functions as a regular character, and no special
; processing will be done based on it.  Call <mdstxo()> to resume XOFF
; recognition.

_mdnoxo	proc

	cli
	mov	xoff_enable, FALSE
	sti
	ret

_mdnoxo	endp

; --------------------------
; ***** void mdstxo() *****
; --------------------------
;
; Sets the XOFF recognition mode.  After calling this function, output will be
; suspended if an XOFF character is received at any time.  Also, the XOFF
; character itself will be placed in the input buffer for retrieval by the MDIN
; function.  Higher level functions may choose to either monitor the XOFF mode
; via the <mdxost()> function, or may ignore it completely.

_mdstxo	proc

	cli
	mov	xoff_enable, TRUE
	sti
	ret

_mdstxo	endp

; --------------------------
; ***** void mdnohs() *****
; --------------------------
;
; Disables hardware handshake mode through the CTS line.

_mdnohs proc

	cli
        mov     hs_enable, FALSE
	sti
	ret

_mdnohs endp

; --------------------------
; ***** void mdsths() *****
; --------------------------
;
; Enables hardware handshaking through the CTS line.

_mdsths proc

	cli
        mov     hs_enable, TRUE
	sti
	ret

_mdsths endp

; ------------------------
; ***** int mdcts() *****
; ------------------------
;
; Reads the current value of the Clear To Send lead.

_mdcts  proc

        mov     ax, cts_status
	ret

_mdcts  endp

; ------------------------
; ***** int mddsr() *****
; ------------------------
;
; Reads the current value of the Data Set Ready lead.

_mddsr  proc

        mov     ax, dsr_status
	ret

_mddsr  endp

; -----------------------
; ***** int mdri() *****
; -----------------------
;
; Reads the current value of the Ring Indicator lead.

_mdri   proc

        mov     ax, ri_status
	ret

_mdri   endp

; ------------------------
; ***** int mddcd() *****
; ------------------------
;
; Reads the current value of the Data Carrier Detect lead (also called
; Received Line Signal Detect in the IBM/Intel literature).

_mddcd	proc

	mov	ax, dcd_status
	ret

_mddcd  endp

; ----------------------------------
; ***** void mddtr(int value) *****
; ----------------------------------
;
; Sets the Data Terminal Ready lead of the port to a desired state.
; VALUE is either TRUE (1) or FALSE (0).

_mddtr	proc

	push	bp
	mov	bp, sp				; look at stack
	mov	cx, PARAM1			; get parameter
	or	cx, cx				; check parameter
        jz      setdtr                          ; if 0 set pin low
        mov     cl, 00000001b                   ; otherwise set bit 0

setdtr:
        mov     dx, mcr_port                    ; get modem control register
	in	al, dx
        and     al, 11111110b                   ; mask out bit 0
	or	al, cl				; OR with CL register
	out	dx, al				; send it
	pop	bp
	ret					; let C clean stack

_mddtr	endp

; ----------------------------------
; ***** void mdrts(int value) *****
; ----------------------------------
;
; Sets the Request To Send lead of the port to a desired state.
; VALUE is either TRUE (1) or FALSE (0).

_mdrts  proc

	push	bp
	mov	bp, sp				; look at stack
	mov	cx, PARAM1			; get parameter
	or	cx, cx				; check parameter
        jz      setrts                          ; if 0 set pin low
        mov     cl, 00000010b                   ; otherwise set bit 1

setrts:
        mov     dx, mcr_port                    ; get modem control register
	in	al, dx
        and     al, 11111101b                   ; mask out bit 1
	or	al, cl				; OR with CL register
	out	dx, al				; send it
	pop	bp
	ret					; let C clean stack

_mdrts  endp

; -------------------------------------------
; ***** int mdbaud(unsigned baud_rate) *****
; -------------------------------------------
;
; Sets serial port to the baud rate indicated the parameter.
;
; The function returns TRUE if the baud rate specified was valid for this
; hardware, and FALSE if it is not in the table of recognized values.  The
; serial port is actually set to the new baud rate only if it has already
; been initialized via the <mpopen()> function.  If not, the return value
; from this function can still be used to check the validity of baud rates
; entered by users, but the actual port settings will not be changed.  Note
; that the baud rate is an unsigned integer; this allows the 38,400 bps rate
; to be accommodated in a one-word parameter.

_mdbaud	proc

	push	bp
	mov	bp, sp				; look at stack
	xor	ax, ax				; preset to error condition

	; get binary code set for baud rate
	; see table 7.37, page 431 in "The Programmer's PC Sourcebook"

	lea	bx, rates			; find rate table

findbd:
	mov	cx, [bx]			; get test rate
	or	cx, cx				; at bottom of list?
	jz	nogood				; yes, is illegal input value
	inc	bx				; point to rate code
	inc	bx
	cmp	cx, PARAM1			; what we're seeking?
	je	stbaud				; yes, proceed
	inc	bx				; no, look at next entry
	inc	bx
	jmp	findbd

stbaud:
	; bail out if port is closed

	or	ax, _mp_ready			; check flag (AX already 0)
	jz	good				; port closed, but return TRUE
	cli					; while accessing chip
        mov     dx, lcr_port                    ; enable access to divisor port
	in	al, dx
	push	ax				; save it
	push	dx
	or	al, 10000000b
	out	dx, al
	mov	ax, [bx]			; get word-sized divisor value
        mov     dx, out_port
	out	dx, al				; send low byte
	mov	al, ah				; and high byte
        mov     dx, dlm_port
	out	dx, al
	pop	dx				; restore LCR port address
	pop	ax				; disable access to divisors
	and	al, 01111111b
	out	dx, al
	sti					; restore interrupts

good:
	mov	ax, TRUE			; successful return code

nogood:
	pop	bp
	ret					; caller cleans up stack

_mdbaud	endp

; -------------------------------------------------------------------
; ***** int mdparams(int data_bits, int parity, int stop_bits) *****
; -------------------------------------------------------------------
;
; This function sets the serial port to the specified communications
; parameters.
;
; <data_bits> is the number of data bits (5, 6, 7 or 8)
;
; <parity> is an ASCII character denoting the parity status (it must be
; in upper case):
;
;       'N' - no parity
;       'O' - odd parity
;       'E' - even parity
;       'M' - mark parity
;       'S' - space parity
;
; <stop_bits> is the number of stop bits encoded as:
;
;       1 - one stop bit
;       2 - two stop bits
;       3 - 1.5 stop bits
;
; If the parameters are out of range, the function returns FALSE, otherwise
; it sets the port and returns TRUE.

_mdparams proc

	push	bp
	mov	bp, sp				; look at stack
	xor	ax, ax				; preset to return error

	; first, check data and stop bit numbers for valid range
	; build value in CL to send to LCR

	mov	cx, PARAM1			; get data bits
	cmp	cx, 5				; check bounds
        jl      params_bad                      ; note signed comparison
	cmp	cx, 8
        jg      params_bad
        sub     cx, 5                           ; convert to 8250, etc. form
	and	cl, 00000011b			; zero remaining bits
        mov     bx, PARAM3                      ; get stop bits
	cmp	bl, 1				; check bounds
        jl      params_bad
	cmp	bl, 3
        jg      params_bad
	cmp	bl, 1
	je	stopok				; 0 is correct SB value
	or	cl, 00000100b			; 1.5 or 2 stop bits

stopok:
        cmp     PARAM2, 'N'                     ; check for no parity
        je      params_ok                       ; no parity, return OK
        cmp     PARAM2, 'O'                     ; check for odd parity
        jne     ckpareven                       ; no, try even parity
        or      cl, 00001000b                   ; enable parity odd
        jmp     params_ok

ckpareven:
        cmp     PARAM2, 'E'                     ; check for even parity
        jne     ckparmark                       ; no, try mark parity
        or      cl, 00011000b                   ; enable parity and even bit
        jmp     params_ok

ckparmark:
        cmp     PARAM2, 'M'                     ; check for mark parity
        jne     ckparspace                      ; no, try space parity
        or      cl, 00101000b                   ; enable parity and stick bit
        jmp     params_ok

ckparspace:
        cmp     PARAM2, 'S'                     ; check for space parity
        jne     params_bad                      ; no, it's a bad value
        or      cl, 00111000b                   ; set parity, even, stick bits

params_ok:
	mov	al, cl				; retrieve data
        mov     dx, lcr_port
	out	dx, al				; and send to chip
	mov	ax, TRUE			; successful return code

params_bad:
	pop	bp
	ret					; C cleans up stack

_mdparams endp

; --------------------------------------
; ***** int mdfifo(int threshold) *****
; --------------------------------------
;
; When using a 16550 UART, this function sets the threshold level of the
; receive FIFO buffer to the specified depth.  The <mpopen()> function must
; be called before this function to insure the port is initialized properly
; and the UART type has been successfully detected.  If the port has not
; been previously initialized, the function does nothing and returns FALSE.
;
; <threshold> is the requested FIFO buffer depth.  If it is 0, the FIFOs
; will be disabled entirely; otherwise the actual FIFO depth will be rounded
; down to the closest available value on the 16550: 1, 4, 8 or 14 bytes.
;
; The function returns TRUE if the port is open *and* the UART is a 16550,
; or FALSE otherwise.

_mdfifo proc

        cmp     is_16550, FALSE                 ; is it a 16550?
        je      mdfifo0                         ; no, return FALSE
        cmp     _mp_ready, FALSE                ; is port open?
        jne     mdfifo1                         ; yes, continue

mdfifo0:
        xor     ax, ax                          ; return FALSE
        ret

mdfifo1:
        push    bp
        mov     bp, sp                          ; look at stack

        ; build value in AL to send to FCR

        cmp     PARAM1, 1                       ; is parameter 1 or more?
        jae     ckfifo1                         ; yes
        mov     al, 0                           ; parameter is 0 or less,
                                                ;  turn off FIFOs
        mov     t_bytes, 1                      ; cancel xmit FIFO also
        jmp     fifo_true                       ; load FIFO control and exit

ckfifo1:
        mov     t_bytes, 15                     ; all values > 0 will
        mov     al, 00000001b                   ;  enable xmit and FIFOs
        cmp     PARAM1, 4                       ; is parameter 4 or above?
        jae     ckfifo4
        jmp     fifo_true                       ; no, use 00 for fifo = 1

ckfifo4:
        cmp     PARAM1, 8                       ; is parameter 8 or above?
        jae     ckfifo8
        or      al, 01000000b                   ; set FIFO level 4
        jmp     fifo_true

ckfifo8:
        cmp     PARAM1, 14                      ; is parameter 14 or above?
        jae     ckfifo14
        or      al, 10000000b                   ; set FIFO level 8
        jmp     fifo_true

ckfifo14:
        or      al, 11000000b                   ; set FIFO level 14

fifo_true:

        ; on entry, we assume AL contains valid value for FCR

        mov     dx, fcr_port                    ; output new value to FCR
        out     dx, al
	mov	ax, TRUE			; successful return code
	pop	bp
	ret					; C cleans up stack

_mdfifo endp

; -----------------------------------------
; ***** void mdsend(char *outstring) *****
; -----------------------------------------
;
; Sends the parameter <outstring> to the serial port.

_mdsend	proc

	push	bp
	mov	bp, sp				; look at stack
	mov	bx, PARAM1			; pointer from stack
        or      bx, bx                          ; check for null pointer
        jz      sddon

sdloop:
	mov	al, [bx]			; get next character
        or      al, al                          ; at end yet?
	jz	sddon				; yes, exit
	push	bx				; <mdout()> changes it
	push	ax				; send AX
	call	_mdout
	pop	ax				; clean up stack
	pop	bx
	inc	bx				; next character
	jmp	sdloop				; through whole string

sddon:
	pop	bp
	ret					; C cleans the stack

_mdsend	endp

; ----------------------------------------------------------
; ***** int mpopen(unsigned port, unsigned interrupt, *****
; *****            unsigned ibuf, unsigned obuf)      *****
; ----------------------------------------------------------
;
; Initializes the serial port interrupts as required by the buffered input
; routines.
;
; Accepts four parameters:
;	<port> is the base physical port address of the modem's UART
;	<interrupt> is the IRQ number of the modem port
;	<ibuf> is the size of the input buffer in bytes
;	<obuf> is the size of the output buffer in butes
;
; Returns 1 (TRUE) if the port is successfully opened or 0 (FALSE) if there
; is an error.

_mpopen	proc

	; bail out if port is already open

	mov	ax, _mp_ready			; check flag
	or	ax, ax
	jz	open_port			; not open, continue
        ret                                     ; return TRUE (already in AX)

open_port:
        push    bp
	mov	bp, sp				; look at stack

	; get parameters from stack into memory

	mov	ax, PARAM1			; port number
        mov     base_port, ax

        ; keep incrementing base port number to fill various other
        ; port addresses

        inc     ax
        mov     ier_port, ax
        inc     ax
        mov     iir_port, ax
        inc     ax
        mov     lcr_port, ax
        inc     ax
        mov     mcr_port, ax
        inc     ax
        mov     lsr_port, ax
        inc     ax
        mov     msr_port, ax
	mov	cx, PARAM2			; get IRQ (must be 0 - 7)
	cmp	cx, 7				; check range
	jbe	mpok				; in range
	xor	ax, ax				; FALSE error return value
	jmp	mporet

mpok:
        mov     al, 1                           ; make 8259A interrupt mask
	shl	al, cl				; based on IRQ number
	mov	imask, al
	add	cl, 8				; make actual interrupt #
        mov     int_com, cl

        ; pulse the OUT2 line high momentarily to reset interrupts
        ; this is important when using multiple com ports on one IRQ

        mov     dx, mcr_port                    ; address MCR
        in      al, dx                          ; get current value
        mov     ah, al                          ; save it
        or      al, 00001000b                   ; enable OUT2 (interrupts)
        out     dx, al
        mov     al, ah
        out     dx, al

        ; save old interrupt vector and set the new one

	push	es
	push	ds
	mov	al, int_com			; get current interrupt vector
	mov	ah, DOSF_INTVEC
	int	DOSI_FUNC
	mov	old_int_sgmt, es		; store old vector segment
	mov	old_int_ofst, bx		; and offset
	mov	al, int_com			; set comm port vector
        push    cs                              ; interrupt handler segment
        pop     ds                              ; into DS
        lea     dx, _mpint                      ; offset into DX
        cli
	mov	ah, DOSF_SIVEC			; DOS set interrupt function
	int	DOSI_FUNC
	pop	ds				; restore segments
        pop     es
        sti

        ; store all the current values in the UART ports

        mov     dx, lcr_port                    ; enable access to divisor port
	in	al, dx
	mov	old_lcr, al			; store this value as old LCR
	push	dx				; save these to change below
	push	ax
	or	al, 10000000b			; set divisor access bit on
	out	dx, al
	mov	dx, base_port			; save low divisor value
	in	al, dx
	mov	old_dll, al
	inc	dx				; save high divisor value
	in	al, dx
	mov	old_dlm, al
	pop	ax				; retrieve LCR port and value
	pop	dx
	and	al, 01111111b			; turn off access bit
	out	dx, al				; and write back out to chip
        mov     dx, ier_port                    ; store original IER
	in	al, dx
	mov	old_ier, al
        inc     dx                              ; skip past IIR
        in      al, dx
        inc     dx                              ; skip past LCR
        in      al, dx
	inc	dx				; store original MCR
	in	al, dx
	mov	old_mcr, al
	inc	dx				; store original LSR
	in	al, dx
	mov	old_lsr, al
	inc	dx				; store original MSR
	in	al, dx
	mov	old_msr, al
	
	; initialize the UART

        mov     t_bytes, 1                      ; assume not a 16550A for now
        mov     is_16550, FALSE
        mov     dx, in_port                     ; read receive buffer to clear
	in	al, dx				; important!
        mov     dx, lcr_port                    ; address line control register
	mov	al, 00110011b			; 8/n/1 and DLAB off - MUST!
        out     dx, al
        mov     dx, fcr_port                    ; trial enable 16550 FIFOs
        mov     al, 00000001b                   ; FIFO on, threshold 1 byte
        out     dx, al
        mov     dx, iir_port                    ; read IIR
        in      al, dx                          ; bits 6 & 7 set if FIFOs on
        and     al, 11000000b                   ;  indicating a 16550A present
        cmp     al, 11000000b                   ; both bits must be set
        jne     mpono550                        ; no, it is not a 16550A
        mov     is_16550, TRUE                  ; yes, it is a 16550A
        mov     t_bytes, 15                     ; use threshold 1 and
                                                ;  xmit FIFO by default; use
                                                ;  <mdfifo()> to change if
                                                ;  desired

mpono550:
        mov     dx, ier_port                    ; enable UART interrupts
        mov     al, 00001101b                   ; status, RX, rx error
        out     dx, al                          ; (but no TX for now)
        mov     current_ier, al                 ; store value for later

	; read back the value just sent to IER to see if port is really there
	
	in	al, dx
        cmp     al, current_ier
	mov	ax, FALSE			; don't use XOR here
	jne	mporet				; return error code
        mov     dx, lsr_port                    ; read LSR to reset
        in      al, dx
        call    _mdclr                          ; initialize buffers
        call    _mdoclr
        call    _mdnoxo                         ; disable XOFF recognition
        call    _mdxon                          ; reset XOFF mode
        call    _mdnohs                         ; disable hardware handshake
        mov     _mderrs, 0                      ; reset total error count
        mov     _mdoerrs, 0                     ; ... overrun error count
        mov     _mdperrs, 0                     ; ... parity error count
        mov     _mdferrs, 0                     ; ... framing error count
        mov     _mdbreaks, 0                    ; ... and break count
        mov     dx, msr_port                    ; get modem status
	in	al, dx
        call    setst                           ; set up status words

        ; enable the interrupt through the 8259A
        ; store the old 8259A mask register setting for this port's IRQ
        ; ONLY in <old_imr>

        cli
        in      al, IMP_8259A                   ; read interrupt mask
        mov     bl, al                          ; save a copy of it
        mov     ah, imask                       ; IRQ bit mask
        and     bl, ah                          ; save this IRQ bit only
        mov     old_imr, bl                     ; so IMR can be restored later
        not     ah                              ; turn OFF IMR bit to activate
	and	al, ah
        out     IMP_8259A, al                   ; write it back
        sti
        mov     dx, mcr_port                    ; address MCR
	mov	al, 00001000b			; enable OUT2 (interrupts)
	out	dx, al
        mov     ax, TRUE                        ; return TRUE for success

mporet:
	mov	_mp_ready, ax			; tell other functions here
	pop	bp
	ret					; done

_mpopen	endp

; -----------------------------------------
; ***** void mpclose(int assert_dtr) *****
; -----------------------------------------
;
; Deinitializes the serial port interrupts before termination of the program,
; and restores the UART hardware to its configuration before calling the
; <mpopen()> function.
;
; If the parameter <assert_dtr> is nonzero, the Data Terminal Ready lead of
; the modem port will be left ON regardless of its original status.  Other
; UART registers are left in their original state as recorded by <mpopen()>
; when the port was initialized.  The purpose of this option is so that
; communications programs can exit while leaving a modem in the "off-hook"
; condition; many modems otherwise would hang up when DTR went low.

_mpclose	proc

	; bail out if port is already closed

	mov	ax, _mp_ready			; check flag
	or	ax, ax
	jnz	close_port			; port open, continue
	ret

close_port:
	push	bp
        mov     bp, sp                          ; address stack

        ; disable UART interrupts by setting IER to 0

        mov     dx, ier_port                    ; disable all interrupts
        xor     ax, ax
        out     dx, al
        mov     current_ier, al

        ; disable serial port interrupt through command to 8259A
        ; restore original 8259A setting for this port's IRQ
        ; DON'T alter any bits that may have changed in the interim
        ; EXCEPT the one controlling the IRQ for this serial port

        cli
        in      al, IMP_8259A                   ; read current interrupt mask
        mov     ah, imask                       ; get mask bit pattern
        not     ah                              ; invert it
        and     al, ah                          ; set IRQ bit to 0
        or      al, old_imr                     ; restore original IRQ bit
	out	IMP_8259A, al			; write it back

        ; restore original interrupt vector

	mov	bx, old_int_sgmt		; retrieve old segment
	mov	dx, old_int_ofst		; and offset
	mov	al, int_com			; before DS changes
	mov	ah, DOSF_SIVEC			; via DOS function
	push	ds
	mov	ds, bx				; reset original segment
	int	DOSI_FUNC
	pop	ds				; restore segments

	; restore all the original values in the UART ports

        mov     dx, lcr_port                    ; enable access to divisor port
	in	al, dx
	push	ax				; save register and contents
	push	dx
	or	al, 10000000b			; set divisor access bit on
	out	dx, al
	mov	dx, base_port
	mov	al, old_dll			; get old low divisor value
	out	dx, al
	inc	dx				; restore high divisor value
	mov	al, old_dlm
	out	dx, al
	pop	dx				; get LCR port address
	pop	ax				; and contents
	and	al, 01111111b			; turn off access bit
	out	dx, al				; and write back out to chip
        mov     dx, in_port                     ; read receive reg to clear it
	in	al, dx
        mov     dx, ier_port                    ; restore original IER
	mov	al, old_ier
        out     dx, al
        mov     current_ier, al
        inc     dx                              ; disable 16550 FIFOs
        cmp     is_16550, FALSE                 ; don't write if not a 16550
        je      cp1
        mov     al, 0                           ; this is write-only register
        out     dx, al

cp1:
        inc     dx                              ; restore original LCR
	mov	al, old_lcr
	out	dx, al
	inc	dx				; restore original MCR
	mov	al, old_mcr
        mov     cx, PARAM1                      ; check DTR parameter
	or	cx, cx				; is it true?
	jz	out_mcr				; no, use original value
        or      al, 00000011b                   ; set DTR and RTS true
	and	al, 11111011b			; force OUT1 low (no reset)

out_mcr:
	out	dx, al
	inc	dx				; restore original LSR
	mov	al, old_lsr
	out	dx, al
	inc	dx				; restore original MSR
	mov	al, old_msr
        out     dx, al
        mov     _mp_ready, FALSE                ; tell other functions here
        sti                                     ; restore interrupts
        pop     bp
	ret					; done

_mpclose	endp

; ------------------
; ***** mpint *****
; ------------------
;
; serial port interrupt handler
;
; NOT to be called directly by any program

_mpint	proc

        push    dx                              ; save registers
        push    cx
	push	bx
	push	ax
	push	ds
	mov	ax, @data			; set DS to our data segment
        mov     ds, ax

again:
        mov     dx, iir_port
	in	al, dx				; identify the interrupt
	test	al, 00000001b			; if no interrupt pending,
        jnz     exit2                           ;  exit handler
        and     ax, 0000000000001110b           ; mask only legitimate values
        mov     bx, ax                          ; save it

        ; create an interrupt edge by disabling, then reenabling all
        ; interrupts; this ensures the 8259A will see all interrupts.
        ; this procedure suggested in National Semiconductor Applications
        ; Note AN-493, reprinted in the 1988 Data Communications / Local
        ; Area Networks / UARTs Handbook, page 4-88
        ;
        ; uncomment the following three instructions to experiment with
        ; software generated interrupt edges in 16450/16550 systems only --
        ; this causes problems with 8250s

        ; mov     dx, ier_port
        ; xor     al, al
        ; out     dx, al

        ; use interrupt ID register value as an index into the array of
        ; pointers at <itabl> to jump to the appropriate handler code
        ; all handlers jump to <exit>

	jmp	word ptr itabl[bx]

	; exit the interrupt handler

exit:
        ; uncomment the following three instructions to experiment with
        ; software generated interrupt edges in 16450/16550 systems only --
        ; this causes problems with 8250s
        ;
        ; mov     dx, ier_port
        ; mov     al, current_ier                 ; restore correct IER
        ; out     dx, al

exit2:
        mov     al, 20h                         ; 8259A interrupt acknowledge
        out     BASE_8259A, al
        pop     ds                              ; restore registers
	pop	ax
        pop     bx
        pop     cx
	pop	dx
        sti                                     ; interrupts back on
        iret                                    ; leave

	; modem status change interrupt handler

stchg:
        mov     dx, msr_port                    ; get status in AL
	in	al, dx
        call    setst                           ; set status words

        ; this procedure for checking the THRE empty interrupt is as
        ; suggested in National Semiconductor Applications Note AN-493,
        ; reprinted in the 1988 Data Communications / Local Area Networks /
        ; UARTs Handbook, page 4-88

        mov     dx, lsr_port                    ; check THRE bit
        in      al, dx
        test    al, 00100000b                   ; service xmtr if true
        jz      again                            ; else fall through

	; transmit buffer empty interrupt handler
	; get the top character from the buffer and send it through the
	; port

xmit:
        cmp     xoff_mode, FALSE                ; do not transmit if in XOFF
        jne     xmit0
        cmp     hs_mode, FALSE                  ; do not transmit if CTS low
        jne     xmit0
        mov     cx, t_bytes                     ; how many characters to send
        mov     dx, out_port                    ; does not change in loop

xmit2:
	cmp	out_chars, 0			; check # of chars in buffer
        jnz     xmit3                           ; there are some, so transmit

        ; if there are no characters in the output buffer, turn off the
        ; transmit interrupt
        ;
        ; comment out the following five instructions and uncomment the
        ; <and> following them to experiment with software generated
        ; interrupt edges in 16450/16550 systems only -- this causes
        ; problems with 8250s

        mov     dx, ier_port                    ; get interrupt enable reg
        mov     al, current_ier
        and     al, 11111101b                   ; turn off xmit interrupt
        out     dx, al
        mov     current_ier, al                 ; save new value
        ; and     byte ptr current_ier, 11111101b ; turn off xmit interrupt
        jmp     again

xmit3:
	mov	bx, first_out			; get offset into buffer
	mov	al, out_buf[bx]			; get char from buffer
        out     dx, al                          ; send it
	dec	out_chars			; one less in buffer

	; increment FIRST_OUT or reset to beginning if now at top

	cmp	bx, O_BUF_SIZE - 1		; at top of buffer?
	jb	xmit1				; no
	mov	first_out, -1			; reset to start of buffer

xmit1:
        inc     first_out                       ; increment buffer position
        cmp     out_chars, 0                    ; don't loop if no more
        loopnz  xmit2                           ; if in 16550 mode send more

xmit0:
        jmp     again

	; received character available interrupt handler

rcv:
        mov     dx, in_port
	in	al, dx				; get input character
	cmp	al, XOFF			; is incoming char XOFF?
	jne	noxof				; no, continue

	; set XOFF mode indicator and cancel pending output if XOFF
	; received

xof:
	cmp	xoff_enable, FALSE		; is XOFF enabled?
	je	noxon				; no, treat as regular char
	mov	xoff_mode, TRUE			; set XOFF mode on
        jmp     ckrcvst

noxof:
	cmp	xoff_mode, FALSE		; currently in XOFF?
	je	noxon				; no, continue

	; terminate XOFF upon receipt of another character

        mov     xoff_mode, FALSE                ; cancel XOFF

xmitjmp:
        jmp     xmit                            ; transmit first pending char

noxon:
	cmp	in_chars, I_BUF_SIZE		; is input buffer full?
	jae	rcverr				; yes, flag an error, ignore
	mov	bx, last_in			; offset into buffer
	mov	in_buf[bx], al			; move char into buffer
	cmp	bx, I_BUF_SIZE - 1		; at end of buffer?
	jb	mpi1				; no, continue
	mov	last_in, -1			; reset to start of buffer

mpi1:
	inc	last_in				; increment buffer address
        inc     in_chars                        ; increment char count

        ; check receiver status -- with 8250/16450 this should always be
        ; false, but will be true with 16550 that has characters in FIFO;
        ; we can loop to receive all characters in one interrupt service

ckrcvst:
        mov     dx, lsr_port                    ; look at receive status
        in      al, dx
        test    al, 00000001b                   ; check receive status bit
        jnz     rcv                             ; another byte, so read it

        ; this procedure for checking the THRE empty interrupt is as
        ; suggested in National Semiconductor Applications Note AN-493,
        ; reprinted in the 1988 Data Communications / Local Area Networks /
        ; UARTs Handbook, page 4-88

        ; note AL contains current line status from above

        test    al, 00100000b                   ; service xmtr if true
        jnz     xmitjmp
        jmp     again                           ; no more bytes in buffer

	; receiver error interrupt handler

rcverr:
        mov     dx, lsr_port                    ; read line status register
	in	al, dx				; to determine error source
	test	al, 00011110b			; check for break or error
	jz	rcver0				; false alarm
        inc     _mderrs                         ; increment master error count
        test    al, 00010000b                   ; is it a break?
        jz      rcver1                          ; no
        inc     _mdbreaks                       ; yes, increment break count

rcver1:
        test    al, 00001000b                   ; is it a framing error?
        jz      rcver2                          ; no
        inc     _mdferrs                        ; yes, increment frame errors

rcver2:
        test    al, 00000100b                   ; is it a parity error?
        jz      rcver3                          ; no
        inc     _mdperrs                        ; yes, increment parity errors

rcver3:
        test    al, 00000010b                   ; is it an overrun error?
        jz      rcver0                          ; no
        inc     _mdoerrs                        ; yes, increment overruns

rcver0:
        jmp     again                           ; and do nothing more

_mpint	endp

; ---------------------------
; serial functions data area
; ---------------------------

.data						; initialized data area

; table of acceptable baud rates
;
;                baud   code
;               -------------
rates   dw      38400, 0003h
        dw      19200, 0006h
	dw	 9600, 000ch
	dw	 7200, 0010h
	dw	 4800, 0018h
	dw	 3600, 0020h
	dw	 2400, 0030h
	dw	 2000, 003ah
	dw	 1800, 0040h
	dw	 1200, 0060h
	dw	  600, 00c0h
	dw	  300, 0180h
	dw	  150, 0300h
	dw	  134, 0359h
	dw	  110, 0417h
	dw	   75, 0600h
	dw	   50, 0900h
	dw	    0				; terminator

; table of addresses of various interrupt condition handlers

itabl   dw      stchg                           ; modem status change
        dw      xmit                            ; transmitter ready
        dw      rcv                             ; receiver ready
        dw      rcverr                          ; receive error
        dw      exit                            ; not a legitimate value
        dw      exit                            ; not a legitimate value
        dw      rcv                             ; timeout same as receive
        dw      exit                            ; not a legitimate value

; miscellaneous initialized global data used by modem functions
; (initialized to default values)

; public data

_mp_ready       dw      FALSE                   ; port initialized (public)
_mderrs         dw      0                       ; total RX error count
_mdoerrs        dw      0                       ; overrun error count
_mdperrs        dw      0                       ; parity error count
_mdferrs        dw      0                       ; framing error count
_mdbreaks       dw      0                       ; break count

; local data

int_com		db	0bh			; comm port interrupt (COM2:)
base_port       dw      DEFAULT_BASE            ; comm port number
ier_port        dw      DEFAULT_BASE + 1        ; interrupt enable register
iir_port        dw      DEFAULT_BASE + 2        ; interrupt ID register
lcr_port        dw      DEFAULT_BASE + 3        ; line control register
mcr_port        dw      DEFAULT_BASE + 4        ; modem control register
lsr_port        dw      DEFAULT_BASE + 5        ; line status register
msr_port        dw      DEFAULT_BASE + 6        ; modem status register
imask           db      00001000b               ; 8259A interrupt enable mask
is_16550        dw      FALSE                   ; is UART a 16550?
cts_status      dw      TRUE                    ; clear to send state
dsr_status      dw      TRUE                    ; data set ready state
ri_status       dw      FALSE                   ; ring indicator state
dcd_status      dw      TRUE                    ; carrier detect state
t_bytes         dw      1                       ; bytes to send per interrupt

.data?                                          ; uninitialized data area

old_int_ofst	dw	?			; original interrupt offset
old_int_sgmt	dw	?			; original interrupt segment
in_chars	dw	?			; chars in *in* buffer
first_in	dw	?			; first in buffer position
last_in		dw	?			; last in buffer position
out_chars	dw	?			; chars in *out* buffer
first_out	dw	?			; first out buffer position
last_out	dw	?			; last out buffer position
xoff_mode	dw	?			; current state of XOFF
xoff_enable     dw      ?                       ; is XOFF function enabled?
hs_mode         dw      ?                       ; current state of handshake
hs_enable       dw      ?                       ; hardware handshake enabled?
current_ier     db      ?                       ; last value written to IER
old_dll		db	?			; old divisor low register
old_dlm		db	?			; old divisor high register
old_ier		db	?			; old interrupt enable register
old_lcr		db	?			; old line control register
old_mcr		db	?			; old modem control register
old_lsr		db	?			; old line status register
old_msr         db      ?                       ; old modem status register
old_imr         db      ?                       ; old 8259A interrupt mask reg

; input and output buffers

in_buf		db	I_BUF_SIZE dup (?)	; input buffer
out_buf		db	O_BUF_SIZE dup (?)	; output buffer

end

; end of file
