/* INT2GUID.C
 *
 * from:
 * INT2QH.C
 *
 * Author:   Kai Uwe Rommel
 * Date:     Sun 07-Oct-1990
 * Update:   Sat 20-Oct-1990
 * Update:   Sun 11-Nov-1990  Ralf Brown
 *
 * Compiler: MS C 5.00 and newer / compact model, or TC 2.0 / compact model
 * System:   PC/MS-DOS 3.20 and newer, OS/2 1.0 and newer
 *
 *
 * INT2GUID.C is a GUIDE/MAKEHELP version of INT2QH.C. GUIDE and MAKEHELP
 * are programs for pop-up help included the TurboPower Software package
 * Turbo Professional.
 *
 * Transscriptor:	Bent Lynggaard
 * Date:	1.00	Sun 24-Mar-1991
 * Update:	1.01	Tue 02-Apr-1991	- test '!', ';' in line pos 1.
 *					- test disk full
 *		1.02	Sun 14-Apr-1991	- smaller index pages
 *					- main index always leads to start of
 *					  an INT #
 *		1.03	Sun 12-May-1991	- configuration file.
 *		1.04	Sat 18-May-1991 - conditional mask in *.cfg
 *		1.05	Sun 29-Dec-1991 - fixed \n bug in configurate()
 *		1.06	Sun 09-May-1992 - adjusted for change in file format
 *					- "dividerline" info as headings
 *					- all registers can carry parameters
 *
 * This program creates output to the standard output device. The output
 * should be redirected to a file INTERRUP.TXT. The created file should
 * be compiled by the TurboPower MAKEHELP program to a INTERRUP.HLP file,
 * which can be interpreted by the TurboPower GUIDE TSR program:
 *	INT2GUID > [ramdisk:]INTERRUP.TXT
 *	MAKEHELP [ramdisk:INTERRUPT] INTERRUP[.HLP] /Q
 *	GUIDE INTERRUP	(or enter GUIDE with a hot key, press F3,
 *			enter INTERRUP)
 *
 * TurboPower Software supplies a program called POPHELP in their Object
 * Professional package, which is a successor to GUIDE. INT2GUID has
 * facilities for conditional interpretation of supplementary files, so
 * these files can include code optimized for both GUIDE and POPHELP, and
 * the parts compiled depends on a mask defined in the configuration file.
 *
 * The following is considered in creating the topic (and popup window)
 * headers:
 * The first word in the header is the interrupt number, so that GUIDE's
 * search mechanism can find the entry if the hot key is pressed when
 * the cursor is at an interrupt number.
 * MAKEHELP restricts the (length of longest header + 1) times the number
 * of topics to 64 kB. INTER191 had about 2200 topics, so the length of
 * the headers should be limited to 25 characters. However, rather than
 * truncating the INTERRUP.LST header lines to some nonsence, this
 * program uses only the interrupt number as topic headings, plus the
 * AH or AX values where applicable.
 * (v. 1.06: "divider line" info (e.g. "214C" in "--------214C------...")
 * is used for headings, thus allowing a more selective search by GUIDE's
 * cursor-word search.)
 * The main index references some subindeces. The subindeces use the
 * MAKEHELP cross reference facility. MAKEHELP limits the number of cross
 * references to 50 per topic, however, this program limits each subindex
 * to 18 entries, so each "topic" can be shown on one single screen with
 * height 20 lines. For each interrupt number, the entries are temporarily
 * stored, and written to the current subindex page only if all entries
 * for the interrupt number fit on the page.
 *
 * MAKEHELP's text wrapping mechanism is disabled, and as the active
 * window is limited to 76 characters, some lines are missing one or two
 * characters.
 * The amount of text that can be displayed per topic is limited by
 * GUIDE's setting of pages/topic (default = 20) and the screen height
 * defined when GUIDE was initialized.
 *
 */

#define LABEL    "int2guid.c"
#define VERSION  "1.06"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#define divider_line(s) (strncmp(s,"--------",8)==0)
#define maxIndeces 18
  /* max 50, 18 gives one page with height 20 (18 active lines) */
#define mainIndex 200
  /* 1-99 reserved for program, 100-199 reserved for user */
#define false 0
#define true 1

FILE *input, *output, *topics, *subtopics, *config;

char line1[128];
char line2[128];
char tempBuffer[50 * 128];
char *tempPtr;
char category[4] = "--";
char nextHeader[14] = "??";
char infilename[14] = "interrup.lst";
#define infileExt 9
int splitInfile = 0;

char configfile[14] = "int2guid.cfg";
char configmarker[18] = "INT2GUIDE CONFIG";

char missingFile[] = "Missing include file.";

int sub = 0, indexed = 1;
unsigned currentID = 1, activeID = 1, indeces = 0, indexNo = 0, buffered = 0,
  subindeces, activeSub, mask;

void exitfunc(void)
{
  fcloseall();
  unlink("topic.tmp");
  unlink("subtopic.tmp");
}

void errorexit(char *msg)
/* writes msg to stderr and exits with error code 1 */
{
  fputs(msg, stderr);
  exit(1);
} /* errorexit */

void diskFull(int temporary)
/* reports disk full and exits */
{
  char msg[80];
  sprintf(msg,"\n\nDisk full, %s file\n", temporary ? "temporary" : "output");
  errorexit(msg);
} /* diskFull */


int _fputs(char *line, FILE *stream)
/* filters TABs to spaces, and inserts a leading space in lines starting
   with the MAKEHELP command and comment characters '!' and ';'. "_fputs"
   should be used when copying from unknown sources. Use "fputs" when
   copying files with information specifically for this program.
*/
{
  char buffer[128];
  int cnt = 0;

  if ( (*line=='!') || (*line==';') ) /* MAKEHELP command/comment? */
    buffer[cnt++] = ' '; /* start with a space */

  while ( *line )
  {
    switch ( *line )
    {
      case '\t': do buffer[cnt++] = ' '; while ( cnt & 7 ); break;
			/* MAKEHELP does not interpret tabs */
      default  : buffer[cnt++] = *line;
    }
    line++;
  }

  buffer[cnt] = 0;

  if ( (cnt = fputs(buffer, stream)) == EOF )
    diskFull(stream != output);

  return cnt;
}

char *_fgets(char *s, int n, FILE *stream)
{
  char *ptr;
  ptr = fgets(s, n, stream);
  if ( (ptr==NULL) && (stream==input) && splitInfile )
  {
    fclose(input);
    infilename[infileExt]++;
    input = fopen(infilename, "r");
    if ( input != NULL )
    {
      fprintf(stderr, "%s\n", infilename);
      ptr = fgets(s, n, input);
    }
  }
  return ptr;
} /* _fgets */

void Initialize(void)
{
  input     = fopen(infilename, "r");
  if ( input == NULL )
  {
    infilename[infileExt] = 'a';
    infilename[infileExt+1] = 0;
    input = fopen(infilename, "r");
    if ( input == NULL )
    {
      fputs("Cannot open input file (INTERRUP.LST or INTERRUP.A)\n", stderr);
      exit(1);
    }
    splitInfile = 1;
  }
  fprintf(stderr, "%s\n", infilename);
  output    = stdout;
  topics    = fopen("topic.tmp", "w");
  subtopics = fopen("subtopic.tmp", "w");

  tempPtr = tempBuffer;

  fprintf(output,
    ";INTERRUPT help text for MAKEHELP/GUIDE.\n;\n!WIDTH 80\n;\n");
  fprintf(topics, "!TOPIC 1 Interrupt List\n!INDEX %u\n", mainIndex);
}


void testTopic(void) /* limit xrefs/topic to maxIndeces */
{
  if ( indeces+buffered >= maxIndeces ) /* leave one entry for forw. ref */
  {
    currentID++;
    fprintf(topics, "\004%u\005INT %s\005 (index continued)\n"
      "!TOPIC %u INT %s (cont)\n!INDEX %u\n"
      "(continued \004%u\005from\005)\n",
      currentID, category, currentID, category, ++indexNo, activeID);
    indeces = 1; /* the backwards ref */
    activeID = currentID;
  }
} /* testTopic */

void copyBuffer(void)
{
  if ( buffered == 0 )
    return;

  testTopic();
  if ( fputs(tempBuffer, topics) == EOF )
      diskFull(true);
  indeces += buffered;
  buffered = 0;
  tempPtr = tempBuffer;
} /* copyBuffer */

void Cleanup(void)
{
  copyBuffer();
  fclose(topics);
  fclose(subtopics);
  fputs("Cleaning up\n", stderr);

  topics = fopen("topic.tmp", "r");
  subtopics = fopen("subtopic.tmp", "r");

  while ( fgets(line1, sizeof(line1), topics) )
    if ( fputs(line1, output) == EOF )
      diskFull(false);

  while ( fgets(line1, sizeof(line1), subtopics) )
    if ( fputs(line1, output) == EOF )
      diskFull(false);
}


int CopyFile(char *name, int commands)
/* copies a file to the database, returns 0 for success or 1 for error */

/* If commands!=0, also interprets lines starting with "!! <number>" as
   an update to variable "condition", and copies lines to the database
   only if condition == 0 or (condition & mask) != 0
*/
{
  int condition = 0;
  FILE *temp = fopen(name, "r");
  fprintf(stderr, "%s\n", name);
  if ( temp == NULL )
  {
    fprintf(stderr, "WARNING: Could not open %s\n", name);
    fputs("Information was not available\n", output);
    return 1;
  }
  else
  {
    while ( fgets(line2, sizeof(line2), temp) )
      if ( !commands )
	_fputs(line2, output);
      else
	/* does line start with "!! <number>" ? */
	if ( sscanf(line2, "!!%i", &condition) != 1 )
	  /* yes: condition updated, sscanf returns 1 */
	  if ( (condition==0) || (condition & mask) )
	    fputs(line2, output);

    fputs("!NOWRAP\n", output); /* in case it was left in !WRAP state */
    fclose(temp);
    return 0;
  }
} /* CopyFile */

void testTemp(void)
/* v. 1.06: allow no more than 50 entries in tempBuffer */
{
  if (buffered >= 50)
  {
    copyBuffer();
    fprintf(stderr,"WARNING: Too many entrees for GUIDE in INT %s.\n"
      "Manual editing of topic %u may be desirable.\n", category, currentID);
  }
} /* testTemp */

void testSubtopic(char *marker)
{
  if ( ++subindeces >= maxIndeces )
  {
    testTemp();
    sprintf(tempPtr, "\004%u\005%s\005  (list cont.)\n",
      ++currentID, marker);
    tempPtr += strlen(tempPtr);
    buffered++;
    fprintf(subtopics,
      "\004%u\005%s\005  (list cont.)\n!TOPIC %u %s (list cont)\n!NOINDEX\n"
      "(continued \004%u\005from\005)\n",
      currentID, marker, currentID, category, activeSub);
      activeSub = currentID;
    subindeces = 2;
  }
} /* testSubtopic */


void StartTopic(char *header, char *marker, char *desc)
{
  if ( sub )
  {
    testSubtopic(marker);
    if ( fprintf(subtopics, "\004%u\005%s\005  %s",
      ++currentID, marker, desc) == EOF )
	diskFull(true);
  }
  else
  {
    testTemp();
    sprintf(tempPtr, "\004%u\005%s\005  %s", ++currentID, marker, desc);
    tempPtr += strlen(tempPtr);
    buffered++;
  }

  fprintf(output, "!TOPIC %u %s\n", currentID, header);
  if ( indexed )
    indexNo++;
  fprintf(output, indexed ? "!INDEX %u\n" : "!NOINDEX\n", indexNo);
}


void StartList(void)
{
  testTemp();
  sprintf(tempPtr, "\004%u\005INT %s\005  (list)\n",
    ++currentID, category);
  tempPtr += strlen(tempPtr);
  buffered++;
  fprintf(subtopics, "!TOPIC %u %s (list)\n!NOINDEX\n",
    currentID, category);

  sub = 1;
  subindeces = 0;
  activeSub = currentID;
}


/* void EndList(void) - not used */


/* char *NextID(void) - not used */


int RecognizedTopic(void)
{
  char *ptr, *pdesc, topic[4], reg[4], header[16], marker[20];
  int i;

  if ( _fgets(line2, sizeof(line2), input) == NULL )
    return 0;

/* v. 1.06: interrupt number suggested to start after line1[4] */
/* line1 is already tested to start with "INT " */
  if ( (pdesc = strchr(line1, '-')) == NULL )
    return 0;

  for (ptr = line1+3; isspace(*ptr); ptr++)
    ;
  strncpy(topic, ptr, 2); /* interrupt number */
  topic[2] = '\0';

  if ( strcmp(category, topic) )
  {
    sub = 0;
    copyBuffer();
    strcpy(category, topic);
    fprintf(stderr, "%s\015", topic); /* show progress */
  }
  else
    fprintf(output, "\004%u\005INT %s\005 (continued)\n",
      currentID+1, topic);
    /* insert a reference to this one in the former topic */

  strcpy(header, topic);

/* v. 1.06: accept any register as a parameter carrier */
  for (ptr = line2 ; isspace(*ptr) ; ptr++)
     ;
  reg[0] = toupper(*ptr);
  reg[1] = toupper(ptr[1]);
  reg[2] = 0;

  if ( !isspace(reg[1]) &&
	strstr("AX AH AL BX BH BL CX CH CL DX DH DL SI DI BP ES DS", reg)
	!= NULL &&
	!strncmp(ptr+2, " =", 2)
      )
  {
    for ( i = 5; (tolower(ptr[i]) != 'h') && !isspace(ptr[i]); i++ )
      ;
    if ( i < 11 )
    {
      if ( !sub )
	StartList();
      strcat(header, "  ");
      strncat(header, ptr, i);
    }
  }

  strcpy(marker, "INT ");
  strcat(marker, header);

/* v. 1.06: use global "nextHeader" rather than local "header" */
  StartTopic(nextHeader, marker, pdesc);

  _fputs(line1, output);
  if ( !divider_line(line2) )
    _fputs(line2, output);

  return 1;
}


void CopyTopic(void)
{
  char *ptr;
  if ( divider_line(line2) )
  { /* kludge for one-line interrupts */
    _fgets(line1, sizeof(line2), input);
    return;
  }

  for (;;)
  {
    if ( _fgets(line1, sizeof(line1), input) == NULL )
      break;

    if ( !divider_line(line1) )
      _fputs(line1, output);
    else
    {
      if ( _fgets(line2, sizeof(line2), input) == NULL )
	break;

      if ( strncmp(line2, "INT ", 4) )
      {
	_fputs(line1, output);
	_fputs(line2, output);
      }
      else
      {
	/* v. 1-06: store divider line info as a header */
	for (ptr=line1+6; *ptr=='-' || isspace(*ptr); ptr++)
	  ;
	strncpy(nextHeader,ptr,sizeof(nextHeader));
	ptr = strchr(nextHeader, '-');
	if (ptr==NULL || ptr>nextHeader+sizeof(nextHeader))
	  nextHeader[sizeof(nextHeader)-1] = 0;
	else
	  *ptr = 0;
	strcpy(line1, line2);
	break;
      }
    }
  }
}

void configError(void)
{
  errorexit("\nFormat error in configuration file.\n");
} /* configError */

void readconfig(void)
/* reads one line from file config to line2 */
{
  if ( fgets(line2, sizeof(line2), config) == NULL )
    configError();
} /* readconfig */

void copyline(char *str, int len)
/* copies line2 (after deleting the terminating '\n') to str
	for max len characters. */
{
  line2[strlen(line2)-1] = 0; /* ignore '\n' */
  strncpy(str, line2, len);
  str[len] = 0; /* edited: v. 1.05 */
} /* copyline */

void configurate(void)
/* parses configuration file */
{
#define maxHeader 14
#define markerLen 12
  int confv, confsubv, command, extraTopics, i;
  char header[(maxHeader+2) & 0xFE], marker[(markerLen+2) & 0xFE],
    desc[(76-markerLen) & 0xFE], filename[80];
  config = fopen(configfile, "r");
  if ( config == NULL )
  {
    fputs("\nWarning: No configuration file.\n", stderr);
    return;
  }
  readconfig();
  if ( strncmp(line2, configmarker, strlen(configmarker)) )
    configError();
  readconfig();
  if ( sscanf(line2, "%u%u", &confv, &confsubv) < 2 )
    configError();
  if ( ((confv << 8) + confsubv) < 0x104 )
    errorexit ("\nThe configuration file is incompatible with this"
    " version of INT2GUID");
  while ( !feof(config) )
  {
    while ( (fgets(line2, sizeof(line2), config) != NULL)
      && (line2[0] == ';') ) ;
    if feof(config) break;
    copyline(filename, 79);
    readconfig();
    copyline(header, maxHeader);
    readconfig();
    copyline(marker, markerLen);
    i = strlen(marker);
    while ( i<markerLen-1 ) /* pad with spaces if short */
      marker[i++] = ' '; /* is already 0-terminated at markerLen */
    readconfig();
    copyline(desc, 76-markerLen-2);
    i = strlen(desc); /* edited: v. 1.05 */
    desc[i] = '\n';
    desc[++i] = 0;
    readconfig();
    if ( sscanf(line2, "%u%u%i", &command, &extraTopics, &mask) < 3 )
      configError();

    StartTopic(header, marker, desc);
    CopyFile(filename, command);
    currentID += extraTopics;
  }
  fclose(config);
#undef maxHeader
#undef markerLen
} /* configurate */

void main(void)
{
  fprintf(stderr, "\nINT2GUID %s - (c) Kai Uwe Rommel/Bent Lynggaard - %s\n",
    VERSION, __DATE__);
  atexit(exitfunc);

  Initialize(); /* uses topic 1 */

  fputs("Including:\n",stderr);
  StartTopic("Copyright etc.", "Copyright  ", "and references.\n");
  if ( CopyFile("int2guid.ref", true) ) /* topic 2 */
    errorexit(missingFile);

  StartTopic("INTERRUP.LST", "HEADER     ", "Overview of the Interrupt List\n");
  fputs("(See also \4""4\5INTERRUP.1ST\5)\n", output); /* insert reference */
  CopyTopic(); /* topic 3 */

  StartTopic("INTERRUP.1ST", "Mail etc.  ", "How to get/update INTERRUP.LST\n");
  if ( CopyFile("interrup.1st", false) ) /* topic 4 */
    errorexit(missingFile);

  StartTopic("GUIDE Program", "GUIDE      ", "Popup Reference Engine.\n");
  if ( CopyFile("int2guid.gui", true) ) /* topic 5 */
    errorexit(missingFile);

  configurate();

  indexed = 0;
  indexNo = mainIndex;

  while ( RecognizedTopic() )
    CopyTopic();

  Cleanup();

  exit(0);
}

/* End of INT2GUID.C */
