
#include "stdafx.h"
#include "ccanvas.h"
#include <wchar.h>


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

cCanvas::cCanvas(CSize p_size)
{
    m_size = p_size;
	m_curAttribute = 0x07;
}

cCanvas::~cCanvas()
{
}

BOOL cCanvas::Resize(CSize p_newsize)
{
    m_size = p_newsize;
    Fill(7,32);
    return TRUE;
}

void cCanvas::FlipX()
{
    tCanvasLine pOldLine = new sCanvasElement[m_size.cx];
    tCanvasLine pNewLine = new sCanvasElement[m_size.cx];
    LONG lRow;
    LONG lCol;

    for (lRow = 0; lRow < m_size.cy; lRow++)
    {       
        GetLine(CPoint(0,lRow), pOldLine);
        for (lCol = 0; lCol < m_size.cx; lCol++)
        {
            pNewLine[m_size.cx-lCol-1] = pOldLine[lCol];

        }
        SetLine(CPoint(0,lRow), pNewLine);
    }

    delete [] pOldLine;
    delete [] pNewLine;
}

void cCanvas::FlipY()
{
    tCanvasLine pOldLine = new sCanvasElement[m_size.cx];
    tCanvasLine pNewLine = new sCanvasElement[m_size.cx];
    LONG lRow;

    for (lRow = 0; lRow < m_size.cy/2; lRow++)
    {       
        GetLine(CPoint(0,lRow), pOldLine);
        GetLine(CPoint(0,m_size.cy-lRow-1), pNewLine);
        SetLine(CPoint(0,lRow), pNewLine);
        SetLine(CPoint(0,m_size.cy-lRow-1), pOldLine);
    }

    delete [] pOldLine;
    delete [] pNewLine;
}

void cCanvas::Rotate()
{
    cMemCanvas mc(CSize(m_size.cy,m_size.cx));

    CPoint pt;
	for (pt.y = 0; pt.y < m_size.cy; pt.y++)
    {       
	    for (pt.x = 0; pt.x < m_size.cx; pt.x++)
		{       
			mc.Set(CPoint(pt.y, pt.x), Get(pt));
		}
	}
	Resize(CSize(m_size.cy, m_size.cx));
	Set(CPoint(0,0), &mc, CPoint(0,0), m_size);
}

void cCanvas::FillBack(tElementPart p_back, CRect p_rectSelect)
{
	CRect rect = (p_rectSelect.IsRectNull()) ? CRect(CPoint(0,0), m_size) : p_rectSelect;
	CPoint pt(rect.TopLeft());

    for (pt.y = rect.top; pt.y < rect.bottom; pt.y++)
    {       
	    for (pt.x = rect.left; pt.x < rect.right; pt.x++)
		{       
			SetBack(pt, p_back);
		}
	}
}

void cCanvas::FillFore(tElementPart p_fore, CRect p_rectSelect)
{
	CRect rect = (p_rectSelect.IsRectNull()) ? CRect(CPoint(0,0), m_size) : p_rectSelect;
	CPoint pt(rect.TopLeft());

    for (pt.y = rect.top; pt.y < rect.bottom; pt.y++)
    {       
	    for (pt.x = rect.left; pt.x < rect.right; pt.x++)
		{       
			SetFore(pt, p_fore);
		}
	}
}

void cCanvas::FillAttr(tElementPart p_attribute, CRect p_rectSelect)
{
	CRect rect = (p_rectSelect.IsRectNull()) ? CRect(CPoint(0,0), m_size) : p_rectSelect;
	CPoint pt(rect.TopLeft());

    for (pt.y = rect.top; pt.y < rect.bottom; pt.y++)
    {       
		SetLine(pt, p_attribute, CANVAS_TRANSPARENT, rect.Width());
	}
}

void cCanvas::FillChar(tElementPart p_character, CRect p_rectSelect)
{
	CRect rect = (p_rectSelect.IsRectNull()) ? CRect(CPoint(0,0), m_size) : p_rectSelect;
	CPoint pt(rect.TopLeft());

    for (pt.y = rect.top; pt.y < rect.bottom; pt.y++)
    {       
		SetLine(pt, CANVAS_TRANSPARENT, p_character, rect.Width());
	}
}

void cCanvas::Fill(tElementPart p_attribute, tElementPart p_character, CRect p_rectSelect)
{
	CRect rect = (p_rectSelect.IsRectNull()) ? CRect(CPoint(0,0), m_size) : p_rectSelect;
	CPoint pt(rect.TopLeft());

    for (pt.y = rect.top; pt.y < rect.bottom; pt.y++)
    {       
        SetLine(pt, p_attribute, p_character, rect.Width());
/*
        for (pt.x = 0; pt.x < m_size.cx; pt.x++)
        {
            if (p_attribute != CANVAS_TRANSPARENT)
            {
                SetAttr(pt, p_attribute);
            }
            if (p_character != CANVAS_TRANSPARENT)
            {
                SetChar(pt, p_character);    
            }
        }
*/
    }
}

void cCanvas::Set(const cCanvas* p_pCanvas)
{
    Resize(p_pCanvas->m_size);
    LONG lRow;
    LONG lCol;

    for (lRow = 0; lRow < m_size.cy; lRow++)
    {       
        for (lCol = 0; lCol < m_size.cx; lCol++)
        {
            Set(CPoint(lCol,lRow),p_pCanvas->Get(CPoint(lCol, lRow)));
        }
    }
}

void cCanvas::Set(CPoint p_source, const cCanvas* p_pCanvas, CPoint p_dest, CSize p_size)
{
    tCanvasLine pLine = new sCanvasElement[p_size.cx];
    LONG lRow;
    if (p_dest.x + p_size.cx > GetSize().cx) p_size.cx = GetSize().cx - p_dest.x;
    if (p_dest.y + p_size.cy > GetSize().cy) p_size.cy = GetSize().cy - p_dest.y;

    if (p_source.x + p_size.cx > p_pCanvas->GetSize().cx) p_size.cx = p_pCanvas->GetSize().cx - p_source.x;
    if (p_source.y + p_size.cy > p_pCanvas->GetSize().cy) p_size.cy = p_pCanvas->GetSize().cy - p_source.y;
	for (lRow = 0; lRow < p_size.cy; lRow++)
	{       
		p_pCanvas->GetLine(CPoint(p_source.x,p_source.y+lRow), pLine, p_size.cx);
		SetLine(CPoint(p_dest.x,p_dest.y+lRow), pLine, p_size.cx);
	}

    delete [] pLine;
}

void cCanvas::Set(CPoint p_source, const cCanvas* p_pCanvas, CPoint p_dest, CSize p_size, cOptions* p_pOptions, BOOL p_bUnder, BOOL p_bTransparent)
{
    tCanvasLine pLine = new sCanvasElement[p_size.cx];
    LONG lRow;
	LONG lCol;
    if (p_dest.x + p_size.cx > GetSize().cx) p_size.cx = GetSize().cx - p_dest.x;
    if (p_dest.y + p_size.cy > GetSize().cy) p_size.cy = GetSize().cy - p_dest.y;

    if (p_source.x + p_size.cx > p_pCanvas->GetSize().cx) p_size.cx = p_pCanvas->GetSize().cx - p_source.x;
    if (p_source.y + p_size.cy > p_pCanvas->GetSize().cy) p_size.cy = p_pCanvas->GetSize().cy - p_source.y;
	if (p_bUnder || p_bTransparent)
	{
	    tCanvasLine pOriginalLine = new sCanvasElement[p_size.cx];
		for (lRow = 0; lRow < p_size.cy; lRow++)
		{       
			p_pCanvas->GetLine(CPoint(p_source.x,p_source.y+lRow), pLine, p_size.cx);
			GetLine(CPoint(p_dest.x,p_dest.y+lRow), pOriginalLine, p_size.cx);
			for (lCol = 0; lCol < p_size.cx; lCol++)
			{       
				sCanvasElement& ce = pOriginalLine[lCol];
				sCanvasElement& ceBuf = pLine[lCol];
				BOOL bPaste = TRUE;
				if (p_bUnder)
					bPaste &= (ce.character == 32 && (ce.GetBack() == 0 || (ce.GetBack() == 8 && p_pOptions->GetBackground8())));
				if (p_bTransparent)
					bPaste &= !(ceBuf.character == 32 && (ceBuf.GetBack() == 0 || (ceBuf.GetBack() == 8 && p_pOptions->GetBackground8())));

                if (bPaste) ce = ceBuf;
			}
			SetLine(CPoint(p_dest.x,p_dest.y+lRow), pOriginalLine, p_size.cx);
		}
		delete [] pOriginalLine;
	}
	else 
		for (lRow = 0; lRow < p_size.cy; lRow++)
		{       
			p_pCanvas->GetLine(CPoint(p_source.x,p_source.y+lRow), pLine, p_size.cx);
			SetLine(CPoint(p_dest.x,p_dest.y+lRow), pLine, p_size.cx);
		}

    delete [] pLine;
}

void cCanvas::Set(CPoint p_source, const cCanvas* p_pCanvas, CPoint p_dest, LONG p_lWidth)
{
    LONG lCol;
    for (lCol = 0; lCol < p_lWidth; lCol++)
    {
        Set(CPoint(p_dest.x+lCol,p_dest.y),p_pCanvas->Get(CPoint(p_source.x+lCol,p_source.y)));
    }
}

void cCanvas::Set(CPoint p_point, sCanvasElement p_element)
{
    SetAttr(p_point, p_element.attribute);
    SetChar(p_point, p_element.character);
}

void cCanvas::Set(CPoint p_point, tElementPart p_attribute, tElementPart p_character)
{
    SetAttr(p_point, p_attribute);
    SetChar(p_point, p_character);
}

void cCanvas::Set(CRect p_rect, sCanvasElement p_element)
{
    CPoint pt;
    for (pt.y=p_rect.top; pt.y<p_rect.bottom; pt.y++)
    {
        for (pt.x=p_rect.left; pt.x<p_rect.right; pt.x++)
        {
            Set(pt, p_element);
        }
    }
}

void cCanvas::Set(CRect p_rect, tElementPart p_attribute, tElementPart p_character)
{
    CPoint pt;
    for (pt.y=p_rect.top; pt.y<p_rect.bottom; pt.y++)
    {
        for (pt.x=p_rect.left; pt.x<p_rect.right; pt.x++)
        {
            Set(pt, p_attribute, p_character);
        }
    }
}


sCanvasElement cCanvas::Get(CPoint p_point) const
{
    sCanvasElement elem;
    elem.attribute = GetAttr(p_point);
    elem.character = GetChar(p_point);
    return elem;
}


void cCanvas::SetAttr(CPoint p_point, tElementPart p_attribute)
{
    if (p_attribute != CANVAS_TRANSPARENT)
    {
        SetAttr(p_point, (UCHAR)p_attribute);
    }
}

void cCanvas::SetChar(CPoint p_point, tElementPart p_character)
{
    if (p_character != CANVAS_TRANSPARENT)
    {
        SetChar(p_point, (UCHAR)p_character);
        if (p_character >= 256)
        {
            SetAttr(p_point,(UCHAR)(GetAttr(p_point) & 0x08));
        }
    }
}

void cCanvas::SetFore(CPoint p_point, tElementPart p_fore)
{
    if (p_fore != CANVAS_TRANSPARENT)
    {
        SetFore(p_point, (UCHAR)p_fore);
    }
}

void cCanvas::SetBack(CPoint p_point, tElementPart p_back)
{
    if (p_back != CANVAS_TRANSPARENT)
    {
        SetBack(p_point, (UCHAR)p_back);
    }
}

UCHAR cCanvas::GetFore(CPoint p_point) const
{
    return GetAttr(p_point) & 0xF;
}

UCHAR cCanvas::GetBack(CPoint p_point) const
{
    return GetAttr(p_point) >> 4;
}


void cCanvas::SetLine(CPoint p_point, const tCanvasLine p_line, LONG p_lWidth)
{
    LONG lCol;
    p_lWidth = (p_lWidth == 0) ? GetSize().cx : p_lWidth;
    for (lCol = 0; lCol < p_lWidth; lCol++)
    {
        Set(p_point+CSize(lCol,0),p_line[lCol].attribute,p_line[lCol].character);
    }
}

void cCanvas::GetLine(CPoint p_point, tCanvasLine p_line, LONG p_lWidth) const
{
    LONG lCol;
    p_lWidth = (p_lWidth == 0) ? GetSize().cx : p_lWidth;
    for (lCol = 0; lCol < p_lWidth; lCol++)
    {
        p_line[lCol] = Get(p_point+CSize(lCol,0));
    }
}

void cCanvas::SetLine(CPoint p_point, tElementPart p_attribute, tElementPart p_character, LONG p_lWidth)
{
    LONG lCol;
    p_lWidth = (p_lWidth == 0) ? GetSize().cx : p_lWidth;
    for (lCol = 0; lCol < p_lWidth; lCol++)
    {
        Set(p_point+CSize(lCol,0),p_attribute, p_character);
    }
}

void cCanvas::SetLine(CPoint p_point, sCanvasElement p_element, LONG p_lWidth)
{
	SetLine(p_point, p_element.attribute, p_element.character, p_lWidth);
}

void cCanvas::InsertLine(int p_iLine)
{
	LONG lRow;
	LONG lCol;
	sCanvasElement element;
	for (lRow = GetSize().cy-1; lRow > p_iLine; lRow--)
	{
		for (lCol = 0; lCol < GetSize().cx; lCol++)
		{
			element = Get(CPoint(lCol, lRow-1));
			Set(CPoint(lCol, lRow), element);
		}
	}
	SetLine(CPoint(0,p_iLine), 7, 32);
}
void cCanvas::DeleteLine(int p_iLine)
{
	LONG lRow;
	LONG lCol;
	sCanvasElement element;
	for (lRow = p_iLine; lRow < GetSize().cy-1; lRow++)
	{
		for (lCol = 0; lCol < GetSize().cx; lCol++)
		{
			element = Get(CPoint(lCol, lRow+1));
			Set(CPoint(lCol, lRow), element);
		}
	}
	SetLine(CPoint(0,GetSize().cy-1), 7, 32);
}

void cCanvas::Delete(CRect p_rectSelect)
{
	p_rectSelect.NormalizeRect();
	cMemCanvas mc(CSize(m_size.cx, p_rectSelect.Height()));
	mc.Set(CPoint(p_rectSelect.right, p_rectSelect.top), this, CPoint(0,0), CSize(m_size.cx-p_rectSelect.Width(), p_rectSelect.Height()));
	Set(CPoint(0,0), &mc, p_rectSelect.TopLeft(), CSize(m_size.cx, p_rectSelect.Height()));
}


void cCanvas::SetColumn(CPoint p_point, tElementPart p_attribute, tElementPart p_character, LONG p_lHeight)
{
    LONG lRow;
    p_lHeight = (p_lHeight == 0) ? GetSize().cy : p_lHeight;
    for (lRow = 0; lRow < p_lHeight; lRow++)
    {
        Set(p_point+CSize(0,lRow),p_attribute, p_character);
    }
}
void cCanvas::SetColumn(CPoint p_point, sCanvasElement p_element, LONG p_lHeight)
{
	SetColumn(p_point, p_element.attribute, p_element.character, p_lHeight);
}

void cCanvas::InsertColumn(int p_iColumn)
{
	LONG lRow;
	LONG lCol;
	sCanvasElement element;
	for (lCol = GetSize().cx-2; lCol >= p_iColumn; lCol--)
	{
		for (lRow = 0; lRow < GetSize().cy; lRow++)
		{
			element = Get(CPoint(lCol, lRow));
			Set(CPoint(lCol+1, lRow), element);
		}
	}
	SetColumn(CPoint(p_iColumn,0), 7, 32);
}

void cCanvas::DeleteColumn(int p_iColumn)
{
	LONG lRow;
	LONG lCol;
	sCanvasElement element;
	for (lCol = p_iColumn; lCol < GetSize().cx-1; lCol++)
	{
		for (lRow = 0; lRow < GetSize().cy; lRow++)
		{
			element = Get(CPoint(lCol+1, lRow));
			Set(CPoint(lCol, lRow), element);
		}
	}
	SetColumn(CPoint(GetSize().cx-1,0), 7, 32);
}



int cCanvas::FindEndX(int p_iLine, int p_iStart, int p_iEnd)
{
	sCanvasElement element;
	if (p_iEnd == 0) p_iEnd = GetSize().cx;
    for (int lCol = p_iEnd-1; lCol >= p_iStart; lCol--)
	{
		element = Get(CPoint(lCol, p_iLine));
        if ((element.character != ' ' && element.character != 0) || (element.attribute > 15))
        {
            return lCol;
        }
	}
    return -1;
}

int cCanvas::FindEndY()
{
    for (int lRow = GetSize().cy-1; lRow >= 0; lRow--)
    {
        if (FindEndX(lRow) != -1)
        {
            return lRow;
        }
    }
    return -1;
}


void cCanvas::DeleteChar(CPoint p_point, BOOL p_bInsert)
{
	if (p_bInsert)
	{
	    if (p_point.x < GetSize().cx-1)
		{
			CSize size(GetSize().cx-p_point.x-1,1);
			cMemCanvas mc(size);
			mc.Set(p_point+CSize(1,0),this,CPoint(0,0),size);
			Set(CPoint(0,0),&mc,p_point,size);
		}
		Set(CPoint(GetSize().cx-1,p_point.y), 7, 32);
	}
	else
	{
		Set(p_point, 7, 32);
	}
}

void cCanvas::InsertChar(CPoint p_point, sCanvasElement p_element, BOOL p_bInsert)
{
	if (p_bInsert && p_point.x < GetSize().cx-1)
	{
	    CSize size(GetSize().cx-p_point.x-1,1);
		cMemCanvas mc(size);
		mc.Set(p_point+CSize(1,0),this,CPoint(0,0),size);
		Set(p_point,this,p_point+CSize(1,0),size);
	}
	Set(p_point, p_element);
}


// cMemCanvas

cMemCanvas::cMemCanvas(CSize p_size) :
    cCanvas(p_size)
{
    m_pData = NULL;
    Resize(p_size);
}

cMemCanvas::~cMemCanvas()
{
    CSingleLock sl(&m_mutex, TRUE);
    if (sl.IsLocked())
    {
        int iCount;
        if (m_pData)
        {
            for (iCount=0; iCount<m_size.cy; iCount++)
            {
                if (m_pData[iCount]) delete [] m_pData[iCount];
            }
            delete [] m_pData;
            m_pData = NULL;
        }
    }
}

BOOL cMemCanvas::Resize(CSize p_newsize)
{
    int iCount;
    
    CSingleLock sl(&m_mutex, TRUE);
    if (sl.IsLocked())
    {
        if (m_pData)
        {
            for (iCount=0; iCount<m_size.cy; iCount++)
            {
                if (m_pData[iCount]) delete [] m_pData[iCount];
            }
            delete [] m_pData;
            m_pData = NULL;
        }
        m_pData = new tCanvasLine[p_newsize.cy];
        for (iCount=0; iCount<p_newsize.cy; iCount++)
        {
            m_pData[iCount] = NULL; //new sCanvasElement[p_newsize.cx];
        }
        cCanvas::Resize(p_newsize);
        return TRUE;
    }
    return FALSE;
}

void cMemCanvas::SetAttr(CPoint p_point, UCHAR p_ucAttribute)
{
    CSingleLock sl(&m_mutex, TRUE);
    if (sl.IsLocked())
    {
		if (p_point.y < m_size.cy && p_point.x < m_size.cx)
		{
			if (!m_pData[p_point.y] && p_ucAttribute != 7)
			{
				m_pData[p_point.y] = new sCanvasElement[GetSize().cx];
				wmemset((wchar_t*)m_pData[p_point.y], 0x2007, GetSize().cx);
			}
			if (m_pData[p_point.y]) m_pData[p_point.y][p_point.x].attribute = p_ucAttribute;
		}
    }
}

void cMemCanvas::SetChar(CPoint p_point, UCHAR p_ucCharacter)
{
    CSingleLock sl(&m_mutex, TRUE);
    if (sl.IsLocked())
    {
		if (p_point.y < m_size.cy && p_point.x < m_size.cx)
		{
			if (!m_pData[p_point.y] && p_ucCharacter != 32)
			{
				m_pData[p_point.y] = new sCanvasElement[GetSize().cx];
				wmemset((wchar_t*)m_pData[p_point.y], 0x2007, GetSize().cx);
			}
			if (m_pData[p_point.y]) m_pData[p_point.y][p_point.x].character = p_ucCharacter;
		}
    }
}

void cMemCanvas::SetFore(CPoint p_point, UCHAR p_ucFore)
{
    CSingleLock sl(&m_mutex, TRUE);
    if (sl.IsLocked())
    {
        if (!m_pData[p_point.y] && p_ucFore != 7)
        {
            m_pData[p_point.y] = new sCanvasElement[GetSize().cx];
            wmemset((wchar_t*)m_pData[p_point.y], 0x2007, GetSize().cx);
        
        }
        if (m_pData[p_point.y]) 
        {
            sCanvasElement* pElem = &m_pData[p_point.y][p_point.x];
            pElem->attribute = (pElem->attribute & 0xF0) | (p_ucFore & 0x0F);
        }
    }
}

void cMemCanvas::SetBack(CPoint p_point, UCHAR p_ucBack)
{
    CSingleLock sl(&m_mutex, TRUE);
    if (sl.IsLocked())
    {
        if (!m_pData[p_point.y] && p_ucBack != 0)
        {
            m_pData[p_point.y] = new sCanvasElement[GetSize().cx];
            wmemset((wchar_t*)m_pData[p_point.y], 0x2007, GetSize().cx);
        }
        if (m_pData[p_point.y]) 
        {
            sCanvasElement* pElem = &m_pData[p_point.y][p_point.x];
            pElem->attribute = (pElem->attribute & 0x0F) | (p_ucBack << 4);
        }
    }
}

UCHAR cMemCanvas::GetAttr(CPoint p_point) const
{
    if (!m_pData[p_point.y])
    {
        return 7;
    }
    return m_pData[p_point.y][p_point.x].attribute;
}

UCHAR cMemCanvas::GetChar(CPoint p_point) const
{
    if (!m_pData[p_point.y])
    {
        return 32;
    }
    return m_pData[p_point.y][p_point.x].character;
}

cCanvas* cMemCanvas::Create(CSize p_size)
{
    return new cMemCanvas(p_size);
}

void cMemCanvas::SetLine(CPoint p_point, tElementPart p_attribute, tElementPart p_character, LONG p_lWidth)
{
    if (p_attribute == 7 && p_character == 32 && ((p_lWidth == 0) || (p_lWidth == m_size.cx && p_point.x == 0)))
    {
        CSingleLock sl(&m_mutex, TRUE);
        if (sl.IsLocked())
        {
            if (m_pData[p_point.y])
            {
                delete [] m_pData[p_point.y];
                m_pData[p_point.y] = NULL;
            }
        }
    }
    else cCanvas::SetLine(p_point, p_attribute, p_character, p_lWidth);
}

void cMemCanvas::InsertLine(int p_iLine)
{
    CSingleLock sl(&m_mutex, TRUE);
    if (sl.IsLocked())
    {
	    LONG lRow;
	    if (m_pData[m_size.cy-1]) delete [] m_pData[m_size.cy-1];
	    for (lRow = m_size.cy-1; lRow > p_iLine; lRow--)
	    {
		    m_pData[lRow] = m_pData[lRow-1];
	    }
	    m_pData[p_iLine] = NULL; 
    }
	//SetLine(CPoint(0,p_iLine), 7, 32);
}

void cMemCanvas::DeleteLine(int p_iLine)
{
	if (p_iLine < 0 || p_iLine >= m_size.cy) return;
    CSingleLock sl(&m_mutex, TRUE);
	if (sl.IsLocked())
	{
	    LONG lRow;
	    if (m_pData[p_iLine]) delete [] m_pData[p_iLine];
	    for (lRow = p_iLine; lRow < m_size.cy-1; lRow++)
	    {
		    m_pData[lRow] = m_pData[lRow+1];
	    }
        m_pData[m_size.cy-1] = NULL;
	}
}

void cMemCanvas::InsertColumn(int p_iColumn)
{
    CSingleLock sl(&m_mutex, TRUE);
	if (sl.IsLocked())
	{
	    LONG lRow;
		LONG lLastY = FindEndY();
	    for (lRow = 0; lRow < lLastY+1; lRow++)
	    {
		    if (m_pData[lRow])
			{
				sCanvasElement* pCanvasLine = new sCanvasElement[GetSize().cx];
				if (p_iColumn > 0) wmemcpy((wchar_t*)&pCanvasLine[0], (wchar_t*)&m_pData[lRow][0], p_iColumn);
				if (p_iColumn < m_size.cy-1) wmemcpy((wchar_t*)&pCanvasLine[p_iColumn+1], (wchar_t*)&m_pData[lRow][p_iColumn], m_size.cx-p_iColumn-1);
				pCanvasLine[p_iColumn].value = 0x2007;
				delete [] m_pData[lRow];
				m_pData[lRow] = pCanvasLine;
			}
	    }
	}
}

void cMemCanvas::DeleteColumn(int p_iColumn)
{
    CSingleLock sl(&m_mutex, TRUE);
	if (sl.IsLocked())
	{
	    LONG lRow;
		LONG lLastY = FindEndY();
	    for (lRow = 0; lRow < lLastY+1; lRow++)
	    {
		    if (m_pData[lRow])
			{
				sCanvasElement* pCanvasLine = new sCanvasElement[GetSize().cx];
				if (p_iColumn > 0) wmemcpy((wchar_t*)&pCanvasLine[0], (wchar_t*)&m_pData[lRow][0], p_iColumn);
				if (p_iColumn < m_size.cy-1) wmemcpy((wchar_t*)&pCanvasLine[p_iColumn], (wchar_t*)&m_pData[lRow][p_iColumn+1], m_size.cx-p_iColumn-1);
				pCanvasLine[m_size.cx-1].value = 0x2007;
				delete [] m_pData[lRow];
				m_pData[lRow] = pCanvasLine;
			}
	    }
	}
}

int cMemCanvas::FindEndY()
{
    CSingleLock sl(&m_mutex, TRUE);
	if (sl.IsLocked())
	{
	    LONG lRow;
		for (lRow = m_size.cy-1; lRow >= 0; lRow--)
		{
			if (m_pData[lRow] && FindEndX(lRow) != -1) return lRow;
		}
	}
	return -1;
}


/*

cRemoteCanvas::cRemoteCanvas(CSize p_size) :
    cCanvas(p_size)
{
    HRESULT hr;
    hr = CoInitializeEx(NULL, COINIT_MULTITHREADED);
    ASSERT(SUCCEEDED(hr));
    hr = CoCreateInstance(CLSID_RemoteCanvasComponent, NULL, CLSCTX_ALL, IID_IRemoteCanvas, (void**)&m_pIRemoteCanvas);
    ASSERT(SUCCEEDED(hr));
    Resize(p_size);
}

cRemoteCanvas::~cRemoteCanvas()
{
    m_pIRemoteCanvas->Release();
    CoUninitialize();
}

BOOL cRemoteCanvas::Resize(CSize p_newsize)
{
    HRESULT hr;
    hr = m_pIRemoteCanvas->Resize(p_newsize.cx,p_newsize.cy);
    cCanvas::Resize(p_newsize);
    return SUCCEEDED(hr);
}

void cRemoteCanvas::SetAttr(CPoint p_point, UCHAR p_ucAttribute)
{
    m_pIRemoteCanvas->SetAttr(p_point.x, p_point.y, p_ucAttribute);
}

void cRemoteCanvas::SetChar(CPoint p_point, UCHAR p_ucCharacter)
{
    m_pIRemoteCanvas->SetChar(p_point.x, p_point.y, p_ucCharacter);
}

void cRemoteCanvas::SetFore(CPoint p_point, UCHAR p_ucFore)
{
    m_pIRemoteCanvas->SetFore(p_point.x, p_point.y, p_ucFore);
}

void cRemoteCanvas::SetBack(CPoint p_point, UCHAR p_ucBack)
{
    m_pIRemoteCanvas->SetBack(p_point.x, p_point.y, p_ucBack);
}

UCHAR cRemoteCanvas::GetAttr(CPoint p_point) const
{
    UCHAR ucAttr;
    m_pIRemoteCanvas->GetAttr(p_point.x, p_point.y, &ucAttr);
    return ucAttr;
}

UCHAR cRemoteCanvas::GetChar(CPoint p_point) const
{
    UCHAR ucChar;
    m_pIRemoteCanvas->GetChar(p_point.x, p_point.y, &ucChar);
    return ucChar;
}

void cRemoteCanvas::Fill(tElementPart p_attribute, tElementPart p_character)
{
    m_pIRemoteCanvas->Fill(p_attribute,p_character);
}

void cRemoteCanvas::SetLine(CPoint p_point, const tCanvasLine p_line, LONG p_lWidth)
{
    p_lWidth = (p_lWidth == 0) ? GetSize().cx : p_lWidth;
    m_pIRemoteCanvas->SetLine(p_point.x,p_point.y, (SHORT*)p_line, p_lWidth);
}

void cRemoteCanvas::GetLine(CPoint p_point, tCanvasLine p_line, LONG p_lWidth) const
{
    p_lWidth = (p_lWidth == 0) ? GetSize().cx : p_lWidth;
    m_pIRemoteCanvas->GetLine(p_point.x,p_point.y, (SHORT*)p_line, p_lWidth);
}



cCanvas* cRemoteCanvas::Create(CSize p_size)
{
    return new cRemoteCanvas(p_size);
}

*/