<?php

define (WRAP, 80);
define (TABSTOP, 8);

/**
* class ansiparse
*
* parse an ansi
*
* @author Stefan Kesseler <poti@kaoz.org>
* @author Bob Copeland <email@bobcopeland.com>
* @package blockshow
*
*/
class ansiparse {

    /**
    * ansiparse::$chars
    *
    * reference to ansiconv::$chars
    *
    */
    var $chars;
    
    /**
    * ansiparse::$attribs
    *
    * reference to ansiconv::$attribs
    *
    */
    var $attribs;

    /**
    * ansiparse::$curr_x
    *
    * current x position
    *
    */
    var $curr_x;
    
    /**
    * ansiparse::$curr_y
    *
    * current y position
    *
    */
    var $curr_y;
    
    /**
    * ansiparse::$curr_atrribs
    *
    * current attributes (foreground & background colors)
    *
    */
    var $curr_attribs;

    /**
    * ansiparse::$savepos_x
    *
    * saved x position
    *
    */
    var $savepos_x;
    
    /**
    * ansiparse::$savepos_y
    *
    * saved y position
    *
    */
    var $savepos_y;

    /**
    * ansiparse::ansiparse()
    *
    * parse an ansifile
    *
    * @param string $ansifile filename of the ansifile to parse
    * @param array $chars reference on char-array (2dim)
    * @param array $attribs reference on attrib-array (2dim)
    * @return bool true for success, false for failure
    *
    */
    function ansiparse($ansifile,&$chars,&$attribs) {

        $this->chars = &$chars;
        $this->attribs = &$attribs;

        if (!$fp = @fopen($ansifile,"r") ) {
            trigger_error("file open error",E_USER_WARNING);
            $this = false;
            return false;
        }
        else
        {

            $this->parse($fp);
            return true;
        }
    }

    /**
    * ansiparse::getMaxy()
    *
    * get number of lines the read in ansi has
    *
    * @return int $number of lines
    *
    */
    function getMaxy() {
        $max = 0;
        $keys = array_keys($this->chars);
        for ($i=0;$i<sizeof($keys);$i++) {
            if ($keys[$i]>$max) $max = $keys[$i];
        }
        return $max+1;
    }

    /**
    * ansiparse::store()
    *
    * Stores a byte in the current position, wrapping if
    * necessary, and increments the position.
    *
    * @param char $char element to store
    *
    */
    function store($char) {
        $this->chars[$this->curr_y][$this->curr_x]=ord($char);
        $this->attribs[$this->curr_y][$this->curr_x]=$this->curr_attribs;
        $this->curr_x++;
        if ( $this->curr_x == WRAP ) {
            $this->curr_x = 0;
            $this->curr_y++;
        }
    }

    /**
    * ansiparse::moveUp()
    *
    * move current position up by some lines
    *
    * @param integer $arg number of lines
    * @see ansiparse::$curr_y
    *
    */
    function moveUp( $arg=1 ) {
        $this->curr_y -= $arg;
    }

    /**
    * ansiparse::moveDown()
    *
    * move current position down by some lines
    *
    * @param integer $arg number of lines
    * @see ansiparse::$curr_y
    *
    */
    function moveDown( $arg=1 ) {
        $this->curr_y += $arg;
    }

    /**
    * ansiparse::clearScreen()
    *
    * clear the whole screen
    *
    */
    function clearScreen() {
      $this->chars = array();
      $this->attribs = array();
    }

    /**
    * ansiparse::clearLine()
    *
    * clear one line
    *
    */
    function clearLine() {
        for ($i=0;$i<WRAP;$i++) {
            $this->chars[$this->curr_y][$i] = "";
            $this->attribs[$this->curr_y][$i] = $this->curr_atrribs;
        }
    }

    /**
    * ansiparse::moveForward()
    *
    * move current position to the right
    *
    * @param integer $arg number of positions
    * @see ansiparse::$curr_x
    *
    */
    function moveForward( $arg=1 ) {
        $this->curr_x += $arg;
        if ( $this->curr_x > WRAP-1 ) $this->curr_x = WRAP-1;
    }

    /**
    * ansiparse::moveBackward()
    *
    * move current position to the left
    *
    * @param integer $arg number of positions
    * @see ansiparse::$curr_x
    *
    */
    function moveBackward( $arg=1 ) {
        $this->curr_x -= $arg;
        if ( $this->curr_x < 0 ) $this->curr_x = 0;
    }

    /**
    * ansiparse::setAttributes()
    *
    * change the attributes
    *
    * @param array $arg attribute change codes
    * @see ansiparse::$curr_attribs
    *
    */
    function setAttributes($arg=array()) {
        for ($i=0;$i<sizeof($arg);$i++) {
            if ( $arg[$i] == 0 ) {
                // reset attribute
                $this->curr_attribs = 0x07;
            } else if ( $arg[$i] == 1 ) {
                // set bold on
                $this->curr_attribs |= 0x08;
            } else if ( $arg[$i] == 5 ) {
                // set blink on
                $this->curr_attribs |= 0x80;
            } else if ( $arg[$i] >= 30 & $arg[$i] <= 37 ) {
                // foreground
                $this->curr_attribs &= 0xf8;
                $this->curr_attribs |= ($arg[$i] - 30);
            } else if ( $arg[$i] >= 40 && $arg[$i] <= 47 ) {
                // background
                $this->curr_attribs &= 0x8f;
                $this->curr_attribs |= (($arg[$i] - 40)<<4);
            }
        }
    }

    /**
    * ansiparse::parse()
    *
    * parse and read an ansifile
    *
    * @param int $fp file pointer
    *
    */
    function parse($fp) {

        $this->curr_x = $this->saved_x = 0;
        $this->curr_y = $this->saved_y = 0;

        $ch = "";
        $state = "S_TXT"; // php does not have enum
        $argbuf = "";
        $count=0;

        while (!feof( $fp )) {
            $ch = fgetc( $fp );
            if($state == "S_TXT") {
                switch( $ch ) {
                  case "\x1a":
                    $state = "S_END";
                    break;
                  case "\x1b":
                    $state = "S_CHK_B";
                    break;
                  case "\n":
                    $this->curr_y++;
                    $this->curr_x = 0;
                    break;
                  case "\r":
                    break;
                  case "\t":
                    $count = $this->curr_x % TABSTOP;
                    if ($count) {
                      $count = TABSTOP - $count;
                    }
                    for ($i=0; $i<$count; $i++) {
                      $this->store(' ');
                    }
                    break;
                  default:
                    $this->store( $ch );
                    break;
                }
                //break;

            } elseif ($state == "S_CHK_B") {
                if ( $ch != '[' ) {
                  $this->store( ESC);
                  $this->store( $ch );
                  $state = "S_TXT";
                } else  {
                  $state = "S_WAIT_LTR";
                }
                //break;

            } elseif ($state == "S_WAIT_LTR") {

                //if ( isalpha( ch ) ) {
                if (($ch>='a' && $ch<='z') || ($ch>='A' && $ch<='Z')) {
               // if (ereg_replace("[^[:alpha:]]","",$ch)==$ch) {
                  // handle the ansi code
                  $args = explode(";",$argbuf);
                  switch( $ch ) {
                    case "m":                   // set attrs
                      $this->setAttributes ($args);
                      break;
                    case "H":                   // set position
                    case "f":
                      $this->setPosition ($args[0],$args[1]);
                      break;
                    case "A":                   // move up
                        $this->moveUp (settype($args[0],"integer"));
                      break;
                    case "B":                   // down
                      $this->moveDown ($args[0]);
                      break;
                    case "C":                   // right
                      $this->moveForward ($args[0]);
                      break;
                    case "D":                   // left
                      $this->moveBackward ($args[0]);
                      break;
                    case "s":                   // save
                      $this->savepos_x = $this->curr_x;
                      $this->savepos_y = $this->curr_y;
                      break;
                    case "u":                   // restore
                      $this->curr_x = $this->save_x;
                      $this->curr_y = $this->save_y;
                      break;
                    case "J":                   // clear page
                      $this->clearScreen ();
                      break;
                    case "K":                   // clear line
                      $this->clearLine ();
                      break;
                    default:
                      break;
                  }
                  unset($args);
                  $argbuf = "";
                  $state = "S_TXT";
                } else {
                  $argbuf .= $ch;
                }
              
            } elseif ($state == "S_END") {
                return 1;
              
            } else {
                $state = "S_TXT";
            }
        }
        fclose($fp);
    }
}

?>
