<?php

require_once("class.ansiparse.php");

/**
* class ansiconv
*
* convert a given ansi into a png file
*
* @author Stefan Kesseler <poti@kaoz.org>
* @package blockshow
*
*/
class ansiconv {

    /**
    * ansiconv::$im
    *
    * image resource
    *
    */
    var $im;

    /**
    * ansiconv::$chars
    *
    * stores single characters of read in ansi at a certain position
    * (2 dimensional [y][x] )
    *
    */
    var $chars;
    
    /**
    * ansiconv::$attribs
    *
    * stores attributes of read in ansi at a certain position
    * (2 dimensional [y][x] )
    *
    */
    var $attribs;

    /**
    * ansiconv::$outfile
    *
    * stores the given (via the constructor) output file
    *
    */
    var $outfile;

    /**
    * ansiconv::ansiconv()
    *
    * class to convert an ansi file to the png format
    *
    * @param string ansifile filename for the ansi file to be converted
    * @param bool $size640 true for resulting png in 640 width, false for 80
    * @param string $outfile  filename to store the png to
    * @return bool true for success, false for failure
    */
    function ansiconv($ansifile,$size640=true,$outfile="") {

        $this->outfile = $outfile;

        $this->chars = array();                // initialize
        $this->atrribs = array();

        // parse ans. read data into this->chars and this->attribs
        if (!$ansi = new ansiparse($ansifile,$this->chars,$this->attribs)) {
            trigger_error("Could not parse input file",E_USER_ERROR);
        }

        $width=640;
        $height = $ansi->getMaxy()*16;
        if ($size640==false) {
            $width = 80;
            $height = $ansi->getMaxy()*2;
        }
        
        if (function_exists('imagecreatetruecolor')){
            if (!$this->im = imagecreatetruecolor($width,$height)) {
                trigger_error("Cannot Initialize new GD image stream",E_USER_WARNING);
                return false;
            }
        } else {
            if (!$this->im = imagecreate($width,$height)) {
                trigger_error("Cannot Initialize new GD image stream",E_USER_WARNING);
                return false
                ;
            }
        }
        
        if ($size640==true) {
           $this->buildPng640($ansi->getMaxy());
        }
        else {
           $this->buildPng80($ansi->getMaxy());
        }
        
        return true;
    }
        

    /**
    * ansiconv::buildPng640()
    *
    * writes to png image (640*x size) 
    *
    */
    function buildPng640($height) {

        $color_table = $this->getBasicColors();

        $color = array();
        for ($i=0;$i<16;$i++) {
           $color[$i] = imagecolorallocate($this->im,$color_table[$i][0],$color_table[$i][1],$color_table[$i][2]);
        }
        
        if (!$dosfont = &imageloadfont(dirname(__FILE__)."/fonts/msdos.fnt")) {
            trigger_error("Cannot Load Fontfile",E_USER_ERROR);
            return false;
        }

        // write everything (chars and attribs) to the png
        for ($y=0;$y<$height;$y++) {
            for ($x=0;$x<80;$x++) {
                if (($this->attribs[$y][$x] >> 4)!=0) imagefilledrectangle($this->im,$x*8,$y*16,($x+1)*8,($y+1)*16-1,$color[$this->attribs[$y][$x] >> 4]);
                imagechar($this->im,$dosfont,$x*8,$y*16,chr($this->chars[$y][$x]),$color[$this->attribs[$y][$x] & 0x0f]);
            }
        }
    }
    
    /**
    * ansiconv::buildPng80()
    *
    * writes to png image (80*x size) 
    *
    */
    function buildPng80($height) {

        $color_table = $this->getBasicColors();
        
        $font = $this->getBasicFont();
        
        for ($x=0;$x<8;$x++) {
            for ($y=0;$y<16;$y++) {
                for ($z=0;$z<=8;$z++) {
                    $color[($y*8)+$x][8-$z] = imagecolorallocate($this->im,
                           (($z/8) * $color_table[$x][0]) + (((8-$z)/8) * $color_table[$y][0]),
                           (($z/8) * $color_table[$x][1]) + (((8-$z)/8) * $color_table[$y][1]),
                           (($z/8) * $color_table[$x][2]) + (((8-$z)/8) * $color_table[$y][2])
                    );
                }
            }
        }

        for ($y=0;$y<($height);$y++) {
            for ($x=0;$x<80;$x++) {
                $offset = (($this->attribs[$y][$x] & 0x0f)*8)+($this->attribs[$y][$x] >> 4);
                imagesetpixel($this->im,$x,$y*2, $color[$offset][$font[$this->chars[$y][$x]] >> 4]);
                imagesetpixel($this->im,$x,$y*2+1, $color[$offset][$font[$this->chars[$y][$x]] & 0x00f]);
            }
        }

    }

    /**
    * ansiconv::output()
    *
    * output the png, either to stdout or to the given outputfile
    *
    */
    function output() {
        if (empty($this->outfile)) {
            header("Content-type: image/png");
            imagepng($this->im);
        }
        else
            imagepng($this->im,$outfile);
    }

    /**
    * ansiconv::getBasicColors()
    *
    * returns a table of the basic colors used
    *
    */
    function getBasicColors() {
        return array (
            array(0,0,0),                               // black
            array(0xaa,0,0),                            // red
            array(0,0xaa,0),                            // green
            array(0xaa,0x55,0),                         // yellow
            array(0,0,0xaa),                            // blue
            array(0xaa,0,0xaa),                         // magenta
            array(0,0xaa,0xaa),                         // cyan
            array(0xaa,0xaa,0xaa),                      // white
            array(0x55,0x55,0x55),                      // black     // bright
            array(254,0x55,0x55),                       // red
            array(0x55,0xff,0x55),                      // green
            array(254,254,0x55),                        // yellow
            array(0x55,0x55,0xff),                      // blue
            array(254,0x55,254),                        // magenta
            array(0x55,0xff,0xff),                      // cyan
            array(254,254,254)                          // white
        );
    }
    
    /**
    * ansiconv::getBasicFonts()
    *
    * returns an array with 8step-information on lower and upper part of each
    * letter. so lower 4bits are for lower part of the letter (meaning: x/8
    * opacity of the foreground color), upper 4bits are the same for the upper
    * part of the letter.
    *
    */
    function getBasicFont() {

        return array (0=>0, 1=>50, 2=>83, 3=>49, 4=>16, 5=>33, 6=>32, 7=>0, 8=>136, 9=>0, 10=>119, 11=>18, 12=>32, 13=>18, 14=>35, 15=>33, 16=>99,
        17=>16, 18=>16, 19=>33, 20=>48, 21=>66, 22=>4, 23=>17, 24=>16, 25=>0, 26=>16, 27=>16, 28=>33, 29=>16, 30=>18, 31=>48, 32=>0,
        33=>16, 34=>16, 35=>34, 36=>50, 37=>17, 38=>34, 39=>0, 40=>0, 41=>0, 42=>16, 43=>0, 44=>0, 45=>16, 46=>0, 47=>1, 48=>67,
        49=>16, 50=>19, 51=>17, 52=>32, 53=>65, 54=>66, 55=>32, 56=>66, 57=>48, 58=>0, 59=>0, 60=>0, 61=>17, 62=>0, 63=>32, 64=>51,
        65=>51, 66=>50, 67=>50, 68=>50, 69=>50, 70=>50, 71=>50, 72=>83, 73=>0, 74=>2, 75=>50, 76=>34, 77=>83, 78=>83, 79=>50, 80=>50,
        81=>67, 82=>50, 83=>34, 84=>16, 85=>66, 86=>65, 87=>66, 88=>34, 89=>32, 90=>35, 91=>16, 92=>32, 93=>0, 94=>16, 95=>1, 96=>0,
        97=>18, 98=>50, 99=>34, 100=>18, 101=>34, 102=>34, 103=>36, 104=>50, 105=>0, 106=>1, 107=>50, 108=>0, 109=>35, 110=>17, 111=>34, 112=>19,
        113=>35, 114=>18, 115=>17, 116=>16, 117=>34, 118=>17, 119=>35, 120=>17, 121=>36, 122=>18, 123=>0, 124=>0, 125=>0, 126=>16, 127=>18, 128=>50,
        129=>66, 130=>34, 131=>18, 132=>34, 133=>18, 134=>18, 135=>16, 136=>50, 137=>50, 138=>50, 139=>16, 140=>16, 141=>16, 142=>51, 143=>51, 144=>50,
        145=>18, 146=>67, 147=>34, 148=>50, 149=>34, 150=>66, 151=>50, 152=>67, 153=>50, 154=>66, 155=>32, 156=>50, 157=>16, 158=>99, 159=>17, 160=>18,
        161=>0, 162=>34, 163=>50, 164=>49, 165=>99, 166=>16, 167=>16, 168=>18, 169=>18, 170=>16, 171=>66, 172=>66, 173=>0, 174=>16, 175=>17, 176=>17,
        177=>68, 178=>85, 179=>0, 180=>16, 181=>32, 182=>33, 183=>17, 184=>32, 185=>49, 186=>17, 187=>33, 188=>48, 189=>32, 190=>32, 191=>16, 192=>0,
        193=>16, 194=>16, 195=>0, 196=>16, 197=>16, 198=>0, 199=>17, 200=>16, 201=>1, 202=>48, 203=>33, 204=>17, 205=>32, 206=>49, 207=>32, 208=>32,
        209=>32, 210=>17, 211=>16, 212=>0, 213=>0, 214=>1, 215=>33, 216=>32, 217=>16, 218=>0, 219=>136, 220=>24, 221=>119, 222=>0, 223=>112, 224=>35,
        225=>37, 226=>83, 227=>33, 228=>34, 229=>35, 230=>18, 231=>16, 232=>17, 233=>50, 234=>50, 235=>17, 236=>33, 237=>34, 238=>17, 239=>51, 240=>33,
        241=>1, 242=>0, 243=>0, 244=>0, 245=>3, 246=>0, 247=>17, 248=>16, 249=>0, 250=>0, 251=>17, 252=>48, 253=>48, 254=>33);

    }
}

?>
