/*
 * Socket Demon.
 *   - A little program that waits on a internet port and
 *     handles interactive commands.
 *     Most of the routines are taken out of "Unix Network Programming"
 *     by W. Richard Stevens, Prentice Hall, 1990.
 *
 * History:
 *  1994:
 *   Jan. 14 - Initial program completed.
 *   Jan. 17 - Added in nil command (if you just press enter)
 *           - Added in "?" to be the same as help
 *           - Added in the "bye", "ls", "pwd", and "cd" commands.
 *           - Added in lotsa comments
 *           - Added the 'ver' command.
 *           - Added in support for the -port, -log and -command
 *             command line options.
 *           - Added in the 'die' command.
 *           - Added logging of connects, disconnects, die's, and
 *             bad passwords.
 *   Apr. 07 - Changed the password to be encrypted so it cannot be
 *             discovered with a strings(1) command.
 *           - Added the 'id' command.
 *           - Call setuid if uid != euid (so that if program is setuid root
 *           - we can take advantage of the priviledge).
 *           - Took out the annoying uid listing at login
 *   Apr. 11 - Added a debug mode, and moved most of the verbose FTP style
 *             comments to that mode, leaving normal mode more unix-ish.
 *           - Added in a uname command
 *           - Changed the output format of several commands to look more 
 *             unix-ish
 *   Apr. 13 - Split the program up in to a bunch of different files and
 *             made a make file for it. - Speeds up development and
 *             makes if easier to change..
 *   Apr. 14 - Added new commands, cp, mv, rm
 *           - Redirected stdin, stdout and stderr to the client socket.
 *           - Added a shell!
 *   Apr. 15 - Added new commands, who, w, ps, chmod, chgrp, chown, cat,
 *             create
*/

#include "socdemon.h"
#include "socketio.h"
#include "soclog.h"

/*
 * print_usage()
 *   - prints the program name, version, and date, and also
 *     the program parameters. See socdemon.h for all their
 *     definitions.
*/
void print_usage() {
 int i;
   fprintf(stdout,"%s ver %s, %s.\n",SOCDEMON_NAME,SOCDEMON_VER,SOCDEMON_DATE);
   fprintf(stdout,"usage: %s [option] [option]...\n",program_name);
   fprintf(stdout,"where [option] is one of:\n");
   for (i=0;i<parm_nil;i+=2) {
      fprintf(stdout,"     %8.8s - %s\n",parm_table[i],parm_table[i+1]);
   }
}

/*
 * socket_demon_shutdown()
 *   - shuts down the internet socket (the one created by 
 *     the original parent, not the descriptors created by the
 *     accept() call). This is called before the program terminates,
 *     (under an error condition, or when a SIGTERM signal is caught).
*/
void socket_demon_shutdown() {
   soc_fprintf(stdout,"%s: Shutting down internet socket...",SOCDEMON_NAME);

   if (shutdown(socket_descriptor, 2) != 0) {
      soc_fprintf(stdout,"\n");
      soc_fprintf(stderr,"ERROR(%d): Unable to shutdown socket [%d].\n",
                  CANT_SHUTDOWN_SOCKET, errno);
      /* dont exit, try and close it */
   }

   soc_fprintf(stdout,"done.\n");
   soc_fprintf(stdout,"%s: Closing internet socket...",SOCDEMON_NAME);

   if (close(socket_descriptor) != 0) {
      soc_fprintf(stdout,"\n");
      soc_fprintf(stderr,"ERROR(%d): Unable to close socket [%d].\n",
                  CANT_CLOSE_SOCKET, errno);
      exit(CANT_CLOSE_SOCKET);
   }
   soc_fprintf(stdout,"done.\n");
}

/*
 * socket_demon_death_action()
 *   - this is the signal handler for the signals SIGTERM and SIGKILL,
 *     however, I don't think it catches the SIGKILL's, so always kill
 *     with a plain kill, and never a -9 parm.
*/
void socket_demon_death_action() {
   soc_fprintf(stdout,"%s: Received shutdown notice.\n",SOCDEMON_NAME);

   socket_demon_shutdown();
   soc_fprintf(stdout,"%s: Exiting..\n",SOCDEMON_NAME);
   exit(0);
}

/*
 * soc_demon_main_proc()
 *   - This is sorta the main process that the forked process 
 *     handling the incoming user sits in.  It waits for input
 *     and then calls the handler proc for that action.
 *   - The daemon forks, the parent executing this routine, and
 *     the child returning to wait on the port.  The benefits of
 *     doing it this way, instead of having the parent go back
 *     to wait on the port is that 
 *       i) You don't get defunct processes in you process tables
 *      ii) The process id of the main daemon changes after every
 *          use.
*/
void soc_demon_main_proc() 
{
 char in_msg[MAX_INPUT_LEN];
 char in_parm[MAX_INPUT_LEN];
 char *cmd_ptr;
 char out_msg[MAX_OUTPUT_LEN];
 int  msg_len = 0;
 int  i;

   for (;;) {
      tell_user(client_descriptor, SOCDEMON_PROMPT);
      msg_len = soc_read(client_descriptor, in_msg, MAX_INPUT_LEN);
      if (msg_len < 0) {
         soc_fprintf(stderr,"ERROR(%d): Error reading socket input [%d:%d].\n",
                     SOCKET_INPUT_ERROR, msg_len, errno);
         return;
      }

      if (in_msg[strlen(in_msg)-1] == '\n') {
         in_msg[strlen(in_msg)-2] = 0;
         strcpy(in_parm,in_msg);
         cmd_ptr = (char *) strtok(in_msg," ");
         i = strlen(in_msg);
         cmd_ptr = &in_parm[i];
         strcpy(in_parm,cmd_ptr);
         if (in_parm[0] != 0) {
            cmd_ptr = &in_parm[1];
            strcpy(in_parm,cmd_ptr);
         }
         for (i=0;i<strlen(in_msg);i++) {
            in_msg[i] = toupper(in_msg[i]);
         }
#ifdef LOG_TO_DEATH
         log_command(in_msg);
#endif

         if ((!strcmp(in_msg,command_table[cmd_help]))
            || (!strcmp(in_msg,command_table[cmd_qstn]))) {
            tell_user(client_descriptor, "%3.3d - %s", valid_commands,
                      out_table[valid_commands]);
            for (i=0;i<=cmd_nil;i+=2) {
               tell_user(client_descriptor, "  %-6.6s - %s\n",
                         command_table[i],command_table[i+1]);
            }
         } else if ((!strcmp(in_msg,command_table[cmd_quit])) 
                  || (!strcmp(in_msg,command_table[cmd_bye]))) {
            tell_user(client_descriptor, "%3.3d - %s", goodbye_str,
                      out_table[goodbye_str]);
            break;
         } else if (!strcmp(in_msg,command_table[cmd_ver])) {
            report_version();
         } else if (!strcmp(in_msg,command_table[cmd_pwd])) {
            get_pwd();
         } else if (!strcmp(in_msg,command_table[cmd_whoami])) {
            who_am_i();
         } else if (!strcmp(in_msg,command_table[cmd_id])) {
            get_id();
         } else if (!strcmp(in_msg,command_table[cmd_cp])) {
            do_cp(in_parm);
         } else if (!strcmp(in_msg,command_table[cmd_rm])) {
            do_rm(in_parm);
         } else if (!strcmp(in_msg,command_table[cmd_ps])) {
            do_ps(in_parm);
         } else if (!strcmp(in_msg,command_table[cmd_w])) {
            do_unix_command("w");
         } else if (!strcmp(in_msg,command_table[cmd_who])) {
            do_unix_command("who");
         } else if (!strcmp(in_msg,command_table[cmd_create])) {
            do_create(in_parm);
         } else if (!strcmp(in_msg,command_table[cmd_chmod])) {
            do_chmod(in_parm);
         } else if (!strcmp(in_msg,command_table[cmd_chown])) {
            do_chown(in_parm);
         } else if (!strcmp(in_msg,command_table[cmd_chgrp])) {
            do_chgrp(in_parm);
         } else if (!strcmp(in_msg,command_table[cmd_mv])) {
            do_mv(in_parm);
         } else if (!strcmp(in_msg,command_table[cmd_cat])) {
            do_cat(in_parm);
         } else if (!strcmp(in_msg,command_table[cmd_shell])) {
            do_shell();
         } else if (!strcmp(in_msg,command_table[cmd_uname])) {
            strcpy(out_msg,"uname");
            if (in_parm[0] != 0) {
               strcat(out_msg," ");
               strcat(out_msg,in_parm);
            }
            do_unix_command(out_msg);
         } else if (!strcmp(in_msg,command_table[cmd_cd])) {
            ch_dir(in_parm);
         } else if (!strcmp(in_msg,command_table[cmd_die])) {
            /* ok, to kill both the client and the server, we */
            /* first kill our parent, then ourselves... when  */
            /* our parent get's the TERM signal, we might be  */
            /* knocked off to, but just incase, send ourselves*/
            /* the TERM signal too, so that everything is nice*/ 
            /* and tidy                                       */
#ifdef LOG_TO_DEATH
            log_stuff(killed_by_command);
#endif
            kill(getppid(),SIGTERM); /* bam, our parent is dead */
            kill(getpid(), SIGTERM); /* bam, we're dead.        */
         } else if (!strcmp(in_msg,command_table[cmd_ls])) {
            strcpy(out_msg,"ls");
            if (in_parm[0] != 0) {
               strcat(out_msg," ");
               strcat(out_msg,in_parm);
            }
            do_unix_command(out_msg);
         } else if (!strcmp(in_msg,command_table[cmd_cmd])) {
            do_unix_command(in_parm);
         } else if ((!strcmp(in_msg,command_table[cmd_nil]))
                 || (in_msg[0] == 0)) {
            ;
         } else {
            tell_user(client_descriptor, "%3.3d - %s: %s", unknown_command,
                      in_msg, out_table[unknown_command]);
         }
      } else {
         tell_user(client_descriptor, "%3.3d - %s", input_too_long,
                   out_table[input_too_long]);
      }
   }
}
#ifdef SOCDEMON_PASSWORD
int check_pw(entered_pw, access_pw) 
char *entered_pw;
char *access_pw;
{
 char salt[3];
 char key[8];
 char *encr_pw;

   strncpy(key,entered_pw,8);
   strncpy(salt,access_pw,2);
   encr_pw = (char *) crypt(key,salt);
   return strcmp(access_pw,encr_pw);
}
#endif /* SOCDEMON_PASSWORD */
/*
 * soc_demon_welcome()
 *   - This routine prints up a bunch of greetings and identity info,
 *     all defined in socdemon.h, aswell as its real user id, effective user id
 *     real group id, effective group id, process id, and process group id.
 *     this is so you know what's going on on the server side of things.
 *     See socdemon.h for definitions of all the welcome strings.
 *   - Also, if the define 'SOCDEMON_PASSWORD' is not a nil string,
 *     then the user on the port will be prompted for a password,
 *     and allowed access only if the password is correct.
 *     The password is a compile time constant, and may not be changed other
 *     than by recompiling. The printing of the process id's etc. is
 *     held off until the user has successfully entered the password.
 *     see socdemon.h for the #define of SOCDEMON_PASSWORD.
*/
boolean soc_demon_welcome() {
 char hello_msg[HELLO_MSG_LEN];
 int  hello_msg_len;
 char pword_str[MAX_INPUT_LEN];
 int  pword_len;

   report_version();
#ifdef SOCDEMON_PASSWORD
   if (strlen(SOCDEMON_PASSWORD) > 0) {
      tell_user(client_descriptor, "Password? ");
      pword_len = soc_read(client_descriptor, pword_str, MAX_INPUT_LEN);
      if (pword_str[strlen(pword_str)-1] == '\n') {
         pword_str[strlen(pword_str)-2] = 0;
      }
      if (check_pw(pword_str, SOCDEMON_PASSWORD)) {
#ifdef LOG_TO_DEATH
         strcpy(log_string,pword_str);
         log_stuff(bad_password);
#endif
         tell_user(client_descriptor, "Access Denied.\n");
         return false;
      }
   }
#endif
   return true;
}

/*
 * main()
 *   - This is the main procedure..
 *   - Most of the socket setup stuff is taken from Stevens
 *     (the socket(), bind(), listen() and accept() stuff),
 *     the rest is mine.
*/
main(argc, argv) 
int   argc;
char *argv[];
{
 int     client_address_len;
 t_parms parms;
 int i;

   silent_mode = false;
   debug_mode  = false;
   port_to_use = SOCDEMON_PORT;
   /* get the name of the executable for use in the print_usage() */
   /* call so that the "usage: XXX [options]" will be correct     */
   strcpy(program_name,argv[0]);
#ifdef LOG_TO_DEATH
   log_to_file = false;
   command_log = false;
   strcpy(log_file,program_name);
   strcat(log_file,".log");
   strcpy(cmd_log_file,program_name);
   strcat(cmd_log_file,".log");
#endif
   if (argc > 1) {
      /* Collect and process the program arguments */
      for (i=1;i<argc;i++) {
         if ((!strcmp(argv[i],parm_table[parm_help1])) 
          || (!strcmp(argv[i],parm_table[parm_help2]))) {
            print_usage();
            exit(0);
         } else if ((!strcmp(argv[i],parm_table[parm_silent1])) 
                 || (!strcmp(argv[i],parm_table[parm_silent2]))) {
            silent_mode = true;
         } else if ((!strcmp(argv[i],parm_table[parm_debug1]))
                 || (!strcmp(argv[i],parm_table[parm_debug2]))) {
            debug_mode = true;
         } else if ((!strcmp(argv[i],parm_table[parm_port1]))
                 || (!strcmp(argv[i],parm_table[parm_port2]))) {
            i++;
            if (i == argc) {
               fprintf(stderr,"ERROR(%d): You must specify a port number.\n",
                       BAD_PARAMETERS);
               exit(BAD_PARAMETERS);
            } 
            if (sscanf(argv[i],"%d",&port_to_use) <= 0) {
               soc_fprintf(stderr,out_table[bad_port_use_def]);
               port_to_use = SOCDEMON_PORT;
               /* they prob. forgot to put the port number in and */
               /* tried to read the next parm, back up the index  */
               /* and we'll take a peep at what was there..       */
               i--;
            }
#ifdef LOG_TO_DEATH
         } else if ((!strcmp(argv[i],parm_table[parm_log1])) 
                 || (!strcmp(argv[i],parm_table[parm_log2]))) {
            log_to_file = true;
            i++;
            if (i == argc) {
               soc_fprintf(stderr, out_table[no_log_use_def]);
            } else {
               if (argv[i][0] == '-') {
                  soc_fprintf(stderr, out_table[no_log_use_def]);
                  i--;
               } else {
                  strcpy(log_file,argv[i]);
               }
            }
         } else if ((!strcmp(argv[i],parm_table[parm_cmds1]))
                 || (!strcmp(argv[i],parm_table[parm_cmds2]))) {
            command_log = true;
            i++;
            if (i == argc) {
               soc_fprintf(stderr, out_table[no_cmd_log_use_def]);
            } else {
               if (argv[i][0] == '-') {
                  soc_fprintf(stderr, out_table[no_cmd_log_use_def]);
                  i--;
               } else {
                  strcpy(cmd_log_file,argv[i]);
               }
            }
#endif
         } else {
            fprintf(stderr,"ERROR(%d): Unknown switch \"%s\".\n",
                    BAD_PARAMETERS, argv[i]);
            print_usage();
            exit(BAD_PARAMETERS);
         }
      }
   }

   soc_fprintf(stdout,"%s ver %s.\n",SOCDEMON_NAME,SOCDEMON_VER);

   soc_fprintf(stdout,"%s: Initializing.\n",SOCDEMON_NAME);
   if (getuid() != geteuid()) {
      /* Our user id is *not* equal to our effective user id */
      /* This means that we are a setuid program.. so, let's */
      /* Enjoy all the priviledges that we can, call setuid  */
      /* to set our user id to our effective userid          */
      soc_fprintf(stdout,"%s: Running setuid.\n",SOCDEMON_NAME);
      setuid(geteuid());
   }
   if (getgid() != getegid()) {
      soc_fprintf(stdout,"%s: Running setgid.\n",SOCDEMON_NAME);
      setgid(getegid());
   }
   soc_fprintf(stdout,"%s: Creating internet socket...",SOCDEMON_NAME);

   /* create a socket */
   socket_address_family = AF_INET;
   socket_type           = SOCK_STREAM;
   socket_protocol       = IPPROTO_TCP;
   
   socket_descriptor = socket(socket_address_family, 
                              socket_type, 
                              socket_protocol);

   if (socket_descriptor == -1) {
      soc_fprintf(stdout,"\n");
      soc_fprintf(stderr,"ERROR(%d): Unable to create socket [%d].\n",
                  CANT_MAKE_SOCKET, errno);
      exit(CANT_MAKE_SOCKET);
   }

   soc_fprintf(stdout,"done.\n");
   soc_fprintf(stdout,"%s: Binding socket to port %d ...",
               SOCDEMON_NAME, port_to_use);

   /* now bind it to the port, if the program was just run   */
   /* this call will fail, because it takes about 45 seconds */
   /* after a process exits for the port to be unbound       */
   bzero((char *) &server_address, sizeof(server_address));
   server_address.sin_family      = AF_INET;
   server_address.sin_addr.s_addr = htonl(INADDR_ANY);
   server_address.sin_port        = htons(port_to_use);

   socket_rc = bind(socket_descriptor,
                    (struct sockaddr *) &server_address,
                    sizeof(server_address));

   if (socket_rc != 0) {
      soc_fprintf(stdout,"\n");
      soc_fprintf(stderr,"ERROR(%d): Cannot bind internet socket to specified address and port. [%d]\n",
                  CANT_BIND_SOCKET, errno);
      if (errno == EADDRINUSE) {
         soc_fprintf(stderr,"            This address is inuse. Waiting 15 seconds then re-trying.\n");
         /* The following status message is to stderr instead of to stdout  */
         /* because stdout doesn't flush (at least on HP-UX) until a C/R    */
         /* but stderr flushes right away, so that way we can have the dots */
         /* print one at a time instead of all at once                      */
         soc_fprintf(stderr,"%s: Waiting",SOCDEMON_NAME);
         for (i=1;i<=15;i++) {
            sleep(1);
            soc_fprintf(stderr,".");
         }
         soc_fprintf(stderr,"done.\n");
         soc_fprintf(stdout,"%s: Retrying..",SOCDEMON_NAME);
         socket_rc = bind(socket_descriptor,
                          (struct sockaddr *) &server_address,
                          sizeof(server_address));
         if (socket_rc != 0) {
            soc_fprintf(stdout, " failed.\n");
            soc_fprintf(stderr, "ERROR(%d): This port is inuse, either wait a few moments, or use\n",
                        CANT_BIND_SOCKET);
            soc_fprintf(stderr, "            a different port.\n");
            socket_demon_shutdown();
            exit(CANT_BIND_SOCKET);
         } 
      } else {
         socket_demon_shutdown();
         exit(CANT_BIND_SOCKET);
      }
   }

   soc_fprintf(stdout,"done.\n");

   /* now my favorite part, convert to a daemon.. */
   if (fork() != 0) {
      /* we let the parent die off.. this means that our real parent   */
      /* will not be the shell, as was this thread right here, but     */
      /* because this thread is exiting, I believe that the new parent */
      /* becomes the process group leader, which may be the shell, but */
      /* I think it's the login instead.. whoever to process group     */
      /* is...                                                         */
      if (silent_mode == false) {
         /* now, we hold for a sec, so that the user doesn't get their */
         /* shell back until we've completely started up.. this is so  */
         /* the program doesn't print out it's init messages to the    */
         /* tty when the user has recieved thier prompt again.. if     */
         /* we're in silent mode, however, we don't need to pause      */
         /* because there is no output                                 */
         sleep(SOCDEMON_PAUSE);
      }
      exit(0);
   }
   /* now we dis-associate from our process group leader by creating   */
   /* our own process group.                                           */
   setpgrp();
   /* now we ignore hang up signals, so we are effectively nohup-ing   */
   /* the process now                                                  */
   signal(SIGHUP,SIG_IGN);
 
   /* fork once again to get rid of our parent (the new process group  */
   /* leader) and now our parent is init(1) so we are now a daemon     */
   /* (rejoice)                                                        */
   if (fork() != 0) {
      exit(0);
   }

   soc_fprintf(stdout,"%s: Binding signal handlers to shutdown routines...", 
               SOCDEMON_NAME);

   /* now bind our shutdown routines to catch the TERMINATE and KILL   */
   /* I don't think that the KILL signals are caught however, but i'm  */
   /* not entirely sure.. all I know is that when i kill -9 the daemon */
   /* I don't get the shutdown messages, but that doesn't necessarily  */
   /* mean it's not catching it.. so don't kill with the -9 option and */
   /* all will be well..                                               */
   signal(SIGTERM,socket_demon_death_action);
   signal(SIGKILL,socket_demon_death_action);

   soc_fprintf(stdout,"done.\n");

   /* hokey dokey, now listen to the internet socket... */
   socket_rc = listen(socket_descriptor,SOCDEMON_MAX_BACKLOG);

   if (socket_rc != 0) {
      soc_fprintf(stderr,"ERROR(%d): Listen failed on internet socket [%d].\n",
                  LISTEN_FAILED, errno);
      socket_demon_shutdown();
      exit(LISTEN_FAILED);
   }
  
   for (;;) {
      client_address_len = sizeof(client_address);

      /* now we sit and wait for someone to connect to us...           */
      /* this will not take up any CPU at all, because the process is  */
      /* blocked waiting for a connect... what I might do if there's   */
      /* some need, is to send the process wake up signals every now   */
      /* and then to change it's process ID, or hide or something, but */
      /* until all other little bugs are resolved, that will have to   */
      /* wait                                                          */
      client_descriptor = accept(socket_descriptor, 
                                 (struct sockaddr *) &client_address,
                                 &client_address_len);

      if (client_descriptor < 0) {
         soc_fprintf(stderr,"ERROR(%d): accept() call failed on internet socket [%d]\n",
                     ACCEPT_FAILED, errno);
         exit(ACCEPT_FAILED);
      }

#ifdef LOG_TO_DEATH
      /* ok, now to convert the address to a dotted quad, I guess i */
      /* could just cast it on top of the dotted_quad structure, I  */
      /* dunno if that'd work, so to be safe and portable, let's    */
      /* do it manually... oh, and note the bit shifts, yeah, it's  */
      /* messy, but I couldn't think of anything more elegant at the*/
      /* time.. same with the leading zeros in the bitwise 'and's   */
      client_dotted_q.class    = (client_address.sin_addr.s_addr >> 24);
      client_dotted_q.netid    = (client_address.sin_addr.s_addr >> 16) & 0x00ff;
      client_dotted_q.subnetid = (client_address.sin_addr.s_addr >> 8) & 0x0000ff;
      client_dotted_q.hostid   = client_address.sin_addr.s_addr & 0x000000ff;
      sprintf(dotted_4_str, "%d.%d.%d.%d",
              client_dotted_q.class, client_dotted_q.netid,
              client_dotted_q.subnetid, client_dotted_q.hostid);
#endif
      if (fork() != 0) {
         /* let the parent process handle the user comming in, and       */
         /* make the shild go on to wait for more requests - this        */
         /* will avoid defunct processes and have an added bonus side-   */
         /* effect of having the main daemon have it's process ID change */
         /* periodically.                                                */
#ifdef LOG_TO_DEATH
         log_stuff(accepting_connection);
#endif
     
         /* direct stdin, stdout and stderr to the socket */
         dup2(client_descriptor,0);
         dup2(client_descriptor,1);
         dup2(client_descriptor,2);
         if (soc_demon_welcome()) {
            soc_demon_main_proc();
         } 

         shutdown(client_descriptor, 2);
         close(client_descriptor);
#ifdef LOG_TO_DEATH
         log_stuff(connection_closed);
#endif
         exit(0);
      }

      close(client_descriptor);
   }
}
