/* $Id: nit3open.c,v 2.2 89/10/24 17:53:27 dupuy Exp $ */

#include <sys/param.h>			/* NOFILE */
#include <sys/time.h>			/* timeval */
#include <sys/socket.h>			/* sockaddr_nit (sockaddr) */
#include <sys/ioctl.h>			/* ioctl */

#include <net/nit.h>			/* sockaddr_nit */
#include <netinet/in.h>			/* htons */

#include <strings.h>			/* strncpy */
#include <errno.h>			/* EWOULDBLOCK/EINVAL */

#include "libether.h"

struct timeval *ether_timeout;
struct timeval *ether_timestamp;

static ether_addr promiscuous;

static fd_set multicast;
static gotlocal;
static ether_addr local_addr;

unsigned _ether_types[NOFILE];
ether_addr _ether_multi_addrs[NOFILE];

#define ether_type (_ether_types[fd])
#define multi_addr (_ether_multi_addrs[fd])

/*
 * Returns file descriptor for ethernet device by name ("ie0", "le0", etc.).
 * If name is NULL, uses primary ethernet interface.  Will only receive
 * packets of type specified.  Will receive packets for the ethernet address
 * specified, or local ethernet address if NULL.  If there is an error,
 * returns (-1) and the appropriate value is left in errno.  Normal return
 * status zero. Requires superuser privilege.
 */

int
ether_open (name, type, address)
char *name;
unsigned type;
ether_addr *address;
{
    int fd;
    struct sockaddr_nit snit;
    struct nit_ioc nioc;
    char **interfaces;
    int saved_errno;

    if (name == 0)
    {					/* get default ethernet interface */
	interfaces = ether_interfaces ();
	if (interfaces == 0 || *interfaces == 0)
	    return (-1);

	name = *interfaces;		/* just use the first name in list */
    }

    if ((fd = socket (AF_NIT, SOCK_RAW, NITPROTO_RAW)) < 0)
    {
#ifdef DEBUG
	perror ("ether_open: socket");
#endif
	return (-1);
    }

    snit.snit_family = AF_NIT;
    (void) strncpy (snit.snit_ifname, name, NITIFSIZ);

    if (bind (fd, (struct sockaddr *) &snit, sizeof (snit)) != 0)
    {
	saved_errno = errno;
#ifdef DEBUG
	perror ("ether_open: bind");
#endif
	(void) close (fd);
	errno = saved_errno;
	return (-1);
    }

    bzero ((char *) &nioc, sizeof (nioc));

    if (type != ETHER_ALLTYPES && type > ETHER_MAXTYPE)
    {
	(void) close (fd);
	errno = EINVAL;
	return (-1);
    }

    nioc.nioc_bufalign = sizeof (long);
    nioc.nioc_chunksize = nioc.nioc_bufspace = ETHER_BUFSIZ;
    nioc.nioc_typetomatch =
	(type == ETHER_ALLTYPES) ? type : htons ((u_short) type);
    nioc.nioc_snaplen = ETHER_PKT + ETHER_MAX;

    if (address == 0)			/* not a multicast address */
    {
	multicast.fds_bits[0] &= ~(1 << fd);
	nioc.nioc_flags = NF_TIMEOUT;
    }
    else				/* a multicast address */
    {
	if (ether_cmp (address, &promiscuous))
	{
	    if (!ETHER_MCAST (address))
	    {
#ifdef DEBUG
		(void) printf ("rejecting non-multicast address argument\n");
#endif
		(void) close (fd);
		errno = EINVAL;
		return (-1);
	    }

	    multicast.fds_bits[0] |= (1 << fd);
	    multi_addr = *address;
	    if (gotlocal == 0)
		if (ether_address (fd, &local_addr) != NULL)
		    gotlocal = 1;
	}
	else
	    multicast.fds_bits[0] &= ~(1 << fd);

	nioc.nioc_flags = NF_TIMEOUT | NF_PROMISC;
    }

    if (ether_timeout != 0)		/* use specified value of timeout */
	nioc.nioc_timeout = *ether_timeout;
    else
    {					/* then use default value of timeout */
	nioc.nioc_timeout.tv_sec = 0;
	nioc.nioc_timeout.tv_usec = 10000;
    }

    if (ioctl (fd, SIOCSNIT, (char *) &nioc) != 0)
    {
	saved_errno = errno;
#ifdef DEBUG
	perror ("ether_open: ioctl SIOCSNIT");
#endif
	(void) close (fd);
	errno = saved_errno;
	return (-1);
    }

    ether_type = type;
    return (fd);
}


#ifdef DEBUG

void
dump_nit_hdr (nitbuf)
struct nit_hdr *nitbuf;
{
    switch (nitbuf->nh_state)
    {
      case NIT_QUIET:
	(void) printf ("ether_read: nit quiet\n");
	break;

      case NIT_CATCH:
	(void) printf ("ether_read: nit catch: size %d\n", nitbuf->nh_datalen);
	break;

      case NIT_NOMBUF:
	(void) printf ("ether_read: nit no mbufs: %d lost\n",
		       nitbuf->nh_dropped);
	break;

      case NIT_NOCLUSTER:
	(void) printf ("ether_read: nit no mclusters: %d lost\n",
		       nitbuf->nh_dropped);
	break;

      case NIT_NOSPACE:
	(void) printf ("ether_read: nit no bufspace: %d lost\n",
		       nitbuf->nh_dropped);
	break;

      case NIT_SEQNO:
	(void) printf ("ether_read: nit sequence # %d\n", nitbuf->nh_seqno);
	break;

      default:
	(void) printf ("ether_read: bad nit header state: %d\n",
		       nitbuf->nh_state);
	break;
    }
}

#endif


/*
 * Reads and returns a single packet, filling in all fields.  If pktbuf is
 * NULL, a buffer is allocated for it.	If pktbuf is not NULL, the function
 * assumes that pktbuf is large enough to hold pktlen bytes.  Since read() may
 * return several packets at a time, ether_read() has to buffer them as well.
 * Since socket based nit doesn't handle multicast addresses, we have to check
 * them at the user level.
 *
 * NOTE - the buffering code is non-reentrant if the same fd is used.
 */

static long ether_buf[NOFILE][ETHER_BUFSIZ / sizeof (long)];
static int ether_buflen[NOFILE];
static char *ether_bufptr[NOFILE];

#define buf	(ether_buf[fd])
#define buflen	(ether_buflen[fd])
#define bufptr	(ether_bufptr[fd])

_ether_next_packet (fd, bufp)
int fd;
char **bufp;
{
    struct nit_hdr *nitbuf;
    ether_addr *pbuf;

    if (bufptr == 0)			/* easier than static initialization */
	bufptr = (char *) buf;

    for (;;)
    {
	if ((bufptr - (char *) buf) >= buflen)
	{				/* then buffer is used up */
	    buflen = read (fd, (char *) buf, ETHER_BUFSIZ);
	    bufptr = (char *) buf;
	    if (buflen <= 0)
	    {
		if (errno == EWOULDBLOCK)
		    errno = EAGAIN;
		return (-1);
	    }
	}

	/*
	 * This assignment is safe, since nit guarantees requested alignment
	 */

	nitbuf = (struct nit_hdr *) bufptr;
	while (nitbuf->nh_state != NIT_CATCH)
	{
#ifdef DEBUG
	    dump_nit_hdr (nitbuf);
#endif
	    nitbuf++;
	    if ((char *) nitbuf - (char *) buf > buflen)
	    {
#ifdef DEBUG
		(void) printf ("truncated nit_hdr struct in buffer\n");
#endif
		continue;
	    }
	}
#ifdef DEBUG
	dump_nit_hdr (nitbuf);
#endif
	/*
	 * Set bufptr to next nit_hdr, pbuf to start of ether packet
	 */

	bufptr = ((char *) (pbuf = (ether_addr *) (nitbuf + 1)))
	    + nitbuf->nh_datalen + pad (nitbuf->nh_datalen);

	/*
	 * Check for local, broadcast or multicast packet 
	 */

	if ((multicast.fds_bits[0] & (1 << fd))
	    && (ETHER_MCAST (pbuf) ? (ether_cmp (pbuf, &ether_bcast_addr)
				      && ether_cmp (pbuf, &multi_addr))
		: ether_cmp (pbuf, &local_addr)))
	{
#ifdef DEBUG
	    (void) printf ("discarding incorrectly addressed packet\n");
#endif
	    continue;
	}

	if (nitbuf->nh_wirelen > nitbuf->nh_datalen)
	{
#ifdef DEBUG
	    (void) printf ("discarding truncated packet\n");
#endif
	    continue;
	}
	break;
    }

    if (nitbuf->nh_wirelen - ETHER_PKT < 0)
	errno = EBADMSG;
    else
	*bufp = (char *) pbuf;

    ether_timestamp = &nitbuf->nh_timestamp;

    return (nitbuf->nh_wirelen - ETHER_PKT);
}
