/* pingmod.c - v1.65 - sockets, etherlib, and dlpi support */
/* Michael R. Widner. (atreus) - 9/26/94

 * the routine build_icmp() appends a fake IP packet to an icmp packet.
 * It will send fake DESTINATION UNREACHABLE messages, and will do either
 * host or net unreachable and some other crap depending on options set.
 *
 * v1.65 in the process of adding linux support, since I just got up and
 *      running on linux.  Special thanks to Tim Newsham, who explained
 *      the peculiarities of SOCK_RAW and SOCK_PACKET of linux, and also
 *      for two progs he did that illustrate them perfectly.
 *      Right now only SOCK_RAW is working.  SOCK_PACKET should be in
 *      the not so distant future, I hope.
 *
 * v1.6 added DLPI support.  I'd really like to see more systems support
 *      the DLPI interface.  I've tested it on Solaris 2.3, but nowhere
 *      else.  To use the DLPI etherlib support define USE_DLPI when
 *      compiling, and do the stuff in the Makefile. 
 *
 * v1.5 added etherlib support, rather than just /dev/nit support.  You
 *      must set the etherlib options in the Makefile, and make sure this
 *      is compiled with USE_ETHERLIB defined.
 *
 * v1.x previous versions had only /dev/nit and sockets support.  As of
 *      1.5 the code was almost entirely rewritten.
 *
 * This program may be compiled in either of two modes, depending on whether
 * or not etherlib support is defined.  If it is, you will use the raw ethernet
 * interface to send packets with faked ip addresses.  If you don't use
 * the etherlib library (or if you use -x on the command line) you will
 * be using the your own ip address in your icmp packets.  This will result
 * in packets that could possibly be traced.
 *
 *
 * Someday I'll modify this to send ECHO icmps with a fake return
 * ip address.  Why?  Because pinging a broadcast address on your
 * local net causes a lot of traffic.  Changing the return ip address
 * to something bogus causes 4 'arp who has' requests for every echo
 * requests.  Sending these at a rapid rate can cause kick ass broadcast
 * storms.
 *
 * Just something to think about:  I've put this on a few systems that
 * admins really wouldn't like it to be on.  In those cases I generally
 * disable the help feature.  (-h and -H dont work).  Renaming is always
 * a good idea as well.
 */


#define TEST
/* This should compile under bsd or sysv. */
#ifdef SYSV
#define bcopy(b1,b2,len) memmove(b2, b1, (size_t)(len))
#define bzero(b,len) memset(b, 0, (size_t)(len))
#endif

/* Linux appears to need me to include this.  */
#ifdef LINUX
#include <endian.h>
#endif

/* Pretty generous group of includes.  Someday I'll narrow this list. */
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/file.h>
#include <sys/time.h>
#include <sys/signal.h>

#ifdef LINUX
#include <bsd/netinet/in_systm.h>
#include <bsd/netinet/in.h>
#include <bsd/netinet/ip.h>
#include <bsd/netinet/ip_icmp.h>
#include <bsd/netinet/ip_var.h>
#include <bsd/netinet/tcp.h>
#include <bsd/netinet/udp.h>
#else
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <netinet/ip_var.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>
#endif
#include <netdb.h>
#include <unistd.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>


#if !defined(usleep)
#include <sys/time.h>
#define usleep(x) { \
    struct timeval st_delay; \
    st_delay.tv_usec = (x); \
    st_delay.tv_sec = 0; \
    select(0, NULL, NULL, NULL, &st_delay); \
}
#endif defined(usleep)

/* various options */
u_long options=0;
#define	F_FLOOD		0x001
#define	F_INTERVAL	0x002
#define	F_NUMERIC	0x004
#define	F_ECHO		0x008
#define	F_QUIET		0x010
#define	F_FAKE_SOURCE	0x020
#define	F_NO_ETHER	0x040
#define	F_VERBOSE	0x080
#define	F_BOTH_WAYS	0x100
#define	F_NETWORK	0x200
#define	F_PORT		0x400
#define	F_UDP		0x800

#define	MAXPACKET	(65536 - 60 - 8)/* max packet size - I dont need this */

/* Define your local router here.  Arp can be used to get this is. */
#ifndef ROUTER
#define ROUTER "0:1:2:3:4:5"
#endif

#if defined(USE_ETHERLIB)
#include <ether.h>
#endif
char *egate=NULL;

#ifdef USE_DLPI
#include <sys/ethernet.h>
#include <sys/dlpi.h>
#include <dlcommon.h>
#include <dltest.h>
#endif

/* global variables.  This is uglier than your mother. */
int s;				/* socket file descriptor */

/* usage */
void usage()
{
	(void)fprintf(stderr,
	    "usage: pingmod [options] source_host unreach_host ...\n");
	fprintf(stderr,"  -b     send both ways\n");
	fprintf(stderr,"  -q     do it quietly\n");
#if defined(USE_ETHERLIB) || defined(USE_DLPI)
	fprintf(stderr,"  -f x   address packet is supposedly from\n");
	fprintf(stderr,"  -g x   specify e-net gateway as xx:xx:xx:xx:xx:xx\n");
	fprintf(stderr,"  -x     don't spoof my ip address\n");
#endif
	fprintf(stderr,"  -n     send network unreach (default is host)\n");
	fprintf(stderr,"  -p     send port unreach\n");
	fprintf(stderr,"  -s x-y port range to hit on source_host\n");
	fprintf(stderr,"  -d x-y destination port range which is unreachable\n");
	fprintf(stderr,"  -u     make fake packet udp (default is tcp)\n");
	fprintf(stderr,"  -c x   number of packets to send\n");
	fprintf(stderr,"  -i x   micro-seconds to wait betwen each packet\n");
	fprintf(stderr,"  -I x   milli-seconds betwen packet groups (used with -c)\n");
	fprintf(stderr,"  .      sub dest or source from previous host pair\n");
	fprintf(stderr,"         ie: pingmod 1 2 . 3  = pingmod 1 2 1 3\n");
	fprintf(stderr,"\n");
	fprintf(stderr,"  NOTE:  Using port ranges in conjunction with c,b, or multiple\n");
	fprintf(stderr,"         host names can cause LOTS of traffic.\n");
	exit(1);
}


/*
 * pr_addr --
 *	Return an ascii host address as a dotted quad and optionally with
 * a hostname.
 */
char *
pr_addr(l)
	struct in_addr l;
{
	struct hostent *hp;
	static char buf[80];

	if ((options & F_NUMERIC) ||
	    !(hp = gethostbyaddr((char *)&l, 4, AF_INET)))
		(void)sprintf(buf, "%s", inet_ntoa(l));
	else
		(void)sprintf(buf, "%s (%s)", hp->h_name,
		    inet_ntoa(l));
	return(buf);
}


/*
 * in_cksum --
 *	Checksum routine for Internet Protocol family headers (C Version)
 */
in_cksum(addr, len)
	u_short *addr;
	int len;
{
	register int nleft = len;
	register u_short *w = addr;
	register int sum = 0;
	u_short answer = 0;

	/*
	 * Our algorithm is simple, using a 32 bit accumulator (sum), we add
	 * sequential 16 bit words to it, and at the end, fold back all the
	 * carry bits from the top 16 bits into the lower 16 bits.
	 */
	while (nleft > 1)  {
		sum += *w++;
		nleft -= 2;
	}

	/* mop up an odd byte, if necessary */
	if (nleft == 1) {
		*(u_char *)(&answer) = *(u_char *)w ;
		sum += answer;
	}

	/* add back carry outs from top 16 bits to low 16 bits */
	sum = (sum >> 16) + (sum & 0xffff);	/* add hi 16 to low 16 */
	sum += (sum >> 16);			/* add carry */
	answer = ~sum;				/* truncate to 16 bits */
	return(answer);
}



/*
 * build_icmp -- constructs a fake icmp packet given source addr, source
 *               port, dest addr, and dest port.  options are set in
 *               global variable 'options'.
 *               packet constructed at address of packstart (must be
 *               aligned), build_icmp returns size of packet built.
 */
int build_icmp(start_addr, source, s_port, dest, d_port)
	u_char *start_addr;
	struct in_addr source,dest;
	u_short s_port,d_port;
{
	struct icmp *icp;
	void *next_location;
	struct tcphdr *tcp_start;
	struct udphdr *udp_start;
	int cc,i;

	/* allocated some stuff we use */
	icp=(void *)start_addr;
	next_location=start_addr;

	/* Set header for ECHO or UNREACH type. */
	if (options & F_ECHO)
		icp->icmp_type = ICMP_ECHO;
	else
		icp->icmp_type = ICMP_UNREACH;


	/* And set for network, port, or host unreachable */
	if (options & F_NETWORK )
		icp->icmp_code = ICMP_UNREACH_NET;
	else if (options & F_PORT )
		icp->icmp_code = ICMP_UNREACH_PORT; 
	else
		icp->icmp_code = ICMP_UNREACH_HOST;

	/* Clear out the stuff we're not using. */
	icp->icmp_cksum = 0;
	icp->icmp_seq = 0;
	icp->icmp_id = 0;


	/* Create our fake IP header here. */
	icp->icmp_ip.ip_v=4;	/* version 4 ip */
	icp->icmp_ip.ip_hl=5;	/* header length in 32-bit words */
	icp->icmp_ip.ip_tos=0;	/* type of service is 0 */
	icp->icmp_ip.ip_len=256;/* length of orig (fake) paket.  type short */
	icp->icmp_ip.ip_id=21;	/* identification - do we need it? */
	icp->icmp_ip.ip_off=0;	/* fragment offset */
	icp->icmp_ip.ip_ttl=25;	/* arbitrary ttl remaining */
	if (options & F_UDP)	/* protocol:  6=tcp  17=udp */
		icp->icmp_ip.ip_p=IPPROTO_UDP;
	else
		icp->icmp_ip.ip_p=IPPROTO_TCP;
	icp->icmp_ip.ip_sum=0;	/* type short:  arbitrary fake chksum */
	icp->icmp_ip.ip_src=source;	/* source address */
	icp->icmp_ip.ip_dst=dest;	/* destination address */

	/* We probably don't need this, but I do it anyway. */
	icp->icmp_ip.ip_sum = in_cksum((u_short *)&icp->icmp_ip, 20);

	/* We have to append 8 bytes of data after for an unreachable message */
	/* For tcp & udp the first 4 bytes are source & dest port */
	/* for tcp the next 4 are sequence number.  */
	/* for udp they are length (2 bytes) and checksum (2 bytes) */
	next_location=start_addr+sizeof(struct icmp);
	tcp_start=(void *)next_location;
	if (options & F_UDP)
	{
		udp_start=(void *)next_location;
		udp_start->uh_sport=s_port;
		udp_start->uh_dport=d_port;
		udp_start->uh_ulen=1000;
		udp_start->uh_sum=1000;
	}
	else
	{
		tcp_start->th_sport=s_port;
		tcp_start->th_dport=d_port;
		tcp_start->th_seq=1000;
	}

	/* compute ICMP checksum here */
	cc = sizeof(struct icmp)+8;	/* icmp hd + ip head + 64 bits data */
	icp->icmp_cksum = in_cksum((u_short *)icp, cc);

	return(cc); /* site of packet we built */
}

/* 
 * do_packet - where we create the whole thing and send it. 
 *
 */
int do_packet(whereto, source, s_port, dest, d_port, fake_gw)
	struct sockaddr whereto;
	struct in_addr source,dest,fake_gw;
	u_short s_port,d_port;
{
	int packet_size=0,i,x;
	u_char *whole_packet,*next_loc;
#if defined(USE_ETHERLIB) || defined(USE_DLPI) || defined(USE_SOCK_PACK)
	int ether_fd;
	struct ip *ip;
#if defined(USE_ETHERLIB)
	ether_addr e_address;
	struct etherpacket eth_packet;
#else if defined(USE_DLPI)
	u_char localphys[ETHERADDRL];
	u_char phys[10];
	u_char addr[MAXDLADDR];
	char buf[MAXDLBUF];
	struct ether_header *ehp;
	union DL_primitives *dlp;
#endif
#endif

	whole_packet=(u_char *)malloc(MAXPACKET);
	next_loc=whole_packet;
	
#if defined(USE_ETHERLIB) || defined(USE_DLPI) || defined(USE_SOCK_PACK)
	if (!(options & F_NO_ETHER))
	{
#if defined(USE_ETHERLIB)  /* start set up FD for etherlib stuff */
		eth_packet.pktbuf=(char *)whole_packet;
		ip=(void *)next_loc;
		if ((ether_fd=ether_open(NULL,0x0800,(ether_addr *)NULL))<0)
			perror("ether open");
		ether_a2e((egate ? egate : ROUTER),&e_address);
		bcopy((char *)&e_address,(char *)&eth_packet.dest,
			sizeof(ether_addr));
#endif    /* end of etherlib setup block */
#if defined(USE_DLPI)
          /* this is where we set up the dlpi interface stuff */
		ehp=(void *)whole_packet;
		if ((ether_fd=open("/dev/le",2))<0) {
			printf("ERROR: ether open\n");
			exit(255);
			}
		for (i=0;i<MAXDLBUF;i++)
			buf[i]=(unsigned char) i & 0xff;
		dlattachreq(ether_fd,0);
		dlokack(ether_fd,buf);
		dlbindreq(ether_fd,0x800,0,DL_CLDLS,0,0);
		dlbindack(ether_fd,buf);
		dlphysaddrreq(ether_fd, DL_CURR_PHYS_ADDR);
		dlphysaddrack(ether_fd, buf);
		dlp=(union DL_primitives *)buf;
		stringtoaddr((egate ? egate : ROUTER),phys);
		memcpy(&ehp->ether_dhost, phys, ETHERADDRL);
		memcpy(&ehp->ether_shost,
			OFFADDR(dlp, dlp->physaddr_ack.dl_addr_offset),
			ETHERADDRL);
		ehp->ether_type=(u_short)0x800;
		if (strioctl(ether_fd, DLIOCRAW, -1, 0, 0) < 0) {
			printf("ERROR: DLIOCRAW\n");
			exit(255);
			}
		/* fucking alignment problems... */
		ip=(struct ip *)malloc(1024);
		next_loc=(void *)ip;
#endif /* end of dlpi setup block */
#if defined(USE_SOCK_PACK)
	/* linux SOCK_PACKET specific setup */
		strcpy(add.sa_data,"eth0");
	/* My local routers enet address is 0:0:c:1:a2:39 */		
		whole_packet=0x00;
		whole_packet[1]=0x00;
		whole_packet[2]=0x0c;
		whole_packet[3]=0x01;
		whole_packet[4]=0xa2;
		whole_packet[5]=0x39;
		whole_packet[6]=0x00;
		whole_packet[7]=0x00;
		whole_packet[8]=0x00;
		whole_packet[9]=0x00;
		whole_packet[10]=0x00;
		whole_packet[11]=0x00;
		whole_packet[12]=0x80;
		whole_packet[13]=0x00;
		/* fucking alignment problems again... */
		ip=(struct ip *)malloc(1024);
		next_loc=(void *)ip;
#endif
		/* Create our fake IP header here. */
		ip->ip_v=4;	/* version 4 ip */
		ip->ip_hl=5;	/* header length in 32-bit words */
		ip->ip_tos=0;	/* type of service is 0 */
		ip->ip_len=0; /* length of whole paket. */
		ip->ip_id=0;	/* identification - do we need it? */
		ip->ip_off=0;	/* fragment offset */
		ip->ip_ttl=25;	/* arbitrary ttl remaining */
		ip->ip_p=IPPROTO_ICMP;
		ip->ip_sum=0;	/* type short:  arbitrary fake chksum */
		/* we're sending to 'source', making it look like the
		 * message is from 'dest'.  Get it? */
		ip->ip_src=fake_gw;	/* source address */
		ip->ip_dst=source;	/* destination address */
		next_loc=next_loc+sizeof(struct ip);
		packet_size=packet_size+sizeof(struct ip);
	}
#endif
	/* tack on the icmp packet */
	packet_size=packet_size+build_icmp(next_loc,source,s_port,dest,d_port);

#if defined(USE_ETHERLIB) || defined (USE_DLPI) || (USE_SOCK_PACK)
	if (!(options & F_NO_ETHER))
	{
		/* set total packet size and compute checksum */
		ip->ip_len=packet_size;
		/* ip->ip_sum=in_cksum((u_short *)ip, packet_size); */
		ip->ip_sum=in_cksum((u_short *)ip, 20);
#ifdef USE_ETHERLIB
		eth_packet.pktlen=packet_size;
#ifdef TEST
  for(x=0;x<packet_size;x++)
    printf("%02x%c",(unsigned char)eth_packet[x],((x+1)%8)?' ':'\n');
  printf("\n");
#endif /*TEST*/
		if (ether_write(ether_fd,&eth_packet)<0)
			perror("ether write");
		if (ether_blocking(ether_fd,0))
			perror("ether blocking");
#else
		/* Rebuild packet because of fucking enet header align */
#ifdef USE_DLPI
		next_loc=whole_packet+sizeof(struct ether_header);
#else
		next_loc=whole_packet+14;
#endif
		memcpy(next_loc,ip,packet_size);
		free(ip);
		if (write(ether_fd, whole_packet,
			packet_size+sizeof(struct ether_header))<0)
			printf("ERROR:  failed ether write\n");
#endif
		close(ether_fd);
	}
#endif
	/* Do it with the the sockets interface */
	if (options & F_NO_ETHER) {
		printf("Using sockets interface.  Real IP addr. is visible.\n");
		i = sendto(s, (char *)whole_packet, packet_size, 0, &whereto,
		    sizeof(struct sockaddr));
#ifdef TEST
  for(x=0;x<packet_size;x++)
    printf("%02x%c",(unsigned char)whole_packet[x],((x+1)%8)?' ':'\n');
  printf("\n");
#endif /*TEST*/

		/* check that correct number of bytes were sent */
		if (i < 0 || i != packet_size)  {
			if (i < 0)
				perror("pingmod: sendto");
			printf("pingmod: wrote %d chars, ret=%d\n", packet_size, i);
		} 
	}
	free (whole_packet);
	return(0);
}


/*
 * main - this is where it all happens.
 */
main(argc, argv)
	int argc;
	char **argv;
{
	extern int optind;
	extern char *optarg;
	extern int opterr;
	struct hostent *hp;
	struct sockaddr whereto;	/* who to ping */
	struct sockaddr_in *to;
	struct sockaddr wherefrom;	/* who are we faking from */
	struct sockaddr_in *from;
	struct sockaddr fakegw;		/* who are we faking from */
	struct sockaddr_in *fgw;
	struct protoent *proto;
	register int i;
	long npackets = 1;		/* packets to transmit */
	int major_interval = 1;		/* interval between each packet */
	int minor_interval = 1;		/* interval between packet cycles */
	int ch, fdmask, packlen, argloop;
	u_char *datap, *packet;
	u_short start_sport=1024,start_dport=23,dport,sport;
	u_short end_sport=1024,end_dport=23;
	char *target, *whosunreach, hnamebuf[MAXHOSTNAMELEN];
	char *hostname, *fgwname=NULL;
	

#if !defined(USE_ETHERLIB) && !defined(USE_DLPI)
	options |= F_NO_ETHER;
#endif

	opterr=0;
	while ((ch = getopt(argc, argv, "bd:f:s:c:i:g:npquxhH")) != EOF)
		switch(ch) {
		case 'b':
			options |= F_BOTH_WAYS;
			break;
		case 'c':
			npackets = atoi(optarg);
			if (npackets <= 0) {
				(void)fprintf(stderr,
				    "pingmod: bad number of packets to transmit.\n");
				exit(1);
			}
			break;
		case 's':
			start_sport = atoi(optarg);
			if (strchr(optarg,'-'))
				end_sport=0-atoi(strchr(optarg,'-'));
			else
				end_sport=start_sport;
			break;
		case 'g':
			egate = optarg;
			break;
		case 'd':
			start_dport = atoi(optarg);
			if (strchr(optarg,'-'))
				end_dport=0-atoi(strchr(optarg,'-'));
			else
				end_dport=start_dport;
			break;
		case 'F':
			options |= F_FLOOD;
			setbuf(stdout, (char *)NULL);
			break;
		case 'f':
			options |= F_FAKE_SOURCE;
			fgwname=optarg;
			break;
		case 'i':		/* wait between sending packets */
			minor_interval = atoi(optarg);
			if (minor_interval <= 0) {
				fprintf(stderr,
				    "pingmod: bad timing interval.\n");
				exit(1);
			}
			options |= F_INTERVAL;
			break;
		case 'I':		/* wait between sending packets */
			major_interval = atoi(optarg);
			if (major_interval <= 0) {
				fprintf(stderr,
				    "pingmod: bad timing interval.\n");
				exit(1);
			}
			options |= F_INTERVAL;
			break;
		case 'x':
			options |= F_NO_ETHER;
			break;
		case 'n':
			options |= F_NETWORK;
			break;
		case 'p':
			options |= F_PORT;
			break;
		case 'q':
			options |= F_QUIET;
			break;
		case 'u':
			options |= F_UDP;
			break;
		case 'h':
		case 'H':
			usage();
		default:
			exit(1);
		}
	argc -= optind;
	argv += optind;

	if (argc < 2)
		exit(1);

	
	/* You cant flood at an interval, you dumbass */
	/* of course I don't have flooding implemented yet */
	if (options & F_FLOOD && options & F_INTERVAL) {
		(void)fprintf(stderr,
		    "pingmod: -f and -i incompatible options.\n");
		exit(1);
	} 

	if (options & F_NO_ETHER) {
		/* We're gonna do it with a socket.  Get icmp protocol. */
		if (!(proto = getprotobyname("icmp"))) {
			fprintf(stderr, "pingmod: unknown protocol icmp.\n");
			exit(1);
		}

		/* create the socket */
#ifndef USE_SOCK_PACK
		if ((s = socket(AF_INET, SOCK_RAW, proto->p_proto)) < 0) {
#else
		if ((s = socket(AF_INET, SOCK_PACKET, proto->p_proto)) < 0) {
#endif
			perror("pingmod: socket");
			exit(1);
		}
	}

	/* initialize target and whosunreach so 'pingmod . .' will behave */
	target=*argv;
	whosunreach=*argv;

	while(npackets)
	{
	argloop=0;
	
	while (argloop<(argc-1))
	{
	if (strcmp(argv[argloop],"."))
		target = argv[argloop];
	argloop++;
	if (strcmp(argv[argloop],"."))
		whosunreach = argv[argloop];
	argloop++;

	if(!(options & F_FAKE_SOURCE))
		fgwname=whosunreach;

	/* clear and assign socket struct */
	bzero((char *)&whereto, sizeof(struct sockaddr));
	to = (struct sockaddr_in *)&whereto;

	bzero((char *)&wherefrom, sizeof(struct sockaddr));
	from = (struct sockaddr_in *)&wherefrom;

	/* If we are choosing a gateway to fake from, this will be unique. */
	/* Often, it will be the same as whosunreach (aka - from)*/
	bzero((char *)&fakegw, sizeof(struct sockaddr));
	fgw = (struct sockaddr_in *)&fakegw;

	/* build a sockaddr for our fake source gateway. */
	fgw->sin_family = AF_INET;
	fgw->sin_addr.s_addr = inet_addr(fgwname); /* 1.2.3.4 to u_long */
	if (fgw->sin_addr.s_addr != (u_int)-1)
		hostname = target;
	else {                                   /* it's a name, look it up */
		hp = gethostbyname(fgwname);
		if (!hp) {
			(void)fprintf(stderr,
			    "pingmod: unknown dest %s\n", fgwname);
			exit(1);
		}
		fgw->sin_family = hp->h_addrtype;
		bcopy(hp->h_addr, (caddr_t)&fgw->sin_addr, hp->h_length);
		strncpy(hnamebuf, hp->h_name, sizeof(hnamebuf) - 1);
		hostname = hnamebuf;
	}


	/* build a sockaddr for our fake unreachable host. */
	from->sin_family = AF_INET;
	from->sin_addr.s_addr = inet_addr(whosunreach); /* 1.2.3.4 to u_long */
	if (from->sin_addr.s_addr != (u_int)-1)
		hostname = target;
	else {                                   /* it's a name, look it up */
		hp = gethostbyname(whosunreach);
		if (!hp) {
			(void)fprintf(stderr,
			    "pingmod: unknown dest %s\n", target);
			exit(1);
		}
		from->sin_family = hp->h_addrtype;
		bcopy(hp->h_addr, (caddr_t)&from->sin_addr, hp->h_length);
		strncpy(hnamebuf, hp->h_name, sizeof(hnamebuf) - 1);
		hostname = hnamebuf;
	}

	/* Now do same thing for source (who we're sending to)  */
	to->sin_family = AF_INET;
	to->sin_addr.s_addr = inet_addr(target); /* 1.2.3.4 to u_long */
	if (to->sin_addr.s_addr != (u_int)-1)
		hostname = target;
	else {                                   /* it's a name, look it up */
		hp = gethostbyname(target);
		if (!hp) {
			(void)fprintf(stderr,
			    "pingmod: unknown source %s\n", target);
			exit(1);
		}
		to->sin_family = hp->h_addrtype;
		bcopy(hp->h_addr, (caddr_t)&to->sin_addr, hp->h_length);
		strncpy(hnamebuf, hp->h_name, sizeof(hnamebuf) - 1);
		hostname = hnamebuf;
	}

	/* Let the user know what's going on */
	if (!(options & F_QUIET)) {
#if defined(USE_ETHERLIB) || defined(USE_DLPI)
		if (!(options & F_NO_ETHER))
		printf("faking from: %s\n",pr_addr(fgw->sin_addr));
#endif
		printf(" sending to: %s  port: %i - %i\n",pr_addr(to->sin_addr),
			start_sport,end_sport);
		printf(" cant reach: %s  port: %i - %i\n",pr_addr(from->sin_addr),
			start_dport,end_dport);
	}


	/* Send the packets */
	for(dport=start_dport;dport<=end_dport;dport++) {
	for(sport=start_sport;sport<=end_sport;sport++)
		do_packet(whereto,to->sin_addr,sport,from->sin_addr,dport,
			fgw->sin_addr);
		if (options & F_BOTH_WAYS )  /* and reverse if desired */
			do_packet(wherefrom,from->sin_addr,dport,to->sin_addr,sport,
				fgw->sin_addr);
		usleep(minor_interval);
		}
	} /* end of arguments while loop */

	if (--npackets) { usleep(major_interval*1000); }
	} /* end of npackets while loop */
	exit(0);
}
