/*
 * network.c 
 *     Network library code.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>

#include "network.h"

/*
 * Socket table which stores whether a socket is valid or not.
 * 1: currently open
 * 0: currently closed or not being checked
 */
static char sockets[256];

/*
 * Get a tcp socket and return it
 */
int 
get_tcp_socket() {
    int sock;

#ifdef DEBUG
    printf("get_tcp_socket(): entered\n");
#endif /* DEBUG */

/*
 * Grabs next available socket of protocol family AF_INET and of type 
 * SOCK_STREAM.  This is the tcp protocol.
 */
    sock = socket(AF_INET, SOCK_STREAM, 0);

#ifdef DEBUG
    printf("get_tcp_socket(): got socket(%d)\n", sock);
#endif /* DEBUG */

    return (sock);
}

/*
 * Get a udp socket and return it
 */
int 
get_udp_socket() {
    int sock;

/*
 * Grabs next available socket of protocol family AF_INET and of type 
 * SOCK_DGRAM.  This is the udp protocol.
 */
    sock = socket(AF_INET, SOCK_DGRAM, 0);

    return (sock);
}

/*
 * Connects a socket to another socket specified by the sockaddr_in
 */
int 
connect_tcp_socket(int sock, struct sockaddr_in *to) {
    int out;    /* Return value */
    int len;    /* Used to store size of sockaddr_in structure */

    out = 0;    /* Default return value is 0. Fail */

    len = sizeof(struct sockaddr_in);

    out = connect(sock, (struct sockaddr *)to, len);

    return (out);
}

/*
 * Binds tcp socket to what is specified in to
 */
int 
bind_tcp_socket(int sock, struct sockaddr_in *to) {
    int out;    /* Return value */
    int len;    /* Used to store size of sockaddr_in structure */

#ifdef DEBUG
    printf("bind_tcp_socket: entered\n");
#endif /* DEBUG */
 
    len = sizeof(struct sockaddr_in);

    out = bind(sock, (struct sockaddr *)to, len);

#ifdef DEBUG
    printf("bind_tcp_socket: done\n");
#endif /* DEBUG */
    return (len);
}

/*
 * Initializes the sockaddr_in structure
 */
int
init_sockaddr(struct sockaddr_in *to, int port, unsigned long addr) {

#ifdef DEBUG
    printf("init_sockaddr: entered\n");
#endif /* DEBUG */

    /*
     * Zeroes out to
     */
    bzero((char *) to, sizeof(struct sockaddr_in));

    to->sin_family = AF_INET;    	/* Using INET family of protocols */
    to->sin_addr.s_addr = addr;		/* Setting address */
    to->sin_port = htons(port);		/* Setting port */

#ifdef DEBUG
    printf("init_sockaddr: exited\n");
#endif /* DEBUG */
}

/*
 * sends data out a given socket
 */
int
send_data(int sock, char *data) {
    int out;	/* return value */
    int len;	/* Used to store the length of data */

#ifdef DEBUG
    printf("send_data: entered\n");
    printf("send_data: sending to socket: %d\n", sock);
#endif /* DEBUG */

    len = strlen(data);

    out = write(sock, data, len);

#ifdef DEBUG
    printf("send_data: done\n");
#endif /* DEBUG */

    return (out);
}

/*
 * Initializes the socket table by setting all entries to 0
 */
void
init_sockets() {
int i;

#ifdef DEBUG
    printf("init_sockets(): entered\n");
#endif /* DEBUG */

for( i = 0; i < 256; i++)
    sockets[i] = 0;

#ifdef DEBUG
    printf("init_sockets(): finished\n");
#endif /* DEBUG */
}

/*
 * Removes a socket from the socket table and closes it.
 */
int
close_socket(int sock) {

#ifdef DEBUG
    printf("close_socket(): entered\n");
    printf("close_socket(): sockets[%d]: %d\n", sock, sockets[sock]);
#endif /* DEBUG */

if(sockets[sock] == 0)
    return (0); /* Already closed */
sockets[sock] = 0;
close(sock);

#ifdef DEBUG
    printf("close_socket(): finished\n");
    printf("close_socket(): sockets[%d]: %d\n", sock, sockets[sock]);
#endif /* DEBUG */

return (1);
}

/*
 * Checks to see if any data is coming in from any of the sockets
 * flag = 1, specifies server, flag = 0 specifies client.
 */
int
get_data(int sockfd, char *data, int len, int flag) {
    int out;			/* return value */
    int i, j;			/* dummy variables */
    fd_set fdvar, 		/* Master fd structure */
           fdread,		/* Stores read events */
           fdwrite, 		/* Stores write events */
           fdexcept;		/* Stores exception events */
    struct timeval timeout;	/* Stores select timeout value */
    struct sockaddr_in cliaddr; /* Stores client information */
    int clilen;			/* Stores length of cliaddr */
    static int sockmax = 4; /* 0,1,2 are stdin/out/err and 3 is sockfd */

#ifdef DEBUG
    printf("get_data(): entered\n");
#endif /* DEBUG */

    sockets[sockfd] = 1;	/* Enables listen socket in socket table */

    FD_ZERO(&fdvar);		/* Zeroes out the fdvar structure */

    for(i = sockfd; i < sockmax; i++)	/* Sets fdvar bit field to include */
        if(sockets[i])			/* all valid sockets */
            FD_SET(i, &fdvar);		

    timeout.tv_sec = 5;			/* Sets timeout to 5 seconds */
    timeout.tv_usec = 0;		/* Perhaps this should get passed */
					/* in so it is program selectible? */
    out = 0;				/* Default return value */

    for(i = 0; i < len; i++)		/* Zeroes out data to make sure */
        data[i] = '\0';			/* it is written to from the */
					/* beginning */

    /* 
     * Copy fdvar into fdread 
     */
    bcopy((char *) &fdvar, (char *) &fdread, sizeof(fd_set));
    /*
     * Zero out fdexcept and fdwrite as they are not checked for
     */
    FD_ZERO(&fdexcept);
    FD_ZERO(&fdwrite);
    
    /*
     * Select waits timeout to see if any events occur up to sockmax + 1
     * using the bitfields of fdread, fdwrite, & fdexcept.
     * I HIGHLY recommend looking at the man page of this call.
     */
    i = select(sockmax + 1, &fdread, &fdwrite, &fdexcept, &timeout);
     
    if (i > 0) {		/* If i > 0 then an event occurred */

    /*
     * Checks if listening socket has data waiting.
     * If so then there is a new connection and it must be accepted.
     * The accepted socket is then placed on the active list of connections.
     * Also, if the accepted socket is of a number greater than or equal
     * to sockmax, then sockmax is increased by one.
     * Return value of -5 is sent back to specify new connection, and 
     * data[0] is set to the accepted socket.
     */
    if(FD_ISSET(sockfd, &fdread) && (flag == 1)) {
        clilen = sizeof(cliaddr);
        len = accept(sockfd, (struct sockaddr *)&cliaddr, (int *)&clilen); 
        if(len >= sockmax)
            sockmax = sockmax + 1;
        out = -5; /* New Connection */
        data[0] = len;
        sockets[len] = 1;

#ifdef DEBUG
    printf("get_data(): got new connection\n");
#endif /* DEBUG */
	goto done;
	}

        /*
         * Check for data on any of the other valid sockets. 
         * As soon as valid data is found it returns it along with the
         * the socket it came from.
         */
        for(j = 0; j < sockmax + 1; j++) { 
            if(sockets[j] == 1) {
                if(FD_ISSET(j, &fdread)) {
                    out = j;
                    read(j, data, len);
                    goto done;
                }

            }
        }
}

done:

#ifdef DEBUG
    printf("get_data(): finished\n");
#endif /* DEBUG */

    return (out);
}
