/*
 * fastcracker.c
 *
 * main file of fastcracker project
 * cracks faster than tunder
 *
 * (c) Olivier Daigle, 1997
 * beagle@mediom.qc.ca
 *
 */


#include <stdio.h>
#include <stdlib.h>
#include <pwd.h>
#include <string.h>
#include <getopt.h>
#include <ctype.h>
#include <time.h>

#include "chain.h"
#include "modify.h"
#include "message.h"
#include "deslib/des.h"

#define VERSION 1.5

struct data
{
  FILE *passwd, *output, *error, *word;
  char *fpasswd, *foutput, *ferror, *fword, *skip;
  uint dual, modify, beep, recover, no_passwd, is_skip, only_passwd;
};

void parse_cmdline(uint, char**, struct data*);
void parse_passwd(struct data*, struct Header*);
struct Header* sort_passwd(struct data*, struct Header*);
unsigned long int get_memory_usage(struct Header*);
void final_out(struct data*, struct Header*);
void crack(struct data*, struct Header*);
void crack_with_word(struct data*, struct Header *);
void crack_with_passwd(struct data*, struct Header*);
void calibrat(struct Header*);
void test_password(char*, struct data*, struct Header*);
void test_password_with_modifiers(char*, struct data*, struct Header*);
int modify_entry(uchar*);
uchar* read_word(FILE*, char*, uint);

void main(int argc, char **argv)
{
  struct Header *list=new_list();  
  struct data *stuff=(struct data*)malloc(sizeof(struct data));
  
  printf("\n");
  
  parse_cmdline(argc,argv,stuff);
  
  parse_passwd(stuff, list);
  
  list=sort_passwd(stuff, list);
  get_memory_usage(list);
  
  calibrat(list);
  
  crack(stuff, list);

  printf("\n\n");
}


void calibrat(struct Header *list)
{
  struct Fcb *tmp, *tmp2;
  time_t t, u;
  unsigned long int loop;
  uint salts, exit_flag;
  
  printf("\nDetermining crack speed...\n");
  
  for(u=t=time(&t); u==time(&t););  /* synchronise with seconds */
  
  for(loop=0, u=t=time(&t); u==time(&t); loop++)
    crypt("az", "aaaaaaaa");
  
  for(tmp=list->first, salts=0; tmp; salts++)  /* get amount of different of salts */
  {
    for(tmp2=tmp, exit_flag=0; !exit_flag && tmp2;)  /* seek until next different salt */
    {
      if(strcmp(tmp->salt, tmp2->salt))
        exit_flag=1;
      else
        tmp2=tmp2->next;
    }  
      
        
   tmp=tmp2;
  }
  
  salts--;
    
  printf("System performance: %lu crypts/sec\n", loop);
  printf("Cracking %lu users with %d different salts...\n  Average speed should be %lu trys/sec\n", list->longeur, salts, loop*(list->longeur/salts));
  
}
  

void crack(struct data *stuff, struct Header *list)
{
  if(!stuff->no_passwd)
  {
    printf("\nNow cracking using password file...\n");
    crack_with_passwd(stuff, list);
  }
  
  if(stuff->only_passwd)
    return;
  
  if(stuff->word)
  {
    printf("\nNow cracking using dictionnary...\n");
    crack_with_word(stuff, list);
  }
  
  
}    

/*
 * void crack_with_passwd(struct data*, struct Header*);
 *
 * crack using fields in password file
 *
 */

void crack_with_passwd(struct data *stuff, struct Header *list)
{
  struct Fcb *tmp;
  char *real_name, *tmp_name, *memory_pos, letter;
  uint exit_flag;
  int count;
  
  test_password("", stuff, list);  /* test for NULL passwords */
  
  for(tmp=list->first; tmp; tmp=tmp->next)
  {
    test_password_with_modifiers(tmp->pw->pw_name, stuff, list);  /* test with user's login name */
    
    memory_pos=tmp_name=real_name=(char*)malloc(sizeof(char)*(strlen(tmp->pw->pw_gecos)+1));
    strcpy(real_name, tmp->pw->pw_gecos);
    
    for(exit_flag=count=0; !exit_flag; tmp_name++, count++)  /* test with user's real name */
    {
      if(!(*tmp_name))
      {
        test_password_with_modifiers(real_name, stuff, list);
        exit_flag=1;
      }
        
      else if(((*tmp_name)==0x20) || (count==7))  /* space */
      {
        letter=*tmp_name;
        *tmp_name=0;
        test_password_with_modifiers(real_name, stuff, list);
        real_name=tmp_name+1;
        *tmp_name=letter;
        if(count==7)
          while((*(++tmp_name+1)) && ((*(tmp_name+1)) != 0x20));  /* seek to next word */ 
        count=-1;
      }
    }
    free(memory_pos);
  
  }   
    
}

void test_password(char *word, struct data *stuff, struct Header *list)
{
  struct Fcb *tmp, *tmp2;
  char *crypted_passwd;
  
  if(stuff->is_skip)
    if(!strncmp(word, stuff->skip, strlen(stuff->skip)))
      return;
  
  for(tmp=list->first; tmp->next;)
  {
    crypted_passwd=crypt(word, tmp->salt);
    
    tmp2=tmp;
    
    do
    {
      if(!tmp2->is_done)
      {
        if(!strcmp(crypted_passwd, tmp2->pw->pw_passwd))  /* found a password */
        {
          if(stuff->beep)
            putchar('\a');
          
          if(stuff->dual)
            printf("%s:%s:%d:%d:%s:%s:%s\n", tmp2->pw->pw_name, word, tmp2->pw->pw_uid, tmp2->pw->pw_gid, tmp2->pw->pw_gecos, tmp2->pw->pw_dir, tmp2->pw->pw_shell);
            
          fprintf(stuff->output, "%s:%s:%d:%d:%s:%s:%s\n", tmp2->pw->pw_name, word, tmp2->pw->pw_uid, tmp2->pw->pw_gid, tmp2->pw->pw_gecos, tmp2->pw->pw_dir, tmp2->pw->pw_shell);
          fflush(stuff->output);
          tmp2->is_done=1;
        }
      }
      
      tmp2=tmp2->next;
    
    } while(!strcmp(tmp2->salt, tmp->salt));
    
    tmp=tmp2;
    
  }
}

void test_password_with_modifiers(char *word, struct data *stuff, struct Header *list)
{
  static uint (*modifiers[9])(char*, uint*);  /* increase size of it if you add more functions */
  static uint is_first=1;
  
  char *tmp_word;
  uint count=0, modifiers_count=0;
  
  if(stuff->is_skip)
    if(!strncmp(word, stuff->skip, strlen(stuff->skip)))
      return;
  
  if(is_first)  /* do this only once */
  {
    is_first=0;
    switch(stuff->modify)
    {
      case 4:
        modifiers[is_first++]=&add_signs;
        
      case 3:
        modifiers[is_first++]=&walking_lower;
        modifiers[is_first++]=&walking_upper;
        
      case 2:
        modifiers[is_first++]=&mirror;
        modifiers[is_first++]=&remove_letter;
        modifiers[is_first++]=&reverse;
        
      case 1:
        modifiers[is_first++]=&change_limit;
      
      case 0:
        modifiers[is_first++]=&nothing;
        break;
    }
    
    modifiers[is_first]=NULL;
    is_first=0;
  }
  
  tmp_word=(char*)malloc(sizeof(char)*9);  /* reset tmp_word pointer position */
  
  strcpy(tmp_word, word);  /* set temporary word */
  
  for(modifiers_count=0;modifiers[modifiers_count]; modifiers_count++) /* while end of modifications are not reach */     
  {
    for(count=0;(*modifiers[modifiers_count])(tmp_word, &count);count++)  /* modify word */
    {
      test_password(tmp_word, stuff, list);
      strcpy(tmp_word, word);  /* reset word */
    }
    strcpy(tmp_word, word);  /* reset word */
  }
}

/*
 * void crack_with_word(struct data*, struct Header*);
 *
 * crack passwords with a dictionnary
 *
 */

void crack_with_word(struct data *stuff, struct Header *list)
{
  FILE *whereweare;
  char *word=NULL;
  
  while((word=read_word(stuff->word, word, 8)) != NULL)  /* read word in dict */
  {
    test_password_with_modifiers(word, stuff, list);  /* reset tmp_word pointer position */
    
    if((whereweare=fopen(".recovery", "w")))
    {
      fprintf(whereweare, "%s\n", word);
      fclose(whereweare);
    }
    
    free(word);
  }
}

/*
 * uchar* read_word(struct data*, char *word);
 *
 * reads an entry from the dictionnary stuff->word
 * and returns a pointer to it 
 *
 */

uchar* read_word(FILE *fptr, char *word, uint number_letters)
{
  uint letter, comment_flag=0, pos=0, exit_flag=0;
  
  if(number_letters)
    word=(char*)malloc(sizeof(char)*(number_letters+1));
  else
  {
    word=(char*)malloc(sizeof(char)*256);
    number_letters=255;
  }
  while(!exit_flag)  /* read word in dict */
  {
    letter=getc(fptr);
    switch(letter)
    {
      case EOF:  /* End Of File reacehd */
        if(pos)
          exit_flag=1;
        else
          return NULL;  /* no more words found */
        break;
        
      case 0x0a:
        if(pos)  /* not an empty word */
          exit_flag=1;
        else  /* we have not find a word yet */
          comment_flag=0;
        break;
                
      case 0x0d:
        comment_flag=0;
        break;
      
      case '#':
        if(!pos)
          comment_flag=1;  /* remove lines beginning by '#' */  
                         /* no break!! */
      default:  /* write letter to array */
        if(!comment_flag)
        {
          if (pos <= (number_letters-1))
            word[pos++]=letter;
          else  /* trucante words to number_letters letters */
            while(!exit_flag)
              switch(getc(fptr))  /* seek 'till end of line or file */
              {
                case 0x0a:
                case EOF:
                  exit_flag=1;
                  break;
              }                
        } 
        break;
    }
  }
  word[pos]=0;
  return word;
}


/*
 * void sort(struct Header*);
 *
 * sort data in list by salt and remove duplicate entries
 *
 */

struct Header* sort_passwd(struct data *stuff, struct Header *list)
{
  FILE *fptr;
  struct Fcb *tmp, *tmp2;
  struct Fcb *sorted_fcb;
  struct Header *sorted_list=new_list();
  uint pos, exit_flag=0;
  char *entry=NULL;
  
  printf("\nSorting entries...\n");
  
  if(strcmp(stuff->foutput, "stdout")  && stuff->recover)  /* if recover, remove cracked entries */
  {
    fptr=fopen(stuff->foutput, "r");
    while((entry=read_word(fptr, entry, 81))!=NULL)
    {
      for(pos=0; entry[pos]!=':' && entry[pos]; pos++);
      
      entry[pos]=0;  /* terminate string */
      for(tmp=list->first; !exit_flag && tmp;)
      {
        if(!strcmp(tmp->pw->pw_name, entry))
          exit_flag=1;
        else
          tmp=tmp->next;
      }
      
      exit_flag=0;
      /*free(entry);*/
        
      if(tmp)
      {
        fprintf(stuff->error, "\nRemoving %s... already cracked!", tmp->pw->pw_name);
        tmp->is_done=1;
      }
    }
    fclose(fptr);
  }
        
  sorted_fcb=(struct Fcb*)malloc(sizeof(struct Fcb));  /* creates an empty(garbaged) fcb */
  sorted_list->first=sorted_fcb;
  
  for(tmp=list->first; tmp; tmp=tmp->next)
  {
    for(tmp2=tmp; tmp2; tmp2=tmp2->next)
    {
      if(!tmp2->salt_done)
      {
        if(!strcmp(tmp->salt, tmp2->salt))  /* salts match! */
        {
          if(tmp != tmp2) 
            if((!strcmp(tmp->pw->pw_name, tmp2->pw->pw_name)) && (!strcmp(tmp->pw->pw_passwd, tmp2->pw->pw_passwd))) /* duplicate found */ 
            {
              fprintf(stuff->error, "\nDuplicate found for %s... deleting it.", tmp->pw->pw_name);
              tmp2=remove_node_by_addr(list,tmp2);
            }            
            
          if(!tmp2->salt_done)  /* same salt but not a duplicate */
          {
            sorted_fcb->next=(struct Fcb*)malloc(sizeof(struct Fcb));
            sorted_fcb=sorted_fcb->next;
         
            sorted_fcb->pw=tmp2->pw;
            sorted_fcb->salt=tmp2->salt;
            sorted_fcb->salt_done=0;
            sorted_fcb->is_done=tmp->is_done;
            sorted_list->longeur++;
       
            tmp2->salt_done=1;
          }
          
          if(tmp != tmp2)
            tmp2=remove_node_by_addr(list, tmp2);
        }
      }
    }    
  }
  remove_chain(list);
  
  remove_node_by_addr(sorted_list, sorted_list->first);  /* first entry is garbage */
  
  sorted_list->last=sorted_fcb;
  sorted_fcb->next=NULL;  /* terminates chain */
    
  list=sorted_list;  /* makes the list point to the sorted chain */
  
  return(list);
}  
  

unsigned long int get_memory_usage(struct Header *list)
{
  struct Fcb *tmp;
  unsigned long int memory;
  
  memory=sizeof(struct Header);
  
  for(tmp=list->first; tmp->next; tmp=tmp->next)
  {
    memory+=sizeof(struct Fcb);
    memory+=sizeof(char)*(strlen(tmp->pw->pw_name)+1);
    memory+=sizeof(char)*(strlen(tmp->pw->pw_passwd)+1);
    memory+=sizeof(char)*(strlen(tmp->pw->pw_gecos)+1);
    memory+=sizeof(char)*(strlen(tmp->pw->pw_dir)+1);
    memory+=sizeof(char)*(strlen(tmp->pw->pw_shell)+1);
    memory+=sizeof(char)*(strlen(tmp->salt)+1);
    memory+=sizeof(int)*2;
  }
    
  printf("%lu nodes, %lu bytes used\n", list->longeur, memory);
  
  return memory;
}

void final_out(struct data *stuff, struct Header *list)
{
  struct Fcb *tmp;

  for(tmp=list->first; tmp->next; tmp=tmp->next)
    fprintf(stuff->output, "%s:%s:%d:%d:%s:%s:%s\n", tmp->pw->pw_name, tmp->pw->pw_passwd, tmp->pw->pw_uid, tmp->pw->pw_gid, tmp->pw->pw_gecos, tmp->pw->pw_dir, tmp->pw->pw_shell);

  fflush(stuff->output);
/*  if(stuff->output != stdout)
    fclose(stuff->output); */

}

void parse_passwd(struct data *stuff, struct Header *list)
{
  struct Fcb *tmp=NULL;
  uchar tmp2[80];
  uint letter, exit_flag=0;
  uint field=0,pos=0,i;
  
  printf("\nReading %s...\n", stuff->fpasswd);
  
  while(!exit_flag)
  {
    letter=getc(stuff->passwd);
    
    if(!field)
    {
      tmp=create_node(list);
      field++;
    }
    
    switch(letter)
    {
      case 0x0d:
        break;
      
      case EOF:
        exit_flag=1;
      
      case 0x0a:  /* end of a passwd line */
        if(field>=7)
        {
          tmp2[pos]=0;
          if((strlen(tmp->pw->pw_passwd) != 13) && strlen(tmp->pw->pw_passwd))
          {
            fprintf(stuff->error, "\n%s is having a locked password '%s'... deleting it", tmp->pw->pw_name, tmp->pw->pw_passwd);
            fflush(stuff->error);
            remove_node_by_addr(list, tmp);
          }
          else
          {
            tmp->pw->pw_shell=(uchar*)malloc(sizeof(uchar)*(strlen(tmp2)+1));
            strcpy(tmp->pw->pw_shell, tmp2);
          }
        }
        else
          remove_node_by_addr(list, tmp);
          
        field=0;
        pos=0;
        break;
        
      case ':':
        tmp2[pos]=0;
        switch(field)
        {
          case 1:
            tmp->pw->pw_name=(uchar*)malloc(sizeof(uchar)*(strlen(tmp2)+1));
            strcpy(tmp->pw->pw_name, tmp2);
            break;
          
          case 2:
            tmp->pw->pw_passwd=(uchar*)malloc(sizeof(uchar)*(strlen(tmp2)+1));
            strcpy(tmp->pw->pw_passwd, tmp2);

            tmp->salt=(uchar*)malloc(sizeof(uchar)*3);
            for(i=0; i<=1; i++)
              *(tmp->salt+i)=tmp2[i];
            *(tmp->salt+2)=0;
            break;
          
          case 3:
            tmp->pw->pw_uid=atoi(tmp2);
            break;
          
          case 4:
            tmp->pw->pw_gid=atoi(tmp2);
            break;
          
          case 5:
            tmp->pw->pw_gecos=(uchar*)malloc(sizeof(uchar)*(strlen(tmp2)+1));
            strcpy(tmp->pw->pw_gecos, tmp2);
            break;
          
          case 6:
            tmp->pw->pw_dir=(uchar*)malloc(sizeof(uchar)*(strlen(tmp2)+1));
            strcpy(tmp->pw->pw_dir, tmp2);
            break;
          
          default:
            fprintf(stuff->error, "\nIn %s, too many fields encountered for %s... but keeping it.", stuff->fpasswd, tmp->pw->pw_name);
            fflush(stuff->error);
        }
        field++;
        pos=0;
        break;
    
      default:
        if(pos <= 79)
          tmp2[pos++]=letter;

        else if(tmp2[pos-1])
        {
          fprintf(stuff->error, "\nA field in %s a field is longer than 79 characters and has been trucanted.", stuff->fpasswd);
          fflush(stuff->error);
          tmp2[pos-1]=0;
        }
        break;          
    }
  }
} 

void parse_cmdline(uint argc, char **argv, struct data *stuff)
{
  FILE *stats;
  uint c=0;
  extern char *optarg;
  char *word=NULL, *tmp;
  
  stuff->passwd=stuff->output=stuff->error=stuff->word=NULL;
  stuff->fpasswd=stuff->foutput=stuff->fword=stuff->ferror=NULL;
  stuff->skip=NULL;
  stuff->only_passwd=stuff->no_passwd=stuff->dual=stuff->beep=stuff->recover=stuff->is_skip=0;
  stuff->modify=2;
  
  printf("\nParsing command line");
  
  while(c!=EOF)
  {
    c=getopt(argc,argv,"s:w:e:o:p:l:bdhnrvy?");
    putchar('.');
    switch(c)
    {
      case 'b':  /* beep --> on */
        stuff->beep=1;
        break;
        
      case 'd':  /* dual output --> on */
        stuff->dual=1;
        break;
      
      case 'e':  /* open error file */
        if(!(stuff->error=fopen(optarg,"w")))
        {
          fprintf(stderr, "\nCan't open %s for writing\n", optarg);
          exit(1);
        }
        stuff->ferror=(uchar*)malloc(sizeof(uchar)*(strlen(optarg)+1));
        strcpy(stuff->ferror, optarg);
        break;
      
      case 'l':  /* don't modify */
        stuff->modify=atoi(optarg);
        if(stuff->modify>4)
          stuff->modify=4;
        break;
              
      case 'n':  /* don't crack using passwd file */
        stuff->no_passwd=1;
        break;
      
      case 'o':  /* open output file */
        if(!(stuff->output=fopen(optarg,"w")))
        {
          fprintf(stderr,"\nCan't open %s for writing\n", optarg);
          exit(1);
        }
        
        stuff->foutput=(uchar*)malloc(sizeof(uchar)*(strlen(optarg)+1));
        strcpy(stuff->foutput, optarg);
        break;
        
      case 'p':  /* select passwd file */
        if(!(stuff->passwd=fopen(optarg,"r")))
      	{
          fprintf(stderr,"\nCan't open %s for reading\n", optarg);
          exit(1);
        }
        
        stuff->fpasswd=(uchar*)malloc(sizeof(uchar)*(strlen(optarg)+1));
        strcpy(stuff->fpasswd, optarg);
        break;
      
      case 'h':  /* HELP!! */
      case '?':
        printf("\nUsage: fastcracker [-bdh?v] [-e ERROR_FILE] [-o OUTPUT_FILE] [-p PASSWORD_FILE] [-s SKIP_WORD] [-w WORD_FILE]");
        printf("\n  -b    beep when a password is found");
        printf("\n  -d    print results to both OUTPUT file and stdout");
        printf("\n  -h -? show this screen");
        printf("\n  -l    set crack level");
        printf("\n  -n    don't crack using fields of passwd file");
        printf("\n  -r    recover a crashed job");
        printf("\n  -v    show version");   
        printf("\n  -y    crack only using passwd file");
        printf("\n");
        printf("\n  -e ERROR_FILE     redirect error logs to ERROR_FILE");
        printf("\n  -o OUTPUT_FILE    redirect output (cracked passwords) to OUTPUT_FILE");
        printf("\n  -p PASSWORD_FILE  use  PASSWORD_FILE as passwd-like input file"); 
        printf("\n  -s SKIP_WORD      don't test words beginning by SKIP_WORD");
        printf("\n\n");
        exit(0);
        
      case 'r':  /* recover */
        stuff->recover=1;
        
        if(!(stats=fopen(".fastcracker_options", "r")))
        {
          fprintf(stderr, "\nCan't open .fastcracker_options for reading\nCan't recover!\n");
          exit(1);
        }
        
        word=read_word(stats, word, 0);  /* first entry is passwd file */
        if(!(stuff->passwd=fopen(word,"r")))
        {
           fprintf(stderr,"\nCan't open %s for reading\n", word);
           exit(1);
        }
        stuff->fpasswd=(uchar*)malloc(sizeof(uchar)*(strlen(word)+1));
        strcpy(stuff->fpasswd, word);
        free(word);
        
        word=read_word(stats, word, 0);  /* second entry is output file */
        if(!(stuff->output=fopen(word,"a")))
          fprintf(stderr,"\nCan't open %s for writing\n  Output will go to stdout", word);
        else
        {
          stuff->foutput=(uchar*)malloc(sizeof(uchar)*(strlen(word)+1));
          strcpy(stuff->foutput, word);
        }
        free(word);
        
        word=read_word(stats, word, 0);  /* third entry is error file */
        if(!(stuff->error=fopen(word,"a")))
          fprintf(stderr,"\nCan't open %s for writing\n Error will go to stderr", word);
        else
        {
          stuff->ferror=(uchar*)malloc(sizeof(uchar)*(strlen(word)+1));
          strcpy(stuff->ferror, word);
        }
        free(word);
        
        word=read_word(stats, word, 0);  /* fourth entry is word file */
        if(!(stuff->word=fopen(word, "r")))
        {
          fprintf(stderr, "\nCan't open %s for reading\n", word);
          exit(1);
        }
        if(!strcmp(word, " internaly created dictionnary"))
          stuff->word=NULL;
        else
        {
          stuff->fword=(uchar*)malloc(sizeof(uchar)*(strlen(word)+1));
          strcpy(stuff->fword, word);
        }
        free(word);  
        
        word=read_word(stats, word, 1);  /* read dual */
        stuff->dual=atoi(word);
        free(word);
        
        word=read_word(stats, word, 1);  /* read beep */
        stuff->beep=atoi(word);
        free(word);
        
        word=read_word(stats, word, 1);  /* read modify */
        stuff->modify=atoi(word);
        free(word);
        
        word=read_word(stats, word, 1);  /* read no_passwd */
        stuff->no_passwd=atoi(word);
        free(word);
        
        word=read_word(stats, word, 1);  /* read is_skip  */
        if((stuff->skip=atoi(word)))
        {
          free(word);
          word=read_word(stats, word, 0);
          stuff->skip=(char*)malloc(sizeof(char)*(strlen(word)+1));
          strcpy(stuff->skip, word);
        }  
        free(word);
        fclose(stats);
        
        if(!(stats=fopen(".recovery","r")))
          fprintf(stderr, "\nCan't open .recovery for reading\nCrack will have to start all over again");
        else  /* seek 'til recovery word in word file */
        {
          word=read_word(stats, word, 8);
          fclose(stats);
          for(tmp=(char*)malloc(1), *tmp=0; strcmp(tmp, word); tmp=read_word(stuff->word, tmp, 8))
            free(tmp);
        }
        
        break;
      
      case 's':
        stuff->is_skip=1;
        stuff->skip=(char*)malloc(sizeof(char)*(strlen(optarg)+1));
        strcpy(stuff->skip, optarg);
        break;
      
      case 'v':  /* version */
        printf("\nfastcracker version %.02f\nBy beagle@Aeternum.deathleaf.org\n", VERSION);
        exit(0);
        
      case 'y':
        stuff->only_passwd=1;
        break;
      
      case 'w':  /* word file */
        if(!(stuff->word=fopen(optarg,"r")))
        {
          fprintf(stderr, "\nCan't open %s for reading\n", optarg);
          exit(1);
        }
        
        stuff->fword=(uchar*)malloc(sizeof(uchar)*(strlen(optarg)+1));
        strcpy(stuff->fword, optarg);
        break;
    }
  }
  
  putchar('\n');
  
  if(!stuff->word)
    stuff->fword=" internaly created dictionnary";
  
  if(!stuff->error)
  {
    stuff->error=stderr;
    stuff->ferror="stderr";
  }
  
  if(!stuff->output)
  {
    stuff->output=stdout;
    stuff->foutput="stdout";
    if(stuff->dual)
      stuff->dual=0;
  }
  
  if(!stuff->passwd)
  {
    if(!(stuff->passwd=fopen("/etc/passwd","r")))
    {
      fprintf(stderr,"\nCan't open /etc/passwd for reading\n");
      exit(1);
    }
    stuff->fpasswd="/etc/passwd";
  }
  if(!(stats=fopen(".fastcracker_options", "w")))
  	fprintf(stderr, "Warinig! can't open .fastcracker_options... crash recovery will not work properly!\n");
  else
  {
    fprintf(stats, "%s\n%s\n%s\n%s\n%d\n%d\n%d\n%d\n%d\n%s", stuff->fpasswd, stuff->foutput, stuff->ferror, stuff->fword, stuff->dual, stuff->beep, stuff->modify, stuff->no_passwd, stuff->is_skip, stuff->is_skip?stuff->skip:"");
    fclose(stats);
  }
  
  printf("\nCommand-line options:\n");
  printf("\tPASSWORD-style file: %s\n\tOUTPUT file: %s\n\tERROR file: %s\n\tDICTIONNARY file: %s\n\tbeep is %s\n\tdual output is %s\n\tcrack level set to %d%s%s%s%s\n", stuff->fpasswd, stuff->foutput, stuff->ferror, stuff->fword, stuff->beep?"enabled":"disabled", stuff->dual?"enabled":"disabled", stuff->modify, stuff->recover?"\n\tusing recover mode":" ", stuff->is_skip?"\n\tskipping words beginning by ":" ", stuff->is_skip?stuff->skip:" ", stuff->no_passwd?"\n\tcrack will not use passwd file":"\n\tcrack will use the passwd file too");
}

