/*
 * xforward.c
 *
 * Program that acts as a x relay
 */

/*              Copyright 1992, 1993 Digital Equipment Corporation
 *                        All Rights Reserved
 *
 * Permission to use, copy, and modify this software and its documentation is
 * hereby granted only under the following terms and conditions.  Both the
 * above copyright notice and this permission notice must appear in all copies
 * of the software, derivative works or modified versions, and any portions
 * thereof, and both notices must appear in supporting documentation.
 *
 * Users of this software agree to the terms and conditions set forth herein,
 * and hereby grant back to Digital a non-exclusive, unrestricted, royalty-free
 * right and license under any changes, enhancements or extensions made to the
 * core functions of the software, including but not limited to those affording
 * compatibility with other hardware or software environments, but excluding
 * applications which incorporate this software.  Users further agree to use
 * their best efforts to return to Digital any such changes, enhancements or
 * extensions that they make and inform Digital of noteworthy uses of this
 * software.  Correspondence should be provided to Digital at:
 *
 *                      Director of Licensing
 *                      Cambridge Research Laboratory
 *                      Digital Equipment Corporation
 *                      One Kendall Square, Bld. 700
 *                      Cambridge, MA 02139  
 *
 * This software may be distributed (but not offered for sale or transferred 
 * for compensation except on systems manufactured by Digital Equipment 
 * Corporation) to third parties, provided such third parties agree to abide 
 * by the terms and conditions of this notice.  
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND DIGITAL EQUIPMENT CORP. DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS.   IN NO EVENT SHALL DIGITAL EQUIPMENT
 * CORPORATION BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <netdb.h>
#include <signal.h>
#include <errno.h>
#include <sys/param.h>
#include <syslog.h>
#include <arpa/inet.h>

#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <Xm/Xm.h>
#include <Xm/MessageB.h>

#include <sos.h>
#include <interface.h>

#define BS_X11_PORT 6000

extern int errno;

#define min(a,b) ((a) < (b) ? (a) : (b))
#define max(a,b) ((a) > (b) ? (a) : (b))

#define TRUE 1
#define FALSE 0

#define CANCEL                  4
#define OK                      7

#define MAXHOSTRULES	16

void usage(), doxmit(), readwrite();

void sigchld();

void alert_user_CB();

struct sockaddr_in forward_to;

struct pend_conn {
    pid_t child;
    int newsock;
    int done;
    struct pend_conn *next;
};

struct rwbuf {
    char readbuffer[BUFSIZ+5];
    char writebuffer[BUFSIZ+5];
    int rbytes;				/* #bytes active in readbuffer */
    int wbytes;				/* #bytes active in wbytes */
    int connto;				/* fd this one should be connected to */
    int wclose;				/* if set, close after finished writing */
};

struct hostrule {
    unsigned long addr;
    unsigned long mask;
};

fd_set rinit, winit;

int net16;

int AlertResponse = 0;
 
struct rwbuf *rwbuf;


main(argc, argv)
int argc;
char *argv[];
{
    struct hostent *hp;
    int idle_timeout_secs = 3600*12*4/3;	/* Four days */
    int empty_timeout_secs = 3600/3;		/* One hour */
    char myhostname[MAXHOSTNAMELEN];
    struct sockaddr_in local, dummy;
    struct hostrule okhosts[MAXHOSTRULES];
    unsigned int nokhosts = 0;
    unsigned short my_port;
    unsigned short disp_port;
    register int i;
    int lsock;
    int dummylen;
    int one = 1;
    int nready, nfds, nsessions = 0;
    fd_set readable, writable;
    int dtsize;
    Display *display;
    int nhosts;
    Bool state;
    char *hname;
    size_t len;
    pid_t child_pid;
    struct pend_conn *pend_head = NULL;
    int tcount = 0;
    char *disp_str = NULL;
    char *disphost = NULL;
    struct in_addr dispaddr;
    int terse = 0;

    if (argc < 3)
	usage();

    /* Get the current hostname, and make sure it's fully qualifed.*/
    if (sos_fqhname(myhostname, MAXHOSTNAMELEN) == NULL)
      {
	perror("sos_fqhname");
	exit(1);
      }

    for ( i = 1; i < argc; i++) {
	if (strcmp(argv[i],"-display") == 0) {
	    disp_str = argv[i+1];
	    i++;
        } else if (strcmp(argv[i], "-terse") == 0) {
            terse = 1;
        } else if (strcmp(argv[i], "-timeout") == 0) {
            idle_timeout_secs = atoi(argv[++i]) / 3;
        } else if (strcmp(argv[i], "-etimeout") == 0) {
            empty_timeout_secs = atoi(argv[++i]) / 3;
	} else if (strcmp(argv[i],"-allow") == 0) {
	    do {
	        char *host = argv[i+1];
	        char *mask;
		struct in_addr addrtmp;

		/* If we have a address mask, use it, otherwise it is a singleton */
		if (mask = strchr(host,'/')) {
		    *mask++ = '\0';


		    if (sos_getabyfoo(mask, &addrtmp) < 0) {
			fprintf(stderr, "Mask <%s> unknown\n",host);
			exit(1);
		    }
		    okhosts[nokhosts].mask = addrtmp.s_addr;
		} else {
		    okhosts[nokhosts].mask = inet_addr("255.255.255.255");
		}

		if (sos_getabyfoo(host,&addrtmp) < 0) {
		    fprintf(stderr, "Host <%s> unknown\n",host);
		    exit(1);
		}
		okhosts[nokhosts].addr = addrtmp.s_addr;

		if (nokhosts++ >= MAXHOSTRULES) {
		    fprintf(stderr, "Too many hosts:  only %d max\n",MAXHOSTRULES);
		    exit(1);
		}
		i++;
	    } while (argv[i+1] != NULL && argv[i+1][0] != '-');
	} else {
	    usage();
	}
    }

    /* if not set in argument list, get DISPLAY from environment variable */
    if (disp_str == NULL) {
	disp_str = getenv("DISPLAY");
	if (disp_str == NULL) {
	    fprintf(stderr,"Display not specified\n");
	    exit(1);
	}
    }
    len = strcspn(disp_str,":");
    hname = (char *)malloc(len+1); /* allocate space, including null */
    (void)strncpy(hname,disp_str,len);
    hname[len] = '\0';
    hp = gethostbyname(hname);
    if (!hp) {
	fprintf(stderr, "No such display host %s\n",hname);
	exit(1);
    }
    bzero(&local, sizeof(local));
    bzero(&forward_to, sizeof(forward_to));

    my_port = 6000;
    local.sin_port = htons(my_port);
    local.sin_family = AF_INET;
    /* assign a local display */
    if ((lsock = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
	perror("socket");
	exit(1);
    }
    if (setsockopt(lsock, SOL_SOCKET, SO_REUSEADDR,
		   &one, sizeof(one)) < 0) {
	perror("SO_REUSEADDR:");
	exit(1);
    }
    /* TCP NO_DELAY--prevent buffered I/O */
    {
      struct protoent *p;
      int one=1;

      if (!(p = getprotobyname("tcp")))
	{
	  perror("getproto");
	  exit(2);
	}
      if(setsockopt(lsock, p->p_proto, TCP_NODELAY, (char *)&one, sizeof(one)) < 0)
	{
	  perror("setsockopt");
	  exit(2);
	}
    }
    while ( bind(lsock, (struct sockaddr *)&local, sizeof(local)) < 0) {
	if (errno == EADDRINUSE) {
	    my_port++;
	    if (my_port > 6100) {
		fprintf(stderr,"all ports in use.\n");
		exit(1);
	    }
	    local.sin_port = htons(my_port);
	} else {
	    perror("bind");
	    exit(1);
	}
    }

    if (listen(lsock, SOMAXCONN) < 0) {
	perror("listen");
	exit(1);
    }

    disp_port = atoi(disp_str+len+1);
#ifdef PORT_NOT_UNSIGNED
    if (disp_port < 0) {
	fprintf(stderr, "display must be a non-negative integer!\n");
	exit(1);
    }
#endif /*PORT_NOT_UNSIGNED*/
    disp_port += 6000;
    if (disp_port < 6000 || disp_port > 6100) {
	fprintf(stderr, "display must be >= X11.0 and <= X11.100\n");
	exit(1);
    }

    if (nokhosts == 0) {
	fprintf(stderr, "must specify at least one allowed host.\n");
	exit(1);
    }

    forward_to.sin_port = htons(disp_port);
    forward_to.sin_family = AF_INET;
    bcopy(hp->h_addr, (char *)&forward_to.sin_addr, sizeof(forward_to.sin_addr));

    /* Check that we can open the destination display. */

    if ( (display = XOpenDisplay(disp_str)) == NULL) {
	fprintf(stderr,"Unable to open display at destination.\n");
	exit(1);
    }

    if (!terse)
       printf("display is ");
    printf("%s:%d\n", myhostname, my_port - 6000);
    fflush(stdout);

    if ((ntohl(forward_to.sin_addr.s_addr) & 0xff000000) == 0x10) {
	/* net 16 */
	net16 = 1;
    } else {
	net16 = 0;
    }

    /* clean up to conserve descriptors */
    close(0);
    close(1);
    close(2);
    openlog("xforward",LOG_PID,LOG_DAEMON);

    signal(SIGPIPE, SIG_IGN);

    dtsize = getdtablesize();

    /* set up buffers & sizes */
    rwbuf = (struct rwbuf *)calloc(dtsize, sizeof *rwbuf);
    if (!rwbuf) {
    	syslog(LOG_CRIT, "can't allocate buffers %m");
	exit(1);
    }
    /* -1 means not connected/invalid */
    while (dtsize--) {
	rwbuf[dtsize].connto = -1;
    }

    FD_ZERO(&rinit);
    FD_ZERO(&winit);
    FD_SET(lsock, &rinit);
    
    nfds = lsock + 1;
    while (1) {
	struct timeval timeout;
	struct pend_conn *cur,*prev;
	pid_t temppid;
#ifdef UNION_WAIT
	union wait wstatus;
#else /*UNION_WAIT*/
	int wstatus;
#endif /*UNION_WAIT*/
	int exitstat,wopts,newoutgoing;

	/* for each entry, call waitpid */
	for (cur = pend_head; cur != NULL; cur=cur->next ) {
	    wopts = WNOHANG;
	    temppid = waitpid(cur->child,&wstatus,wopts);
	    /* if there's no status, try next one */
	    if (temppid == 0) {
		continue;
	    }
	    if (WIFSIGNALED(wstatus)) {
	    	syslog(LOG_CRIT, "child died abnormally (%d)",wstatus);
		cur->done = TRUE;
		continue;
	    }
	    if (WIFEXITED(wstatus)) {
		exitstat = WEXITSTATUS(wstatus);
		if (exitstat == OK) {
		    if ((newoutgoing = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
			perror("socket");
			close(cur->newsock);
			cur->done = TRUE;
			continue;
		    }
		    /* TCP NO_DELAY--prevent buffered I/O */
		    {
		      struct protoent *p;
		      int one=1;

		      if (!(p = getprotobyname("tcp")))
			{
			  perror("getproto");
			  exit(2);
			}
		      if(setsockopt(newoutgoing, p->p_proto, TCP_NODELAY, (char *)&one, sizeof(one)) < 0)
			{
			  perror("setsockopt");
			  exit(2);
			}
		    }
		    if (connect(newoutgoing, (struct sockaddr *)&forward_to,
				sizeof(forward_to)) < 0) {
			perror("connect");
			close(newoutgoing);
			close(cur->newsock);
			cur->done = TRUE;
			continue;
		    }
		    
		    nfds = max(nfds, (cur->newsock)+1);
		    nfds = max(nfds, newoutgoing+1);
		    
		    rwbuf[(cur->newsock)].connto = newoutgoing;
		    rwbuf[newoutgoing].connto = (cur->newsock);
		    nsessions++;
		    
		    rwbuf[newoutgoing].rbytes = rwbuf[(cur->newsock)].rbytes = 0;
		    rwbuf[newoutgoing].wbytes = rwbuf[(cur->newsock)].wbytes = 0;
		    FD_SET(cur->newsock, &rinit);
		    FD_SET(cur->newsock, &winit);
		    FD_SET(newoutgoing, &rinit);
		    FD_SET(newoutgoing, &winit);
		    cur->done = TRUE;
		} else if (exitstat == CANCEL) {
		    close(cur->newsock);
		    cur->done = TRUE;
		} else {
		  syslog(LOG_CRIT, "Unknown exit status of child (%d)",exitstat);
		    close(cur->newsock);
		    cur->done = TRUE;
		}
	    }
	}
	/* clean up list */
	prev = (struct pend_conn *)NULL;
	cur = pend_head;
	while (cur != NULL) {
	    if (cur->done == TRUE) {
		if (prev == NULL) {
		    pend_head = cur->next;
		    free(cur);
		    cur = pend_head;
		} else {
		    prev->next = cur->next;
		    free(cur);
		    cur = prev->next;
		}
	    } else {
		prev = cur;
		cur = cur->next;
	    }
	}

	for (i = 0; i < nfds; i++)
	    if (rwbuf[i].wclose)
		FD_SET(i, &winit);

	timeout.tv_usec = 0;
	timeout.tv_sec = 3;

	readable = rinit;
	writable = winit;
	if ((nready = select(nfds, &readable, &writable, 0, &timeout)) == -1) {
	    if (errno == EINTR)
		continue;
	    perror("select");
	    exit(1);
	}
	if (nready == 0) {
	    /* each increment of tcount represents 3 seconds of idleness */
	    tcount++;
	    if (idle_timeout_secs > 0 && tcount > idle_timeout_secs) {
	    	syslog(LOG_NOTICE, "idle: connections timed out");
	        exit(1);
	    }
	    if (nsessions < 0)
	      nsessions = 0;
	    if (nsessions == 0 && empty_timeout_secs > 0 && tcount > empty_timeout_secs) {
	    	syslog(LOG_NOTICE, "empty: timed out");
	        exit(1);
	    }
	} else {
	    /* reset timeout counter if there is some activity */
	    tcount = 0;
	}

	for (i = 0; i < nfds && nready; i++) {
	    /* loop through descriptors */
	    if (FD_ISSET(i, &writable)) {
		int cc, leftover, connto;
		connto = rwbuf[i].connto;
		nready--;
		/* write what we can */
		if (rwbuf[i].wbytes) {
		    cc = write(i, rwbuf[i].writebuffer, rwbuf[i].wbytes);
		    if (cc == -1) {
			if (errno != EPIPE) {
			    syslog(LOG_CRIT, "fd %d: write: %m",i);
			}
			/* asynchrony on close probs? */
			FD_CLR(i, &winit);
			FD_CLR(i, &rinit);
			close(i);
			if (connto != -1) {	
			    FD_CLR(connto, &rinit);
			    FD_CLR(connto, &winit);
			    close(rwbuf[i].connto);
			    nsessions--;
			    rwbuf[connto].connto = -1;
			}
			rwbuf[i].connto = -1;
			continue;
		    } else {
			leftover = rwbuf[i].wbytes - cc;
			if (leftover) {
			    /* didn't write it all */
			    /* copy down */
			    bcopy(rwbuf[i].writebuffer + cc,
				  rwbuf[i].writebuffer,
				  leftover);
			    rwbuf[i].wbytes = leftover;
			} else
			    rwbuf[i].wbytes = 0; /* buffer empty */
			if (connto != -1 && rwbuf[connto].rbytes) {
			    /* more stuff to copy in */
			    copyfromto(connto, i);
			}
			if (!rwbuf[i].wbytes) {
			    /* nothing left to write */
			    if (rwbuf[i].wclose) {
				/* close after flushing */
				FD_CLR(i, &rinit);
				close(i);
				if (connto != -1)
				  nsessions--;
				rwbuf[i].connto = -1;
				rwbuf[i].wclose = 0;
			    }
			    FD_CLR(i, &winit);
			}
			if (connto != -1)
			/* since we wrote some, go look for more */
			    FD_SET(connto, &rinit);
		    }
		} else {
		    if (connto != -1 && rwbuf[connto].rbytes) {
			    /* more stuff to copy in */
			    copyfromto(connto, i);
			    if (rwbuf[i].wbytes)
				continue;
		    }
		    /* nothing to write at the moment */
		    if (rwbuf[i].wclose) {
			/* close after flushing */
			FD_CLR(i, &rinit);
			close(i);
			if (connto != -1)
			  nsessions--;
			rwbuf[i].connto = -1;
			rwbuf[i].wclose = 0;
		    }
		    /* nothing to write, so clear */
		    FD_CLR(i, &winit);
		}
	    }
	    if (FD_ISSET(i, &readable)) {
		nready--;
		/* something is readable */
		if (i == lsock) {
		    int newsock, i;
		    struct pend_conn *npc;
		    Display *dpy;
		    /* new connection ready */
		    dummylen = sizeof(dummy);
		    if ((newsock = accept(lsock, (struct sockaddr *)&dummy,
					  &dummylen)) < 0) {
			if (errno == EINTR)
			    continue;
			perror("accept");
			exit(1);
		    }
		    for (i = 0; i < nokhosts; i++) {
			if ((dummy.sin_addr.s_addr & okhosts[i].mask) ==
			    (okhosts[i].addr & okhosts[i].mask))
			  break;
		    }
		    if (i == nokhosts) {
			char *taddr = inet_ntoa(dummy.sin_addr);
			syslog(LOG_CRIT, "bad host connect from %s[%s]",
				sos_getnbyfoo(taddr,(char *)NULL,0),
			        taddr);
			close(newsock);
			continue;
		    }

		    /* Brimstone ACL checks */
#ifdef BRIMSTONE
		    if (getenv("BS_CALLED_FROM_TELNET"))
		      {
			static char *user = NULL;
			int ret;

			if (!user)
			  user = getenv("BSUSER");

			if (!disphost)
			  {
			    char *hname, *tmp, *dup = strdup(disp_str);

			    if (!(tmp = strchr(dup,':')))
			      {	/* Something is wrong */
				syslog(LOG_CRIT, "BRIMSTONE XFORWARD from telnet: no findable hostname (%s)",disp_str);
				exit(2);
			      }
			    *tmp = 0;

			    if (!(hname = sos_getnbyfoo(dup, NULL, 0)))
			      {
				syslog(LOG_CRIT, "BRIMSTONE XFORWARD from telnet: cannot lookup hostname (%s)",disp_str);
				exit(2);
			      }

			    if (sos_getabyfoo(dup, &dispaddr) < 0)
			      {
				syslog(LOG_CRIT, "BRIMSTONE XFORWARD from telnet: cannot get address (%s)",disp_str);
				exit(2);
			      }

			    disphost = strdup(hname);
			    free(dup);
			  }

			if (!user)
			  {	/* Something is wrong */
			    syslog(LOG_CRIT, "BRIMSTONE XFORWARD from telnet: no username from %s",disphost);
			    exit(2);
			  }

syslog(LOG_CRIT,"connection from host %s",inet_ntoa(dummy.sin_addr));
			ret = BS_CheckACL_Num(user, dummy.sin_addr, dummy.sin_port, dispaddr, BS_X11_PORT);
			if (ret != AUTH_SUCCEED)
			  {
			    syslog(LOG_CRIT, "BRIMSTONE AUTH from telnet: user:%s:%s:%d:%s:%d",
				   user, inet_ntoa(dummy.sin_addr), dummy.sin_port, disphost, BS_X11_PORT);
			    close(newsock);
			    continue;
			  }
		      }
#endif /*BRIMSTONE*/

		    /* TCP NO_DELAY--prevent buffered I/O */
		    {
		      struct protoent *p;
		      int one=1;

		      if (!(p = getprotobyname("tcp")))
			{
			  perror("getproto");
			  exit(2);
			}
		      if(setsockopt(newsock, p->p_proto, TCP_NODELAY, (char *)&one, sizeof(one)) < 0)
			{
			  perror("setsockopt");
			  exit(2);
			}
		    }

		    /* after check of allowed hosts, create popup on
		     * destination
		     */
		    /* fork here */
		    child_pid = fork();
		    if ( child_pid < 0) {
		    	syslog(LOG_CRIT, "Fork of child failed %m");
			exit(1);
		    }
		    if ( child_pid == 0 ) {
			/* we're the child, so create pop-up */
			int n=0;
			Arg arg[4];
			XtAppContext app_con;
			Widget topshell,alert,help;
			char dialog_message[200];
			XmString msg_str,msg_str2,msg_str3;

			XtToolkitInitialize();
			app_con = XtCreateApplicationContext();
			dpy = XtOpenDisplay(app_con,disp_str,(char *)NULL,"Xforward",
					    (XrmOptionDescRec *)NULL,(Cardinal)0,&argc,(char **)argv);
			topshell = XtAppCreateShell((char const *)NULL,"Xforward",applicationShellWidgetClass, dpy, (ArgList)NULL, (Cardinal)0);
			sprintf(dialog_message,"Allow X connection from %s ?",
				sos_getnbyfoo(inet_ntoa(dummy.sin_addr),(char *)NULL,0));
			msg_str = XmStringCreateSimple(dialog_message);
			XtSetArg(arg[n],XmNmessageString,msg_str); n++;
			msg_str2 = XmStringCreateSimple("Yes");
			XtSetArg(arg[n],XmNokLabelString,msg_str2); n++;
			msg_str3 = XmStringCreateSimple("No");
			XtSetArg(arg[n],XmNcancelLabelString,msg_str3); n++;
			XtSetArg(arg[n],XmNdefaultButtonType,XmDIALOG_CANCEL_BUTTON); n++;
			alert = XmCreateWarningDialog(topshell,"alert",arg,n);
			XmStringFree(msg_str); XmStringFree(msg_str2); XmStringFree(msg_str3);
			XtAddCallback(alert,XmNokCallback,(XtCallbackProc)alert_user_CB,(caddr_t) OK);
			XtAddCallback(alert,XmNcancelCallback,(XtCallbackProc)alert_user_CB,(caddr_t) CANCEL);
			help = XmMessageBoxGetChild(alert,XmDIALOG_HELP_BUTTON);
			XtUnmanageChild(help);
			AlertResponse = -1;
			XtManageChild(alert);
			while(AlertResponse == -1)
			{
			    XEvent event;
			    XtAppNextEvent(app_con,&event);
			    XtDispatchEvent(&event);
			}
			/* use value of AlertResponse as exit code */
			exit(AlertResponse);
		    }
		    /* add child to list of procs that parent must check
		       exit status of */
		    npc = (struct pend_conn *)malloc(sizeof(struct pend_conn));
		    npc->child = child_pid;
		    npc->newsock = newsock;
		    npc->done = FALSE;
		    npc->next = (struct pend_conn *)NULL;
		    /* add npc to beginning of list */
		    if (pend_head == NULL) {
			pend_head = npc;
		    } else {
			npc->next = pend_head;
			pend_head = npc;
		    }
		} else {
		    int cc, connto;
		    connto = rwbuf[i].connto;
		    /* normal fd is readable */
		    if (rwbuf[i].rbytes < BUFSIZ) {
			/* read what we have room for */
			cc = read(i, rwbuf[i].readbuffer + rwbuf[i].rbytes,
				  BUFSIZ-rwbuf[i].rbytes);
			if (cc == -1) {
			    syslog(LOG_CRIT, "fd %d: read: %m",i);
			    /* asynchrony on close probs? */
			    FD_CLR(i, &winit);
			    FD_CLR(i, &rinit);
			    if (connto != -1) {
				FD_CLR(connto, &rinit);
				FD_CLR(connto, &winit);
				close(i);
				nsessions--;
				rwbuf[connto].connto = -1;
			    }
			    close(connto);
			    rwbuf[i].connto = -1;
			    continue;
			} else if (cc == 0) {
			    /* closedown */
			    FD_CLR(i, &rinit);
			    FD_CLR(i, &winit);
			    close(i);
			    if (connto != -1) {
				/* set close after finishing write */
				rwbuf[connto].wclose = 1;
				rwbuf[connto].connto = -1;
				/* force a write cycle to clean up */
				FD_SET(connto, &winit);
				nsessions--;
			    }
			    rwbuf[i].rbytes = 0;
			    rwbuf[i].wbytes = 0;
			    rwbuf[i].connto = -1;
			    /* XXX what else */
			} else {
			    rwbuf[i].rbytes += cc;
			    /* try to put onto write buffer */

			    if (connto != -1)
				copyfromto(i, connto);
			    if (rwbuf[i].rbytes >= BUFSIZ) {
				/* buffer is full */
				FD_CLR(i, &rinit);
			    }
			}
		    }
		}
	    } 
	} /* for loop through descriptors */
    } /* while (1) */
}

void
usage()
{
    fprintf(stderr, "usage: xforward [-terse] [-timeout secs] [-display dispname] -allow host1 [host2 ... host16]\n");
    exit(1);
}

copyfromto(from, to)
int from, to;
{
    int ncopy;
    if (rwbuf[to].wbytes < BUFSIZ) {
	ncopy = min(rwbuf[from].rbytes,
		    BUFSIZ-rwbuf[to].wbytes);
	
	bcopy(rwbuf[from].readbuffer,
	      rwbuf[to].writebuffer + rwbuf[to].wbytes,
	      ncopy);
	rwbuf[to].wbytes += ncopy;
	FD_SET(to, &winit);
	if (ncopy == rwbuf[from].rbytes)
	    rwbuf[from].rbytes = 0;
	else {
	    bcopy(rwbuf[from].readbuffer + ncopy,
		  rwbuf[from].readbuffer,
		  rwbuf[from].rbytes - ncopy);
	    rwbuf[from].rbytes -= ncopy;
	}
	/* we have room */
	FD_SET(to, &rinit);
    }
}

/* ARGSUSED */
void alert_user_CB(w,user_data,call_data)
Widget w;
caddr_t user_data, call_data;
{
	AlertResponse = (int) user_data;
}
