/*
 * This program manages user stack area executability flag for ELF and a.out
 * binaries. The flag only has effect when running the patched Linux kernel.
 * Copyright (c) 1997 by Solar Designer.
 */
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <linux/elf.h>
#include <linux/a.out.h>

#define HF_STACKEXEC	1

struct elf32_hdr header_elf;
struct exec header_aout;
void *header;
int header_size;
int fd;
unsigned long (*get_flags)();
void (*put_flags)(unsigned long);

unsigned long get_flags_elf() {
	return header_elf.e_flags;
}

void put_flags_elf(unsigned long flags) {
	header_elf.e_flags = flags;
}

unsigned long get_flags_aout() {
	return N_FLAGS(header_aout);
}

void put_flags_aout(unsigned long flags) {
	N_SET_FLAGS(header_aout, flags);
}

int read_header(char *name, int mode) {
	if ((fd = open(name, mode)) < 0) return 1;

	if (read(fd, &header_elf, sizeof(header_elf)) != sizeof(header_elf)) {
		close(fd); return 2;
	}
	memcpy(&header_aout, &header_elf, sizeof(header_aout));

	if (!strncmp(header_elf.e_ident, ELFMAG, SELFMAG)) {
		if (header_elf.e_type != ET_EXEC) return 2;
		if (header_elf.e_machine != EM_386) return 3;
		header = &header_elf; header_size = sizeof(header_elf);
		get_flags = get_flags_elf; put_flags = put_flags_elf;
	} else
	if (N_MAGIC(header_aout) == NMAGIC ||
	    N_MAGIC(header_aout) == ZMAGIC ||
	    N_MAGIC(header_aout) == QMAGIC) {
		if (N_MACHTYPE(header_aout) != M_386) return 3;
		header = &header_aout; header_size = 4;
		get_flags = get_flags_aout; put_flags = put_flags_aout;
	} else return 2;

	return 0;
}

int write_header() {
	size_t retval;

	if (lseek(fd, 0, SEEK_SET)) {
		close(fd); return 1;
	}

	retval = write(fd, header, header_size);
	close(fd);

	return retval - header_size;
}

void help(char *name) {
	printf(
		"Usage: %s OPTION FILE...\n"
		"Manage stack area executability flag for binaries\n\n"
		"  -e   enable execution permission\n"
		"  -d   disable execution permission\n"
		"  -v   view current flag state\n\n"
		"The flag only has effect when running the patched Linux kernel\n",
		name);
	exit(1);
}

int main(int argc, char **argv) {
	char **current;
	unsigned long flags;
	int error = 0, retval;

	if (argc < 3) help(argv[0]);
	if (strlen(argv[1]) != 2) help(argv[0]);
	if (argv[1][0] != '-' || !strchr("edv", argv[1][1])) help(argv[0]);

	current = &argv[2];
	do {
		if (argv[1][1] == 'v')
			retval = read_header(*current, O_RDONLY);
		else
			retval = read_header(*current, O_RDWR);

		if (retval) {
			switch (retval) {
			case 1:
				perror(*current);
				break;
			case 2:
				printf("%s: Unknown file type\n", *current);
				break;
			case 3:
				printf("%s: Wrong architecture\n");
			}
			error = 1; continue;
		}

		flags = get_flags();

		switch (argv[1][1]) {
		case 'e':
			put_flags(flags | HF_STACKEXEC);
			break;
		case 'd':
			put_flags(flags & ~HF_STACKEXEC);
			break;
		default:
			printf("%s: %s stack area\n", *current,
				flags & HF_STACKEXEC
				? "Executable" : "Non-executable");
		}

		if (flags == get_flags()) close(fd); else
		if (write_header()) {
			perror(*current); error = 1;
		}
	} while (*++current);

	return error;
}
