/*----------------------------------------------------------------------*
 * Bounds Checking for GCC.						*
 * Copyright (C) 1995 Richard W.M. Jones <rwmj@doc.ic.ac.uk>.		*
 *----------------------------------------------------------------------*
 * This program is free software; you can redistribute it and/or modify	*
 * it under the terms of the GNU General Public License as published by	*
 * the Free Software Foundation; either version 2 of the License, or	*
 * (at your option) any later version.					*
 *									*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	*
 * GNU General Public License for more details.				*
 *									*
 * You should have received a copy of the GNU General Public License	*
 * along with this program; if not, write to the Free Software		*
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		*
 *----------------------------------------------------------------------*
 * File:
 *	treestats/locality.c
 * Summary:
 *	Study the locality of objects in the tree.
 * Other notes:
 *	
 * Author      	Date		Notes
 * RWMJ		28/5/95		Initial implementation.
 *----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>

#include "misc.h"
#include "readfile.h"

static void examine_locality (tree *, struct ext_tree *);

static int percentage[] = {50, 70, 90, 95, 99, 0};

int
main (int argc, char *argv[])
{
  int i;

  if (argc < 2)
    {
      fprintf (stderr, "st-locality dumpfile [dumpfile [...]]\n");
      exit (1);
    }

  printf ("INDEX\tCOUNT");
  for (i = 0; percentage[i] != 0; ++i)
    {
      printf ("\tWS%d", percentage[i]);
      printf ("\tMU%d", percentage[i]);
    }
  putchar ('\n');

  for (i = 1; i < argc; ++i)
    {
      tree *t;
      struct ext_tree *et;

      if (readfile (argv[i], &t, &et) != 0)
	continue;

      printf ("%u\t%d", et->index, counttree (t));
      examine_locality (t, et);

      deltree (t);
    }

  exit (0);
}

struct loc {
  object *n;
  int depth;
};

static int compare_loc (struct loc *, struct loc *);
static struct loc *flatten (object *, struct loc *, int);
static int mu_ws_size (struct loc *, int count, int pc, unsigned tot_hits);
static double mu_avg_depth (struct loc *, int ws_size);

static void
examine_locality (tree *t, struct ext_tree *et)
{
  int count = counttree (t), i;
  struct loc nodes [count];
  double mu;
  int ws;
  unsigned tot_hits = 0;

  /* Flatten the nodes into our array, recording the depth of each node
   * too.
   */
  (void) flatten (t->root, nodes, 0);

  /* Sort the nodes in the tree into descending order of # of hits. (Node[0]
   * will be most frequently used node).
   */
  qsort (nodes, count, sizeof (struct loc), (__compar_fn_t) compare_loc);

#if 0
  /* Print nodes here. */
  for (i = 0; i < count; ++i)
    printf ("node %d, hits %d, depth %d\n",
	    i, nodes[i].n->hits, nodes[i].depth);
#endif

  /* Count total number of hits received.
   */
  for (i = 0; i < count; ++i)
    tot_hits += nodes[i].n->hits;

  /* Calculate average depth of each percentage of objects of interest.
   */
  for (i = 0; percentage[i] != 0; ++i)
    {
      ws = mu_ws_size (nodes, count, percentage[i], tot_hits);
      mu = mu_avg_depth (nodes, ws);
      printf ("\t%d\t%2.2g", ws, mu);
    }
  putchar ('\n');
}

static struct loc *
flatten (object *n, struct loc *nodes, int depth)
{
  if (n == NULL)
    return nodes;

  nodes->n = n;
  nodes->depth = depth;

  nodes ++;

  nodes = flatten (n->left, nodes, depth + 1);
  nodes = flatten (n->right, nodes, depth + 1);

  return nodes;
}

static int
compare_loc (struct loc *l1, struct loc *l2)
{
  return l2->n->hits - l1->n->hits;
}

static int
mu_ws_size (struct loc *nodes, int count, int pc, unsigned tot_hits)
{
  /* Work out the size of the working set that received pc% of the
   * hits.
   */
  int i;
  unsigned pc_hits = pc * tot_hits / 100, sum_hits = 0;

  for (i = 0; i < count && sum_hits < pc_hits; ++i)
    sum_hits += nodes[i].n->hits;

  return i;
}

static double
mu_avg_depth (struct loc *nodes, int ws_size)
{
  /* Work out the average depth of the first ws_size nodes.
   */
  int i;
  double sum = 0;

  if (ws_size == 0) return 0;

  for (i = 0; i < ws_size; ++i)
    sum += nodes[i].depth;

  return sum / ws_size;
}
