/*----------------------------------------------------------------------*
 * Bounds Checking for GCC.						*
 * Copyright (C) 1995 Richard W.M. Jones <rwmj@doc.ic.ac.uk>.		*
 *----------------------------------------------------------------------*
 * This program is free software; you can redistribute it and/or modify	*
 * it under the terms of the GNU General Public License as published by	*
 * the Free Software Foundation; either version 2 of the License, or	*
 * (at your option) any later version.					*
 *									*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	*
 * GNU General Public License for more details.				*
 *									*
 * You should have received a copy of the GNU General Public License	*
 * along with this program; if not, write to the Free Software		*
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		*
 *----------------------------------------------------------------------*
 * File:
 *	tools/find-objects.c
 * Summary:
 *	Look for static data in libraries and other object files. Write
 *	a C program that will add these objects to the tree at run time
 *	automatically.
 * Other notes:
 *	You will need GNU's `BFD' library, which can be found in the GDB
 *	package.
 * Author      	Date		Notes
 * RWMJ		27/8/95		Initial implementation.
 *----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <errno.h>

#ifdef __STDC__
#define PROTO(args) args
#define VPROTO (void)
#else
#define PROTO(args)
#define VPROTO ()
#endif

#define PARAMS(args) PROTO(args)
#include "bfd.h"

void process_file PROTO((CONST char *));
void process_archive_or_object PROTO((bfd *, CONST char *));
void process_object_file PROTO((bfd *, CONST char *));
int  compare_symbols PROTO((asymbol **, asymbol **));
int  miscellaneous_deletable_symbol PROTO((CONST char *));
int  excluded_symbol PROTO((CONST char *));
int  add_excluded_symbols PROTO((CONST char **));
void open_output_file PROTO((CONST char *, CONST char *));
void write_symbol PROTO((CONST char *, CONST char *, CONST char *, CONST char *, long, size_t));
void write_table VPROTO;
void close_output_file PROTO((CONST char *));
void do_bfd_error PROTO((CONST char *));
void do_error PROTO((CONST char *));
void usage VPROTO;

int quiet = 0;
FILE *output_file;

int
main (argc, argv)
  int argc;
  char *argv[];
{
  char *input_filename = NULL, *output_filename = NULL;
  int count = 0, i;

  /* Initialize the BFD library.
   */
  bfd_init ();

  /* Process the command line arguments here.
   */
  for (i = 1; i < argc; ++i)
    {
      if (argv[i][0] == '-' && argv[i][1] == '-')
	{
	  if (strcmp (argv[i], "--quiet") == 0)
	    quiet = 1;
	  else if (strcmp (argv[i], "--exclude") == 0 && i < argc-1)
	    i += add_excluded_symbols ((CONST char **) &argv[i+1]);
	  else if (strcmp (argv[i], "--") == 0)
	    ;
	  else
	    usage ();
	}
      else
	{
	  if (count == 0)
	    input_filename = argv[i];
	  else if (count == 1)
	    output_filename = argv[i];
	  else
	    usage ();
	  count ++;
	}
    }
  if (count != 2)
    usage ();

  open_output_file (output_filename, input_filename);
  process_file (input_filename);
  close_output_file (input_filename);

  exit (0);
}

void
process_file (filename)
  CONST char *filename;
{
  bfd *input_bfd;

  input_bfd = bfd_openr (filename, NULL);
  if (input_bfd == NULL) do_bfd_error ("bfd_openr");

  if (!quiet) printf ("File %s:\n", filename);
  process_archive_or_object (input_bfd, filename);

  bfd_close (input_bfd);
}

void
process_archive_or_object (abfd, filename)
  bfd *abfd;
  CONST char *filename;
{
  /* Autodetect if this is a library (`archive') or object file. Other types
   * of file will be rejected here.
   */
  if (bfd_check_format (abfd, bfd_archive))
    {
      /* It's a library. Go through each object file in the library in turn.
       */
      bfd *subbfd = NULL;

/*      if (!quiet) printf ("This is a library.\n"); */

      while ((subbfd = bfd_openr_next_archived_file (abfd, subbfd)) != NULL)
	process_archive_or_object (subbfd, subbfd->filename);
    }
  else if (bfd_check_format (abfd, bfd_object))
    {
      process_object_file (abfd, filename);
    }
  else
    {
      /* Unrecognized format, or miscellaneous error reading the input file.
       */
      do_bfd_error ("file type");
    }
}

void
process_object_file (abfd, filename)
  bfd *abfd;
  CONST char *filename;
{
  long storage_needed, number_of_symbols, number_deleted, i;
  asymbol **symbol_table;
  int symbols_written = 0;

  if (!quiet)
    {
      printf ("  Object file %s: ", filename);
      fflush (stdout);
    }

  /* Read the symbol table from the object file. This code is copied almost
   * directly from the BFD info pages.
   */
  storage_needed = bfd_get_symtab_upper_bound (abfd);
  if (storage_needed < 0)
    do_bfd_error ("bfd_get_symtab_upper_bound");
  if (storage_needed == 0)
    {
      printf ("Warning: object file contains no symbols.\n");
      return;
    }
  symbol_table = (asymbol **) malloc (storage_needed);
  if (symbol_table == NULL)
    do_error ("malloc");

  /* This actually reads the symbol table into memory.
   */
  number_of_symbols = bfd_canonicalize_symtab (abfd, symbol_table);
  if (number_of_symbols < 0)
    do_bfd_error ("bfd_canonicalize_symtab");
  if (number_of_symbols == 0)
    goto end;

  /* Now we have the symbol table in memory, delete undefined references
   * and sort the remaining symbols into ascending order, so we can easily
   * find the location and size of each one. If several symbols map to
   * a single location, we ensure that globally named symbols appear first.
   */
  number_deleted = 0;
  for (i = 0; i < number_of_symbols; ++i)
    {
      if (symbol_table[i]->flags == 0
	  || (symbol_table[i]->flags & BSF_DEBUGGING)
	  || (symbol_table[i]->flags & BSF_SECTION_SYM)
	  || (symbol_table[i]->flags & BSF_WARNING)
	  || (symbol_table[i]->flags & BSF_INDIRECT)
	  || (symbol_table[i]->flags & BSF_FILE)
	  || (symbol_table[i]->flags & BSF_DYNAMIC)
	  || miscellaneous_deletable_symbol (symbol_table[i]->name)
	  || excluded_symbol (symbol_table[i]->name))
	{
	  symbol_table[i] = 0; /* Mark symbol for deletion. */
	  number_deleted ++;
	}
    }
  qsort (symbol_table, number_of_symbols, sizeof (asymbol *), compare_symbols);

#if 0
  for (i = 0; i < number_of_symbols; ++i)
    if (symbol_table[i])
      printf ("    Symbol: %s. Value 0x%08X (%s) Section %s\n",
	      symbol_table[i]->name,
	      symbol_table[i]->value,
	      symbol_table[i]->flags & BSF_GLOBAL ? "global" : "local",
	      symbol_table[i]->section->name);
    else
      printf ("    <Deleted symbol>\n");
#endif

  number_of_symbols -= number_deleted;
  if (number_of_symbols == 0)
    goto end;

  /* From this list of symbols, derive a list of objects. This is not as
   * easy as it may seem. For a start, we can't generate pointers to
   * static declarations in object files, because these symbols aren't
   * exported. In that case, we use offsets to global symbols in the
   * same segment. Also, symbols don't have explicit sizes, we have to
   * guess the size from the position of the next object, or the size
   * of the section.
   */
  {
    long next_i, global_symbol_in_this_section = -1;
    size_t size;

    for (i = 0; i < number_of_symbols; i = next_i)
      {
	/* Set `next_i' to point to the next distinct symbol after the
	 * current one, skipping symbols which point to the same address.
	 */
	next_i = i + 1;
	while (next_i < number_of_symbols
	       && symbol_table[next_i]->value == symbol_table[i]->value
	       && symbol_table[next_i]->section == symbol_table[i]->section)
	  next_i ++;

	/* Try to compute the size of this symbol. If the next symbol is
	 * in the same section, then the size is simply the difference
	 * between the two symbols. If the next symbol is not in the same
	 * section, we must look at the size of the section itself.
	 */
	if (next_i < number_of_symbols
	    && symbol_table[next_i]->section == symbol_table[i]->section)
	  size = symbol_table[next_i]->value - symbol_table[i]->value;
	else
	  {
	    long section_size
	      = bfd_get_section_size_before_reloc (symbol_table[i]->section);
	    if (section_size > symbol_table[i]->value)
	      size = section_size - symbol_table[i]->value;
	    else
	      size = 1;
	  }

	/* If any of the symbols pointing to this address are global, then
	 * they should have been sorted to the beginning, so `i' should
	 * point to a global symbol if there is one.
	 */
	if (symbol_table[i]->flags & BSF_GLOBAL)
	  {
	    /* There is a global symbol: Generate a pointer to this
	     * object directly.
	     */
	    write_symbol (filename, symbol_table[i]->name,
			  symbol_table[i]->section->name,
			  symbol_table[i]->name, 0,
			  size);
	    symbols_written ++;
	    global_symbol_in_this_section = i;
	  }
	else
	  {
	    /* There is no global symbol for this object: We can still attempt
	     * to get a pointer to a global in the same section and use
	     * an offset from that symbol. If there are no globals at all in
	     * this section, print a warning and continue.
	     */
	    if (global_symbol_in_this_section >= 0
		&& (symbol_table[global_symbol_in_this_section]->section
		    == symbol_table[i]->section))
	      {
		write_symbol (filename, symbol_table[i]->name,
			      symbol_table[i]->section->name,
			      symbol_table[global_symbol_in_this_section]->name,
			      symbol_table[i]->value - symbol_table[global_symbol_in_this_section]->value,
			      size);
		symbols_written ++;
	      }
	    else
	      {
		long j;

		global_symbol_in_this_section = -1;
		for (j = next_i; j < number_of_symbols; ++j)
		  if (symbol_table[j]->section == symbol_table[i]->section)
		    {
		      if (symbol_table[j]->flags & BSF_GLOBAL)
			{
			  global_symbol_in_this_section = j;
			  break;
			}
		    }
		  else
		    break;
		if (global_symbol_in_this_section >= 0)
		  {
		    write_symbol (filename, symbol_table[i]->name,
				  symbol_table[i]->section->name,
				  symbol_table[global_symbol_in_this_section]->name,
				  symbol_table[i]->value - symbol_table[global_symbol_in_this_section]->value,
				  size);
		    symbols_written ++;
		  }
		else
		  {
		    if (!quiet) fputc ('\n', stdout);
		    printf ("Warning: Cannot generate a ref. to local symbol `%s' in file `%s'.\n",
			    symbol_table[i]->name, filename);
		  }
	      }
	  }
      }
  }

  end:
  if (!quiet) printf ("%d objects found.\n", symbols_written);
}

int
compare_symbols (sym1_ptr, sym2_ptr)
  asymbol **sym1_ptr, **sym2_ptr;
{
  asymbol *sym1 = *sym1_ptr;
  asymbol *sym2 = *sym2_ptr;

  if (sym1 != 0 && sym2 != 0)
    {
      /* Order symbols according to their section, then according to their
       * value. If they have the same section and value, we favour placing
       * global references before local ones.
       * Notice that we simple use the section pointers to order the
       * sections: it doesn't actually matter if `.data' comes before
       * or after `.text', etc.
       */
      if (sym1->section != sym2->section)
	return sym1->section - sym2->section;
      else if (sym1->value != sym2->value)
	return sym1->value - sym2->value;
      else if (sym1->flags & BSF_GLOBAL)
	return -1;
      else
	return 1;
    }
  else if (sym1 != 0)	/* Deleted symbols migrate to the end. */
    return -1;
  else
    return 1;
}

int
miscellaneous_deletable_symbol (name)
  CONST char *name;
{
  /* Delete certain symbols which may cause problems because they are
   * built into GCC or they don't really exist.
   */
  if (strncmp (name, "__builtin_", 10) == 0
      || strcmp (name, "sccsid") == 0
      || strcmp (name, "rcsid") == 0
      || ((strncmp (name, "__CTOR_", 7) == 0
	   || strncmp (name, "__DTOR_", 7) == 0)
	  &&
	  (strcmp (name+7, "LIST__") == 0
	   || strcmp (name+7, "END__") == 0)))
    return 1;
  else
    return 0;
}

/*----------------------------------------------------------------------*/

/* A primitive hash table that lists symbols which the user wants excluded
 * from the output. We hash on the first letter of the symbol.
 */
static struct {
  int nr_symbols;
  CONST char **symbols;
} hash_table[256];

static int some_excluded_symbols = 0;

int
add_excluded_symbols (argv)
  CONST char *argv[];
{
  int i;

  if (!some_excluded_symbols)
    {
      memset (hash_table, 0, sizeof hash_table);
      some_excluded_symbols = 1;
    }
  for (i = 0; argv[i] != NULL && (argv[i][0] != '-' || argv[i][1] != '-'); ++i)
    {
      /* Append this symbol to the end of the current hash table list.
       */
      int c = argv[i][0], new_size;

      hash_table[c].nr_symbols ++;
      new_size = hash_table[c].nr_symbols * sizeof (char *);
      if (hash_table[c].symbols != NULL)
	hash_table[c].symbols = realloc (hash_table[c].symbols, new_size);
      else
	hash_table[c].symbols = malloc (new_size);
      if (hash_table[c].symbols == NULL) { perror ("malloc"); exit (1); }
      hash_table[c].symbols[hash_table[c].nr_symbols - 1] = argv[i];
    }

  if (i == 0)	/* no symbols defined */
    usage ();

  return i;
}

/* Test to see if `name' is in the list of excluded symbols provided by
 * the user.
 */

int
excluded_symbol (name)
  CONST char *name;
{
  int c, i;

  if (!some_excluded_symbols) return 0;
  c = name[0];
  for (i = 0; i < hash_table[c].nr_symbols; ++i)
    if (strcmp (hash_table[c].symbols[i], name) == 0)
      return 1;
  return 0;
}

/*----------------------------------------------------------------------*/

char *prologue = "\
/* CAUTION: Automatically generated file. Edits will be lost!\n\
 *--\n\
 * Initialization code for objects in the file `%s'.\n\
 * Link this file with programs that use this library or object file so\n\
 * that they can check objects declared in this file correctly.\n\
 */\n\
\n\
typedef unsigned long size_t; /* FIXME - can't include `stdio.h' */\n\
\n\
/* The following table format is shared with `lib/bounds-lib.h' too.\n\
 */\n\
typedef struct {\n\
  void *address;			/* Object address */\n\
  size_t size;				/* Size (bytes) */\n\
  size_t align;				/* Alignment */\n\
  char *filename;			/* Filename where found */\n\
  int line;				/* Line where found */\n\
  char *name;				/* Name of object */\n\
} external_statics_table;\n\
\n\
void __bounds_add_static_objects_table (external_statics_table *);\n\
\n\
/* Ensure that `init' gets called automatically at run time.\n\
 */\n\
static void init (void) __attribute__((constructor));\n\
\n\
";

char *epilogue = "\
/* Constructor function that stuffs the above table into the checking\n\
 * library's splay tree.\n\
 */\n\
static void\n\
init (void)\n\
{\n\
  __bounds_add_static_objects_table (table);\n\
}\n\
\n\
/* EOF */\n\
";

void
open_output_file (filename, title)
  CONST char *filename, *title;
{
  output_file = fopen (filename, "w");
  if (output_file == NULL) do_error ("fopen");

  fprintf (output_file, prologue, title);
}

void
close_output_file (title)
  CONST char *title;
{
  write_table ();
  fprintf (output_file, epilogue, title);
  fclose (output_file);
}

static struct _table {
  struct _table *next;
  CONST char *filename, *real_name, *section_name, *address_name;
  long address_offset;
  size_t size;
} *hd_table = NULL, *end_table = NULL;

void
write_symbol (filename, real_name, section_name, address_name, address_offset,
              size)
  CONST char *filename;		/* Object file where this was found. */
  CONST char *real_name;	/* Name of this symbol (may be static). */
  CONST char *section_name;	/* Name of section where symbol found. */
  CONST char *address_name;	/* Name of symbol used to get address. */
  long address_offset;		/* Optional offset to `address_name'. */
  size_t size;			/* Size of the symbol. */
{
  struct _table *p = malloc (sizeof (struct _table));
  if (p == NULL) do_error ("malloc");

  if (hd_table == NULL)
    hd_table = end_table = p;
  else
    {
      end_table->next = p;
      end_table = p;
    }

  p->next = NULL;
  p->filename = filename;
  p->real_name = real_name;
  p->section_name = section_name;
  p->address_name = address_name;
  p->address_offset = address_offset;
  p->size = size;

  fprintf (output_file, "extern char %s;\n", address_name);
}

void
write_table ()
{
  struct _table *p;

  fprintf (output_file,
	   "\n"
	   "static external_statics_table table[] = {\n");

  for (p = hd_table; p != NULL; p = p->next)
    {
      fprintf (output_file, "  { ");
      if (p->address_offset == 0)
	fprintf (output_file, "&%s, ", p->address_name);
      else
	fprintf (output_file, "(void *)((unsigned)&%s %c %ld), ",
		 p->address_name,
		 p->address_offset >= 0 ? '+' : '-',
		 p->address_offset >= 0
		 ? p->address_offset : -(p->address_offset));
      fprintf (output_file,
	       "%u, 1, \"%s\", 0, \"%s\" },\n",
	       p->size, p->filename, p->real_name);
    }
  fprintf (output_file,
	   "  { 0 }\n};\n\n");
}

void
do_bfd_error (msg)
  CONST char *msg;
{
  perror (msg);
  exit (1);
}

void
do_error (msg)
  CONST char *msg;
{
  perror (msg);
  exit (1);
}

void
usage ()
{
  fprintf (stderr,
	   "find-objects: Find position and sizes of the objects in a library or `.o' file\n"
	   "Usage: find-objects [--flags] <input-file> <c-output-file>\n"
	   "Flags:\n"
	   "  --quiet                           Run quietly.\n"
	   "  --exclude symbol [symbol [...]]   Don't list symbols.\n");
  exit (1);
}

#if defined(linux)

/* Dummies for `libbfd' on this machine.
 */
void
hex_init (void)
{
  printf ("hex_init called\n");
}

void
_hex_value (void)
{
  printf ("hex_value called\n");
}

#endif
