/*----------------------------------------------------------------------*
 * Bounds Checking for GCC.						*
 * Copyright (C) 1995 Richard W.M. Jones <rwmj@doc.ic.ac.uk>.		*
 *----------------------------------------------------------------------*
 * This program is free software; you can redistribute it and/or modify	*
 * it under the terms of the GNU General Public License as published by	*
 * the Free Software Foundation; either version 2 of the License, or	*
 * (at your option) any later version.					*
 *									*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	*
 * GNU General Public License for more details.				*
 *									*
 * You should have received a copy of the GNU General Public License	*
 * along with this program; if not, write to the Free Software		*
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		*
 *----------------------------------------------------------------------*
 * File:
 *	tests/general-test.c
 * Summary:
 *	General test of validity of GCC with bounds checking.
 * Other notes:
 *	This is experimental at the moment. I'm not sure if this is an
 *	effective way of testing this.
 * Author      	Date		Notes
 * RWMJ		June,July 95
 *----------------------------------------------------------------------*/

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <stdarg.h>
#include <assert.h>
#include <malloc.h>
#ifndef hpux
#include <alloca.h>
#endif

#ifndef __BOUNDS_CHECKING_ON
#error "Compile this file with bounds checking"
#endif

char *progname;
char *results_dir = ".";

void perform_test (int i);
void call_test (int i);
void testg0 (void);
void testg1 (void);
void testg2 (void);
void testg3 (void);
void testg4 (void);
void testg5 (void);
void testg6 (void);
void testa0 (void);
void testm0 (void);
void testm1 (void);
void testm2 (void);

typedef void (*function_ptr)(void);
function_ptr test_fn[] =
{
  testg0, testg1, testg2, testg3, testg4, testg5, testg6,
  testa0,
  testm0, testm1, testm2
};

/* This should be the same as the number of tests in the `test_fn' table
 * above. Notice that this line is read automatically by a script in the
 * Makefile, so don't do anything fancy here:
 */
#define NR_TESTS 11

int
main (int argc, char *argv[])
{
  int i;
  int test;

  progname = argv[0];

  if (argc >= 3 && strcmp (argv[1], "-$") == 0)
    {
      /* Program called automatically here. Process the automatic
       * arguments.
       */
      for (i = 2; i < argc; ++i)
	{
	  if (i < argc-1)
	    {
	      if (strcmp (argv[i], "-test") == 0)
		{
		  i++;
		  test = atoi (argv[i]);
		  continue;
		}
	      else
		{
		  fprintf (stderr, "%s: Wrong arguments.\n", progname);
		  exit (1);
		}
	    }
	  fprintf (stderr, "%s: Wrong arguments.\n", progname);
	  exit (1);
	}
      perform_test (test);
      exit (0);
    }

  /* Program called by user. Perform the automatic tests.
   */
  if (argc == 3 &&
      strcmp (argv[1], "-dir") == 0)
    results_dir = argv[2];
  else if (argc >= 2)
    {
      fprintf (stderr, "%s [-dir <directory>]\n", progname);
      exit (1);
    }

  printf ("Bounds checking GENERAL-TEST. Date: " __DATE__ ".\n");

  for (i = 0; i < NR_TESTS; ++i)
    call_test (i);

  return 0;
}

void
call_test (int i)
{
  int line;
  char command [strlen (progname) + 128];
  char inputfile [256];
  char buffer1 [256], buffer2 [256], *b1, *b2;
  FILE *pp, *fp;

  printf ("Test %d\r", i); fflush (stdout);
  sprintf (command, "%s -$ -test %d 2>&1", progname, i);
  sprintf (inputfile, "%s/test-%d.res", results_dir, i);
  pp = popen (command, "r");
  if (pp == NULL)
    {
      perror ("popen (command)");
      exit (1);
    }
  fp = fopen (inputfile, "r");
  if (fp == NULL)
    {
      perror ("fopen (result file)");
      exit (1);
    }
  line = 0;
  for (;;)
    {
      int j;

      b1 = fgets (buffer1, 256, fp);
      b2 = fgets (buffer2, 256, pp);

      if (!b1 || !b2) break;

      /* Compare the actual result (in buffer2) with the correct
       * result (in buffer1).
       */

      line ++;
      for (j = 0; j < strlen (buffer1); ++j)
	{
	  if (buffer1[j] == '*')
	    /* means ignore to the end of the line */
	    break;
	  else if (buffer1[j] != '.'
		   /* means ignore this single character */
		   && buffer1[j] != buffer2[j])
	    {
	      fprintf (stderr,
		       "Difference at line %d:\n"
		       "  From command:    %s"
		       "  Expected result: %s",
		       line, buffer2, buffer1);
	      exit (1);
	    }
	}
    }

  if (b1 != NULL)
    {
      fprintf (stderr, "Command generated too few lines of output.\n");
      exit (1);
    }
  else if (b2 != NULL)
    {
      fprintf (stderr, "Command generated too many lines of output.\n");
      exit (1);
    }
  printf ("Test %d: OK\n", i);
}

void
perform_test (int i)
{
  if (0 <= i && i < NR_TESTS)
    (test_fn[i]) ();
  else
    {
      fprintf (stderr, "Test number must be 0 <= i < %d\n", NR_TESTS);
      exit (1);
    }
}

/*----------------------------------------------------------------------*/
/* Replace `printf' so `%p' is replaced by something more informative.  */

int
my_printf (const char *fs, ...)
{
  va_list args;
  char new_fs [1024], replace[64];
  int i = 0, r;

  if (sizeof (void*) == 2)
    strcpy (replace, "ptr16-0x%04x");
  else if (sizeof (void*) == 4)
    strcpy (replace, "ptr32-0x%08x");
  else if (sizeof (void*) == 8)
    strcpy (replace, "ptr64-0x%016x");
  else
    abort ();

  /* Replace `%p' in the format string. */
  while (*fs)
    {
      if (*fs == '%' && *(fs+1) == 'p')
	{
	  strcpy (new_fs+i, replace);
	  while (new_fs[i]) ++i;
	  fs += 2;
	}
      else if (*fs == '%' && *(fs+1) == '%')
	{
	  new_fs[i++] = *fs++;
	  new_fs[i++] = *fs++;
	}
      else
	new_fs[i++] = *fs++;
    }
  new_fs[i] = 0;

  va_start (args, fs);
  r = vprintf (new_fs, args);
  va_end (args);

  fflush (stdout);

  return r;
}

#define printf my_printf

/*----------------------------------------------------------------------*/
/* Tests follow ...						        */

/* ARRAY REFERENCE */

void
testg0 (void)
{
  char a[10];
  int i;

  for (i = 0; i < 100; ++i)
    a[i] = 0;
}

void
testg1 (void)
{
  char a[10];
  int i;

  for (i = 0; i < 10; ++i)
    printf ("&a[%d] = %p\n", i, &a[i]);
}

void
testg2 (void)
{
  char a[10];
  int i;

  for (i = 0; i <= 10; ++i)
    printf ("&a[%d] = %p\n", i, &a[i]);	/* This is OK */
}

void
testg3 (void)
{
  char a[10];
  int i;

  for (i = 0; i <= 11; ++i)
    printf ("&a[%d] = %p\n", i, &a[i]);	/* This is _not_ OK */
}

/* POINTER DIFFERENCE */

void
testg4 (void)
{
  int p[10], q[10];
  ptrdiff_t n;

  n = &p[5] - &p[2];	/* OK */
  n = &q[2] - &q[5];	/* OK */
  n = q - p;		/* bad */
}

/* POINTER + INT */

void
testg5 (void)
{
  int i, a[10], *p;

  for (i = 0; i <= 10; ++i)
    {
      printf ("a+i = %p\n", (p = a+i));
      printf ("*(a+i) = 0\n"); *p = 0;
    }
}

/* COMPONENT REF */

void
testg6 (void)
{
  struct {int a,b,c,d;} *p1, *p2, *p3, *p4;

  p1 = calloc (sizeof (int), 1);
  p2 = calloc (sizeof (int), 2);
  p3 = calloc (sizeof (int), 3);
  p4 = calloc (sizeof (int), 4);

  printf ("&p1->a = %p\n", &p1->a);
  printf ("&p2->b = %p\n", &p2->b);
  printf ("&p3->c = %p\n", &p3->c);
  printf ("&p4->d = %p\n", &p4->d);
  printf ("&p2->a = %p\n", &p2->a);
  printf ("&p3->a = %p\n", &p3->a);
  printf ("&p4->a = %p\n", &p4->a);
  printf ("p1->a = %d\n", p1->a);
  printf ("p2->b = %d\n", p2->b);
  printf ("p3->c = %d\n", p3->c);
  printf ("p4->d = %d\n", p4->d);
  printf ("p2->a = %d\n", p2->a);
  printf ("p3->a = %d\n", p3->a);
  printf ("p4->a = %d\n", p4->a);

  printf ("p3->d = %d\n", p3->d);	/* error */
}

/* INTENSIVE ARRAY-REF TESTS */

struct art0 {
  int datum1, datum2, datum3;
  char text[1];
} __attribute__ ((packed));

void
testa0 (void)
{
  int len, i;
  struct art0 *p;
  int base_size = sizeof (struct art0) - 1;

  /* This should run without error, but in fact fails because of HtB's
   * array ref. patch.
   */
  for (len = 0; len < 20; len ++)
    {
      p = malloc (base_size + len);
      printf ("allocated structure at address %p, size %d\n",
	      p, base_size + len);
      p->datum1 = 1;
      p->datum2 = 2;
      p->datum3 = 3;
      for (i = 0; i < len; ++i)
	p->text[i] = 'a';
      p->datum3 = 4;
      p->datum2 = 5;
      p->datum1 = 6;
      free (p);
    }
  printf ("OK\n");
}

/* FREE TWICE */

void
testm0 (void)
{
  void *p = malloc (10);

  free (p);
  free (p);
}

/* FREE NON-HEAP MEM */

void
testm1 (void)
{
  int a[10];

  free (a);
}

/* REALLOC NON-HEAP MEM */

void
testm2 (void)
{
  int a[10];

  realloc (a, 10);
}
