/*
**  FINGER.C
**
**  FINGER client, uses ASYNC functions.
**
**  Compare to
**    (1) AFINGR -- also async, but without using ASYNC.C functions.
**    (2) SFINGR -- synchronous version.
*/

#include <windows.h>
#include <winsock.h>

#include "wil.h"
#include "message.h"
#include "paint.h"
#include "about.h"
#include "async.h"
#include "str.h"


#ifdef WIN32
#define USE_INS HINSTANCE
#define USE_PTR PSTR
#else
#define USE_INS HANDLE
#define USE_PTR LPSTR
#endif

LRESULT CALLBACK MainWndProc(HWND, UINT, WPARAM, LPARAM);

/* globals */

HWND hMainWnd;            /* main window handle */

#define BS            8
#define LF           10
#define CR           13
#define MAX_BUF     128
#define MAX_STR      40

#define FINGER_PORT  79

#define USER_CONN_SUCCESS   100
#define USER_WRITE_SUCCESS  101
#define USER_FAILURE        102
#define USER_SUCCESS        103

static HMENU hMenu;
static USE_INS hInstance;
static int WinWidth = 8 * NCOLS;
static int WinHeight = 12 * NROWS + 48;
static char Temp[MAX_BUF+8];
static int  InBufLen = 0;
static char InBuffer[MAX_BUF+1];
static char User[MAX_STR] = "";
static char Host[MAX_STR] = "";
static SOCKET Socket = 0;
static LPSTR  Ptr;
static HCURSOR ArrowCursor;
static HCURSOR WaitCursor;

/* add character to buffer */

static void Add2Buffer(char Chr)
{/* add char to input buffer */
 switch(Chr)
   {case BS:
      if(InBufLen>0)
        {/* backup on screen */
         DisplayChar(BS);
         /* remove last char from buffer */
         InBufLen--;
        }
      break;
    default:
      /* display char */
      DisplayChar(Chr);
      /* put into buffer */
      if(InBufLen<MAX_BUF) InBuffer[InBufLen++] = Chr;
      break;
   }
}

/* display error message */

static void DisplayError(int Code, LPSTR Msg)
{DisplayString("ERROR: ");
 if(Msg) DisplayString(Msg);
 if(Code)
   {wilErrorText(Code,(LPSTR)Temp,50);
    DisplayLine((LPSTR)Temp);
   }
 SetCursor(ArrowCursor);
}

/* WinMain */

#ifdef WIN32
int WINAPI
#else
int PASCAL
#endif
WinMain(USE_INS hInst, USE_INS hPrevInstance,
        USE_PTR szCmdLine,  int nCmdShow)
{WNDCLASS  wc;
 MSG msg;
 BOOL Result;
 if(!hPrevInstance)
   {/* register main window class */
    wc.style = CS_HREDRAW | CS_VREDRAW;
    wc.lpfnWndProc = MainWndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
    wc.hInstance = hInst;
    wc.hIcon = LoadIcon(hInst, "HostIcon");
    wc.hCursor = NULL;
    wc.hbrBackground = (HBRUSH)GetStockObject(WHITE_BRUSH);
    wc.lpszMenuName =  "HostMenu";
    wc.lpszClassName = "HostWClass";
    Result = RegisterClass(&wc);
    if(!Result) return FALSE;
   }

 /* create main window */
 hInstance = hInst;
 hMainWnd = CreateWindow(
        "HostWClass",   "FINGER",    WS_OVERLAPPEDWINDOW,
        CW_USEDEFAULT,  CW_USEDEFAULT,
        WinWidth,       WinHeight,
        NULL,           NULL,
        hInstance,      NULL);
 ShowWindow(hMainWnd, nCmdShow);
 UpdateWindow(hMainWnd);
 hMenu = GetMenu(hMainWnd);

 /* window control loop */

 while(GetMessage(&msg,NULL,0,0))
   {
    TranslateMessage(&msg);
    DispatchMessage(&msg);
   }
 return (msg.wParam);
} /* end WinMain */

#ifdef WIN32
LRESULT CALLBACK
#else
long FAR PASCAL
#endif
MainWndProc(HWND hWindow,UINT iMsg,WPARAM wParam,LPARAM lParam)
{int Code;
 HDC hDC;
 PAINTSTRUCT ps;
#ifdef WIN32
#else
 static FARPROC lpfnAboutDlgProc;
#endif
 hMainWnd = hWindow;
 switch (iMsg)
    {case WM_CREATE:
      /* create cursors */
      ArrowCursor = LoadCursor(NULL, IDC_ARROW);
      WaitCursor = LoadCursor(NULL, IDC_WAIT);
      SetCursor(ArrowCursor);
#ifdef WIN32
#else
       /* create thunk for Win16 */
       lpfnAboutDlgProc = MakeProcInstance(AboutDlgProc,hInstance);
#endif
      /* initialize paint module */
      PaintInit();
      /* attach WINSOCK */
      DisplayString("Attaching WINSOCK...");
      Code = wilAttach();
      DisplayLine("OK");
      if(Code<0) DisplayError(Code,"wilAttach fails:");
      else
        {wsprintf((LPSTR)Temp," Description: %s", wilGetDescription() );
         DisplayLine((LPSTR)Temp);
         wsprintf((LPSTR)Temp," My HostName: %s", wilGetMyHostName() );
         DisplayLine((LPSTR)Temp);
         wsprintf((LPSTR)Temp," My HostAddr: %s", wilGetMyHostDotted(0) );
         DisplayLine((LPSTR)Temp);
        }
      break;

     case WM_COMMAND:
         switch(wParam)
           {case MSG_ABOUT :
#ifdef WIN32
               DialogBox(hInstance, "AboutBox", hMainWnd, AboutDlgProc);
#else
               DialogBox(hInstance, "AboutBox", hMainWnd, lpfnAboutDlgProc);
#endif
               return 0;

            case MSG_EXIT:
              wilRelease();
              DestroyWindow(hMainWnd);
              break;

            case MSG_DEBUG:
              Code = wilDebug(0);
              wsprintf((LPSTR)Temp,"Debug(0) returned %d",Code);
              DisplayLine((LPSTR)Temp);
              break;

            case MSG_BREAK:
              wilCancelBlocking();
              wilAwaitCancel(Socket,hMainWnd);
              break;

            case MSG_FINGER:
              InBufLen = 0;
              DisplayString("Enter user (usr@domain):");
              break;
           }
         break;

    case WM_PAINT:
      HideCaret(hMainWnd);
      hDC = BeginPaint(hMainWnd, &ps);
      SetMapMode(hDC,MM_ANISOTROPIC);
      SelectObject(hDC, GetStockObject(OEM_FIXED_FONT) );
      PaintMain(hDC,&ps);
      EndPaint(hMainWnd,&ps);
      SetCaretPos(PaintGetColPos(),PaintGetRowPos());
      ShowCaret(hMainWnd);
      break;

    case WM_DESTROY:
      PostQuitMessage(0);
      break;

    case WM_USER: /* posted by WIL */

#if 0
wsprintf((LPSTR)Temp,"[%ld]", lParam);
DisplayString((LPSTR)Temp);
#endif

      AsyncProcessMsg(lParam);
      break;

    case WM_USER+1:  /* posted by Async functions */

#if 0
wsprintf((LPSTR)Temp,"<%d>", wParam);
DisplayString((LPSTR)Temp);
#endif

      switch(wParam)
        {
         case USER_CONN_SUCCESS:
           /* we are now connected. send user name to server */
           wsprintf((LPSTR)Temp,"%s\r\n",(LPSTR)User);
           AsyncWrite((LPSTR)Temp, USER_WRITE_SUCCESS, USER_FAILURE);
           break;

         case USER_WRITE_SUCCESS:
           /* now get response text from server */
           AsyncRead(USER_SUCCESS, USER_FAILURE, ASYNC_MULTIPLE_LINES);
           break;

         case USER_FAILURE:
           DisplayLine("Finger has failed.");
           /* fall thru... */

         case USER_SUCCESS:
           DisplayLine("\r\nFinger completed.");
           SetCursor(ArrowCursor);
           wilCloseSocket(Socket);
           break;
        }
      break;

    case WM_CHAR:
      if(wParam==CR)
        {/* do the CR */
         DisplayChar((char)wParam);
         /* user has completed input */
         DisplayChar(LF);
         InBuffer[InBufLen] = '\0';
         /* execute command */
         wsprintf((LPSTR)Temp,"User@Host: %s",(LPSTR)InBuffer);
         DisplayLine((LPSTR)Temp);
         /* extract user & host names */
         Ptr = StringChar((LPSTR)InBuffer,'@');
         if(Ptr==NULL)
           {DisplayError(Code, "Cannot recognize User@Host");
            break;
           }
         *Ptr = '\0';
         lstrcpy((LPSTR)User, (LPSTR)InBuffer);
         lstrcpy((LPSTR)Host, (LPSTR)(++Ptr));
         wsprintf((LPSTR)Temp,"User ='%s'", (LPSTR)User);
         DisplayLine(Temp);
         wsprintf((LPSTR)Temp,"Host ='%s'", (LPSTR)Host);
         DisplayLine(Temp);
         SetCursor(WaitCursor);
         /* connect to server */
         Socket = AsyncConnect(hMainWnd,"Finger",(LPSTR)Host,
                          FINGER_PORT,  USER_CONN_SUCCESS,
                          USER_FAILURE, ASYNC_NONE);
        }
      else
        {/* add char to input buffer */
         Add2Buffer((char)wParam);
        }
      break;

    default:
      return (DefWindowProc(hMainWnd, iMsg, wParam, lParam));
   }
 return 0;

} /* end MainWndProc */                                                                                                          