/*
    FastStr.h

    Defines class CFastString --
    FAST replacement for MFC CString.

    By Dan Shappir, 1997.
*/
#ifndef _FASTSTR_H
#define _FASTSTR_H

class CFastString : public CString {
public:
    CFastString()                                                  {}
    CFastString(const CFastString& stringSrc) : CString(stringSrc) {}
    CFastString(const CString& stringSrc) : CString(stringSrc)     {}
    CFastString(TCHAR ch, int nRepeat = 1) : CString(ch, nRepeat)  {}
    CFastString(LPCSTR lpsz) : CString(lpsz)                       {}
    CFastString(LPCWSTR lpsz) : CString(lpsz)                      {}
    CFastString(LPCTSTR lpsz, int nLength)
        : CString(lpsz, nLength)                                   {}
    CFastString(const unsigned char* psz) : CString(psz)           {}
    CFastString(const class CStringSumBase&);

    const CFastString& operator=(const CFastString& stringSrc)
    {
        CString::operator=(stringSrc);
        return *this;
    }
    const CFastString& operator=(const CString& stringSrc)
    {
        CString::operator=(stringSrc);
        return *this;
    }
    const CFastString& operator=(TCHAR ch)
    {
        CString::operator=(ch);
        return *this;
    }
#ifdef _UNICODE
    const CFastString& operator=(char ch)
    {
        CString::operator=(ch);
        return *this;
    }
#endif
    const CFastString& operator=(LPCSTR lpsz)
    {
        CString::operator=(lpsz);
        return *this;
    }
    const CFastString& operator=(LPCWSTR lpsz)
    {
        CString::operator=(lpsz);
        return *this;
    }
    const CFastString& operator=(const unsigned char* psz)
    {
        CString::operator=(psz);
        return *this;
    }
    const CFastString& operator=(const CStringSumBase& sum)
    {
        return *this = CFastString(sum);
    }

    const CFastString& operator+=(const CFastString& string)
    {
        CString::operator+=(string);
        return *this;
    }
    const CFastString& operator+=(const CString& string)
    {
        CString::operator+=(string);
        return *this;
    }
    const CFastString& operator+=(TCHAR ch)
    {
        CString::operator+=(ch);
        return *this;
    }
#ifdef _UNICODE
    const CFastString& operator+=(char ch)
    {
        CString::operator+=(ch);
        return *this;
    }
#endif
    const CFastString& operator+=(LPCTSTR lpsz)
    {
        CString::operator+=(lpsz);
        return *this;
    }
    const CFastString& operator+=(const CStringSumBase& sum);

    LPTSTR Put(LPTSTR dst) const
    {
        int length = GetLength();
        return static_cast<LPTSTR>(memcpy(dst, LPCTSTR(*this),
                                    length*sizeof(TCHAR)))+length;
    }
};

class CStringSumBase {
    CFastString    m_string;
    CFastString Eval() const
    {
        return const_cast<CStringSumBase*>(this)->m_string = *this;
    }
    
public:
    virtual int GetLength() const       = 0;
    virtual LPTSTR Put(LPTSTR) const   = 0;
    BOOL IsEmpty() const               { return GetLength() == 0; }
    
    TCHAR GetAt(int nIndex) const    { return Eval().GetAt(nIndex); }
    TCHAR operator[](int nIndex) const
        { return Eval().GetAt(nIndex); }
    operator LPCTSTR() const         { return Eval(); }
    
    int Compare(LPCTSTR lpsz) const  { return Eval().Compare(lpsz); }
    int CompareNoCase(LPCTSTR lpsz) const
    {
        return Eval().CompareNoCase(lpsz);
    }
    int Collate(LPCTSTR lpsz) const  { return Eval().Collate(lpsz); }
    
    CFastString Mid(int nFirst, int nCount) const
    {
        return Eval().Mid(nFirst, nCount);
    }
    CString Mid(int nFirst) const    { return Eval().Mid(nFirst); }
    CString Left(int nCount) const   { return Eval().Left(nCount); }
    CString Right(int nCount) const  { return Eval().Right(nCount); }

    CString SpanIncluding(LPCTSTR lpszCharSet) const
    {
        return Eval().SpanIncluding(lpszCharSet);
    }
    CString SpanExcluding(LPCTSTR lpszCharSet) const
    {
        return Eval().SpanExcluding(lpszCharSet);
    }
};

template<class A,class B>
class CStringSum : public CStringSumBase {
protected:
    const A&    m_a;
    const B&    m_b;
public:
    CStringSum(const A& a, const B& b) : m_a(a), m_b(b)  {}
    int GetLength() const
    {
        return m_a.GetLength()+m_b.GetLength();
    }
    LPTSTR Put(LPTSTR dst) const
    {
        return m_b.Put(m_a.Put(dst));
    }
};

template<class A>
class CStringSumTCHAR1 : public CStringSumBase {
protected:
    const A&    m_a;
    const TCHAR    m_b;
public:
    CStringSumTCHAR1(const A& a, const TCHAR b) : m_a(a), m_b(b) {}
    int GetLength() const
    {
        return m_a.GetLength()+1;
    }
    LPTSTR Put(LPTSTR dst) const
    {
        *(dst = m_a.Put(dst)) = m_b;
        return dst+1;
    }
};

template<class B>
class CStringSumTCHAR2 : public CStringSumTCHAR1<B> {
public:
    CStringSumTCHAR2(const TCHAR a, const B& b)
        : CStringSumTCHAR1<B>(b, a)  {}
    LPTSTR Put(LPTSTR dst) const
    {
        *dst = m_b;
        return m_a.Put(dst+1);
    }
};

template<class A>
class CStringSumLPCTSTR1 : public CStringSum<A,TCHAR> {
protected:
    int m_length;
public:
    CStringSumLPCTSTR1(const A& a, const LPCTSTR b)
        : CStringSum<A,TCHAR>(a, *b)  {}
    int GetLength() const
    {
        const_cast<CStringSumLPCTSTR1*>(this)->m_length =
            lstrlen(&m_b);
        return m_a.GetLength()+m_length;
    }
    LPTSTR Put(LPTSTR dst) const
    {
        return static_cast<LPTSTR>(memcpy(m_a.Put(dst),
                        &m_b, m_length*sizeof(TCHAR)))+m_length;
    }
};

template<class B>
class CStringSumLPCTSTR2 : public CStringSumLPCTSTR1<B> {
public:
    CStringSumLPCTSTR2(const LPCTSTR a, const B& b)
        : CStringSumLPCTSTR1<B>(b, a)  {}
    LPTSTR Put(LPTSTR dst) const
    {
        memcpy(dst, &m_b, m_length*sizeof(TCHAR));
        return m_a.Put(dst+m_length);
    }
};

inline CFastString::CFastString(const CStringSumBase& sum)
{
    AllocBuffer(sum.GetLength());
    if ( GetLength() )
        sum.Put(m_pchData);
}

inline const CStringSum<CFastString,CFastString>
   operator+(const CFastString& a, const CFastString& b)
{
    return CStringSum<CFastString,CFastString>(a, b);
}

inline const CStringSum<CFastString,CFastString>
   operator+(const CFastString& a, const CString& b)
{
    return a+static_cast<const CFastString&>(b);
}
inline const CStringSum<CFastString,CFastString>
   operator+(const CString& a, const CFastString& b)
{
    return static_cast<const CFastString&>(a)+b;
}

inline const CStringSumLPCTSTR1<CFastString>
   operator+(const CFastString& a, const LPCTSTR b)
{
    return CStringSumLPCTSTR1<CFastString>(a, b);
}
inline const CStringSumLPCTSTR2<CFastString>
   operator+(const LPCTSTR a, const CFastString& b)
{
    return CStringSumLPCTSTR2<CFastString>(a, b);
}

inline const CStringSumTCHAR1<CFastString>
   operator+(const CFastString& a, const TCHAR& b)
{
    return CStringSumTCHAR1<CFastString>(a, b);
}
inline const CStringSumTCHAR2<CFastString>
   operator+(const TCHAR& a, const CFastString& b)
{
    return CStringSumTCHAR2<CFastString>(a, b);
}

inline const CStringSum<CFastString,CStringSumBase>
   operator+(const CFastString& a, const CStringSumBase& b)
{
    return CStringSum<CFastString,CStringSumBase>(a, b);
}
inline const CStringSum<CStringSumBase,CFastString>
   operator+(const CStringSumBase& a, const CFastString& b)
{
    return CStringSum<CStringSumBase,CFastString>(a, b);
}

inline const CStringSum<CFastString,CStringSumBase>
   operator+(const CString& a, const CStringSumBase& b)
{
    return static_cast<const CFastString&>(a)+b;
}
inline const CStringSum<CStringSumBase,CFastString>
   operator+(const CStringSumBase& a, const CString& b)
{
    return a+static_cast<const CFastString&>(b);
}

inline const CStringSumLPCTSTR1<CStringSumBase>
   operator+(const CStringSumBase& a, const LPCTSTR b)
{
    return CStringSumLPCTSTR1<CStringSumBase>(a, b);
}
inline const CStringSumLPCTSTR2<CStringSumBase>
   operator+(const LPCTSTR a, const CStringSumBase& b)
{
    return CStringSumLPCTSTR2<CStringSumBase>(a, b);
}

inline const CStringSumTCHAR1<CStringSumBase>
   operator+(const CStringSumBase& a, const TCHAR& b)
{
    return CStringSumTCHAR1<CStringSumBase>(a, b);
}
inline const CStringSumTCHAR2<CStringSumBase>
   operator+(const TCHAR& a, const CStringSumBase& b)
{
    return CStringSumTCHAR2<CStringSumBase>(a, b);
}

inline const CStringSum<CStringSumBase,CStringSumBase>
   operator+(const CStringSumBase& a, const CStringSumBase& b)
{
    return CStringSum<CStringSumBase,CStringSumBase>(a, b);
}

inline const CFastString&
   CFastString::operator+=(const CStringSumBase& sum)
{
    return *this = *this+sum;
}

#endif  // _FASTSTR_H
