package bmsi.tui;
import java.awt.*;
import java.awt.event.*;
import java.awt.peer.*;
import java.awt.image.*;
import java.io.*;

/** TUIComponent represents a component on a remote TUI client screen.
 * TUI is an acronym for Text User Interface.  It is basically a GUI
 * minus the pictures. A TUI has menus, windows, and controls, but
 * usually has no Images and only constant width fonts.
 *
 * The TUIComponent could be based on a more general RemoteComponent, but
 * we still want to add some optimizations for TUI.  For instance,
 * since all fonts are the same size, we can return a FontMetrics without
 * consulting the TUI server.
 */

abstract class TUIComponent implements ComponentPeer, RemotePeer {
  protected Component target;
  protected TUIKit toolkit;
  protected int winID = -1;
  TUIGraphics currentGraphics = null;

  public int getID() { return winID; }

  /** Callback from remote peer.
   * For TUI, this is typically to post an event.
   */
  public void remoteMethod(int cmd) throws IOException {
    switch (cmd) {
    case RESHAPE:
      int x = toolkit.readShort();
      int y = toolkit.readShort();
      int w = toolkit.readShort();
      int h = toolkit.readShort();
      // The target posts a ComponentEvent (MOVED and/or RESIZED)
      target.reshape(x,y,w,h);
      return;
    case REQUESTFOCUS:
      if (toolkit.hasFocus != null)
	toolkit.theQueue.postEvent(new FocusEvent(toolkit.hasFocus.target,
	      FocusEvent.FOCUS_LOST));
	//JDK1.0: target.postEvent(new Event(toolkit.hasFocus.target,
	//    Event.LOST_FOCUS,null));
      toolkit.hasFocus = this;
      toolkit.theQueue.postEvent(new FocusEvent(target,FocusEvent.FOCUS_GAINED));
      //JDK1.0: target.postEvent(new Event(target,Event.GOT_FOCUS,null));
      return;
    case KEYPRESS:
      int keycode = toolkit.readShort();
      int modifiers = toolkit.readShort();
      //JDK1.0: target.postEvent(new Event(target,System.currentTimeMillis(),
      //Event.KEY_PRESS, 0,0,keycode,modifiers));
      long ts = System.currentTimeMillis();
      toolkit.theQueue.postEvent(
	new KeyEvent(target,KeyEvent.KEY_PRESSED,ts - 1, modifiers,keycode));
      toolkit.theQueue.postEvent(
	new KeyEvent(target,KeyEvent.KEY_RELEASED,ts, modifiers,keycode));
      return;
    }
  }

  /** This and all derived peer classes must define initialize() to copy
   all relevant attributes from the target component.
   */
  protected void initialize() {
    Color c = target.getForeground();
    if (c != null)
      setForeground(c);
    c = target.getBackground();
    if (c != null)
      setBackground(c);
    Font f = target.getFont();
    if (f != null)
      setFont(f);
    if (!target.isEnabled())
      setEnabled(false);
    try {
      // jdk 1.0 has getCursor in frame only
      Cursor cursor = target.getCursor();
      if (cursor.getType() != Cursor.DEFAULT_CURSOR)
	setCursor(cursor);
    }
    catch (NoSuchMethodError e) { }
    Rectangle r = target.getBounds();
    if (!r.isEmpty())
      setBounds(r.x,r.y,r.width,r.height);
    if (target.isVisible())
      setVisible(true);
  }

  /** Derived classes must define create() to create the proper
   type of RemotePeer. */
  protected abstract void create(TUIContainer parent);

  protected TUIComponent(Component comp,TUIKit toolkit) {
    this.toolkit = toolkit;
    target = comp;
    Container parent = target.getParent();
    TUIContainer parentpeer = null;
    if (parent != null)
      parentpeer = (TUIContainer)parent.getPeer();
    create(parentpeer);	// initialize winID
    initialize();
  }

  protected void finalize() { dispose(); }

  public void dispose() {
    if (toolkit.hasFocus == this) {
      toolkit.theQueue.postEvent(new FocusEvent(target,FocusEvent.FOCUS_LOST));
      //theQueue.postEvent(new Event(target,Event.LOST_FOCUS,null));
      toolkit.hasFocus = null;
    }
    toolkit.removePeer(winID);
    winID = -1;
    target = null;
  }

  public void setVisible(boolean vis) {
    if (winID < 0) return;	// AWT sometimes calls dispose before hide!
    if (vis)
      toolkit.writeCmd(this,SHOW);
    else
      toolkit.writeCmd(this,HIDE);
  }

  public void setEnabled(boolean a) {
    if (a)
      toolkit.writeCmd(this,ENABLE);
    else
      toolkit.writeCmd(this,DISABLE);
  }

  public void paint(Graphics g) {
    g.setColor(target.getForeground());
    g.setFont(target.getFont());
    target.paint(g);
  }

  /** Add rectangle to list of areas to be repainted in ms time.
   * Most TUIComponents are painted entirely by the remote program.
   * @see TUICanvas
   */
  public void repaint(long ms,int x,int y,int w,int h) {
    //target.update(getGraphics());
  }

  public void print(Graphics g) {
    g.setColor(target.getForeground());
    g.setFont(target.getFont());
    // draw box around component for printing
    Dimension dim = target.size();
    g.drawRect(0,0,dim.width - 1,dim.height - 1);
    target.print(g);
  }

  public Point getLocationOnScreen() {
    Point loc = target.getLocation();
    Point ploc = target.getParent().getLocationOnScreen();
    ploc.translate(loc.x,loc.y);
    return ploc;
  }

  public void setBounds(int x,int y,int w,int h) {
    Container p = target.getParent();
    if (p != null) {
      Insets i = p.insets();
      x -= i.left;
      y -= i.top;
    }
    toolkit.writeCmd(this,RESHAPE,x,y,w,h);
    target.repaint();
  }

  public boolean isFocusTraversable() { return true; }

  public Object getTarget() { return target; }

  public Dimension getMinimumSize() { return target.size(); }
  public Dimension getPreferredSize() { return getMinimumSize(); }
  public ColorModel getColorModel() { return toolkit.getColorModel(); }
  public Toolkit getToolkit() { return toolkit; }

  public Graphics getGraphics() { return new TUIGraphics(this); }

  public FontMetrics getFontMetrics(Font font) {
    return new TUIFontMetrics(font,toolkit);
  }

  int pixelFromColor(Color c) {
    return TUIColorModel.getGrayLevel(c) / 100;
  }

  public void setCursor(Cursor cursor) {
    toolkit.writeCmd(this,SETCURSOR,cursor.getType());
  }

  public void setForeground(Color c) {
    toolkit.writeCmd(this,SETFOREGROUND,pixelFromColor(c));
  }
  public void setBackground(Color c) {
    toolkit.writeCmd(this,SETBACKGROUND,pixelFromColor(c));
  }
  /** Our TUI has only one font. */
  public void setFont(Font f) {
    //toolkit.writeCmd(this,SETFONT,f.getName(),f.getSize(),f.getStyle());
  }

  public void requestFocus() { toolkit.writeCmd(this,REQUESTFOCUS); }

  public Image createImage(ImageProducer prod) { return null; }
  public Image createImage(int w,int h) { return null; }
  public boolean prepareImage(Image img,int w,int h,ImageObserver obs) {
    return false;
  }
  public int checkImage(Image img,int w,int h,ImageObserver obs) {
    return ImageObserver.ERROR | ImageObserver.ABORT;
  }
  public void handleEvent(AWTEvent e) { }

  /* jdk 1.0 compatibility */
  public final boolean handleEvent(Event e) { return false; }
  public final void reshape(int x,int y,int w,int h) { setBounds(x,y,w,h); }
  public final void show() { setVisible(true); }
  public final void hide() { setVisible(false); }
  public final void enable() { setEnabled(true); }
  public final void disable() { setEnabled(false); }
  public final Dimension minimumSize() { return getMinimumSize(); }
  public final Dimension preferredSize() { return getPreferredSize(); }
}
