/*
	Source code is copyright 1997 by Viperware(tm)
	Permission to re-use code is granted
	
	Note: this code is not intended to compile- there are too many interdependancies 
		  to make this feasible.  However, this code is being used in real products
		  being sold by Viperware, and it should furthur illustrate the concepts
		  introduced in the "Tagged Data Storage Architecture" article.
 */

#ifndef __TK_PACKEROBJECT__
#include "TK_PackerObject.h"
#endif
#ifndef __FT_TAGMANAGER__
#include "FT_TagManager.h"
#endif
#ifndef __LB_TYPES__
#include "LB_Types.h"
#endif
#ifndef __LB_HANDLECHUNK__
#include "LB_ResourceChunk.h"
#endif
#ifndef __LB_CHUNKDATA__
#include "LB_ChunkData.h"
#endif
#ifndef __EX_EXTERMINATOR__
#include "EX_Exterminator.h"
#endif

TK_PackerObject::TK_PackerObject()
{
}

TK_PackerObject::~TK_PackerObject()
{
	Dispose(false, false);
}

SN_Error TK_PackerObject::Create()
{
	SN_Error			result;
	
	EX_TRY
	{
	}
	catch (SN_Exception& rException)
	{
		result = rException;
	}
	
	return result;
}

SN_Error TK_PackerObject::Dispose(Boolean destroy, Boolean stopOnError)
{
	SN_Error			result;
	
	EX_TRY
	{ 
		if (stopOnError)
		{
			EX_THROW_ERROR(FT_TaggableObject::Dispose(destroy, stopOnError));
		}
		else
		{
			FT_TaggableObject::Dispose(destroy, stopOnError);
		}	
	}
	catch (SN_Exception& rException)
	{
		result = rException;
	}
	
	return result;
}

SN_Error TK_PackerObject::KeepPackingData(LB_DataChunk& rDataChunk, long& rStuffOffset)
{
	TK_TagHeader			tagHeader;
	FT_TagIndex				tagCount;
	FT_Tag*					pTag;
	FT_TagDataSize 			headerSize = sizeof(TK_TagHeader);
	SN_Error				result;
	
	// KeepPackingData() stuffs the tag information at a specified offset into the
	// chunk of data.
	EX_TRY
	{
		EX_THROW_ERROR(CountTags(tagCount));
					
		for (FT_TagIndex tagIndex = 1; tagIndex <= tagCount; tagIndex++)
		{
			// Retrieve the current tag.
			EX_THROW_ERROR(RetrieveTagWithIndex(tagIndex, pTag));
			EX_THROW_NIL(pTag);
			
			// Stuff the tag information into the tag header.
			tagHeader.tkt_tagType = pTag->GetType();
			tagHeader.tkt_tagName = pTag->GetName();
			tagHeader.tkt_tagDataSize = pTag->GetDataSize();
			
			// Stuff the tag info into the data chunk.
			rDataChunk.StuffAndOffsetData((Ptr) &tagHeader, rStuffOffset, headerSize);
			rDataChunk.StuffAndOffsetData(FT_GetTagDataReference(pTag->GetData()), rStuffOffset, pTag->GetDataSize());
		}
	}
	catch (SN_Exception& error)
	{
		result = error;
		
		// The data chunk is no longer valid.
		rDataChunk.DestroyData();
	}
	
	return result;
}

SN_Error TK_PackerObject::PackData(LB_DataChunk& rDataChunk)
{
	SN_Error			result;
	
	EX_TRY
	{
		// Pack the object's tags into a chunk of data.
		EX_THROW_ERROR(PrepareForPackingData(rDataChunk));
		EX_THROW_ERROR(PackMyData(rDataChunk));
		EX_THROW_ERROR(FinishPackingData(rDataChunk));
	}
	catch (SN_Exception& rException)
	{
		result = rException;
		
		FailPackingData(rDataChunk);
	}
	
	return result;
}

SN_Error TK_PackerObject::PackMyData(LB_DataChunk& rDataChunk)
{
	LB_DataSize				totalSize, chunkSize;
	SN_Error				result;
	
	EX_TRY
	{
		EX_THROW_ERROR(CalculateTotalDataSizeWithHeader(totalSize, sizeof(TK_TagHeader)));
				
		// Resize the chunk of data to fit all of the packed tag data.
		chunkSize = totalSize;
		rDataChunk.SetDataSize(chunkSize);
	
		// Pack the object's tags into a chunk of data.
		long			stuffOffset = 0;
		EX_THROW_ERROR(KeepPackingData(rDataChunk, stuffOffset));
	}
	catch (SN_Exception& rException)
	{
		result = rException;
	}
	
	return result;
}
	
SN_Error TK_PackerObject::UnpackMyData(LB_DataChunk& rPackedData)
{
	TK_TagHeader*	pTagHeader;
	FT_TagData		tagData = nil;
	SN_Error		result;
	LB_DataSize		unpackOffset = 0;
	LB_DataSize		dataSize = rPackedData.GetDataSize();
	
	// Unpack the packed data to restore the object.
	// Add each tag as we come upon it in the data chunk.
	// For now, assume all data is packed tag data.
	EX_TRY
	{
		// Make sure that if the data is a handle, it doesn't get purged.
		LB_ChunkData			chunkData;
		EX_THROW_ERROR(rPackedData.RetrieveData(chunkData));
		EX_THROW_NIL(chunkData.GetData());
		
		// Remove all existing tags.
		EX_THROW_ERROR(RemoveAllTags());
		
		// This object has been stored with tags- extract them.
		while (unpackOffset < dataSize)
		{
			// Extract the current tag header.
			pTagHeader = (TK_TagHeader*) ((long) chunkData.GetData() + unpackOffset);
		
			tagData = nil;
			
			// Advance past the tag header.
			unpackOffset += sizeof(TK_TagHeader);
			
			if (pTagHeader->tkt_tagDataSize > 0)
			{
				// This tag has some data.  Allocate space in memory for the tag data.
				tagData = GetAllocatedTagData(pTagHeader->tkt_tagDataSize);
				
				EX_THROW_NIL(tagData);
				EX_THROW_NIL(FT_GetTagDataReference(tagData));
				{
					FT_UseTagData			useTagData(tagData);
					
					// Copy the tag data into the pointer.
					BlockMove((Ptr) ((long) pTagHeader + sizeof(TK_TagHeader)), FT_GetTagDataReference(tagData),
							  FT_GetTagDataSize(tagData));
					
					// Advance past the tag data.
					unpackOffset += pTagHeader->tkt_tagDataSize;
				}
			}
			
			// Attach each tag.
			EX_THROW_ERROR(AttachTag(pTagHeader->tkt_tagName, pTagHeader->tkt_tagType, tagData, FT_GetTagDataSize(tagData)));
		}
	}
	catch (SN_Exception& rException)
	{
		result = rException;
			
		if (tagData)
			// Release the unused tag data from memory.
			DisposeTagData(tagData);
	}
	
	return result;
}

Boolean TK_PackerObject::IsTaggedData(LB_DataChunk& rPackedData)
{
	Boolean			result = false;
	
	// For now, assume all data is tagged.
	result = true;
	
	return result;
}

SN_Error TK_PackerObject::LoadObjectFromResource(ResType resourceType, short resourceID)
{
	SN_Error			result;
	
	EX_TRY
	{
		LB_ResourceChunk			resourceChunk(resourceType, resourceID);
		
		// Remove all tags.
		EX_THROW_ERROR(RemoveAllTags());
		
		// Unpack tags from a chunk of data (the resource).
		EX_THROW_ERROR(UnpackData(resourceChunk));
	}
	catch (SN_Exception& rException)
	{
		result = rException;
	}
	
	return result;
}

SN_Error TK_PackerObject::PrepareForPackingData(LB_DataChunk& rDataChunk)
{
	SN_Error			result;
	
	// Do any necessary allocation, preparation for packing the
	// attached tags into a chunk of data.
	
	return result;
}

SN_Error TK_PackerObject::FinishPackingData(LB_DataChunk& rDataChunk)
{
	SN_Error			result;
	
	// Clean up after the process of packing the attached
	// tags into a chunk of data.
	
	return result;
}

void TK_PackerObject::FailPackingData(LB_DataChunk& rDataChunk)
{
	// This routine is called whenever the process of packing 
	// the attached tags into a chunk of data fails.
	FinishPackingData(rDataChunk);
}

SN_Error TK_PackerObject::UnpackData(LB_DataChunk& rPackedData)
{
	SN_Error			result;
	
	EX_TRY
	{
		// Unpack tags from a chunk of data.
		EX_THROW_ERROR(PrepareForUnpackingData(rPackedData));
		EX_THROW_ERROR(UnpackMyData(rPackedData));
		EX_THROW_ERROR(FinishUnpackingData(rPackedData));
	}
	catch (SN_Exception& rException)
	{
		result = rException;
		
		FailUnpackingData(rPackedData);
	}
	
	return result;
}

SN_Error TK_PackerObject::PrepareForUnpackingData(LB_DataChunk& rPackedData)
{
	SN_Error			result;
	
	// Do any necessary allocation, preparation before attempting
	// to unpack tags from a chunk of data.
	
	return result;
}

SN_Error TK_PackerObject::FinishUnpackingData(LB_DataChunk& rPackedData)
{
	SN_Error			result;
	
	// Clean up after we have unpacked the tags from the chunk of data.
	
	return result;
}

void TK_PackerObject::FailUnpackingData(LB_DataChunk& rPackedData)
{
	// This routine is called whenever the process of unpacking
	// the tags from a chunk of data fails.
	FinishUnpackingData(rPackedData);
}

//---------------------------------------------------------------
// TK_TagHeader
//---------------------------------------------------------------

TK_TagHeader::TK_TagHeader()
:	tkt_tagType('????'),
	tkt_tagName('????'),
	tkt_unused(0),
	tkt_tagDataSize(0)
{
}