///////////////////////////////////////////////////////////////////////////
// Copyright (c) Panos Kougiouris 1997
///////////////////////////////////////////////////////////////////////////

#include <kfThread.h>
#include <kfReferenceCount.h>
#include <assert.h>
#include <pthread.h>
#include "kfRunnable.h"

class CKFMTThreadImplementation : public CKFReferenceCount {
  public:
    pthread_t            m_thread;
    pthread_mutex_t     *m_mutex;
    pthread_cond_t      *m_cond;
    volatile CKFBool     m_can_run;
    CKFMT::CRunnable*     m_runnable;
    CKFMT::CThread*       m_thread_object;

    CKFMTThreadImplementation() {}

    static void start_thread(CKFMTThreadImplementation* a_th)
    {
        // Wait until they make us runnable
        pthread_mutex_lock(a_th->m_mutex);
	while (!a_th->m_can_run)
	    pthread_cond_wait(a_th->m_cond, a_th->m_mutex);
        pthread_mutex_unlock(a_th->m_mutex);

        // Since these are not used again delete them to 
        // free some recources
        pthread_mutex_destroy(a_th->m_mutex);
        delete a_th->m_mutex;
	a_th->m_mutex = NULL;
        pthread_cond_destroy(a_th->m_cond);
	delete a_th-> m_cond;
	a_th->m_cond = NULL;

	// Now run
        a_th->m_runnable->run();
	a_th->m_runnable->m_impl->completed();
	delete a_th->m_thread_object;
    }
    
    CKFMTThreadImplementation(CKFMT::CThread* a_th, CKFMT::CRunnable* a_run, int stack_size)
    {
        // Create the mutex and condition used to wait until we
        // become runnable
        m_can_run = CKFFalse;
        m_runnable = a_run;
        m_mutex = new pthread_mutex_t;
	pthread_mutex_init(m_mutex, NULL);
	m_cond  = new pthread_cond_t;
	pthread_cond_init(m_cond, NULL);
	m_thread_object = a_th;

        // Now create the thread
        pthread_attr_t attr;
        pthread_attr_init(&attr); /* initialize attr with default attributes */
        pthread_attr_setscope(&attr, PTHREAD_SCOPE_SYSTEM);  
                                                 /* system-wide contention */

        pthread_create(&m_thread,
                       &attr,
                      (void*(*)(void*))CKFMTThreadImplementation::start_thread, 
                      (void*) this); 
    }

  private:
    ~CKFMTThreadImplementation()
    {
      //if (pthread_self() != m_thread)) {
           ;// XXX PK;
      // }
    }
};

//--------------------------------------------------------

CKFMT::CThread::CThread(CKFMT::CRunnable* a_run, int stack_size)
{
    m_impl = new CKFMTThreadImplementation(this, a_run, stack_size);   
}

//--------------------------------------------------------

CKFMT::CThread::CThread(const CThread& a_m)
{
    m_impl = a_m.m_impl;
    m_impl->increment();
}

//--------------------------------------------------------

CKFMT::CThread& 
CKFMT::CThread::operator= (const CKFMT::CThread& a_m)
{ 
    m_impl->decrement();
    m_impl = a_m.m_impl;
    m_impl->increment();

    return *this;
}

//--------------------------------------------------------

CKFMT::CThread::~CThread()
{
    m_impl->decrement();
}

//--------------------------------------------------------

void
CKFMT::CThread::make_runnable()
{
    assert(m_impl->m_mutex != NULL);
    assert(m_impl->m_cond != NULL);

    pthread_mutex_lock(m_impl->m_mutex);
    m_impl->m_can_run = CKFTrue;
    pthread_cond_signal(m_impl->m_cond);
    pthread_mutex_unlock(m_impl->m_mutex);   
}

//--------------------------------------------------------

unsigned long
CKFMT::CThread::get_thread_id()
{
    return m_impl->m_thread;
}

//--------------------------------------------------------

void
CKFMT::CThread::exit(int code)
{
    assert(m_impl->m_thread != pthread_self());
    pthread_cancel(m_impl->m_thread);
    m_impl->m_runnable->decrement();
    delete this;
}

//--------------------------------------------------------

unsigned long
CKFMT::CThread::current_id()
{
    return pthread_self();
}

//--------------------------------------------------------

void
CKFMT::CThread::current_exit(int code)
{
    pthread_exit(&code);
}

