//
// decl.cpp - translate C++ declarations into English
//
// Copyright (C) 1996, 1997 by Dan Saks.
// May be copied for private, non-commercial use,
// provided this copyright notice remains intact.
// All other rights reserved.
//
// To build with Borland C++ 5 under Windows 95:
//
// bcc32 decl.cpp scanner.cpp
//
// To build with Microsoft C++ 5 under Windows 95:
//
// cl -GX decl.cpp scanner.cpp
//

#include <iostream>

#include "scanner.h"

using namespace std;

class parser
	{
public:
	parser(istream &, ostream &);
private:
	enum declarator_category { ABSTRACT, CONCRETE, EITHER };
	enum type_category
		{ SIMPLE, SIZED_ARRAY, UNSIZED_ARRAY, FUNCTION, POINTER, REFERENCE };
	struct recoverable_error { };
	scanner input;
	ostream &output;
	string indent;

	void error(const string &);
	void must_be(token::category);
	string array_suffix(type_category &);
	string cv_qualifier_seq();
	string declarator
		(declarator_category, type_category &);
	string direct_declarator
		(declarator_category, type_category &);
	string function_suffix();
	string parameter_declaration();
	string parameter_list();
	string ptr_operator(type_category &);
	string simple_declaration();
	string type_specifier_seq();

	parser(parser const &);
	parser &operator=(parser const &);
	};

void parser::error(string const &why)
	{
	output << "error: " << why << '\n';
	input.reset();
	indent = "";
	throw recoverable_error();
	}

void parser::must_be(token::category tc)
	{
	if (input.current().kind() == tc)
		input.get();
	else
		error(string("\'") + image(tc) + "' expected");
	}

//
// array-suffix =
//     "[" [ constant-name | integer-literal ] "]" .
//
string parser::array_suffix(type_category &outer)
	{
	must_be(token::LEFT_BRACKET);
	string as = "array with ";
	token::category tc = input.current().kind();
	if (tc == token::NAME || tc == token::INT_LITERAL)
		{
		outer = SIZED_ARRAY;
		as += input.current().text() + ' ';
		input.get();
		}
	else
		{
		outer = UNSIZED_ARRAY;
		as += "unspecified number of ";
		}
	as += "elements of type...\n";
	must_be(token::RIGHT_BRACKET);
	return indent + as;
	}

//
// cv-qualifier-seq =
//     { "const" | "volatile" } .
//
string parser::cv_qualifier_seq()
	{
	bool cq = false;
	bool vq = false;
	token::category tc;
	for (;;)
		{
		tc = input.current().kind();
		if (tc == token::CONST)
			{
			if (cq)
				error("redundant 'const' qualifier");
			else
				cq = true;
			}
		else if (tc == token::VOLATILE)
			{
			if (vq)
				error("redundant 'volatile' qualifier");
			else
				vq = true;
			}
		else
			break;
		input.get();
		}
	string t;
	if (cq)
		t += "const ";
	if (vq)
		t += "volatile ";
	return t;
	}

//
// declarator =
//     direct-declarator | ptr-operator declarator .
//
string parser::declarator
	(declarator_category dc, type_category &outer)
	{
	token::category tc = input.current().kind();
	if (tc == token::AMPERSAND || tc == token::STAR
	|| tc == token::NAME)
		{
		type_category ptc;
		string p = ptr_operator(ptc);
		type_category out;
		string d = declarator(dc, out);
		if (ptc == REFERENCE)
			{
			if (out == SIZED_ARRAY || out == UNSIZED_ARRAY)
				error("can't have array of reference");
			if (out == POINTER)
				error("can't have pointer to reference");
			if (out == REFERENCE)
				error("can't have reference to reference");
			}
		outer = ptc;
		return d + p;
		}
	else
		return direct_declarator(dc, outer);
	}

//
// direct-declarator =
//     [ declarator-id | "(" declarator ")" ]
//         { array-suffix | function-suffix } .
//
string parser::direct_declarator
	(declarator_category dc, type_category &outer)
	{
	string dd;
	token::category tc = input.current().kind();
	if (tc == token::IDENTIFIER)
		{
		if (dc == ABSTRACT)
			error("can't have declarator-id in "
				"abstract-declarator");
		dd = indent + input.current().text() + " is ...\n";
		input.get();
		outer = SIMPLE;
		}
	else if (tc == token::LEFT_PAREN)
		{
		bool its_grouping = false;
		input.mark();
		tc = input.get().kind();
		if (tc == token::IDENTIFIER
		|| tc == token::AMPERSAND || tc == token::STAR
		|| tc == token::LEFT_BRACKET || tc == token::LEFT_PAREN
		|| (tc == token::NAME && input.get().kind() == token::SCOPE))
			its_grouping = true;
		input.backup();
		if (its_grouping)
			{
			input.get();
			dd = declarator(dc, outer);
			must_be(token::RIGHT_PAREN);
			}
		}
	if (dd == "")
		{
		if (dc == CONCRETE)
			error("declarator-id missing or obscured"
				" by something before it");
		dd = indent + "<type> is ...\n";
		outer = SIMPLE;
		}
	for (;;)
		{
		tc = input.current().kind();
		if (tc == token::LEFT_BRACKET)
			{
			if (outer == FUNCTION)
				error("can't have function returning array");
			type_category tc;
			dd += array_suffix(tc);
			if (tc == UNSIZED_ARRAY)
				if (outer == SIZED_ARRAY || outer == UNSIZED_ARRAY)
					error("only 1st dimension can be unspecified");
			outer = tc;
			}
		else if (tc == token::LEFT_PAREN)
			{
			if (outer == SIZED_ARRAY || outer == UNSIZED_ARRAY)
				error("can't have array of function");
			if (outer == FUNCTION)
				error("can't have function returning function");
			dd += function_suffix();
			outer = FUNCTION;
			}
		else
			break;
		}
	return dd;
	}

//
// function-suffix =
//     "(" [ parameter-list ] ")" cv-qualifier-seq .
//
string parser::function_suffix()
	{
	must_be(token::LEFT_PAREN);
	string fs = "function with ";
	token::category tc = input.current().kind();
	if (tc == token::NAME || tc == token::TYPE_KEYWORD
	|| tc == token::CONST || tc == token::VOLATILE)
		fs += "parameter(s) ...\n" + parameter_list();
	else
		fs += "no parameters ...\n";
	must_be(token::RIGHT_PAREN);
	fs += indent + "returning ...\n";
	string cvs = cv_qualifier_seq();
	if (cvs != "")
		fs = cvs + "member " + fs;
	return indent + fs;
	}

//
// parameter-declaration =
//     type-specifier-seq declarator .
//
string parser::parameter_declaration()
	{
	string dss = indent + type_specifier_seq();
	type_category outer;
	string pd = declarator(EITHER, outer) + dss + '\n';
	return pd;
	}

//
// parameter-list =
//     parameter-declaration { "," parameter-declaration } .
//
string parser::parameter_list()
	{
	string const tab_stop = "    ";
	indent += tab_stop;
	string pl = parameter_declaration();
	while (input.current().kind() == token::COMMA)
		{
		input.get();
		pl += parameter_declaration();
		}
	indent = string(indent, 0, indent.length() - tab_stop.length());
	return pl;
	}

//
// ptr-operator =
//    "&" | [ type-name "::" ] "*" cv-qualifier-seq .
//
string parser::ptr_operator(type_category &cat)
	{
	token t = input.current();
	if (t.kind() == token::AMPERSAND)
		{
		input.get();
		cat = REFERENCE;
		return indent + "reference to ...\n";
		}
	else
		{
		string p = "pointer to ";
		if (t.kind() == token::NAME)
			{
			p += "member of " + t.text()
				+ " with type ";
			input.get();
			must_be(token::SCOPE);
			}
		must_be(token::STAR);
		cat = POINTER;
		return indent + cv_qualifier_seq() + p + "...\n";
		}
	}

//
// simple-declaration =
//     type-specifier-seq declarator { "," declarator } .
//
string parser::simple_declaration()
	{
	type_category outer;
	string d = type_specifier_seq();
	string sd = declarator(CONCRETE, outer) + d + '\n';
	while (input.current().kind() == token::COMMA)
		{
		input.get();
		sd += declarator(CONCRETE, outer) + d + '\n';
		}
	return sd;
	}

//
// type-specifier-seq =
//     {
//     "const" | "volatile" | type-keyword | type-name
//     } .
//
string parser::type_specifier_seq()
	{
	bool cq = false;
	bool vq = false;
	string tn;
	token::category tc;
	for (;;)
		{
		tc = input.current().kind();
		if (tc == token::NAME)
			{
			input.mark();
			tc = input.get().kind();
			input.backup();
			if (tc == token::SCOPE)
				break;
			tc = input.current().kind();
			}
		if (tc == token::CONST)
			{
			if (!cq)
				cq = true;
			else
				error("redundant 'const' qualifier");
			}
		else if (tc == token::VOLATILE)
			{
			if (!vq)
				vq = true;
			else
				error("redundant 'volatile' qualifier");
			}
		else if (tc == token::TYPE_KEYWORD
		|| tc == token::NAME)
			{
			if (tn == "")
				tn = input.current().text();
			else
				break;
			}
		else
			break;
		input.get();
		}
	if (tn == "")
		{
		if (!(cq | vq))
			error("type specifier missing");
		tn = "int";
		}
	string t;
	if (cq)
		t += "const ";
	if (vq)
		t += "volatile ";
	return t + tn;
	}

//
// parser =
//     { simple-declaration ";" } .
//
parser::parser(istream &is, ostream &os)
:	input(is), output(os)
	{
	for (;;)
		try
			{
			while (input.get().kind() != token::NO_MORE)
				{
				string s = simple_declaration();
				if (input.current().kind() != token::SEMICOLON)
					error("';' expected");
				else
					output << s;
				}
			break;
			}
		catch (recoverable_error const &)
			{
			}
	}

int main()
	{
	parser declarations(cin, cout);
	return 0;
	}
